/*
 * blancoIg
 * Copyright (C) 2004-2005 Yasuo Nakanishi
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */
package blanco.ig.generator;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import blanco.ig.io.SourceFinder;
import blanco.ig.io.SourceWriter;
import blanco.ig.tool.ConsolePrinter;

/**
 * @author Yasuo Nakanishi
 */
public class RuntimeGenerator {
    private GeneratorSetting _setting = null;
    private List _runtimeResorces = new ArrayList();
    private SourceWriter _writer = null;
    private SourceFinder _finder = null;
    private ConsolePrinter _printer = ConsolePrinter.get();
    private PackageTransformer _transformer = null;

    public RuntimeGenerator(GeneratorSetting setting) {
        _setting = setting;
        _finder = new SourceFinder();
        _transformer = new PackageTransformer();
    }

    public void addSourceDirectory(String path) {
        _finder.addSourceDirectory(path);
    }

    public void addPackagePair(String originalPackage, String userPackage) {
        _transformer.addPackagePair(originalPackage, userPackage);
    }

    public void generate() throws IOException {

        Iterator i = _runtimeResorces.iterator();
        while (i.hasNext()) {
            generateRuntime((Class) i.next());
        }
    }

    private void generateRuntime(Class runtimeClass) throws IOException {
        String outputPath = _setting.getMainOutputDirectory();
        String sourcePath = _finder.getSourcePath(runtimeClass);
        String resourcePath = _finder.getResourcePath(runtimeClass);
        String targetPath = _transformer
                .getTargetPath(outputPath, resourcePath);
        BufferedReader br = null;
        List code = new ArrayList();

        _writer = new SourceWriter(_setting.getSoruceEncoding());
        String line = "";

        String assertionRegex = "^[\\s]*?assert\\s*.*";
        String headerRegex = "^[(package)|(import)].*";

        boolean isFileHeader = true;
        try {
            br = new BufferedReader(
                    new InputStreamReader(getStream(sourcePath)));

            while (true) {
                line = br.readLine();
                if(line == null) {
                    break;
                }
                assert line != null;

                if(isFileHeader) {
                    if(line.matches(headerRegex)) {
                        isFileHeader = false;
                        code.add("/*");
                        code
                                .add(" * This code is generated by blanco Framework.");
                        code.add(" */");
                    }
                }

                if(!isFileHeader && line.matches(assertionRegex)) {
                    code.add("");
                } else if(!isFileHeader) {
                    code.add(_transformer.replacePackage(line));
                }
            }
            _writer.write(targetPath, code);

            String name = runtimeClass.getName();
            if(_writer.isCreated()) {
                _printer.printInfo("NX:" + name);
            } else if(_writer.isUpdated()) {
                _printer.printInfo("XVNX:" + name);
            }
        } finally {
            if(br != null) {
                br.close();
            }
        }
    }

    private InputStream getStream(String targetPath)
            throws FileNotFoundException {
        InputStream result = null;
        File f = new File(targetPath);
        if(f.exists()) {
            result = new FileInputStream(f);
        } else {
            result = getClass().getClassLoader().getResourceAsStream(targetPath);
        }
        return result;
    }

    public void addRuntime(Class runtimeClass) {
        _runtimeResorces.add(runtimeClass);
    }
}
