/*
 * blancoReport
 * Copyright (C) 2005 Yasuo Nakanishi
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */
package blanco.api.excel.oo;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import blanco.api.excel.CellParser;
import blanco.api.excel.CellPointer;
import blanco.api.excel.CellRangeParser;
import blanco.api.excel.CellType;
import blanco.api.excel.ExcelBook;
import blanco.api.excel.ExcelSheet;

import com.sun.star.beans.PropertyVetoException;
import com.sun.star.beans.UnknownPropertyException;
import com.sun.star.beans.XPropertySet;
import com.sun.star.lang.IllegalArgumentException;
import com.sun.star.lang.IndexOutOfBoundsException;
import com.sun.star.lang.WrappedTargetException;
import com.sun.star.sheet.CellDeleteMode;
import com.sun.star.sheet.CellInsertMode;
import com.sun.star.sheet.XCellRangeMovement;
import com.sun.star.sheet.XSpreadsheet;
import com.sun.star.table.CellAddress;
import com.sun.star.table.CellContentType;
import com.sun.star.table.CellRangeAddress;
import com.sun.star.table.XCell;
import com.sun.star.table.XTableRows;
import com.sun.star.text.XText;

/**
 * @author Yasuo Nakanishi
 */
public class OoExcelSheet implements ExcelSheet {

    private CalcService _service = null;

    private ExcelBook _book = null;

    private XSpreadsheet _sheet = null;

    private static DateFormat _format = new SimpleDateFormat(
            "yyyy/MM/dd HH:mm:ss");

    public OoExcelSheet(CalcService service, ExcelBook book, XSpreadsheet sheet) {
        _service = service;
        _book = book;
        _sheet = sheet;
    }

    public void setText(int column, int row, String value) {
        XCell cell = null;
        XText text = null;
        try {
            cell = _sheet.getCellByPosition(column, row);
            text = _service.getText(cell);
            text.setString(value);
        } catch (IndexOutOfBoundsException e) {
            String message = "sȃ|WVn܂B";
            message += "Column:" + column;
            message += ", Row:" + row;
            throw new RuntimeException(message, e);
        }
    }

    public void setText(CellPointer pointer, String value) {
        setText(pointer.getColumn(), pointer.getRow(), value);
    }

    public void copyRange(String sourceRange, String destinationCell) {
        CellRangeAddress sourceAddress = _service.getCellRangeAddress(_sheet,
                sourceRange);
        CellAddress destinationAddress = _service.getCellAddress(_sheet,
                destinationCell);

        XCellRangeMovement movement = _service.getCellRangeMovement(_sheet);

        CellRangeParser parser = new CellRangeParser(sourceRange);
        String insertRange = destinationCell + ":" + destinationCell;
        CellRangeAddress insertAddress = _service.getCellRangeAddress(_sheet,
                insertRange);
        for (int i = 0; i < parser.getRowCount(); i++) {
            movement.insertCells(insertAddress, CellInsertMode.ROWS);
        }

        movement.copyRange(destinationAddress, sourceAddress);
        //		XCellRangeFormula formula =
        // _service.getCellRangeFormula(sourceAddress);

        List heights = getRowHeights(sourceRange);
        setRowHeights(destinationCell, heights);
        copyCellPropertiesOfRange(sourceRange, destinationCell);
    }

    private void copyCellPropertiesOfRange(String sourceRange,
            String destinationCell) {
        CellRangeParser parser = new CellRangeParser(sourceRange);
        CellPointer source = null;
        CellPointer distination = null;

        for (int row = 0; row <= parser.getRowCount(); row++) {
            for (int column = 0; column <= parser.getColumnCount(); column++) {
                source = parser.getStartPointer().move(row, column);
                distination = new CellPointer(destinationCell)
                        .move(row, column);
                copyCellProperties(source, distination);
            }
        }
    }

    private void copyCellProperties(CellPointer sourceCell,
            CellPointer destinationCell) {
        XPropertySet source = _service.getCellPropertySet(getCell(sourceCell));
        XPropertySet destination = _service
                .getCellPropertySet(getCell(destinationCell));

        String value = "";
        try {
            value = "NumberFormat";
            destination.setPropertyValue(value, source.getPropertyValue(value));
        } catch (UnknownPropertyException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } catch (PropertyVetoException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } catch (IllegalArgumentException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } catch (WrappedTargetException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }

    private void setRowHeights(String cell, List heights) {
        Iterator iterator = heights.iterator();
        Integer height = null;
        int rowNumber = CellParser.parseRowNumber(cell);
        XPropertySet propertySet = null;

        XTableRows rows = _service.getColumnRowRange(_sheet).getRows();
        Object row = null;

        try {
            while (iterator.hasNext()) {
                height = (Integer) iterator.next();
                row = rows.getByIndex(rowNumber);
                propertySet = _service.getCellPropertySet(row);
                propertySet.setPropertyValue("Height", height);
                rowNumber++;
            }
        } catch (IndexOutOfBoundsException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } catch (WrappedTargetException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } catch (UnknownPropertyException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } catch (PropertyVetoException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } catch (IllegalArgumentException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }

    private List getRowHeights(String range) {
        List result = new ArrayList();
        XTableRows rows = _service.getColumnRowRange(_sheet).getRows();
        CellRangeParser parser = new CellRangeParser(range);
        int startRow = parser.getStartPointer().getRow();

        Object row = null;
        XPropertySet propertySet = null;
        try {
            for (int i = 0; i < parser.getRowCount(); i++) {
                row = rows.getByIndex(startRow + i);
                propertySet = _service.getCellPropertySet(row);
                result.add(propertySet.getPropertyValue("Height"));
            }
        } catch (IndexOutOfBoundsException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } catch (WrappedTargetException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } catch (NumberFormatException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        } catch (UnknownPropertyException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
        return result;
    }

    public int getRowCount() {
        return _service.getColumnRowRange(_sheet).getRows().getCount();
    }

    public int getColumnCount() {
        return _service.getColumnRowRange(_sheet).getColumns().getCount();
    }

    public String getText(int column, int row) {
        return _service.getText(getCell(column, row)).getString();
    }

    public String getValue(String address) {
        return getText(CellParser.parse(address));
    }

    private XCell getCell(CellPointer pointer) {
        return getCell(pointer.getColumn(), pointer.getRow());
    }

    private XCell getCell(int column, int row) {
        try {
            return _sheet.getCellByPosition(column, row);
        } catch (IndexOutOfBoundsException e) {
            String message = "sCellAddressn܂B";
            message += "Column:" + column;
            message += ", Row:" + row;
            throw new RuntimeException(message, e);
        }
    }

    public void removeRange(String range) {
        CellRangeAddress rangeAddress = _service.getCellRangeAddress(_sheet,
                range);

        XCellRangeMovement movement = _service.getCellRangeMovement(_sheet);
        movement.removeRange(rangeAddress, CellDeleteMode.ROWS);
    }

    public String getText(CellPointer address) {
        return getText(address.getColumn(), address.getRow());
    }

    public ExcelBook getBook() {
        return _book;
    }

    public String getName() {
        assert _sheet != null;
        assert _service.getNamed(_sheet) != null;
        return _service.getNamed(_sheet).getName();
    }

    public double getNumber(int column, int row) {
        return getCell(column, row).getValue();
    }

    public double getNumber(CellPointer pointer) {
        return getNumber(pointer.getColumn(), pointer.getRow());
    }

    public void setNumber(int column, int row, double value) {
        setFormulaLocal(column, row, Double.toString(value));
    }

    public void setNumber(CellPointer pointer, double value) {
        setNumber(pointer.getColumn(), pointer.getRow(), value);
    }

    public CellType getCellType(int column, int row) {
        CellType result = null;
        CellContentType type = getCell(column, row).getType();
        if (type.equals(CellContentType.TEXT)) {
            result = CellType.TEXT;
        } else if (type.equals(CellContentType.VALUE)) {
            result = CellType.NUMBER;
        } else {
            result = CellType.OTHER;
        }
        return result;
    }

    public CellType getCellType(CellPointer pointer) {
        return getCellType(pointer.getColumn(), pointer.getRow());
    }

    public void setDate(int column, int row, Date value) {
        setFormulaLocal(column, row, _format.format(value));
    }

    private void setFormulaLocal(int column, int row, String value) {
        XCell cell = getCell(column, row);
        XPropertySet propertySet = _service.getCellPropertySet(cell);
        try {
            propertySet.setPropertyValue("FormulaLocal", value);
        } catch (Exception e) {
            String message = "FormulaLocalvpeB̐ݒɎs܂B";
            throw new RuntimeException(message);
        }
    }

    public void setDate(CellPointer pointer, Date value) {
        setDate(pointer.getColumn(), pointer.getRow(), value);
    }
}
