/*
 * blancoReport
 * Copyright (C) 2005 Yasuo Nakanishi
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */
package blanco.report.collector;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import blanco.api.excel.ExcelBook;
import blanco.api.excel.ExcelBookCreator;
import blanco.api.excel.ExcelSheet;
import blanco.api.excel.oo.CalcService;
import blanco.api.excel.oo.OoExcelBookCreator;
import blanco.api.oo.launchar.OOoLauncher;
import blanco.ig.tool.ConsolePrinter;
import blanco.report.definition.FormSet;
import blanco.report.definition.FormSetSpecification;
import blanco.report.definition.FormSpecification;
import blanco.report.definition.ReportServiceDefinition;
import blanco.report.generator.BlancoReportSetting;

/**
 * @author Yasuo Nakanishi
 * @author Tosiki Iga svȃtB[hϐ
 */
public class BlancoReportCollector {

    private List _formSetPathList = new ArrayList();

    private List _specificationPathList = new ArrayList();

    private ReportServiceDefinition _definition = null;

    private BlancoReportSetting _setting = new BlancoReportSetting();

    private OOoLauncher _launcher = null;

    private ConsolePrinter _printer = ConsolePrinter.get();

    private CalcService _service = null;

    private boolean _safeMode = false;

    public BlancoReportCollector(BlancoReportSetting setting) {
        _setting = setting;
        _definition = new ReportServiceDefinition();

        String executeFile = _setting.getOpenOfficeExecuteFile();
        int port = _setting.getUnoPort();
        boolean hide = _setting.isHideOpenOffce();

        _launcher = new OOoLauncher(executeFile, port, hide);
        _service = new CalcService("localhost", port, hide);
    }

    public void addFormSet(String formSetPath, String specificationPath) {
        _formSetPathList.add(formSetPath);
        _specificationPathList.add(specificationPath);
    }

    public ReportServiceDefinition collect() {

        _printer.printInfo("^̎WJn܂B");

        try {
            if (!_safeMode) {
                startOpenOffice();
            }
            collectAll();
        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            if (!_safeMode) {
                stopOpenOffice();
            }
        }

        _printer.printInfo("^̎WI܂B");

        return _definition;
    }

    private void startOpenOffice() throws Exception {
        _launcher.start();
        _service.connect();
    }

    private void stopOpenOffice() {
        try {
            if (_service.isAvailable()) {
                _service.close();
            }
        } finally {
            if (_launcher.isAlive()) {
                _launcher.stop();
            }
        }
    }

    private void collectAll() {
        String reportPath = "";
        String specificationPath = "";
        FormSet formSet = null;

        for (int i = 0; i < _formSetPathList.size(); i++) {
            reportPath = (String) _formSetPathList.get(i);
            specificationPath = (String) _specificationPathList.get(i);

            _printer.printInfo("tH[Zbg[" + reportPath + "]̃^WB");

            try {
                formSet = collectFormSet(reportPath, specificationPath);
                _definition.addFormSet(formSet);
            } catch (Exception e) {
                String message = "|[g̍쐬𒆎~܂BtH[ZbgpX:" + reportPath;
                _printer.printWarning(message);
                e.printStackTrace();
            }
        }
    }

    private FormSet collectFormSet(String formSetPath, String specificationPath)
            throws Exception {
        FormSet result = null;

        try {
            if (_safeMode) {
                startOpenOffice();
            }

            ExcelBookCreator creator = new OoExcelBookCreator(_service);

            String path = "";
            File file = null;

            ExcelBook formSetBook = null;
            path = getFormSetPath(formSetPath);
            file = new File(path);
            checkFileCanRead(file, "tH[Zbgt@C");
            formSetBook = creator.openBook(path);

            ExcelBook specificationBook = null;
            path = getFormSetSpecificationPath(specificationPath);
            file = new File(path);
            checkFileCanRead(file, "tH[Zbg`t@C");
            specificationBook = creator.openBook(path);

            result = collectFormsSpecificatoin(formSetBook, formSetPath,
                    specificationBook);

        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            if (_safeMode) {
                stopOpenOffice();
            }
        }
        return result;
    }

    private FormSet collectFormsSpecificatoin(ExcelBook formSetBook,
            String formSetPath, ExcelBook specificationBook) throws Exception {
        FormSet result = null;

        validateSpecificationBook(specificationBook);
        result = new FormSet(formSetPath,
                collectFormSetSpecificatoin(specificationBook));

        try {
            String name = "";
            ExcelSheet sheet = null;
            int sheetCount = specificationBook.getSheetCount();
            for (int i = 0; i < sheetCount; i++) {
                sheet = specificationBook.getSheet((short) i);
                name = sheet.getName();
                try {
                    if (isFormSpecificationSheet(name)) {
                        result.addForm(collectFormSpecification(formSetBook,
                                specificationBook, sheet));
                    }
                } catch (Exception e) {
                    String message = "tH[̍쐬𒆎~܂BtH[:" + name;
                    _printer.printWarning(message);
                    e.printStackTrace();
                }
            }
        } finally {
            if (specificationBook != null) {
                specificationBook.close();
                specificationBook = null;
            }
        }
        return result;
    }

    private void validateSpecificationBook(ExcelBook specificationBook) {
        String sheetName = getFormSetSpecificationSheetName();
        if (!specificationBook.hasSheet(sheetName)) {
            String message = "tH[Zbg`t@CɃtH[Zbg`V[g[";
            message += sheetName;
            message += "]܂łB";
            throw new RuntimeException(message);
        }
    }

    private boolean isFormSpecificationSheet(String name) {
        boolean result = false;
        if (!name.equals(getFormSetSpecificationSheetName())
                && !name.equals("blancoReportConfiguration")) {
            result = true;
        }
        return result;
    }

    private void checkFileCanRead(File file, String name) {
        if (!file.exists()) {
            String message = name + "܂łBt@C:"
                    + file.getAbsolutePath();
            throw new RuntimeException(message);
        } else if (!file.canRead()) {
            String message = name + "ǂݍ݂ł܂Bt@C:"
                    + file.getAbsolutePath();
            throw new RuntimeException(message);
        }
    }

    private String getFormSetPath(String path) {
        return _setting.getFormSetDirectory() + "/" + path;
    }

    private String getFormSetSpecificationPath(String path) {
        return _setting.getFormSetSpecificatoinDirctory() + "/" + path;
    }

    private FormSpecification collectFormSpecification(ExcelBook formSetBook,
            ExcelBook specificationBook, ExcelSheet formSpecificatoinSeet)
            throws Exception {
        String nameSpace = _setting.getRootNameSpace();
        FormSpecificationCollector collector = new FormSpecificationCollector(
                formSetBook, specificationBook, formSpecificatoinSeet,
                nameSpace);
        collector.parse();
        collector.validate();
        return collector.collect();
    }

    private FormSetSpecification collectFormSetSpecificatoin(
            ExcelBook specificationBook) throws Exception {
        ExcelSheet sheet = specificationBook
                .getSheet(getFormSetSpecificationSheetName());
        String nameSpace = _setting.getRootNameSpace();
        FormSetSpecificationCollector collector = new FormSetSpecificationCollector(
                specificationBook, sheet, nameSpace);
        collector.parse();
        collector.validate();
        return collector.collect();
    }

    private String getFormSetSpecificationSheetName() {
        return "FormSet";
    }

    protected void finalize() throws Throwable {
        super.finalize();

        if (_launcher != null) {
            System.out
                    .println("BlancoReportCollector#stopOpenOffice()\bhĂ΂܂łBKstop()\bhĂԂ悤ɂĂB");
        }
    }
}
