/*
 * blancoReport
 * Copyright (C) 2005 Yasuo Nakanishi
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */
package blanco.report.runtime;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.apache.log4j.Logger;

import blanco.api.excel.CellPointer;
import blanco.api.excel.CellType;
import blanco.api.excel.ExcelBook;
import blanco.api.excel.ExcelBookCreator;
import blanco.api.excel.ExcelSheet;
import blanco.api.excel.oo.CalcService;
import blanco.api.excel.oo.OoExcelBook;
import blanco.api.excel.oo.OoExcelBookCreator;
import blanco.api.oo.launchar.OOoLauncher;
import blanco.report.definition.OutputType;

/**
 * @author Yasuo Nakanishi
 */
public abstract class AbstractReportProcessor {

    private String _formSetPath = "";

    private OutputType _outputType = null;

    private boolean _hasBlockCount = false;

    private int _blockCount = 0;

    private String _outputPath = "";

    private String _currentFormName = "";

    private ExcelBook _currentBook = null;

    private FormSheet _currentForm = null;

    private TargetSheet _targetSheet = null;

    private List _formSheetNames = new ArrayList();

    private Map _formSheetMap = new Hashtable();

    private OOoLauncher _launcher = null;

    private Map _formIndexMap = new Hashtable();

    private Logger _logger = Logger.getLogger(getClass());

    private List _bindedValues = new ArrayList();

    private static DateFormat _format = new SimpleDateFormat(
            "yyyy/MM/dd HH:mm:ss");

    private CalcService _service = null;

    public AbstractReportProcessor(String formSetPath, OutputType outputType) {
        _formSetPath = formSetPath;
        _outputType = outputType;
        initialize();
    }

    public AbstractReportProcessor(String formSetPath, OutputType outputType,
            int blockCount) {
        _formSetPath = formSetPath;
        _outputType = outputType;
        _hasBlockCount = true;
        _blockCount = blockCount;

        initialize();
    }

    private void initialize() {
        String executeFile = getOpenOfficeExecuteFile();
        int port = getUnoPort();
        _launcher = new OOoLauncher(executeFile, port);
    }

    private String getFormPath() {
        return getFormSetDirectory() + "/" + _formSetPath;
    }

    public void newReport(String path) throws IOException {
        _outputPath = getOutputDirectory() + "/" + path;
        copyReportForm();

        String host = "localhost";
        int port = getUnoPort();

        _launcher.start();

        _service = new CalcService(host, port);
        ExcelBookCreator creator = null;
        try {
            _service.connect();
            creator = new OoExcelBookCreator(_service);

            _currentBook = creator.openBook(_outputPath);
            int count = _currentBook.getSheetCount();
            for (short i = 0; i < count; i++) {
                _formSheetNames.add(_currentBook.getSheet(i).getName());
            }
        } catch (Exception e) {
            String message = "t@C̃I[vɎs܂Bt@C:" + _outputPath;
            IOException ioe = new IOException(message);
            ioe.initCause(e);
            throw ioe;
        }

        _logger.info("t@C:" + _outputPath + "֏o͊JnB");
    }

    private void copyReportForm() throws IOException {
        createDirectory(getOutputDiractory());
        copy(getFormPath(), getOutputPath());
    }

    protected void newPage(String formName) {
        _hasBlockCount = false;
        _blockCount = 0;
        createNewPage(formName);
    }

    protected void newPage(String formName, int blockCount) {
        _hasBlockCount = true;
        _blockCount = blockCount;
        createNewPage(formName);
    }

    public void checkReportCreated() {
        if (_currentBook == null) {
            String message = "|[gVy[W쐬łԂł͂܂BnewReport()\bhĂяoĂ邩mFĂB";
            throw new IllegalStateException(message);
        }
    }

    protected void checkCurrentForm(String formName) {
        if (!_currentFormName.equals(formName)) {
            String message = "";

            if (_currentFormName.equals("")) {
                message = "y[W쐬OɃy[Wf[^n܂BO" + _currentFormName
                        + "쐬邽߂̃\bhĂяoĂB";
            } else {
                message = "݂̃y[W̃f[^Ƃ͈قȂ^̃f[^n܂BO" + _currentFormName
                        + "쐬邽߂̃\bhĂяoĂB݂̃tH[:" + _currentFormName;
            }
            throw new IllegalStateException(message);
        }
    }

    protected void createNewPage(String formName) {
        checkReportCreated();

        if (_targetSheet != null) {
            _targetSheet.clean();
        }

        _bindedValues.clear();
        _currentFormName = formName;
        String destinationName = formName + "_" + getFormIndex(formName);
        _currentBook.copySheet(formName, destinationName);
        _currentForm = getForm(formName);
        ExcelSheet target = _currentBook.getSheet(destinationName);
        _targetSheet = new TargetSheet(_currentForm, target);

        _logger.debug("V[gRs[B" + formName + " -> " + destinationName);
    }

    protected void collectPageParameters(String endAddress) {
        _currentForm.collectParameters(endAddress);
    }

    protected void collectLoopParameters(String loopName, String range) {
        _currentForm.collectLineParameters(loopName, range);
    }

    private FormSheet getForm(String formName) {
        if (!_formSheetMap.containsKey(formName)) {
            FormSheet sheet = new FormSheet(_currentBook.getSheet(formName));
            _formSheetMap.put(formName, sheet);
        }
        return (FormSheet) _formSheetMap.get(formName);
    }

    private int getFormIndex(String formName) {
        int result = 0;
        if (!_formIndexMap.containsKey(formName)) {
            _formIndexMap.put(formName, new Integer(0));
        }
        result = ((Integer) _formIndexMap.get(formName)).intValue();
        result++;
        _formIndexMap.put(formName, new Integer(result));
        return result;
    }

    private void removeForms() {
        Iterator i = _formSheetNames.iterator();
        String name = "";
        while (i.hasNext()) {
            name = (String) i.next();
            _currentBook.removeSheet(name);
        }
    }

    public void saveReport() throws IOException {

        removeForms();
        _targetSheet.clean();
        _targetSheet = null;

        try {
            _currentBook.save();
            if (_outputType.equals(OutputType.XLS_PDF)) {
                saveToPdf();
            }
            _currentBook.close();
            _currentBook = null;
        } finally {
            clear();
        }
        _logger.info("o͏IB");
    }

    private void clear() {
        if (_targetSheet != null) {
            _targetSheet = null;
            _currentFormName = "";
        }

        if (_currentBook != null) {
            _currentBook.close();
            _currentBook = null;
        }

        _formIndexMap.clear();
        _formSheetMap.clear();
        _formSheetNames.clear();

        if (_service != null) {
            _service.close();
            _service = null;
        }

        _outputPath = "";

        if (_launcher.isAlive()) {
            _launcher.stop();
        }
    }

    public void cancel() {
        new File(getOutputPath()).delete();
        clear();
    }

    public void close() {
        clear();
    }

    private String getOutputPath() {
        assert !_outputPath.equals("");
        return _outputPath;
    }

    private String getOutputDiractory() {
        File f = new File(_outputPath);
        assert f.isFile();
        return f.getParentFile().getAbsolutePath();
    }

    private void createDirectory(String path) {
        File f = new File(path);
        assert f.isDirectory();
        if (!f.exists()) {
            if (f.mkdirs()) {
                System.out.println("fBNg쐬܂BpX:" + path);
            } else {
                throw new RuntimeException("fBNg쐬Ɏs܂BpX:" + path);
            }
        }
    }

    protected void setValue(String parameter, CellType type, String value) {
        Iterator i = _currentForm.getParameterIterator(parameter);

        if (value == null || (!type.equals(CellType.TEXT) && value.equals(""))) {
            setValue(parameter, CellType.TEXT, "");
            return;
        }

        String text = "";
        if (value != null) {
            text = value;
        }

        CellPointer pointer = null;
        while (i.hasNext()) {
            pointer = (CellPointer) i.next();
            _targetSheet.setText(pointer, value);
        }
        _bindedValues.add(new BindedValue(parameter, type, value));
    }

    protected void setValue(String parameter, CellType type, BigDecimal value) {
        Iterator i = _currentForm.getParameterIterator(parameter);

        if (value == null || value.equals("")) {
            setValue(parameter, CellType.TEXT, "");
            return;
        }

        CellPointer pointer = null;
        while (i.hasNext()) {
            pointer = (CellPointer) i.next();
            _targetSheet.setValue(pointer, value.doubleValue());
        }

        _bindedValues.add(new BindedValue(parameter, type, value));
    }

    protected void setValue(String parameter, CellType type, Date value) {
        Iterator i = _currentForm.getParameterIterator(parameter);

        if (value == null || value.equals("")) {
            setValue(parameter, CellType.TEXT, "");
            return;
        }

        CellPointer pointer = null;
        while (i.hasNext()) {
            pointer = (CellPointer) i.next();
            _targetSheet.setDate(pointer, value);
        }

        _bindedValues.add(new BindedValue(parameter, type, value));
    }

    protected void setBlockValue(String blockName, String parameter,
            CellType type, String value) {
        Iterator i = _currentForm
                .getLineParameterIterator(blockName, parameter);

        if (value == null || (!type.equals(CellType.TEXT) && value.equals(""))) {
            setBlockValue(blockName, parameter, CellType.TEXT, "");
            return;
        }

        String text = "";
        if (value != null) {
            text = value;
        }

        CellPointer pointer = null;
        while (i.hasNext()) {
            pointer = (CellPointer) i.next();
            _targetSheet.setBlockText(blockName, pointer, text);
        }
    }

    protected void setBlockValue(String blockName, String parameter,
            CellType type, BigDecimal value) {
        Iterator i = _currentForm
                .getLineParameterIterator(blockName, parameter);

        if (value == null || value.equals("")) {
            setBlockValue(blockName, parameter, CellType.TEXT, "");
            return;
        }

        CellPointer pointer = null;
        while (i.hasNext()) {
            pointer = (CellPointer) i.next();
            _targetSheet
                    .setBlockNumber(blockName, pointer, value.doubleValue());
        }
    }

    protected void setBlockValue(String blockName, String parameter,
            CellType type, Date value) {
        Iterator i = _currentForm
                .getLineParameterIterator(blockName, parameter);

        if (value == null || value.equals("")) {
            setBlockValue(blockName, parameter, CellType.TEXT, "");
            return;
        }

        CellPointer pointer = null;
        while (i.hasNext()) {
            pointer = (CellPointer) i.next();
            _targetSheet.setBlockDate(blockName, pointer, value);
        }
    }

    private void saveAs() throws IOException {
        OoExcelBook book = (OoExcelBook) _currentBook;

        book.saveAs(getPdfFileName());
    }

    private void saveToPdf() throws IOException {
        OoExcelBook book = (OoExcelBook) _currentBook;

        // saveAsExcelubNɂȂĂ܂܂BPDFɕۑȂĂ͂Ȃ܂B
        // book.saveAs(getPdfFileName());
        // TODO Asł͂ȂToŗǂ̂ǂ𐸍ׂ
        book.saveToPdf(getPdfFileName());
    }

    private String getPdfFileName() {
        return _outputPath.replaceAll("(.XLS)|(.xls)$", ".pdf");
    }

    protected void finalize() throws Throwable {
        super.finalize();

        if (_launcher != null) {
            System.out
                    .println("AbstractReportProcessor#stop()\bhĂ΂܂łBKstop()\bhĂԂ悤ɂĂB");
        }

        cancel();
    }

    public void newLine(String loopName) {

        if (_hasBlockCount) {
            if (_targetSheet.getBlockCount() == _blockCount) {
                breakPage();
            }
        }
        _targetSheet.newLine(loopName);
    }

    private void breakPage() {
        List list = new ArrayList();
        Iterator i = _bindedValues.iterator();
        while (i.hasNext()) {
            list.add(i.next());
        }
        newPage(_currentFormName, _blockCount);
        i = list.iterator();
        BindedValue bindedValue = null;
        Object value = null;
        while (i.hasNext()) {
            bindedValue = (BindedValue) i.next();
            value = bindedValue.getValue();
            if (value == null || value instanceof String) {
                setValue(bindedValue.getName(), bindedValue.getType(),
                        (String) value);
            } else if (value instanceof BigDecimal) {
                setValue(bindedValue.getName(), bindedValue.getType(),
                        (BigDecimal) value);
            } else if (value instanceof Date) {
                setValue(bindedValue.getName(), bindedValue.getType(),
                        (Date) value);
            }
        }
    }

    public static final void copy(String source, String destination)
            throws IOException {
        BufferedInputStream bis = null;
        BufferedOutputStream bos = null;

        try {
            bis = new BufferedInputStream(new FileInputStream(source));
            bos = new BufferedOutputStream(new FileOutputStream(destination));

            byte[] buffer = new byte[4096];
            int readSize = 0;

            readSize = bis.read(buffer, 0, buffer.length);
            while (readSize != -1) {
                bos.write(buffer, 0, readSize);
                readSize = bis.read(buffer, 0, buffer.length);
            }
            bos.flush();
        } finally {
            try {
                if (bis != null) {
                    bis.close();
                }
            } finally {
                if (bos != null) {
                    bos.close();
                }
            }
        }
    }

    private String getOpenOfficeExecuteFile() {
        ResourceBundle _bundle = ResourceBundle.getBundle("blanco-report");
        try {
            return _bundle.getString("blanco.report.setting.open.office.exec");
        } catch (MissingResourceException e) {
            System.out.println("x: blanco-report \[Xoh̓ǂݍ݂Ɏs܂.:"
                    + e.toString());
        }
        return "";
    }

    private int getUnoPort() {
        ResourceBundle _bundle = ResourceBundle.getBundle("blanco-report");
        try {
            String strPort = _bundle
                    .getString("blanco.report.setting.uno.port");
            if (strPort != null) {
                return Integer.parseInt(strPort);
            }
        } catch (NumberFormatException e) {
            System.out.println("x: blanco-report \[XoĥȂ̃|[gԍsK؂ł.:"
                    + e.toString());
        } catch (MissingResourceException e) {
            System.out.println("x: blanco-report \[Xoh̓ǂݍ݂Ɏs܂.:"
                    + e.toString());
        }
        return 8100;
    }

    private String getFormSetDirectory() {
        ResourceBundle _bundle = ResourceBundle.getBundle("blanco-report");
        try {
            String strResult = _bundle
                    .getString("blanco.report.setting.formset.dir");
            if (strResult != null) {
                return strResult;
            }
        } catch (MissingResourceException e) {
            System.out.println("x: blanco-report \[Xoh̓ǂݍ݂Ɏs܂.:"
                    + e.toString());
        }
        return "./report/form";
    }

    private String getOutputDirectory() {
        ResourceBundle _bundle = ResourceBundle.getBundle("blanco-report");
        try {
            String strResult = _bundle
                    .getString("blanco.report.setting.output.dir");
            if (strResult != null) {
                return strResult;
            }
        } catch (MissingResourceException e) {
            System.out.println("x: blanco-report \[Xoh̓ǂݍ݂Ɏs܂.:"
                    + e.toString());
        }
        return "./report/out";
    }
}
