/*
 * blancoIg
 * Copyright (C) 2004-2005 Yasuo Nakanishi
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */
package blanco.ig.expander;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import blanco.ig.expander.field.FieldExpander;
import blanco.ig.expander.implementor.ImplementData;
import blanco.ig.expander.javadoc.ClassJavaDoc;
import blanco.ig.expander.method.MethodExpander;

/**
 * @author Yasuo Nakanishi
 */
public abstract class ClassExpander implements Expander {
    private Type _type = null;
	private ImportList _importList = new ImportList();
    private ImplementData _staticImplement = new ImplementData();
	private List _fieldList = new ArrayList();
	private Map _filedMap = new Hashtable();
	private List _methodList = new ArrayList();
	private List _code = null;
	private boolean _existSuperClass = false;
	private Type _superClass = null;
	private ClassJavaDoc _javaDoc = new ClassJavaDoc();
	private static NameAdjuster _nameAdjuster = new NameAdjuster();
	private List _fileComments = new ArrayList();
    private boolean _final = false;

    public ClassExpander(Type type){
	    _type = _nameAdjuster.adjust(type);
		
		createFileComment();
	}
	
	protected static NameAdjuster getNameAdjuster() {
	    return _nameAdjuster;
	}
	
	public void createFileComment() {
		addFileComment("This code is generated by blanco Framework.");
	}

	public void addFileComment(String comment) {
	   _fileComments.add(comment);
	}

	public ClassJavaDoc  getJavaDoc(){
		return _javaDoc;
	}

	public void addImport(Type type){
		_importList.add(type);
	}
	
	public void addField(FieldExpander field) {
		_importList.add(field.getTypeValue().getType());
        _importList.add(field.getUsingTypeList());
		_fieldList.add(field);
		_filedMap.put(field.getTypeValue().getName(), field);
	}

	public void setSuperClass(Type type){
		_existSuperClass = true;
		addImport(type);
		_superClass = type;
	}
	
	protected boolean existSuperClass(){
		return _existSuperClass;
	}

    protected FieldExpander getField(int index) {
        return (FieldExpander) _fieldList.get(index);
    }

    protected FieldExpander getField(String name) {
        return (FieldExpander) _filedMap.get(name);
    }

    public Value getFieldValue(String fieldName) {
        Value result = null;
        Iterator i = _fieldList.iterator();
        FieldExpander field = null;
        while(i.hasNext()) {
            field = (FieldExpander) i.next();
            if(field.getTypeValue().getName().equals(fieldName)) {
                result = field.getFieldValue();
                break;
            }
        }
        assert result != null : "tB[hȂBtB[h:" + fieldName;
        return result;
    }

	protected int getFieldCount() {
		return _fieldList.size();
	}

	public void addMethod(MethodExpander method) {
		_methodList.add(method);
        method.setupSignature();
		method.setClassSource(this);
	}

	protected MethodExpander getMethod(int index) {
		return (MethodExpander) _methodList.get(index);
	}

	protected int getMethodCount() {
		return _methodList.size();
	}

	private int getMethodSeparatorLineCount(){
		int result = 0;
		int count = getMethodCount();
		
		if(count != 0){
			result = count - 1;
		}
		return result;
	}

	private int getFieldSeparatorLineCount(){
		int result = 0;
		int count = getFieldCount();
		
		if(count != 0){
			result = count - 1;
		}
		return result;
	}

    protected void implementStaticBlock() {
    }

    protected abstract void expandClassStruct();

	private void implementAll(){
        implementStaticBlock();
        _importList.add(_staticImplement.getUsingTypeList());

        MethodExpander m = null;
		for(int i = 0; i < getMethodCount(); i++){
		    m = getMethod(i);
			m.implement();
			_importList.add(m.getUsingTypeList());
		}
	}

	public final List expand(){

		expandClassStruct();
		implementAll();

		_code = new ArrayList();
		
		if(_fileComments.size() != 0) {
			_code.add("/*");
		    for(int i = 0; i < _fileComments.size(); i++) {
		        _code.add(" * " + _fileComments.get(i));
		    }
			_code.add(" */");
		}

		if(!_type.getNameSpace().equals("")){
			_code.add("package " + getNameSpace() + ";");
			_code.add("");
		}

		_code.addAll(_importList.expand());
		_code.add("");

		if(!_javaDoc.isEmpty()){
			List list = _javaDoc.expand();
			for(int i = 0; i < list.size(); i++){
				_code.add(list.get(i));
			}
		}
		_code.add(getClassDeclaration());

        expandAllFields();
		_code.add("");

        expandStaticBlock();
        _code.add("");

        expandAllMethod();
		_code.add("}");
		return _code;
	}
    
    private void expandStaticBlock() {
        if(!_staticImplement.isEmpty()) {
            _code.add("\tstatic {");
            List codes = _staticImplement.getImplementList();
            for(int i = 0; i < codes.size(); i++) {
                _code.add(codes.get(i));
            }
            _code.add("\t}");
            
        }
    }
    
    private void expandAllMethod() {
        for(int i = 0; i < getMethodCount(); i++){
            if(i != 0){
                _code.add("");
            }
            makeMethodCode(getMethod(i));
        }

    }
    
    private void expandAllFields() {
        for(int i = 0; i < getFieldCount(); i++){
            if(i != 0){
                _code.add("");
            }
            makeFieldCode(getField(i));
        }
    }
	
	private String getClassDeclaration(){
		assert !_type.getName().equals("");
		StringBuffer sb = new StringBuffer();
		sb.append("public");
        if(isFinal()) {
            sb.append(" final");
        }
        if(isAbstract()) {
            sb.append(" abstract");
        }
        sb.append(" class ");
		sb.append(_type.getName());
		if(existSuperClass()){
			sb.append(" extends ");
			sb.append(_superClass.getName());
		}
		sb.append(" {");
		return new String(sb);
	}
    
    protected boolean isAbstract() {
        return false;
    }

	private void makeMethodCode(MethodExpander method){
		List methodCode = method.expand();
		for(int i = 0; i < methodCode.size(); i++){
			_code.add(methodCode.get(i));
		}
	}

	private void makeFieldCode(FieldExpander field){
		List fieldCode = field.expand();
		
		for(int i = 0; i < fieldCode.size(); i++){
			_code.add(fieldCode.get(i));
		}
	}

	public String getName() {
		return _type.getName();
	}

	public String getNameSpace() {
		return _type.getNameSpace();
	}
    
	public Type getType() {
	    return _type;
	}
    protected ImplementData getStaticImplementData() {
        return _staticImplement;
    }

    public void clear() {
        _fieldList.clear();
        
        MethodExpander method = null;
        Iterator i = _methodList.iterator();
        while(i.hasNext()) {
            method = (MethodExpander) i.next();
            method.clear();
        }
        _methodList.clear();
        _fileComments.clear();
        _javaDoc = null;
        _staticImplement = null;
        _code.clear();
        _importList.clear();
    }

    public boolean isFinal() {
        return _final;
    }

    public void setFinal(boolean final1) {
        _final = final1;
    }

}
