/*******************************************************************************
 * blanco Framework
 * Copyright (C) 2012 Toshiki IGA
 * 
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *******************************************************************************/
/*******************************************************************************
 * Copyright (c) 2012 Toshiki IGA and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *      Toshiki IGA - initial API and implementation
 *******************************************************************************/
/*******************************************************************************
 * Copyright 2012 Toshiki IGA and others.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *******************************************************************************/
package blanco.excelapi;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Date;

import org.apache.poi.xssf.usermodel.XSSFCell;
import org.apache.poi.xssf.usermodel.XSSFCellStyle;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;

/**
 * Apache POI を利用して Excel ブックを書き出すためのユーティリティ。
 * 
 * @author Toshiki Iga
 */
class BlancoExcelBookWriterPoiXlsxImpl implements BlancoExcelBookWriter {
	/**
	 * 現在処理している書き込み可能なワークブック
	 */
	private XSSFWorkbook workbook = null;

	/**
	 * 現在処理中の書き込み可能なシート
	 */
	private XSSFSheet currentSheet = null;

	/**
	 * 出力先 Excel ブックのストリーム。
	 */
	private OutputStream outStream = null;

	@Override
	public void open(final OutputStream outStream,
			final InputStream inStreamTemplate) throws IOException {
		this.outStream = outStream;

		// 雛形となる Excel ブックを読み込みます。
		workbook = new XSSFWorkbook(inStreamTemplate);
		if (workbook == null) {
			throw new IOException("Fail to create Excel book.");
		}
	}

	@Override
	public void close() throws IOException {
		if (workbook == null) {
			return;
		}

		try {
			// ここでワークブックを書き出します。

			workbook.write(outStream);
			outStream.flush();
			// シートの記憶を破棄します。
			currentSheet = null;
		} finally {
			outStream.close();
			// ワークブックの記憶を破棄します。
			workbook = null;
			outStream = null;
		}
	}

	@Override
	public void selectSheet(final int sheetNo) throws IOException {
		currentSheet = workbook.getSheetAt(sheetNo);

		if (currentSheet == null) {
			throw new IOException("Specified sheet number [" + sheetNo
					+ "] is not exist.");
		}
	}

	@Override
	public void selectSheet(final String sheetName) throws IOException {
		currentSheet = workbook.getSheet(sheetName);

		if (currentSheet == null) {
			throw new IOException("Specified sheet [" + sheetName
					+ "] is not exist.");
		}
	}

	@Override
	public void setSheetName(final String sheetName) throws IOException {
		ensureSheetSelection();

		final String origSheetName = currentSheet.getSheetName();
		int sheetIndex = 0;
		for (int index = 0; index < workbook.getNumberOfSheets(); index++) {
			if (origSheetName.equals(workbook.getSheetName(index))) {
				sheetIndex = index;
				break;
			}
		}

		workbook.setSheetName(sheetIndex, sheetName);
	}

	@Override
	public String getText(final int column, final int row) throws IOException {
		ensureSheetSelection();

		final XSSFCell cell = getCell(column, row);
		if (cell == null) {
			return null;
		}

		return BlancoExcelBookReaderPoiXlsxImpl.getTextInner(cell);
	}

	@Override
	public void setText(final int column, final int row, final String value)
			throws IOException {
		setText(column, row, value, column, row);
	}

	@Override
	public void setText(final int column, final int row, final String value,
			final int templateCellColumn, final int templateCellRow)
			throws IOException {
		ensureSheetSelection();

		final XSSFCell templateCell = getCell(templateCellColumn,
				templateCellRow);
		XSSFCellStyle templateCellStyle = null;
		if (templateCell != null) {
			templateCellStyle = templateCell.getCellStyle();
		}

		XSSFRow newRow = currentSheet.getRow(row);
		if (newRow == null) {
			newRow = currentSheet.createRow(row);
		}

		final XSSFCell cell = newRow.createCell(column);
		if (templateCellStyle != null) {
			cell.setCellStyle(templateCellStyle);
		}
		cell.setCellType(XSSFCell.CELL_TYPE_STRING);
		cell.setCellValue(value);
	}

	@Override
	public void setNumber(final int column, final int row, final double value)
			throws IOException {
		setNumber(column, row, value, column, row);
	}

	@Override
	public void setNumber(final int column, final int row, final double value,
			final int templateCellColumn, final int templateCellRow)
			throws IOException {
		ensureSheetSelection();

		final XSSFCell templateCell = getCell(templateCellColumn,
				templateCellRow);

		XSSFRow newRow = currentSheet.getRow(row);
		if (newRow == null) {
			newRow = currentSheet.createRow(row);
		}

		final XSSFCell cell = newRow.createCell(column, row);
		cell.setCellStyle(templateCell.getCellStyle());
		cell.setCellType(XSSFCell.CELL_TYPE_NUMERIC);
		cell.setCellValue(value);
	}

	@Override
	public void setDateTime(final int column, final int row, final Date value)
			throws IOException {
		setDateTime(column, row, value, column, row);
	}

	@Override
	public void setDateTime(final int column, final int row, final Date value,
			final int templateCellColumn, final int templateCellRow)
			throws IOException {
		ensureSheetSelection();

		final XSSFCell templateCell = getCell(templateCellColumn,
				templateCellRow);

		XSSFRow newRow = currentSheet.getRow(row);
		if (newRow == null) {
			newRow = currentSheet.createRow(row);
		}

		final XSSFCell cell = newRow.createCell(column, row);
		cell.setCellStyle(templateCell.getCellStyle());
		cell.setCellType(XSSFCell.CELL_TYPE_NUMERIC);
		cell.setCellValue(value);
	}

	/**
	 * Apache POI のワークブック・オブジェクトを直接取り出して利用するための API です。
	 * 
	 * @deprecated 基本的にこれは利用しないでください。
	 * @return 書き込み可能なワークブック・オブジェクト。
	 */
	public XSSFWorkbook getWorkbook() {
		return workbook;
	}

	/**
	 * Apache POI のシート・オブジェクトを直接取り出して利用するための API です。
	 * 
	 * @deprecated 基本的にこれは利用しないでください。
	 * @return 書き込み可能なシート・オブジェクト。
	 */
	public XSSFSheet getSheet() {
		return currentSheet;
	}

	/**
	 * セル・オブジェクトを取得します。
	 * 
	 * @param column
	 * @param row
	 * @return
	 * @deprecated 基本的にこれは外部から直接は利用しないでください。
	 */
	public XSSFCell getCell(final int column, final int row) {
		try {
			final XSSFRow lookup = currentSheet.getRow(row);
			if (lookup == null) {
				return null;
			}

			return lookup.getCell(column);
		} catch (ArrayIndexOutOfBoundsException ex) {
			return null;
		}
	}

	/**
	 * シートが選択されていることを確認します。
	 * 
	 * @throws IOException
	 */
	private void ensureSheetSelection() throws IOException {
		if (workbook == null) {
			throw new IOException("Workbook not selected.");
		}

		if (currentSheet == null) {
			throw new IOException("Sheet not selected.");
		}
	}
}
