/*******************************************************************************
 * blanco Framework
 * Copyright (C) 2011 Toshiki IGA
 * 
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *******************************************************************************/
/*******************************************************************************
 * Copyright (c) 2011 Toshiki IGA and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *      Toshiki IGA - initial API and implementation
 *******************************************************************************/
package blanco.eclipseast2cg.util;

import java.util.List;

import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jdt.core.dom.AST;
import org.eclipse.jdt.core.dom.ASTParser;
import org.eclipse.jdt.core.dom.ASTVisitor;
import org.eclipse.jdt.core.dom.ArrayType;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.FieldDeclaration;
import org.eclipse.jdt.core.dom.ParameterizedType;
import org.eclipse.jdt.core.dom.PrimitiveType;
import org.eclipse.jdt.core.dom.SimpleType;

/**
 * 名前関連のユーティリティ。
 * 
 * @author Toshiki IGA
 */
public class BlancoEclipseASTNameUtil {
    /**
     * Java プログラミング言語の予約語一覧。
     */
    public static final String[] JAVA_LANGUAGE_RESERVED_KEYWORD = { "void", "byte", "short", "int", "long", "char",
            "float", "double", "boolean" };

    /**
     * 与えられたローカル名を FQCN に変換します。
     * 
     * @param localClassName
     * @param currentPackage
     * @param importList
     * @return FQCN 化したクラス名。
     */
    public static String toFullyQualifiedClassName(final String localClassName, final String currentPackage,
            final List<String> importList) {
        // System.out.println("TRACE: IN : " + localClassName);

        final StringBuilder fqcnName = new StringBuilder();
        toFullyQualifiedClassName(localClassName, currentPackage, importList, fqcnName);

        // System.out.println("TRACE: OUT: " + fqcnName.toString());
        return fqcnName.toString();
    }

    /**
     * 与えられたローカル名を FQCN に変換するための内部処理。
     * 
     * @param localClassName
     * @param currentPackage
     * @param importList
     * @param resultFqcnBuilder
     */
    private static void toFullyQualifiedClassName(final String localClassName, final String currentPackage,
            final List<String> importList, final StringBuilder resultFqcnBuilder) {
        final String sourceCode = "public class ToFullyQualifiedClassName {public " + localClassName + " dummyField;}";

        final ASTParser parser = ASTParser.newParser(AST.JLS3);
        parser.setSource(sourceCode.toCharArray());
        final CompilationUnit unit = (CompilationUnit) parser.createAST(new NullProgressMonitor());

        try {
            unit.accept(new ASTVisitor() {
                @Override
                public boolean visit(final FieldDeclaration node) {
                    if (node.getType() instanceof PrimitiveType) {
                        final PrimitiveType type = (PrimitiveType) node.getType();
                        // FQCN への加工は不要。
                        resultFqcnBuilder.append(type.toString());
                    } else if (node.getType() instanceof SimpleType) {
                        final SimpleType type = (SimpleType) node.getType();
                        // FQCN に加工
                        resultFqcnBuilder.append(toFullyQualifiedClassNameWithoutArrayParameter(type.getName()
                                .toString(), currentPackage, importList));
                    } else if (node.getType() instanceof ParameterizedType) {
                        final ParameterizedType t = (ParameterizedType) node.getType();
                        // FQCN に加工
                        resultFqcnBuilder.append(toFullyQualifiedClassNameWithoutArrayParameter(t.getType().toString(),
                                currentPackage, importList));
                        resultFqcnBuilder.append("<");
                        boolean isFirst = true;
                        for (Object obj : t.typeArguments()) {
                            if (isFirst) {
                                isFirst = false;
                            } else {
                                resultFqcnBuilder.append(", ");
                            }
                            // 再帰呼出
                            toFullyQualifiedClassName(obj.toString(), currentPackage, importList, resultFqcnBuilder);
                        }
                        resultFqcnBuilder.append(">");
                    } else if (node.getType() instanceof ArrayType) {
                        final ArrayType t = (ArrayType) node.getType();
                        // TODO ここで FQCN に加工
                        resultFqcnBuilder.append(toFullyQualifiedClassNameWithoutArrayParameter(t.getElementType()
                                .toString(), currentPackage, importList));
                        for (int index = 0; index < t.getDimensions(); index++)
                            resultFqcnBuilder.append("[]");
                    } else {
                        System.out.println("サポート外: " + node.getType().getClass());
                    }

                    return super.visit(node);
                }
            });

        } catch (Exception ex) {
            System.out.println("BlancoEclipseASTNameUtil#toFullyQualifiedClassName(" + localClassName
                    + "): Unexpected exception occured." + ex.toString());
            ex.printStackTrace();
        }
    }

    /**
     * 与えられたローカル名を FQCN に変換するための内部処理のうち、配列やパラメータの無いもの。
     * 
     * @param localClassName
     * @param currentPackage
     * @param importList
     * @param resultFqcnBuilder
     */
    private static String toFullyQualifiedClassNameWithoutArrayParameter(final String localName,
            final String currentPackage, final List<String> importList) {
        // Java 言語のキーワードだったら、何もしません。
        for (String keyword : JAVA_LANGUAGE_RESERVED_KEYWORD) {
            if (keyword.equals(localName))
                return localName;
        }

        try {
            Class.forName(localName);
            // 実はすでに存在するクラス?
            return localName;
        } catch (ClassNotFoundException exIgnore) {
        }

        try {
            Class.forName("java.lang." + localName);
            // java.lang 以下のパッケージ名省略に該当します。
            return "java.lang." + localName;
        } catch (ClassNotFoundException exIgnore) {
        }

        // import 文のリストに該当があれば、それをそのまま利用します。
        for (String importValue : importList) {
            if (importValue.endsWith("." + localName)) {
                return importValue;
            }
        }

        // FQCN なのであれば、加工の必要はありません。
        // ポイント: ただし、これだと Entry.Set のような場合の考慮がありません。
        if (localName.indexOf('.') > 0)
            return localName;

        // いずれにも該当しなければ、現在のパッケージに所属するクラスなのでしょう。
        return currentPackage + "." + localName;
    }
}
