package blanco.filemanager.core;

import java.io.File;
import java.io.IOException;
import java.util.Date;

import jp.sf.amateras.stepcounter.CountResult;
import jp.sf.amateras.stepcounter.StepCounter;
import jp.sf.amateras.stepcounter.StepCounterFactory;
import blanco.filemanager.file.BlancoFileManagerFileAttrInfo;
import blanco.filemanager.file.BlancoFileManagerFileInfo;

public abstract class BlancoFileManagerFileParser {
    /**
     * 発生したファイル情報を処理するためのコールバック。
     * 
     * @param fileInfo
     */
    public abstract void fireFileInfo(final BlancoFileManagerFileInfo fileInfo);

    public void parseDir(final File root, final File dir) {
        final File[] files = dir.listFiles();
        if (files == null) {
            return;
        }

        // 最初にカレントディレクトリのファイルを処理します。
        for (File file : files) {
            if (file.isDirectory()) {
                // ディレクトリの処理は後回しです。
            } else if (file.isFile()) {
                parseFile(root, file);
            }
        }

        // 次にサブディレクトリを処理します。ｌ
        for (File file : files) {
            if (file.isDirectory()) {
                // 絶対に関係のない ソース管理システム類のもののみ除外とします。
                if (".svn".equals(file.getName())) {
                    // SVN 管理フォルダはスキップします。
                    continue;
                } else if ("CVS".equals(file.getName())) {
                    // CVS 管理フォルダはスキップします。
                    continue;
                }

                parseDir(root, file);
            } else if (file.isFile()) {
                // ファイルの処理はすでに実施済みです。
            }
        }
    }

    public void parseFile(final File root, final File file) {
        // System.out.println("name: " + file.getName());
        if (file.canRead() == false) {
            return;
        }

        final BlancoFileManagerFileInfo fileInfo = new BlancoFileManagerFileInfo();
        fileInfo.setFileName(file.getName());

        final BlancoFileManagerFileAttrInfo attrInfo = new BlancoFileManagerFileAttrInfo();
        fileInfo.getAttrInfoList().add(attrInfo);

        // file size
        attrInfo.setFileSize(file.length());

        // last modified. ただし、今回はこの値は利用しない見込みです。
        final Date lastModified = new Date(file.lastModified());
        // System.out.println("  update: " + new
        // SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(lastModified));

        try {
            fileInfo.setFilePath(file
                    .getCanonicalFile()
                    .getParent()
                    .substring(
                            Math.min(file.getCanonicalFile().getParent().length(), root.getCanonicalPath().length() + 1)));
            // System.out.println("  dir : " + fileInfo.getFilePath());

            // MD5
            attrInfo.setFileMd5(BlancoFileManagerFileDigestUtil.getMD5(file));

            // SHA1
            attrInfo.setFileSha1(BlancoFileManagerFileDigestUtil.getSHA1(file));

            attrInfo.setFileSha256(BlancoFileManagerFileDigestUtil.getSHA256(file));

            // StepCounter
            {
                final StepCounter counter = StepCounterFactory.getCounter(file.getName());
                if (counter != null) {
                    // カウンターが取得できた場合にのみ処理します。
                    // FIXME: 文字コードですが、さしあたり UTF-8 想定としています。
                    final CountResult cntResult = counter.count(file, "UTF-8");

                    fileInfo.setFileType(cntResult.getFileType());

                    // 注意: long を int に丸めています。int
                    // を超えるステップ数のファイル、というのは、想像しがたいです、、、。
                    attrInfo.setValidLineCount((int) cntResult.getStep());
                    attrInfo.setCommentLineCount((int) cntResult.getComment());
                    attrInfo.setBlankLineCount((int) cntResult.getNon());
                }
            }

            // コールバックを呼び出します。
            fireFileInfo(fileInfo);
        } catch (IOException ex) {
            ex.printStackTrace();
        }
    }
}
