package blanco.filemanager.db.dao;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import blanco.filemanager.db.query.Sbfm0101Iterator;
import blanco.filemanager.db.query.Sbfm0102Iterator;
import blanco.filemanager.db.query.Sbfm0103Iterator;
import blanco.filemanager.db.row.Sbfm0101Row;
import blanco.filemanager.db.row.Sbfm0102Row;
import blanco.filemanager.db.row.Sbfm0103Row;
import blanco.filemanager.db.runtime.exception.NoRowFoundException;
import blanco.filemanager.dbaccessor.query.BfmFileInfoI01Invoker;
import blanco.filemanager.dbaccessor.query.BfmFileInfoS01Iterator;
import blanco.filemanager.dbaccessor.query.BfmFileInfoU01Invoker;
import blanco.filemanager.dbaccessor.row.BfmFileInfoS01Row;
import blanco.filemanager.file.BlancoFileManagerFileInfo;
import blanco.fw.BlancoInject;

public abstract class AbstractBlancoFileManagerFileInfoDao {
    /**
     * 指定された条件のファイルがすでに登録されたかどうかを調べます。
     * 
     * @param conn
     * @param iteSearch
     * @param fileInfo
     * @return あればファイルID。なければ -1。
     * @throws SQLException
     */
    public int getFileIdIfExists(final Connection conn, @BlancoInject final Sbfm0102Iterator iteSearch,
            final BlancoFileManagerFileInfo fileInfo) throws SQLException {
        try {
            iteSearch.setInputParameter(fileInfo.getProject(), fileInfo.getFileName(), fileInfo.getFilePath());
            final Sbfm0102Row rowSearch = iteSearch.getSingleRow();
            return rowSearch.getFileId();
        } catch (NoRowFoundException ex) {
            // はじめてのファイル登録です。
            return -1;
        }
    }

    /**
     * 次のファイルIDを採番します。
     * 
     * @param conn
     * @param iteMax
     * @return
     * @throws SQLException
     */
    public int getNextFileId(final Connection conn, @BlancoInject final Sbfm0101Iterator iteMax) throws SQLException {
        int nextFileId = 0;
        final Sbfm0101Row rowMax = iteMax.getSingleRow();
        if (rowMax.getMaxFileId() == null) {
            nextFileId = 1;
        } else {
            nextFileId = rowMax.getMaxFileId().intValue() + 1;
        }

        return nextFileId;
    }

    /**
     * 新たなデータをINESRTします。子の情報はINSERTしません。別途実施してください。
     * 
     * @param conn
     * @param fileInfo
     * @param invInfo
     * @param now
     * @throws SQLException
     */
    public void insertFileInfo(final Connection conn, final BlancoFileManagerFileInfo fileInfo,
            @BlancoInject final BfmFileInfoI01Invoker invInfo, final Date now) throws SQLException {
        invInfo.setInputParameter(fileInfo.getFileId(), fileInfo.getProject(), fileInfo.getFileName(),
                fileInfo.getFileStatFlg(), fileInfo.getFileChangeFlg(), fileInfo.getFileType(), fileInfo.getFilePath(),
                now, "system");
        invInfo.executeSingleUpdate();
    }

    /**
     * ファイル取り込み時の状況変化に関する最低限の更新(ファイルの変更フラグのみ)を行います。
     * 
     * @param conn
     * @param fileInfo
     * @param invInfo
     * @param now
     * @throws SQLException
     */
    public void updateFileInfo(final Connection conn, final BlancoFileManagerFileInfo fileInfo,
            @BlancoInject final BfmFileInfoU01Invoker invInfo, final Date now) throws SQLException {
        invInfo.setInputParameter(fileInfo.getFileChangeFlg(), now, "system",
                fileInfo.getFileId());
        invInfo.executeSingleUpdate();
    }

    /**
     * プロジェクトにひもづくすべてのファイル情報をリスト取得します。(このテーブルのみの浅い検索結果取得です)
     * 
     * @param conn
     * @param iteList
     * @param project
     * @return
     * @throws SQLException
     */
    public List<BlancoFileManagerFileInfo> getListByProject(final Connection conn,
            @BlancoInject final Sbfm0103Iterator iteList, final String project) throws SQLException {
        iteList.setInputParameter(project);

        final List<BlancoFileManagerFileInfo> infoList = new ArrayList<BlancoFileManagerFileInfo>();
        for (Sbfm0103Row row : iteList.getList(100000)) {
            final BlancoFileManagerFileInfo info = new BlancoFileManagerFileInfo();
            infoList.add(info);
            info.setFileId(row.getFileId());
            info.setProject(project);
            info.setFileName(row.getFileName());
            info.setFileStatFlg(row.getFileStatFlg());
            info.setFileChangeFlg(row.getFileChangeFlg());
            info.setFileType(row.getFileType());
            info.setFilePath(row.getFilePath());
            info.setUpdDt(row.getUpdDt());
            info.setUpdUserId(row.getUpdUserId());
        }

        return infoList;
    }

    /**
     * ファイルIDにより 1件取得します。
     * 
     * @param conn
     * @param ite
     * @param fileId
     * @return
     * @throws SQLException
     */
    public BlancoFileManagerFileInfo getFileInfo(final Connection conn, @BlancoInject final BfmFileInfoS01Iterator ite,
            final int fileId) throws SQLException {
        ite.setInputParameter(fileId);
        final BfmFileInfoS01Row row = ite.getSingleRow();
        final BlancoFileManagerFileInfo info = new BlancoFileManagerFileInfo();
        info.setFileId(fileId);
        info.setProject(row.getProject());
        info.setFileName(row.getFileName());
        info.setFileStatFlg(row.getFileStatFlg());
        info.setFileChangeFlg(row.getFileChangeFlg());
        info.setFileType(row.getFileType());
        info.setFilePath(row.getFilePath());
        info.setUpdDt(row.getUpdDt());
        info.setUpdUserId(row.getUpdUserId());

        return info;
    }
}
