/*******************************************************************************
 * blanco Framework
 * Copyright (C) 2012 NTT DATA BUSINESS BRAINS CORPORATION
 * 
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *******************************************************************************/
/*******************************************************************************
 * Copyright (c) 2012 NTT DATA BUSINESS BRAINS CORPORATION and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *      NTT DATA BUSINESS BRAINS CORPORATION - initial API and implementation
 *******************************************************************************/
package blanco.db.tableaccessor.task;

import java.io.File;
import java.io.IOException;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;

import org.xml.sax.SAXException;

import blanco.db.BlancoDbXml2JavaClass;
import blanco.db.common.BlancoDbTableParser;
import blanco.db.common.BlancoDbXml2SqlInfo;
import blanco.db.common.util.BlancoDbUtil;
import blanco.db.common.util.BlancoDbXmlSerializer;
import blanco.db.common.valueobject.BlancoDbSetting;
import blanco.db.common.valueobject.BlancoDbSqlInfoStructure;
import blanco.db.tableaccessor.BlancoDbTableAccessorConstants;
import blanco.db.tableaccessor.BlancoDbTableAccessorMeta2Xml;
import blanco.db.tableaccessor.BlancoDbTableAccessorSqlGenerator;
import blanco.db.tableaccessor.BlancoDbTableAccessorTargetInfo;
import blanco.db.tableaccessor.BlancoDbTableAccessorXmlParser;
import blanco.db.tableaccessor.task.valueobject.BlancoDbTableAccessorProcessInput;
import blanco.dbmetadata.valueobject.BlancoDbMetaDataColumnStructure;
import blanco.dbmetadata.valueobject.BlancoDbMetaDataTableStructure;

/**
 * DB テーブルアクセサが内部的に利用する処理実装の本体部分そのもの。
 * 
 * @author Toshiki Iga
 */
public class BlancoDbTableAccessorProcessImpl implements BlancoDbTableAccessorProcess {
	/**
	 * 処理実行のエントリポイント。
	 */
    public int execute(final BlancoDbTableAccessorProcessInput input) throws IOException, IllegalArgumentException {
        System.out.println(BlancoDbTableAccessorConstants.getProductName() + " ("
                + BlancoDbTableAccessorConstants.getVersion() + ")");

        // 指定されたメタディレクトリを処理します。
        try {
            new BlancoDbTableAccessorMeta2Xml().processDirectory(new File(input.getMetadir()),
                    new File(input.getTmpdir()) + "/dbtable");
        } catch (TransformerException e) {
            throw new IllegalArgumentException(e.getMessage(), e);
        }

        final BlancoDbSetting dbSetting = new BlancoDbSetting();
        dbSetting.setJdbcdriver(input.getJdbcdriver());
        dbSetting.setJdbcurl(input.getJdbcurl());
        dbSetting.setJdbcuser(input.getJdbcuser());
        dbSetting.setJdbcpassword(input.getJdbcpassword());
        dbSetting.setTargetDir(input.getTargetdir());
        dbSetting.setBasePackage(input.getBasepackage());
        dbSetting.setRuntimePackage(input.getRuntimepackage());
        dbSetting.setSchema(input.getSchema());
        dbSetting.setLoggingsql(input.getLogsql());
        dbSetting.setEncoding(input.getEncoding());

		Connection conn = null;
		try {
			conn = BlancoDbUtil.connect(dbSetting);
			BlancoDbUtil.getDatabaseVersionInfo(conn, dbSetting);

			final Map<String, BlancoDbMetaDataTableStructure> mapDbTable = parseDb(dbSetting, conn);

			final File[] fileMeta = new File(input.getTmpdir() + "/dbtable")
					.listFiles();

			for (int index = 0; index < fileMeta.length; index++) {
				if (fileMeta[index].getName().endsWith(".xml") == false) {
					continue;
				}

				final BlancoDbTableAccessorTargetInfo[] infos = new BlancoDbTableAccessorXmlParser()
						.parse(fileMeta[index]);
				try {
					process(infos, mapDbTable, dbSetting, conn, input.getTmpdir());
				} catch (SQLException e) {
					throw new IllegalArgumentException(e);
				}
			}
		} catch (ClassNotFoundException e) {
			throw new IllegalArgumentException(e.toString(), e);
		} catch (SQLException e) {
			throw new IllegalArgumentException(e);
		} finally {
			BlancoDbUtil.close(conn);
		}

        try {
            new BlancoDbXml2JavaClass() {
                public boolean progress(int progressCurrent, int progressTotal, String progressItem) {
                    return true;
                }
            }.process(dbSetting, new File(input.getTmpdir() + "/dbtable/sql"));
        } catch (SAXException e) {
            throw new IllegalArgumentException(e);
        } catch (IOException e) {
            throw new IllegalArgumentException(e);
        } catch (ParserConfigurationException e) {
            throw new IllegalArgumentException(e);
        } catch (TransformerException e) {
            throw new IllegalArgumentException(e);
        } catch (SQLException e) {
            throw new IllegalArgumentException(e);
        } catch (ClassNotFoundException e) {
            throw new IllegalArgumentException(e);
        } catch (IllegalArgumentException e) {
            throw new IllegalArgumentException(e);
        }

        return 0;
    }

    /**
     * 進捗状況報告
     * 
     * @param argProgressMessage
     * @return 通常は false。
     */
    public boolean progress(String argProgressMessage) {
        System.out.println(argProgressMessage);
        return false;
    }

    /**
     * データベースの表情報をパースします。
     * 
     * @param dbSetting
     * @throws SQLException
     */
	public Map<String, BlancoDbMetaDataTableStructure> parseDb(final BlancoDbSetting dbSetting, final Connection conn) throws SQLException {
		final Map<String, BlancoDbMetaDataTableStructure> result = new HashMap<String, BlancoDbMetaDataTableStructure>();
		System.out.println("blancoDbTableAccessor: read db table schema.");

		try{
			final List<BlancoDbMetaDataTableStructure> tables = new BlancoDbTableParser()
					.parse(conn, dbSetting.getSchema());

			// DB 上のテーブルを列挙。
			for (BlancoDbMetaDataTableStructure tableStructure : tables) {
				result.put(tableStructure.getName().toLowerCase(), tableStructure);
			}
        } catch (SQLException e) {
            throw new IllegalArgumentException(e);
		}
		return result;
	}
    
    /**
     * 本体処理そのものの主処理。
     * 
     * @param infos
     * @param dbSetting
     * @param tmpDir
     * @throws SQLException
     */
	public void process(final BlancoDbTableAccessorTargetInfo[] infos,
			final Map<String, BlancoDbMetaDataTableStructure> mapDbTable,
			final BlancoDbSetting dbSetting, final Connection conn,
			final String tmpDir) throws SQLException {
		System.out.println("blancoDbTableAccessor: build sql.");

		for (BlancoDbTableAccessorTargetInfo targetInfo : infos) {
			final BlancoDbMetaDataTableStructure tableStructure = mapDbTable
					.get(targetInfo.getTableName().toLowerCase());
			if (tableStructure == null) {
				System.err.println("SQL定義ID '" + targetInfo.getSqldefid()
						+ "', 表 '" + targetInfo.getTableName()
						+ "' をデータベース上から見つけることができません。");
				continue;
			} else {
				System.out.println("blancoDbTableAccessor: processing talbe '" + targetInfo.getTableName() + "'.");

				if (tableStructure.getPrimaryKeys().size() == 0) {
					System.err.println("SQL定義ID '" + targetInfo.getSqldefid()
							+ "', プライマリーが含まれない表 '" + targetInfo.getTableName()
							+ "' ではテーブルアクセサーは生成できません。");
				}

				final List<BlancoDbSqlInfoStructure> sqlInfoStructureList = new ArrayList<BlancoDbSqlInfoStructure>();

				// 除外列
				final Map<String, BlancoDbMetaDataColumnStructure> usedExcludeColumnMap = new HashMap<String, BlancoDbMetaDataColumnStructure>();

				// 楽観排他列
				final Map<String, BlancoDbMetaDataColumnStructure> usedFieldNameForOptimisticLockMap = new HashMap<String, BlancoDbMetaDataColumnStructure>();

				if ("INSERT".equals(targetInfo.getSqlType())) {
					final BlancoDbSqlInfoStructure sqlInfo = new BlancoDbTableAccessorSqlGenerator()
							.generateInsert(targetInfo, tableStructure,
									usedExcludeColumnMap);
					sqlInfo.setPackage(targetInfo.getPackageName());
					sqlInfoStructureList.add(sqlInfo);
				} else if ("SELECT".equals(targetInfo.getSqlType())) {
					final BlancoDbSqlInfoStructure sqlInfo = new BlancoDbTableAccessorSqlGenerator()
							.generateSelect(targetInfo, tableStructure,
									usedExcludeColumnMap);
					sqlInfo.setPackage(targetInfo.getPackageName());

					// SQL 実行結果を反映。
					try {
						new BlancoDbXml2SqlInfo().processIterator(conn,
								sqlInfo, dbSetting);
					} catch (IllegalArgumentException ex) {
						System.err.println("SQL定義ID '" + sqlInfo.getName()
								+ "', SQL 文の処理でエラーが発生しました。" + ex.getMessage());
					}

					sqlInfoStructureList.add(sqlInfo);
				} else if ("UPDATE".equals(targetInfo.getSqlType())) {
					final BlancoDbSqlInfoStructure sqlInfo = new BlancoDbTableAccessorSqlGenerator()
							.generateUpdate(targetInfo, tableStructure,
									usedExcludeColumnMap,
									usedFieldNameForOptimisticLockMap);
					sqlInfo.setPackage(targetInfo.getPackageName());
					sqlInfoStructureList.add(sqlInfo);
				} else if ("DELETE".equals(targetInfo.getSqlType())) {
					final BlancoDbSqlInfoStructure sqlInfo = new BlancoDbTableAccessorSqlGenerator()
							.generateDelete(targetInfo, tableStructure,
									usedFieldNameForOptimisticLockMap);
					sqlInfo.setPackage(targetInfo.getPackageName());
					sqlInfoStructureList.add(sqlInfo);
				} else {
					System.err.println("Unknown SQL type '"
							+ targetInfo.getSqlType() + "' is set.");
				}

				for (String excludeColumn : targetInfo.getExcludeColumnList()) {
					if (usedExcludeColumnMap.get(excludeColumn.toLowerCase()) == null) {
						System.err.println("SQL 定義 ID '"
								+ targetInfo.getSqldefid() + "' に問題。指定した除外列 '"
								+ excludeColumn + "' が実際の列の除外処理に利用されませんでした。");
					}
				}

				if (targetInfo.getFieldNameForOptimisticLock() != null
						&& usedFieldNameForOptimisticLockMap.get(targetInfo
								.getFieldNameForOptimisticLock()) == null) {
					System.err.println("SQL 定義 ID '" + targetInfo.getSqldefid()
							+ "' に問題。指定した楽観排他列 '"
							+ targetInfo.getFieldNameForOptimisticLock()
							+ "' がテーブル '" + targetInfo.getTableName()
							+ "' に見つかりませんでした。");
				}

				// 実行結果を tmp に出力。
				new File(tmpDir + "/dbtable/sql/").mkdirs();
				BlancoDbXmlSerializer.serialize(sqlInfoStructureList, new File(
						tmpDir + "/dbtable/sql/" + targetInfo.getSqldefid()
								+ ".xml"));
			}
		}
	}
}
