module BloggerPost
  DRAFT_URL_PREFIX = 'bpdraft:'
  
  STATUS_MODIFIED = 0
  STATUS_POSTED   = 1
  STATUS_DELETED  = 2

  class Entry
    include BloggerPost::Common

    attr_reader :url, :status, :edit_id, :style, :blog_name, :date, :updated, :tag_list, :category_list, :trackback_list
    attr_writer :url, :status, :edit_id, :title, :body, :style, :blog_name, :date, :updated, :tag_list, :category_list, :trackback_list

    def self.new_file(config, style_name, blog_name, tag_list, category_list, trackback_list, filename)
      body = File.read(filename)
      entry = self.new_body(config, body, style_name, blog_name, tag_list, category_list, trackback_list)
      entry.style = BloggerPost.style_new_filename(filename) unless entry.style
      entry
    end

    def self.new_body(config, body, style_name, blog_name, tag_list, category_list, trackback_list)
      self.new(config, nil, STATUS_MODIFIED, nil, nil, body, nil, style_name, blog_name, nil, nil, tag_list, category_list, trackback_list)
    end

    def self.edit_file(config, url, tag_list, category_list, filename)
      self.edit_body(config, url, File.read(filename), tag_list, category_list)
    end

    def self.edit_body(config, url, body, tag_list, category_list)
      entry = self.new_db(config, url)
      entry.body          = body
      entry.tag_list      = tag_list if tag_list
      entry.category_list = category_list if category_list
      entry
    end

    def self.new_db(config, url)
      config.db_open_reader do |db|
        return self.new_unpack(config, db[url])
      end
    end

    def self.new_draft(config)
      url = nil
      number = 1
      config.db_open_reader do |db|
        loop do
          url = "#{DRAFT_URL_PREFIX}#{number}"
          break unless db[url]
          number += 1
        end
      end
      path = "#{config.draft_dir}/#{number}"
      File.new(path, 'w')
      entry = self.new(config, url, STATUS_MODIFIED, nil, '', '', path, config.default_style_name, nil, DateTime.now, DateTime.now, nil, nil, nil)
      entry.save
      entry
    end

    def initialize(config, url, status, edit_id, title, body, path, style_name, blog_name, date, updated, tag_list, category_list, trackback_list)
      @config         = config
      @url            = url
      @status         = status
      @edit_id        = edit_id
      @title          = title
      @body           = body
      @path           = path
      @style          = BloggerPost.style_new(style_name) if style_name
      @blog_name      = blog_name 
      @date           = date
      @updated        = updated
      @tag_list       = tag_list
      @category_list  = category_list
      @trackback_list = trackback_list
      @tag_list       = [] unless @tag_list
      @category_list  = [] unless @category_list
      @trackback_list = [] unless @trackback_list
    end

    def to_html(use_cache = true)
      if use_cache
        return @html if @html
        return @html if !draft? && load_cache
      end
      html = @style.entry_to_html(self).toutf8
      html = apply_filter(html)
      html = delete_lf_in_html(html) if @style.delete_lf?
      @html = html
      if use_cache && !draft?
        dump_cache
      end
      html
    end

    def bpid
      url = URI.parse(@url)
      id = url.path
      id += '?' + url.query if url.query
      id = id[1, id.length]
      delete_extname(id)
    end

    def filename
      url = URI.parse(@url)
      url.host + '/' + bpid + @style.extname
    end

    def delete
      return unless @url
      @config.db_open_writer do |db|
        db.delete(@url)
      end
      if path
        File.delete(path)
      end
    end

    def update_url(url)
      delete unless @url == url
      @url = url
      update_path
    end

    def update_path
      @path = @config.backup_dir + '/' + filename
    end

    def path
      update_path unless @path
      @path
    end

    def cache_filename
      url = URI.parse(@url)
      @config.cache_dir + '/' + url.host + '/' + bpid + '.html'
    end
    
    def dump_backup
      FileUtils.makedirs(File.dirname(path))
      File.open(path, 'w') {|f| f.write(@body) }
    end

    def load_cache
      return nil unless @url
      return nil unless File.exist?(cache_filename)
      @html = File.read(cache_filename)
      @html
    end

    def dump_cache
      return unless @url
      to_html unless @html
      FileUtils.makedirs(File.dirname(cache_filename))
      File.open(cache_filename, 'w') {|f| f.write(@html) }
    end

    def clear_cache
      @html = nil
      File.unlink(cache_filename) unless draft?
    end

    def title
      @title = get_title_text(to_html) unless @title
      @title
    end

    def body
      unless @body
        @body = File.read(@path) if @path
      end
      @body
    end

    def excerpt
      make_excerpt(to_html)
    end

    def blog
      return Blog.new_config(@config, blog_name) if @blog_name
      @config.default_blog
    end

    def save
      @config.db_open_writer do |db|
        db[@url] = pack()
      end
      if body && path
        FileUtils.makedirs(File.dirname(path))
        File.open(path, 'w') {|f| f.write(body) }
      end
    end

    def draft?
      url =~ /^#{DRAFT_URL_PREFIX}/
    end

    def deleted?
      @status == STATUS_DELETED
    end

    def posted?
      !draft? && !deleted?
    end

    def ==(entry)
      @url == entry.url
    end

    private

    def delete_lf_in_html(src)
      dest = ''
      in_pre = false
      src.each_line do |line|
        dest << line.rstrip
        if line =~ /<pre( .*?>|>)/
          in_pre = true
        end
        if line =~ /<\/pre>/
          in_pre = false
        end
        if in_pre
          dest << "\n"
        else
#          dest << ' '
        end
      end
      dest
    end

    def apply_filter(html)
      BloggerPost.each_filter(@config) do |filter|
        html = filter.translate(html)
      end
      html
    end

    def pack
      a = []
      a << @url
      a << @status
      a << @edit_id
      a << title
      a << @path
      a << @style.style_name
      a << blog.blog_name
      a << @date.to_s
      a << @updated.to_s
      a << @tag_list
      a << @category_list
      a << @trackback_list
      Marshal.dump(a)
    end

    def self.new_unpack(config, s)
      a = Marshal.load(s)
      url            = a.shift
      status         = a.shift
      edit_id        = a.shift
      title          = a.shift
      path           = a.shift
      style_name     = a.shift
      blog_name      = a.shift
      date           = DateTime.parse(a.shift)
      updated        = DateTime.parse(a.shift)
      tag_list       = a.shift
      category_list  = a.shift
      trackback_list = a.shift
      self.new(config, url, status, edit_id, title, nil, path, style_name, blog_name, date, updated, tag_list, category_list, trackback_list)
    end
  end
end
