module YahooFS
  PLAYER_URL  = 'http://soccer.yahoo.co.jp/fantasy/jleague/player/'
  PROFILE_URL = 'http://soccer.yahoo.co.jp/fantasy/jleague/profile/'
  
  class Player
    include BloggerPost::Common
    
    attr_reader :id, :name, :team, :position, :salary, :captain
    attr_reader :entry_fp, :defense_fp, :offense_fp, :victory_fp, :caution_fp, :bonus_fp

    POSITION_PRIORITY_TABLE = {
      'GK' => 0,
      'DF' => 1,
      'MF' => 2,
      'FW' => 3,
    }

    TEAM_ABBREVIATION_TABLE = {
      '鹿島アントラーズ'       => '鹿島',
      '柏レイソル'             => '柏',
      '浦和レッズ'             => '浦和',
      '大宮アルディージャ'     => '大宮',
      'ジェフユナイテッド千葉' => '千葉',
      'FC東京'                 => 'F東京',
      '川崎フロンターレ'       => '川崎',
      '横浜F・マリノス'        => '横浜FM',
      '横浜FC'                 => '横浜FC',
      'ヴァンフォーレ甲府'     => '甲府',
      'アルビレックス新潟'     => '新潟',
      '清水エスパルス'         => '清水',
      'ジュビロ磐田'           => '磐田',
      '名古屋グランパスエイト' => '名古屋',
      'ガンバ大阪'             => 'G大阪',
      'ヴィッセル神戸'         => '神戸',
      'サンフレッチェ広島'     => '広島',
      '大分トリニータ'         => '大分'
    }

    def initialize(config, id, captain)
      @id = id.to_i
      @captain = captain
      @entry_fp   = 0
      @defense_fp = 0
      @offense_fp = 0
      @victory_fp = 0
      @caution_fp = 0
      @bonus_fp   = 0
      res = http_get(config, URI.parse(PLAYER_URL + id))
      res.body =~ /<b>\n <a href=\/fantasy\/jleague\/team\/\d*?>(.*?)<\/a>\n (.*?)\n (.*?)\n<\/b>/
      @team = $1.toutf8
      @position = $2.toutf8
      @name = $3.toutf8
      salary_list = res.body.scan(/(\d*(#{'億'.toeuc})?\d*)#{'万'.toeuc}/)
      @salary = salary_list[-1][0]
      @salary =~ /(\d*)(#{'億'.toeuc})?(\d*)/
      if $2
        hundred_million = $1.to_i
        @salary = $3.to_i + $1.to_i * 10000
      else
        @salary = $1.to_i
      end
      fp_line_list = res.body.scan(/<tr bgcolor=#\w*? align=right>.*?<td><small>\d*fp<\/small><\/td>.*?<\/tr>/m)
      if fp_line_list.length > 0
        fp_list = fp_line_list[-1].scan(/<small>(-?\d*)fp<\/small>/)
        @entry_fp   = fp_list[0][0].to_i if fp_list[0]
        @defense_fp = fp_list[1][0].to_i if fp_list[1]
        @offense_fp = fp_list[2][0].to_i if fp_list[2]
        @victory_fp = fp_list[3][0].to_i if fp_list[3]
        @caution_fp = fp_list[4][0].to_i if fp_list[4]
        @bonus_fp   = fp_list[5][0].to_i if fp_list[5]
      end

      def team_short
        TEAM_ABBREVIATION_TABLE[@team]
      end

      def fp
        @entry_fp + @defense_fp + @offense_fp + @victory_fp + @caution_fp + @bonus_fp
      end
    end

    def <=>(other)
      return POSITION_PRIORITY_TABLE[@position] <=> POSITION_PRIORITY_TABLE[other.position] unless @position == other.position
      @id <=> other.id
    end
  end

  class Filter < BloggerPost::Filter
    def translate(html)
      html = filter_attr(html, 'bpyahoofs') do |param|
        display_salary, display_fp, display_bonus_fp, user_id = parse_param(param)
        res = http_get(@config, URI.parse(PROFILE_URL + user_id))
        player_id_list = res.body.scan(/\/fantasy\/jleague\/player\/(\d*)/).collect {|player_id_array| player_id_array[0] }
        player_id_list.uniq!
        res.body =~ res.body.scan(/\/fantasy\/jleague\/player\/(\d*).*?captain/)
        captain_id = $1
        player_list = player_id_list.collect {|player_id| Player.new(@config, player_id, player_id == captain_id) }
        total_salary = 0
        total_fp = 0
        total_bonus_fp = 0
        player_list.sort!
        player_list.each do |player|
          total_salary += player.salary
          total_fp += player.fp
          if player.captain
            total_fp += player.fp
          end
          total_bonus_fp += player.bonus_fp
          if player.captain
            total_bonus_fp += player.bonus_fp
          end
        end
        table = '<table border="1">'
        table += "<thead><tr><th style='text-align: center'>位置</th><th style='text-align: center'>選手</th><th style='text-align: center'>チーム</th>"
        if display_salary
          table += "<th style='text-align: center'>年俸</th>"
        end
        if display_bonus_fp
          table += "<th style='text-align: center'>ボーナス</th>"
        end
        if display_fp
          table += "<th style='text-align: center'>fp</th>"
        end
        table += '</tr></thead>'
        if display_salary || display_fp || display_bonus_fp
          table += "<tfoot><tr><td colspan='3' style='text-align: center'>合計</td>"
          if display_salary
            table += "<td style='text-align: right'>#{convert_salary(total_salary)}</td>"
          end
          if display_bonus_fp
            table += "<td style='text-align: right'>#{total_bonus_fp}fp</td>"
          end
          if display_fp
            table += "<td style='text-align: right'>#{total_fp}fp</td>"
          end
        end
        table += '</tr></tfoot>'
        table += '<tbody>'
        table += player_list.inject('') do |result, player|
          result += '<tr>'
          result += "<td style='text-align: center'>#{player.position}</td>"
          result += "<td>#{player.name}"
          if player.captain
            result += '(C)'
          end
          result += '</td>'
          result += "<td>#{player.team_short}</td>"
          if display_salary
            result += "<td style='text-align: right'>#{convert_salary(player.salary)}</td>"
          end
          if display_bonus_fp
            result += "<td style='text-align: right'>#{player.bonus_fp}fp</td>"
          end
          if display_fp
            result += "<td style='text-align: right'>#{player.fp}fp</td>"
          end
          result + '</tr>'
        end
        table + '</tbody></table>'
      end
      html
    end

    private

    def parse_param(param)
      param_list = param.split(':')
      user_id = nil
      display_salary = false
      display_fp = false
      display_bonus_fp = false
      while param_list.length > 0
        element = param_list.shift
        case element
        when 'salary'
          display_salary = true
        when 'fp'
          display_fp = true
        when 'bonus'
          display_bonus_fp = true
        else
          user_id = element
        end
      end
      [display_salary, display_fp, display_bonus_fp, user_id]
    end

    def convert_salary(salary)
      hundred_million = 0
      while salary > 10000
        hundred_million += 1
        salary -= 10000
      end
      str = ''
      if hundred_million > 0
        str += hundred_million.to_s + '億'
      end
      str + salary.to_s + '万'
    end
  end
end

BloggerPost.register_filter(YahooFS::Filter)
