module BloggerPost
  class Service
    def initialize(config)
      @config = config
    end

    def service_name
      self.class.service_name
    end

    def entry_suffix
      '.html'
    end

    def atom_type
      'normal'
    end

    def auth_type
      'wsse'
    end

    def atom_at_put
      true
    end

    def api
      case api_type
      when 'atom'
        return AtomAPI.new(@config, atom_type)
      when 'xmlrpc'
        return XMLRPCAPI.new(@config)
      end
    end
  end

  class AtomAPI
    include BloggerPost::Common

    GOOGLE_AUTH_URL = 'https://www.google.com/accounts/ClientLogin'

    def initialize(config, atom_type)
      @config = config
      @atom_type = atom_type
    end

    def new_post(entry)
      post(entry, 'new')
    end

    def edit_post(entry)
      post(entry, 'edit')
    end

    def delete_post(entry)
      post(entry, 'delete')
    end

    def post(entry, req_method)
      url_text = entry.edit_id
      url_text = entry.blog.post_url unless url_text
      url = URI.parse(url_text)
      req = make_req(entry, url, req_method)
      res = send_req(url, req)
      response_atom_to_entry(entry, req_method, res.body)
      unless entry.edit_id
        get_edit_id(entry, url)
      end
      Result.new(req_method, 'atom')
    end

    private

    def make_req(entry, url, req_method)
      case req_method
      when 'new'
        req = Net::HTTP::Post.new(url.path)
      when 'edit'
        req = Net::HTTP::Put.new(url.path)
      when 'delete'
        req = Net::HTTP::Post.new(url.path)
#        req = Net::HTTP::Delete.new(url.path)
        req['X-HTTP-Method-Override'] = 'DELETE'
      end
      req['Content-Type'] = 'application/atom+xml'
      set_auth_to_req(req, entry)
      if entry.body
        req.body = make_atom_entry(entry)
      else
        req.body = ''
      end
      req
    end

    def send_req(url, req)
      res = http_request(@config, url, req)
      loop do
        case res
        when Net::HTTPSuccess
          break
        when Net::HTTPRedirection
          url = URI.parse(res['Location'])
          req['Host'] = url.host
          res = http_request(@config, url, req)
        else
          res.error!
        end
      end
      res
    end

    # to avoid REXML bug at Ruby 1.8.6
    def find_element_with_attr(doc, path, key, value)
      finded = nil
      doc.elements.each(path) do |element|
        finded = element if element.attributes[key] == value
      end
      finded
    end

    def parse_atom(src)
      doc = REXML::Document.new(src)
      published_element = doc.elements['/entry/published']
      published_element = doc.elements['/entry/issued'] unless published_element
      published_element = doc.elements['/entry/created'] unless published_element
      updated_element = doc.elements['/entry/updated']
      updated_element = doc.elements['/entry/modified'] unless updated_element
      updated_element = published_element unless updated_element
      # edit_id_element = doc.elements["/entry/link[@rel='service.edit']"]
      edit_id_element = find_element_with_attr(doc, '/entry/link', 'rel', 'service.edit')
      # edit_id_element = doc.elements["/entry/link[@rel='edit']"] unless edit_id_element
      edit_id_element = find_element_with_attr(doc, '/entry/link', 'rel', 'edit') unless edit_id_element
      if edit_id_element
        edit_id = edit_id_element.attributes['href']
      else
        edit_id = nil
      end
      # href = doc.elements["/entry/link[@rel='alternate']"].attributes['href']
      href = find_element_with_attr(doc, '/entry/link', 'rel', 'alternate').attributes['href']
      [published_element.text, updated_element.text, href, edit_id]
    end

    def response_atom_to_entry(entry, method, response_body)
      if method == 'delete'
        entry.updated = DateTime.now
        entry.date = entry.updated unless entry.date
        return
      elsif method == 'new' || entry.blog.service.atom_at_put
        published, updated, url, edit_id = parse_atom(response_body)
      else
        published = entry.date.to_s
        url = entry.url
        edit_id = entry.edit_id
      end
      entry.update_url(url)
      entry.edit_id = edit_id if edit_id
      entry.date = DateTime.parse(published)
      entry.updated = DateTime.parse(updated)
    end

    def get_google_auth_token(user, passwd)
      url = URI.parse(GOOGLE_AUTH_URL)
      req = Net::HTTP::Post.new(url.path)
      req.form_data = {
        'Email' => user,
        'Passwd' => passwd,
        'source' => "bloggerpost-#{Version}",
        'service' => 'blogger'
      }
      res = http_request(@config, url, req)
      res.body =~ /Auth=(.+)/
      $1
    end

    def make_wsse(user, passwd, encode_nonce)
      created = Time.now.utc.iso8601
      nonce = OpenSSL::Digest::SHA1.digest(OpenSSL::Digest::SHA1.digest(Time.now.to_s + sprintf("%f", rand()) + $$.to_s))
      nonce_base64 = [nonce].pack('m').chomp
      if (encode_nonce)
        digest_nonce = nonce_base64
      else
        digest_nonce = nonce
      end
      password_digest = [OpenSSL::Digest::SHA1.digest(digest_nonce + created + passwd)].pack("m").chomp
      wsse = %(UsernameToken Username="#{user}", PasswordDigest="#{password_digest}", Nonce="#{nonce_base64}", Created="#{created}")
      wsse
    end

    def google_auth(req, user, passwd)
      req['Authorization'] = 'GoogleLogin auth=' + get_google_auth_token(user, passwd)
    end

    def wsse_auth(req, user, passwd)
      req["X-WSSE"]  = make_wsse(user, passwd, false)
    end

    def set_auth_to_req(req, entry)
      case entry.blog.service.auth_type
      when 'google'
        google_auth(req, entry.blog.user, entry.blog.passwd)
      when 'wsse'
        wsse_auth(req, entry.blog.user, entry.blog.passwd)
      end
    end

    def get_edit_id(entry, url)
      req = Net::HTTP::Get.new(url.path)
      req.body = make_atom_entry(entry)
      set_auth_to_req(req, entry)
      res = send_req(url, req)
      raise "Couldn't edit url." unless res.code == '200'
      doc = REXML::Document.new(res.body)
      doc.elements.each('/feed/entry') do |entry_element|
        # url = entry_element.elements["link[@rel='alternate']"].attributes['href']
        url = find_element_with_attr(entry_element, 'link', 'rel', 'alternate').attributes['href']
        if entry.url == url
          # edit_link = entry_element.elements["link[@rel='service.edit']"]
          edit_link = find_element_with_attr(entry_element, 'link', 'rel', 'alternate').attributes['href']
          entry.edit_id =  edit_link.attributes['href'] if edit_link
          return
        end
      end
    end

    def make_blogger_atom_entry(entry)
      html = entry.to_html
      title = entry.title
      body = get_body_text(html)
      categories = entry.tag_list.inject('') do |result, tag|
        if tag.length > 0
          result + "<category scheme='http://www.blogger.com/atom/ns#' term='#{tag.toutf8}'/>"
        else
          result
        end
      end
      <<ATOM_ENTRY
<entry xmlns="http://www.w3.org/2005/Atom">
  <title type="text">#{title}</title>
  <content type="xhtml">#{body}</content>
  #{categories}
</entry>
ATOM_ENTRY
    end

    def make_normal_atom_entry(entry)
      html = entry.to_html
      title = entry.title
      body = get_body_text(html)
      dc_subjects = entry.category_list.inject('') do |result, category|
        if category.length > 0
          result + "<dc:subject>#{category.toutf8}</dc:subject>"
        else
          result
        end
      end
      <<ATOM_ENTRY
<entry xmlns="http://purl.org/atom/ns#" xmlns:dc="http://purl.org/dc/elements/1.1/">
  <title type="text">#{title}</title>
  #{dc_subjects}
  <content type="xhtml">#{body}</content>
</entry>
ATOM_ENTRY
    end

    def make_atom_entry(entry)
      case @atom_type
      when 'blogger'
        return make_blogger_atom_entry(entry)
      when 'normal'
        return make_normal_atom_entry(entry)
      end
    end
  end

  class XMLRPCAPI
    def initialize(config)
      @config = config
    end

    def new_post(entry)
      edit_post(entry)
    end

    def edit_post(entry)
      post_id = entry.edit_id
      blog_id = entry.blog.blog_id
      api_url = entry.blog.api_url
      user = entry.blog.user
      passwd = entry.blog.passwd
      html = entry.to_html
      title = entry.title
      body = get_body_text(html)
      category_id_list = nil
      server = xmlrpc_client_new(api_url)
      if entry.category_list.length > 0
        blog_category_list = server.call('mt.getCategoryList', blog_id, user, passwd)
        category_id_list = entry.category_list.collect do |post_category|
          category_id = nil
          post_category = post_category.toutf8
          blog_category_element = blog_category_list.each do |blog_category|
            if blog_category['categoryName'].toutf8 == post_category
              category_id = blog_category['categoryId']
              break
            end
          end
          raise "#{post_category} category is not found." unless category_id
          { 'categoryId' => category_id }
        end
      end
      content_hash = {
        'title'       => title,
        'description' => body,
      }
      if entry.tag_list.length > 0
        content_hash['mt_tags'] = entry.tag_list.collect {|tag| tag.toutf8 }.join(',')
      end
      req_method = nil
      if post_id
        req_method = 'edit'
        server.call('metaWeblog.editPost', post_id, user, passwd, content_hash, true)
      else
        req_method = 'new'
        post_id = server.call('metaWeblog.newPost', blog_id, user, passwd, content_hash, true)
        entry.edit_id = post_id
        post_hash = server.call('metaWeblog.getPost', post_id, user, passwd)
        entry.update_url(post_hash['permaLink'])
        entry.date = DateTime.now
      end
      entry.updated = DateTime.now
      if entry.category_list.length > 0
        category_list = server.call('mt.setPostCategories', post_id, user, passwd, category_id_list)
      end
      Result.new(req_method, 'xmlrpc')
    end

    def delete_post(entry)
      post_id = entry.edit_id
      api_url = entry.blog.api_url
      user = entry.blog.user
      passwd = entry.blog.passwd
      server = xmlrpc_client_new(api_url)
      server.call('blogger.deletePost', '', post_id, user, passwd)
      Result.new('delete', 'xmlrpc')
    end

    private

    def xmlrpc_client_new(api_url)
      url = URI.parse(api_url)
      use_ssl = url.scheme == 'https'
      proxy_host = nil
      proxy_port = nil
      proxy_user =  nil
      proxy_passwd =  nil
      if @config.use_proxy
        proxy_host, proxy_port, proxy_user, proxy_passwd = get_proxy_info(config, use_ssl)
      end
      XMLRPC::Client.new(url.host, url.path, url.port, proxy_host, proxy_port, proxy_user, proxy_passwd, use_ssl)
    end
  end
end
