require 'bloggerpost'
require 'jcode'

def STDIN.eof?
  false
end

module BloggerPost
  class ::BloggerPost::Blog
    attr_reader :entry_list
    attr_writer :entry_list
  end

  class ::BloggerPost::Folder
    include ::BloggerPost::Common
    
    attr_reader :entry_list, :name, :title, :search_list

    def initialize(entry_list, name, title, is_draft)
      @entry_list = entry_list.dup
      @name = name
      @title = title
      @is_draft = is_draft
      @search_list = []
    end

    def entry_list
      if @search_list.length > 0
        @entry_list = @entry_list.find_all {|entry| match_entry?(entry) }
      end
      @entry_list
    end

    def draft?
      @is_draft
    end

    def match_entry?(entry)
      @search_list.each do |search|
        regexp = /#{Regexp.escape(search['word'])}/m
        case search['type']
        when 'Title'
          return false unless to_external_encode(entry.title) =~ regexp
        when 'Body'
          return false unless entry.body =~ regexp
        when 'Tag'
          return false unless entry.tag_list.find {|tag| to_external_encode(tag) =~ regexp }
        when 'Category'
          return false unless entry.category_list.find {|category| to_external_encode(category) =~ regexp }
        when 'Trackback'
          return false unless entry.trackback_list.find {|trackback| to_external_encode(trackback) =~ regexp }
        end
      end
      true
    end
  end

  class ::BloggerPost::Draft
    attr_reader :entry_list, :name, :title
    attr_writer :entry_list

    def initialize(config)
      @config = config
      @name = 'draft'
      @title = '下書き'
    end

    def delete_entry(entry)
      @entry_list.delete(entry)
      entry.delete
    end
  end
  
  class ElBPApp < ElApp
    include ::BloggerPost::Common

    WEEK_DAY = ['日', '月', '火', '水', '木', '金', '土']
    DATE_FORMAT = '%Y年%m月%d日'
    TIME_FORMAT = '%H:%M'
    BODY_SEPARATOR = '--text follows this line--'

    BLOGLIST_BUFFER_NAME = '*bloggerpost blog list*'
    ENTRYLIST_BUFFER_NAME = '*bloggerpost entry list*'
    ENTRY_BUFFER_NAME = '*bloggerpost entry*'
    DRAFT_BUFFER_NAME = '*bloggerpost draft*'

    ENTRYLIST_WINDOW_HEIGHT = 10

    DRAFT_MODE_TABLE = {
      'html'  => :html_mode,
      'bhtml' => :html_mode,
      'rd'    => :rd_mode,
      'rst'   => :rst_mode
    }

    SORT_PROMPT = "Sort by: (D)ate (T)itle: "
    REVERSE_SORT_PROMPT = "Reverse sort by: (D)ate (T)itle: "
    SEARCH_PROMPT = "Search for: (T)itle (B)ody t(A)g (C)ategory t(R)ackback: "

    def initialize(params)
      @config = ::BloggerPost::Config.new
      @draft = ::BloggerPost::Draft.new(@config)
      @blog_list = []
      @entry_list = []
      @current_folder = ::BloggerPost::Folder.new([], '', '', false)
      @current_entry_url = ''
      @current_draft_url = ''
      @boot = false
      define_modes
      define_faces
      elvar.bloggerpost_browse_url_func = :browse_url
      elvar.bloggerpost_preview_func = :browse_url
    end

    private

    def boot
      return if @boot
      update_list
      @boot = true
    end

    # util

    def format_date(date)
      date.strftime(DATE_FORMAT) + "(#{WEEK_DAY[date.wday]})" + date.strftime(TIME_FORMAT)
    end

    def cut_string(s, len)
      t = s.dup
      while t.length > len
        t.chop!
      end
      t
    end

    def define_key_hash(map, keymap)
      keymap.each do |key, cmd|
        define_key(map, key, cmd)
      end
    end

    def insert_with_properties(text, properties)
      prev_point = point
      insert(text)
      set_text_properties(prev_point, point, properties)
    end

    def make_minor_mode_keymap(hash)
      assoc_list = ''
      hash.each do |key, val|
        assoc_list << "(#{el4r_ruby2lisp(key)} . #{el4r_ruby2lisp(el(val))})"
      end
      el('(quote (' + assoc_list + '))')
    end

    def find_blog(blog_name)
      @blog_list.find {|item| item.blog_name == blog_name }
    end

    def find_entry(url)
      @entry_list.find {|item| item.url == url }
    end

    def bloggerpost_browse_url(url)
      apply(elvar.bloggerpost_browse_url_func, url, [])
    end

    def bloggerpost_preview(url)
      apply(elvar.bloggerpost_preview_func, url, [])
    end

    # top

    def update_list
      @config = ::BloggerPost::Config.new
      @draft.entry_list = []
      @blog_list = []
      @config.each_blog do |blog_hash|
        blog = ::BloggerPost::Blog.new_config(@config, blog_hash['name'])
        blog.entry_list = []
        @blog_list << blog
      end
      @entry_list = []
      @config.db_open_reader do |db|
        db.each_value do |value|
          entry = ::BloggerPost::Entry.new_unpack(@config, value)
          @entry_list << entry
          if entry.draft?
            @draft.entry_list << entry
          else
            find_blog(entry.blog.blog_name).entry_list << entry
          end
        end
      end
    end

    defun(:bloggerpost, :interactive => true) do
      boot
      bloggerpost_bloglist
      bloggerpost_bloglist_update
    end

    defun(:bloggerpost_exit, :interactive => true) do
      kill_buffer(get_buffer(BLOGLIST_BUFFER_NAME)) if get_buffer(BLOGLIST_BUFFER_NAME)
      kill_buffer(get_buffer(ENTRYLIST_BUFFER_NAME)) if get_buffer(ENTRYLIST_BUFFER_NAME)
      kill_buffer(get_buffer(ENTRY_BUFFER_NAME)) if get_buffer(ENTRY_BUFFER_NAME)
    end

    defun(:bloggerpost_global_search, :interactive => true) do
      entry_list = @entry_list.find_all {|entry| !entry.draft? }
      @current_folder = ::BloggerPost::Folder.new(entry_list, '', '', false)
      entrylist_read_search_word
      bloggerpost_entrylist
    end

    # blog list

    defun(:bloggerpost_bloglist, :interactive => true) do
      if get_buffer(BLOGLIST_BUFFER_NAME)
        switch_to_buffer(get_buffer(BLOGLIST_BUFFER_NAME))
      else
        switch_to_buffer(get_buffer_create(BLOGLIST_BUFFER_NAME))
        bloggerpost_bloglist_mode
      end
    end

    defun(:bloggerpost_bloglist_select_current, :interactive => true) do
      is_draft = get_text_property(point, :draft)
      if is_draft
        entry_list = @draft.entry_list
        name = @draft.name
        title = @draft.title
      else
        blog = find_blog(get_text_property(point, :blog_name))
        entry_list = blog.entry_list
        name = blog.blog_name
        title = blog.title
      end
      @current_folder = ::BloggerPost::Folder.new(entry_list, name, title, is_draft)
      bloggerpost_entrylist
    end

    defun(:bloggerpost_bloglist_exit, :interactive => true) do
      bloggerpost_exit
    end

    defun(:bloggerpost_bloglist_update, :interactive => true) do
      point_org = point
      with(:save_excursion) do
        let(:buffer_read_only, false) do
          erase_buffer
          blog_name_width = @blog_list.collect {|blog| blog.blog_name.length }.push(@draft.name.length).max
          title_width = @blog_list.collect {|blog| to_external_encode(blog.title).length }.max
          count_width = @blog_list.collect {|blog| blog.entry_list.length.to_s.length }.push(@draft.entry_list.length.to_s.length).max
          draft_line = " [ #{@draft.name}#{' ' * (blog_name_width - @draft.name.length)} ] "
          num = @draft.entry_list.length
          draft_line << "(#{num.to_s})#{' ' * (count_width - num.to_s.length + 1)}"
          draft_line << to_external_encode(@draft.title)
          insert_with_properties(cut_string(draft_line, window_width - 1), [:draft, true, :face, :bloggerpost_blog_face])
          newline
          @blog_list.each do |blog|
            blog_line = ' [ ' + blog.blog_name + ' ' * (blog_name_width - blog.blog_name.length) + ' ] '
            blog_line << '(' + blog.entry_list.length.to_s + ')' + ' ' * (count_width - blog.entry_list.length.to_s.length + 1)
            blog_line << to_external_encode(blog.title) + ' ' * (title_width - to_external_encode(blog.title).length + 1)
            blog_line << blog.url
            insert_with_properties(cut_string(blog_line, window_width - 1), [:blog_name, blog.blog_name, :face, :bloggerpost_blog_face])
            newline
          end
          '' # for performance
        end
      end
      goto_char(point_org)
    end

    defun(:bloggerpost_bloglist_force_update, :interactive => true) do
      update_list
      bloggerpost_bloglist_update
    end

    defun(:bloggerpost_bloglist_browse_url, :interactive => true) do
      is_draft = get_text_property(point, :draft)
      unless is_draft
        blog = find_blog(get_text_property(point, :blog_name))
        bloggerpost_browse_url(blog.url)
      end
    end

    def bloglist_mode_keymap
      {
        "\r" => :bloggerpost_bloglist_select_current,
        " "  => :bloggerpost_bloglist_select_current,
        "q"  => :bloggerpost_bloglist_exit,
        "s"  => :bloggerpost_bloglist_force_update,
        "w"  => :bloggerpost_draft,
        "G"  => :bloggerpost_global_search,
        "b"  => :bloggerpost_bloglist_browse_url
      }
    end

    def define_bloglist_mode
      define_derived_mode(:bloggerpost_bloglist_mode, :fundamental_mode, "BloggerPost Blog List") do
        elvar.buffer_read_only = true
      end
      bloglist_mode_keymap.each do |key, cmd|
        define_key(:bloggerpost_bloglist_mode_map, key, cmd)
      end
    end

    # entry list

    def entrylist_sort(reverse)
      c = 0
      let(:cursor_in_echo_area, true) do
        if reverse
          message(REVERSE_SORT_PROMPT)
        else
          message(SORT_PROMPT)
        end
        c = read_char
      end
      c = c.chr.downcase
      @current_folder.entry_list.sort! do |entry1, entry2|
        case c
        when 't'
          entry1.title <=> entry2.title
        when 'd'
          entry1.date <=> entry2.date
        else
          0
        end
      end
      @current_folder.entry_list.reverse! if reverse
    end
    
    def entrylist_read_search_word
      c = 0
      let(:cursor_in_echo_area, true) do
        message(SEARCH_PROMPT)
        c = read_char
      end
      type = ''
      case c.chr.downcase
      when 't'
        type = 'Title'
      when 'b'
        type = 'Body'
      when 'a'
        type = 'Tag'
      when 'c'
        type = 'Category'
      when 'r'
        type = 'Trackback'
      end
      unless type == ''
        word = read_string("#{type}: ")
        @current_folder.search_list << {'type' => type , 'word' => word}
      end
    end

    defun(:bloggerpost_entrylist, :interactive => true) do
      if get_buffer(ENTRYLIST_BUFFER_NAME)
        switch_to_buffer(ENTRYLIST_BUFFER_NAME)
      else
        switch_to_buffer(get_buffer_create(ENTRYLIST_BUFFER_NAME))
        bloggerpost_entrylist_mode
      end
      bloggerpost_entrylist_update
    end

    defun(:bloggerpost_entrylist_select_current, :interactive => true) do
      url = get_text_property(point, :url)
      if url
        @current_entry_url = get_text_property(point, :url)
        delete_other_windows
        split_window_vertically(ENTRYLIST_WINDOW_HEIGHT)
        with(:save_selected_window) do
          other_window(1)
          bloggerpost_entry
        end
      end
    end

    def entry_scroll_updown(up, count = nil)
      if one_window_p || !get_buffer(ENTRY_BUFFER_NAME) || @current_entry_url != get_text_property(point, :url)
        bloggerpost_entrylist_select_current
      else
        with(:save_selected_window) do
          other_window(1)
          if up
            if count
              scroll_up(count)
            else
              scroll_up
            end
          else
            if count
              scroll_down(count)
            else
              scroll_down
            end
          end
        end
      end
    end

    defun(:bloggerpost_entrylist_read, :interactive => true) do
      entry_scroll_updown(true, 1)
    end

    defun(:bloggerpost_entrylist_scroll_down, :interactive => true) do
      entry_scroll_updown(false)
    end

    defun(:bloggerpost_entrylist_scroll_up, :interactive => true) do
      entry_scroll_updown(true)
    end

    defun(:bloggerpost_entrylist_next, :interactive => true) do
      forward_line(1)
      bloggerpost_entrylist_select_current
    end

    defun(:bloggerpost_entrylist_prev, :interactive => true) do
      forward_line(-1)
      bloggerpost_entrylist_select_current
    end

    defun(:bloggerpost_entrylist_exit, :interactive => true) do
      delete_other_windows
      bloggerpost_bloglist
    end

    defun(:bloggerpost_entrylist_edit, :interactive => true) do
      @current_draft_url = get_text_property(point, :url)
      bloggerpost_edit
    end

    defun(:bloggerpost_entrylist_update, :interactive => true) do
      with(:save_excursion) do
        let(:buffer_read_only, false) do
          erase_buffer
          n = 1
          @current_folder.entry_list.each do |entry|
            entry_line = ' ' + "%#{@current_folder.entry_list.length.to_s.length}d" % n + ' '
            entry_line << to_external_encode(format_date(entry.date)) + ' '
            entry_line << to_external_encode(entry.title)
            face = nil
            case entry.status
            when ::BloggerPost::STATUS_DELETED
              face = :bloggerpost_entry_deleted_face
            when ::BloggerPost::STATUS_MODIFIED
              face = :bloggerpost_entry_modified_face
            else
              face = :bloggerpost_entry_face
            end
            insert_with_properties(cut_string(entry_line, window_width - 1), [:url, entry.url, :face, face])
            newline
            n += 1
          end
          '' # for performance
        end
      end
      goto_line(point_max)
      forward_line(-1)
      elvar.mode_line_buffer_identification = to_external_encode(@current_folder.title) + " [#{@current_folder.name}]"
      force_mode_line_update
    end

    defun(:bloggerpost_entrylist_force_update, :interactive => true) do
      update_list
      bloggerpost_entrylist_update
    end

    def entrylist_delete_entry(entry)
      @entry_list.delete(entry)
      @current_folder.entry_list.delete(entry)
      if entry.draft?
        @draft.delete_entry(entry)
      end
      entry_list_buffer = get_buffer(ENTRYLIST_BUFFER_NAME)
      if entry_list_buffer
        with(:save_excursion) do
          set_buffer(entry_list_buffer)
          with(:save_excursion) do
            beginning_of_buffer
            loop do
              if entry.url == get_text_property(point, :url)
                from_point = point
                forward_line(1)
                end_point = point
                let(:buffer_read_only, false) do
                  delete_region(from_point, end_point)
                end
              end
              forward_line(1)
              break if eobp
            end
          end
        end
      end
    end

    defun(:bloggerpost_entrylist_delete, :interactive => true) do
      entry = find_entry(get_text_property(point, :url))
      unless entry.deleted?
        if y_or_n_p("Delete this entry? ")
          entry_list = nil
          if entry.draft?
            entrylist_delete_entry(entry)
            delete_other_windows
          else
            request = ::BloggerPost::Request.new(@config)
            message("Sending 'DELETE' request...")
            request.delete_post(entry)
            message("Sending 'DELETE' request...done")
            let(:buffer_read_only, false) do
              put_text_property(line_beginning_position, line_end_position, :face, :bloggerpost_entry_deleted_face)
            end
            ''
          end
        end
      end
    end

    defun(:bloggerpost_entrylist_sort, :interactive => 'P') do |reverse|
      entrylist_sort(reverse)
      bloggerpost_entrylist_update
    end

    defun(:bloggerpost_entrylist_search, :interactive => true) do
      entrylist_read_search_word
      bloggerpost_entrylist_update
    end

    defun(:bloggerpost_entrylist_browse_url, :interactive => true) do
      entry = find_entry(get_text_property(point, :url))
      bloggerpost_browse_url(entry.url)
    end

    def entrylist_mode_keymap
      {
        "\r"   => :bloggerpost_entrylist_read,
        " "    => :bloggerpost_entrylist_scroll_up,
        "\177" => :bloggerpost_entrylist_scroll_down,
        "n"    => :bloggerpost_entrylist_next,
        "p"    => :bloggerpost_entrylist_prev,
        "q"    => :bloggerpost_entrylist_exit,
        "s"    => :bloggerpost_entrylist_force_update,
        "e"    => :bloggerpost_entrylist_edit,
        "w"    => :bloggerpost_draft,
        "D"    => :bloggerpost_entrylist_delete,
        "S"    => :bloggerpost_entrylist_sort,
        "g"    => :bloggerpost_entrylist_search,
        "G"    => :bloggerpost_global_search,
        "b"    => :bloggerpost_entrylist_browse_url
      }
    end

    def define_entrylist_mode
      define_derived_mode(:bloggerpost_entrylist_mode, :fundamental_mode, "BloggerPost Entry List") do
        elvar.buffer_read_only = true
      end
      define_key_hash(:bloggerpost_entrylist_mode_map, entrylist_mode_keymap)
    end

    # entry

    defun(:bloggerpost_entry, :interactive => true) do
      if get_buffer(ENTRY_BUFFER_NAME)
        switch_to_buffer(ENTRY_BUFFER_NAME)
      else
        switch_to_buffer(get_buffer_create(ENTRY_BUFFER_NAME))
        bloggerpost_entry_mode
      end
      bloggerpost_entry_update
    end

    defun(:bloggerpost_entry_exit, :interactive => true) do
      delete_other_windows
      bloggerpost_entrylist
    end

    def insert_entry_header(header, content, content_face = :bloggerpost_entry_header_contents_face)
      insert_with_properties(header, [:face, :bloggerpost_entry_header_face])
      insert_with_properties(content, [:face, content_face])
      newline
    end

    defun(:bloggerpost_entry_update, :interactive => true) do
      entry = find_entry(@current_entry_url)
      with(:save_excursion) do
        let(:buffer_read_only, false) do
          erase_buffer
          prev_point = point
          insert_entry_header('Title: ', to_external_encode(entry.title), :bloggerpost_entry_title_face)
          insert_entry_header('Date: ', to_external_encode(format_date(entry.date)), :bloggerpost_entry_date_face)
          insert_entry_header('URL: ', entry.url) unless entry.draft?
          insert_entry_header('Style: ', entry.style.style_name)
          insert_entry_header('Tag: ', entry.tag_list.collect {|tag| to_external_encode(tag) }.join(', ')) unless entry.tag_list.empty?
          insert_entry_header('Category: ', entry.category_list.collect {|category| to_external_encode(category) }.join(', ')) unless entry.category_list.empty?
          insert_entry_header('Trackback: ', entry.trackback_list.collect.join(', ')) unless entry.trackback_list.empty?
          newline
          insert(entry.body)
        end
      end
      elvar.mode_line_buffer_identification = to_external_encode(entry.title) + " [#{entry.blog.blog_name}]"
      force_mode_line_update
    end

    def entry_mode_keymap
      {
        "q" => :bloggerpost_entry_exit,
        "s" => :bloggerpost_entry_update
      }
    end

    def define_entry_mode
      define_derived_mode(:bloggerpost_entry_mode, :fundamental_mode, "BloggerPost Entry") do
        elvar.buffer_read_only = true
      end
      define_key_hash(:bloggerpost_entry_mode_map, entry_mode_keymap)
    end

    # draft

    def get_draft_url
      buffer_name =~ /\/(.*)\z/
      $1
    end

    def insert_draft_header(header, content)
      insert_with_properties(header, [:face, :bloggerpost_draft_header_face, :read_only, true, :front_sticky, true, :rear_nonsticky, true])
      insert(content)
      newline
    end

    def insert_draft_separator
      insert_with_properties(BODY_SEPARATOR, [:face, :bloggerpost_draft_header_separator_face, :read_only, true, :front_sticky, true, :rear_nonsticky, true, :bloggerpost_draft_header_separator, true])
      newline
    end
    
    def make_draft_buffer(edit)
      delete_other_windows
      unless edit
        entry = ::BloggerPost::Entry.new_draft(@config)
        @entry_list << entry
        @draft.entry_list << entry
        if @current_folder.draft?
          @current_folder.entry_list << entry
        end
        @current_draft_url = entry.url
      end
      buffer_name = "#{DRAFT_BUFFER_NAME}/#{@current_draft_url}"
      if get_buffer(buffer_name)
        switch_to_buffer(buffer_name)
      else
        entry = find_entry(@current_draft_url)
        switch_to_buffer(get_buffer_create(buffer_name))
        draft_mode = DRAFT_MODE_TABLE[entry.style.style_name]
        if draft_mode && functionp(draft_mode)
          funcall(draft_mode)
        end
        bloggerpost_draft_mode
        if entry.status == entry.draft?
          insert_draft_header('Blog: ', entry.blog.blog_name)
        end
        insert_draft_header('Title: ', to_external_encode(entry.title))
        insert_draft_header('Tag: ', entry.tag_list.collect {|tag| to_external_encode(tag) }.join(', '))
        insert_draft_header('Category: ', entry.category_list.collect {|category| to_external_encode(category) }.join(', '))
        if entry.status == entry.draft?
          insert_draft_header('Trackback: ', entry.trackback_list.collect.join(', '))
        end
        insert_draft_separator
        insert(entry.body)
        beginning_of_buffer
        search_forward("Title: ", nil, true)
      end
    end

    defun(:bloggerpost_draft, :interactive => true) do
      boot
      make_draft_buffer(false)
    end

    defun(:bloggerpost_edit, :interactive => true) do
      make_draft_buffer(true)
    end

    defun(:bloggerpost_draft_exit, :interactive => true) do
      kill_buffer(current_buffer)
    end

    defun(:bloggerpost_draft_send, :interactive => true) do
      if y_or_n_p("Send current draft? ")
        bloggerpost_draft_save
        entry = find_entry(get_draft_url)
        request = ::BloggerPost::Request.new(@config)
        message("Sending 'POST' request...")
        if entry.draft?
          request.new_post(entry)
          update_list
        else
          request.edit_post(entry)
        end
        message("Sending 'POST' request...done")
        bloggerpost_draft_exit
      end
    end

    defun(:bloggerpost_draft_kill, :interactive => true) do
      entry = find_entry(get_draft_url)
      if entry.draft?
        if y_or_n_p("Kill current draft? ")
          entrylist_delete_entry(entry)
          bloggerpost_draft_exit
        end
      else
        bloggerpost_draft_exit
      end
    end

    def get_header(entry_string, header)
      entry_string =~ /^#{header}: (.*)$/
      to_internal_encode($1.strip)
    end

    def get_header_array(entry_string, header)
      entry_string =~ /^#{header}: (.*)$/
      $1.split(',').collect {|content| to_internal_encode(content.strip) }
    end

    def read_buffer_to_entry
      entry = find_entry(get_draft_url)
      entry_string = bufstr
      blog_name = nil
      trackback_list = nil
      if entry.draft?
        blog_name = get_header(entry_string, 'Blog')
        trackback_list = get_header_array(entry_string, 'Trackback')
      end
      title = get_header(entry_string, 'Title')
      tag_list = get_header_array(entry_string, 'Tag')
      category_list = get_header_array(entry_string, 'Category')
      entry.status = ::BloggerPost::STATUS_MODIFIED
      entry.title = title
      entry.tag_list = tag_list
      entry.category_list = category_list
      entry_string =~ /#{BODY_SEPARATOR}\n(.*)/m
      entry.body = $1
      entry.blog_name = blog_name if blog_name
      entry.trackback_list = trackback_list if trackback_list
      entry.clear_cache
    end

    defun(:bloggerpost_draft_save, :interactive => true) do
      if buffer_modified_p
        message("Saving...")
        entry = find_entry(get_draft_url)
        read_buffer_to_entry
        entry.save
        message("Saving...done")
        set_buffer_modified_p(nil)
      else
        message("(No changes need to be saved)")
      end
    end
    
    defun(:bloggerpost_draft_save_and_exit, :interactive => true) do
      bloggerpost_draft_save
      bloggerpost_draft_exit
    end

    defun(:bloggerpost_draft_preview, :interactive => true) do
      entry = find_entry(get_draft_url)
      read_buffer_to_entry
      File.open(@config.preview_file, 'w') do |file|
        file.write(entry.to_html)
      end
      bloggerpost_preview(@config.preview_file)
    end

    def bloggerpost_draft_header_end
      end_point = 0
      with(:save_restriction) do
        widen
        if get_text_property(point_min, :bloggerpost_draft_header_separator)
          end_point = point_min
        else
          end_point = next_single_property_change(point_min, :bloggerpost_draft_header_separator)
        end
      end
      end_point
    end

    defun(:bloggerpost_draft_beginning_of_line, :interactive => 'p') do |n|
      beginning_of_line(n)
      if point < bloggerpost_draft_header_end
        search_forward(": ", nil, true)
      end
    end

    def draft_mode_keymap
      {
        "\C-c\C-c" => :bloggerpost_draft_send,
        "\C-c\C-k" => :bloggerpost_draft_kill,
        "\C-x\C-s" => :bloggerpost_draft_save,
        "\C-c\C-z" => :bloggerpost_draft_save_and_exit,
        "\C-c\C-p" => :bloggerpost_draft_preview,
        "\C-a"     => :bloggerpost_draft_beginning_of_line
      }
    end

    def add_draft_header_keyword(keyword, content_face = :bloggerpost_draft_header_contents_face)
      font_lock_add_keywords(nil, [["^\\(#{keyword}:\\) \\(.*\\)$", [1, :bloggerpost_draft_header_face], [2, content_face]]])
    end

    def define_draft_mode
      define_minor_mode(:bloggerpost_draft_mode, 'BloggerPost Draft mode', nil, ' BloggerPost-Draft', make_minor_mode_keymap(draft_mode_keymap)) do
        add_draft_header_keyword('Blog', :bloggerpost_draft_blog_face)
        add_draft_header_keyword('Title', :bloggerpost_draft_title_face)
        add_draft_header_keyword('Tag')
        add_draft_header_keyword('Category')
        add_draft_header_keyword('Trackback')
        font_lock_add_keywords(nil, [[BODY_SEPARATOR, [0, :bloggerpost_draft_header_separator_face]]])
        '' # for performance
      end
    end

    # define modes and faces

    def define_modes
      define_bloglist_mode
      define_entrylist_mode
      define_entry_mode
      define_draft_mode
    end

    def define_faces
      el4r_lisp_eval <<FACE
(progn
  (defface bloggerpost-blog-face
    '((((class color) (background light)) (:foreground "Navy"))
      (((class color) (background dark)) (:foreground "SkyBlue")))
    "")
  (defface bloggerpost-entry-face
    '((((class color) (background light)) (:foreground "DarkOliveGreen"))
      (((class color) (background dark)) (:foreground "GreenYellow")))
    "")
  (defface bloggerpost-entry-deleted-face
    '((((class color) (background light)) (:foreground "Tomato3"))
      (((class color) (background dark)) (:foreground "tomato")))
    "")
  (defface bloggerpost-entry-modified-face
    '((((class color) (background light)) (:foreground "khaki4"))
      (((class color) (background dark)) (:foreground "khaki")))
    "")
  (defface bloggerpost-entry-header-face
    '((((class color) (background light)) (:foreground "Gray30" :bold t))
      (((class color) (background dark)) (:foreground "gray" :bold t)))
    "")
  (defface bloggerpost-entry-header-contents-face
    '((((class color) (background light)) (:foreground "purple"))
      (((class color) (background dark)) (:foreground "LightSkyBlue")))
    "")
  (defface bloggerpost-entry-title-face
    '((((class color) (background light)) (:foreground "Navy"))
      (((class color) (background dark)) (:foreground "yellow")))
    "")
  (defface bloggerpost-entry-date-face
    '((((class color) (background light)) (:foreground "DarkSlateBlue"))
      (((class color) (background dark)) (:foreground "orange")))
    "")
  (defface bloggerpost-draft-header-face
    '((((class color) (background light)) (:foreground "Gray30" :bold t))
      (((class color) (background dark)) (:foreground "gray" :bold t)))
    "")
  (defface bloggerpost-draft-header-contents-face
    '((((class color) (background light)) (:foreground "purple"))
      (((class color) (background dark)) (:foreground "LightSkyBlue")))
    "")
  (defface bloggerpost-draft-header-separator-face
    '((((class color) (background light)) (:foreground "black" :background "yellow"))
      (((class color) (background dark)) (:foreground "Black" :background "DarkKhaki")))
    "")
  (defface bloggerpost-draft-blog-face
    '((((class color) (background light)) (:foreground "DarkSlateBlue"))
      (((class color) (background dark)) (:foreground "orange")))
    "")
  (defface bloggerpost-draft-title-face
    '((((class color) (background light)) (:foreground "Navy"))
      (((class color) (background dark)) (:foreground "yellow")))
    "")
  (defvar bloggerpost-blog-face 'bloggerpost-blog-face)
  (defvar bloggerpost-entry-face 'bloggerpost-entry-face)
  (defvar bloggerpost-entry-deleted-face 'bloggerpost-entry-deleted-face)
  (defvar bloggerpost-entry-modified-face 'bloggerpost-entry-modified-face)
  (defvar bloggerpost-entry-header-face 'bloggerpost-entry-header-face)
  (defvar bloggerpost-entry-header-contents-face 'bloggerpost-entry-header-contents-face)
  (defvar bloggerpost-entry-title-face 'bloggerpost-entry-title-face)
  (defvar bloggerpost-entry-date-face 'bloggerpost-entry-date-face)
  (defvar bloggerpost-draft-header-face 'bloggerpost-draft-header-face)
  (defvar bloggerpost-draft-header-contents-face 'bloggerpost-draft-header-contents-face)
  (defvar bloggerpost-draft-header-separator-face 'bloggerpost-draft-header-separator-face)
  (defvar bloggerpost-draft-blog-face 'bloggerpost-draft-blog-face)
  (defvar bloggerpost-draft-title-face 'bloggerpost-draft-title-face)
  )
FACE
    end
  end
end
