package base;
import java.io.*;
import java.nio.charset.*;
import java.nio.*;
import java.util.*;
import java.util.logging.*;

class SortNode{
	char key;
	String value;
	SortNode(char k,String v){
		key=k;
		value=v;
	}
	SortNode(int  k,String v){
		key=(char)k;
		value=v;
	}
	public int compareTo(Object o){ return key - ((SortNode)o).key; }
	public boolean equals(Object obj){ return 0==compareTo(obj); }
}
class JisEscapeType{
	int char_length;
	byte[] escape;
	JisEscapeType(int c,byte[] b){char_length=c;escape=b;}
};

public class ConvertChar {
	// Unicode => Safe
	static SortNode[] kigou={
new SortNode( 0x00a5,"\\"),
new SortNode( 0x00a6,"|"),
new SortNode( 0x00a9,"(C)"),
new SortNode( 0x00ab,"<<"),
new SortNode( 0x00ae,"(R)"),
new SortNode( 0x00b2,"2"),
new SortNode( 0x00b3,"3"),
new SortNode( 0x00b9,"1"),
new SortNode( 0x00bb,">>"),
new SortNode( 0x00bc,"1/4"),
new SortNode( 0x00bd,"1/2"),
new SortNode( 0x00be,"3/4"),
new SortNode( 0x2028,"\r" ),
new SortNode( 0x2029,"\r" ),
new SortNode( 0x2113,"リットル"),
new SortNode( 0x2116,"No."),
new SortNode( 0x2121,"TEL"),
new SortNode( 0x2122,"TM"),
new SortNode( 0x2126,"Ω"),
new SortNode( 0x2160,"I"),
new SortNode( 0x2161,"II"),
new SortNode( 0x2162,"III"),
new SortNode( 0x2163,"IV"),
new SortNode( 0x2164,"V"),
new SortNode( 0x2165,"VI"),
new SortNode( 0x2166,"VII"),
new SortNode( 0x2167,"VIII"),
new SortNode( 0x2168,"IX"),
new SortNode( 0x2169,"X"),
new SortNode( 0x216a,"XI"),
new SortNode( 0x216b,"XII"),
new SortNode( 0x216c,"L"),
new SortNode( 0x216d,"C"),
new SortNode( 0x216e,"D"),
new SortNode( 0x216f,"M"),
new SortNode( 0x2170,"i"),
new SortNode( 0x2171,"ii"),
new SortNode( 0x2172,"iii"),
new SortNode( 0x2173,"iv"),
new SortNode( 0x2174,"v"),
new SortNode( 0x2175,"vi"),
new SortNode( 0x2176,"vii"),
new SortNode( 0x2177,"viii"),
new SortNode( 0x2178,"ix"),
new SortNode( 0x2179,"x"),
new SortNode( 0x217a,"xi"),
new SortNode( 0x217b,"xii"),
new SortNode( 0x217c,"l"),
new SortNode( 0x217d,"c"),
new SortNode( 0x217e,"d"),
new SortNode( 0x217f,"m"),
new SortNode( 0x2211,"Σ"),
new SortNode( 0x22bf,"デルタ"),
new SortNode( 0x2460,"(1)"),
new SortNode( 0x2461,"(2)"),
new SortNode( 0x2462,"(3)"),
new SortNode( 0x2463,"(4)"),
new SortNode( 0x2464,"(5)"),
new SortNode( 0x2465,"(6)"),
new SortNode( 0x2466,"(7)"),
new SortNode( 0x2467,"(8)"),
new SortNode( 0x2468,"(9)"),
new SortNode( 0x2469,"(10)"),
new SortNode( 0x246a,"(11)"),
new SortNode( 0x246b,"(12)"),
new SortNode( 0x246c,"(13)"),
new SortNode( 0x246d,"(14)"),
new SortNode( 0x246e,"(15)"),
new SortNode( 0x246f,"(16)"),
new SortNode( 0x2470,"(17)"),
new SortNode( 0x2471,"(18)"),
new SortNode( 0x2472,"(19)"),
new SortNode( 0x2473,"(20)"),
new SortNode( 0x2474,"(1)"),
new SortNode( 0x2475,"(2)"),
new SortNode( 0x2476,"(3)"),
new SortNode( 0x2477,"(4)"),
new SortNode( 0x2478,"(5)"),
new SortNode( 0x2479,"(6)"),
new SortNode( 0x247a,"(7)"),
new SortNode( 0x247b,"(8)"),
new SortNode( 0x247c,"(9)"),
new SortNode( 0x247d,"(10)"),
new SortNode( 0x247e,"(11)"),
new SortNode( 0x247f,"(12)"),
new SortNode( 0x2480,"(13)"),
new SortNode( 0x2481,"(14)"),
new SortNode( 0x2482,"(15)"),
new SortNode( 0x2483,"(16)"),
new SortNode( 0x2484,"(17)"),
new SortNode( 0x2485,"(18)"),
new SortNode( 0x2486,"(19)"),
new SortNode( 0x2487,"(20)"),
new SortNode( 0x2488,"1."),
new SortNode( 0x2489,"2."),
new SortNode( 0x248a,"3."),
new SortNode( 0x248b,"4."),
new SortNode( 0x248c,"5."),
new SortNode( 0x248d,"6."),
new SortNode( 0x248e,"7."),
new SortNode( 0x248f,"8."),
new SortNode( 0x2490,"9."),
new SortNode( 0x249c,"(a)"),
new SortNode( 0x249d,"(b)"),
new SortNode( 0x249e,"(c)"),
new SortNode( 0x249f,"(d)"),
new SortNode( 0x24a0,"(e)"),
new SortNode( 0x24a1,"(f)"),
new SortNode( 0x24a2,"(g)"),
new SortNode( 0x24a3,"(h)"),
new SortNode( 0x24a4,"(i)"),
new SortNode( 0x24a5,"(j)"),
new SortNode( 0x24a6,"(k)"),
new SortNode( 0x24a7,"(l)"),
new SortNode( 0x24a8,"(m)"),
new SortNode( 0x24a9,"(n)"),
new SortNode( 0x24aa,"(o)"),
new SortNode( 0x24ab,"(p)"),
new SortNode( 0x24ac,"(q)"),
new SortNode( 0x24ad,"(r)"),
new SortNode( 0x24ae,"(s)"),
new SortNode( 0x24af,"(t)"),
new SortNode( 0x24b0,"(u)"),
new SortNode( 0x24b1,"(v)"),
new SortNode( 0x24b2,"(w)"),
new SortNode( 0x24b3,"(x)"),
new SortNode( 0x24b4,"(y)"),
new SortNode( 0x24b5,"(z)"),
new SortNode( 0x2660,"スペード"),
new SortNode( 0x2661,"ハート"),
new SortNode( 0x2662,"ダイヤ"),
new SortNode( 0x2663,"クラブ"),
new SortNode( 0x2664,"すぺーど"),
new SortNode( 0x2665,"はーと"),
new SortNode( 0x2666,"だいや"),
new SortNode( 0x2667,"くらぶ"),
new SortNode( 0x2776,"(1)"),
new SortNode( 0x2777,"(2)"),
new SortNode( 0x2778,"(3)"),
new SortNode( 0x2779,"(4)"),
new SortNode( 0x277a,"(5)"),
new SortNode( 0x277b,"(6)"),
new SortNode( 0x277c,"(7)"),
new SortNode( 0x277d,"(8)"),
new SortNode( 0x277e,"(9)"),
new SortNode( 0x278a,"{1}"),
new SortNode( 0x278b,"{2}"),
new SortNode( 0x278c,"{3}"),
new SortNode( 0x278d,"{4}"),
new SortNode( 0x278e,"{5}"),
new SortNode( 0x278f,"{6}"),
new SortNode( 0x2790,"{7}"),
new SortNode( 0x2791,"{8}"),
new SortNode( 0x2792,"{9}"),
new SortNode( 0x3004,"(JIS)"),
new SortNode( 0x301d,"\""),
new SortNode( 0x301f,"\""),
new SortNode( 0x322a,"(月)"),
new SortNode( 0x322b,"(火)"),
new SortNode( 0x322c,"(水)"),
new SortNode( 0x322d,"(木)"),
new SortNode( 0x322e,"(金)"),
new SortNode( 0x322f,"(土)"),
new SortNode( 0x3230,"(日)"),
new SortNode( 0x3231,"(株)"),
new SortNode( 0x3232,"(有)"),
new SortNode( 0x3233,"(社)"),
new SortNode( 0x3234,"(名)"),
new SortNode( 0x3235,"(特)"),
new SortNode( 0x3236,"(財)"),
new SortNode( 0x3237,"(祝)"),
new SortNode( 0x3238,"(労)"),
new SortNode( 0x3239,"(代)"),
new SortNode( 0x323a,"(呼)"),
new SortNode( 0x323b,"(学)"),
new SortNode( 0x323c,"(監)"),
new SortNode( 0x323d,"(企)"),
new SortNode( 0x323e,"(資)"),
new SortNode( 0x323f,"(協)"),
new SortNode( 0x3240,"(祭)"),
new SortNode( 0x3242,"(自)"),
new SortNode( 0x3243,"(至)"),
new SortNode( 0x3296,"(財)"),
new SortNode( 0x3298,"(労)"),
new SortNode( 0x3299,"(秘)"),
new SortNode( 0x329d,"(優)"),
new SortNode( 0x329e,"(印)"),
new SortNode( 0x32a4,"(上)"),
new SortNode( 0x32a5,"(中)"),
new SortNode( 0x32a6,"(下)"),
new SortNode( 0x32a7,"(左)"),
new SortNode( 0x32a8,"(右)"),
new SortNode( 0x32a9,"(医)"),
new SortNode( 0x3300,"アパート"),
new SortNode( 0x3303,"アール"),
new SortNode( 0x3305,"インチ"),
new SortNode( 0x330d,"カロリー"),
new SortNode( 0x3314,"キロ"),
new SortNode( 0x3315,"キログラム"),
new SortNode( 0x3316,"キロメートル"),
new SortNode( 0x3318,"グラム"),
new SortNode( 0x331e,"コーポ"),
new SortNode( 0x3322,"センチ"),
new SortNode( 0x3323,"セント"),
new SortNode( 0x3326,"ドル"),
new SortNode( 0x3327,"トン"),
new SortNode( 0x332a,"ハイツ"),
new SortNode( 0x332b,"パーセント"),
new SortNode( 0x3331,"ビル"),
new SortNode( 0x3333,"フィート"),
new SortNode( 0x3336,"ヘクタール"),
new SortNode( 0x3339,"ヘルツ"),
new SortNode( 0x333b,"ページ"),
new SortNode( 0x3342,"ホーン"),
new SortNode( 0x3347,"マンション"),
new SortNode( 0x3349,"ミリ"),
new SortNode( 0x334a,"ミリバール"),
new SortNode( 0x334d,"メートル"),
new SortNode( 0x334e,"ヤード"),
new SortNode( 0x3351,"リットル"),
new SortNode( 0x3357,"ワット"),
new SortNode( 0x337b,"平成"),
new SortNode( 0x337c,"昭和"),
new SortNode( 0x337d,"大正"),
new SortNode( 0x337e,"明治"),
new SortNode( 0x337f,"株式会社"),
new SortNode( 0x3385,"KB"),
new SortNode( 0x3386,"MB"),
new SortNode( 0x3387,"GB"),
new SortNode( 0x338e,"mg"),
new SortNode( 0x338f,"kg"),
new SortNode( 0x3390,"Hz"),
new SortNode( 0x3396,"Ml"),
new SortNode( 0x3397,"dl"),
new SortNode( 0x3398,"kl"),
new SortNode( 0x339c,"mm"),
new SortNode( 0x339d,"cm"),
new SortNode( 0x339e,"km"),
new SortNode( 0x339f,"mm2"),
new SortNode( 0x33a0,"cm2"),
new SortNode( 0x33a1,"m2"),
new SortNode( 0x33a2,"km2"),
new SortNode( 0x33a4,"cm3"),
new SortNode( 0x33a5,"m3"),
new SortNode( 0x33b0,"ps"),
new SortNode( 0x33b1,"ms"),
new SortNode( 0x33b3,"ms"),
new SortNode( 0x33c4,"cc"),
new SortNode( 0x33cd,"KK"),
new SortNode( 0x33d4,"mb"),
new SortNode( 0xff02,"\""),
new SortNode( 0xff07,"\'"),
new SortNode( 0xff61,"。"),
new SortNode( 0xff62,"「"),
new SortNode( 0xff63,"」"),
new SortNode( 0xff64,"、"),
new SortNode( 0xff65,"・"),
new SortNode( 0xff66,"ヲ"),
new SortNode( 0xff67,"ァ"),
new SortNode( 0xff68,"ィ"),
new SortNode( 0xff69,"ゥ"),
new SortNode( 0xff6a,"ェ"),
new SortNode( 0xff6b,"ォ"),
new SortNode( 0xff6c,"ャ"),
new SortNode( 0xff6d,"ュ"),
new SortNode( 0xff6e,"ョ"),
new SortNode( 0xff6f,"ッ"),
new SortNode( 0xff70,"ー"),
new SortNode( 0xff71,"ア"),
new SortNode( 0xff72,"イ"),
new SortNode( 0xff73,"ウ"),
new SortNode( 0xff74,"エ"),
new SortNode( 0xff75,"オ"),
new SortNode( 0xff76,"カ"),
new SortNode( 0xff77,"キ"),
new SortNode( 0xff78,"ク"),
new SortNode( 0xff79,"ケ"),
new SortNode( 0xff7a,"コ"),
new SortNode( 0xff7b,"サ"),
new SortNode( 0xff7c,"シ"),
new SortNode( 0xff7d,"ス"),
new SortNode( 0xff7e,"セ"),
new SortNode( 0xff7f,"ソ"),
new SortNode( 0xff80,"タ"),
new SortNode( 0xff81,"チ"),
new SortNode( 0xff82,"ツ"),
new SortNode( 0xff83,"テ"),
new SortNode( 0xff84,"ト"),
new SortNode( 0xff85,"ナ"),
new SortNode( 0xff86,"ニ"),
new SortNode( 0xff87,"ヌ"),
new SortNode( 0xff88,"ネ"),
new SortNode( 0xff89,"ノ"),
new SortNode( 0xff8a,"ハ"),
new SortNode( 0xff8b,"ヒ"),
new SortNode( 0xff8c,"フ"),
new SortNode( 0xff8d,"ヘ"),
new SortNode( 0xff8e,"ホ"),
new SortNode( 0xff8f,"マ"),
new SortNode( 0xff90,"ミ"),
new SortNode( 0xff91,"ム"),
new SortNode( 0xff92,"メ"),
new SortNode( 0xff93,"モ"),
new SortNode( 0xff94,"ヤ"),
new SortNode( 0xff95,"ユ"),
new SortNode( 0xff96,"ヨ"),
new SortNode( 0xff97,"ラ"),
new SortNode( 0xff98,"リ"),
new SortNode( 0xff99,"ル"),
new SortNode( 0xff9a,"レ"),
new SortNode( 0xff9b,"ロ"),
new SortNode( 0xff9c,"ワ"),
new SortNode( 0xff9d,"ン"),
new SortNode( 0xff9e,"゛"),
new SortNode( 0xff9f,"゜"),
new SortNode( 0xffe4,"|"),
new SortNode( 0xfffd,"?"),
};

	public static String FindNode(SortNode[] table,char key){
		int low=0;
		int width = table.length;
		int mid;
		int r;
		while(width>=1){
			mid = (width>>1);
			SortNode sn = table[low + mid];
			if(0==(r = key - sn.key )){
				return sn.value;
			}
			if(r<0){
				width = mid;
			}else{
				low += ++mid;
				width -= mid;
			}
		}
		return null;
	}

	// Unicode(JIS) => Unicode(MS932 or system default)
	// AWT heavy component でも画面に表示できるようにするための変換
	static char[][] ary_ToDisplay={
		{/* \  */ 0x005C, 0x00A5},
		{/* ~  */ 0x007E, 0x203E},
		{/* ￠ */ 0x00A2, 0xFFE0},
		{/* ￡ */ 0x00A3, 0xFFE1},
		{/* ￢ */ 0x00AC, 0xFFE2},
		{/* ― */ 0x2015, 0x2014},
		{/* ∥ */ 0x2016, 0x2225},
		{/* … */ 0x2026, 0x22EF},
		{/* － */ 0x2212, 0xFF0D},
		{/* ～ */ 0x301C, 0xFF5E},
	};


	static final Object[][] encode_escapes=new Object[][]{
		{  null   ,new byte[]{ 0x1b,'(','B'}},
		{"JIS0208",new byte[]{ 0x1b,'$','B'}},
//出力だけ対応しても仕方ない {"JIS0212",new byte[]{ 0x1b,'$','(','D'}},
		{"JIS0201",null},
	};

	private static ByteBuffer decodejis_bb= ByteBuffer.allocate(16);

	static final byte[] kana_escapesJ = new byte[]{0x1b,'(','J'};
	static final byte[] kana_escapesI = new byte[]{0x1b,'(','I'};
	static int kana_escapes_type = 0;
	static final byte[] nojis_result =new byte[]{'?'};
	static ByteArrayOutputStream UnicodeToJIS_bao;

	public static java.util.logging.Logger logger = java.util.logging.Logger.getLogger("base.convertchar");

	// Unicode(MS932) => Unicode(JIS)
	// JISに変換できる状態にするための前変換
	static char[][] aryToJIS;
	static int FirstCall =0;
	static CharsetDecoder base_decorder;
	static CharsetDecoder jis212_decorder;
	static CharsetDecoder x201_decorder;
	static String SJIS = null;
	static String JIS=null;

	// エンティティをデコードするかどうか
	public static boolean decode_NumericCharacterReferences = false;
	// エンティティのデコードが可能かどうかはフォントによって異なる
	static java.awt.Font font=new java.awt.Font("Dialog",0,12);

	// 初期化と環境の調査
	public static void Setup(){
		if(FirstCall==0){
			FirstCall=1;

			char[] c= new char[1];
			String s;
			String s2;
			byte[] b;
			boolean a1,a2;

			// コンバータの有無を確認する
			// アプレットはプロパティを読めない SysEnc= System.getProperty( "file.encoding" );

			// decorder for Shift JIS
			{
				String[] ary_sjis_enc={
					 "MacTEC"
					,"MS932"
					,"SJIS"
				};
				for(int i=0;i<ary_sjis_enc.length;++i){
					try{
						s="ゔ"; // たぶんMacTECしか引っかからない
						String c2 = new String(s.getBytes(ary_sjis_enc[i]),ary_sjis_enc[i]);
						if(!s.equals(c2)) continue;
						SJIS = ary_sjis_enc[i];
						break;
					}catch(UnsupportedEncodingException e){
					}
				}
				if(SJIS==null){
					for(int i=0;i<ary_sjis_enc.length;++i){
						try{
							s="ヴ"; // さっきよりは緩い
							String c2 = new String(s.getBytes(ary_sjis_enc[i]),ary_sjis_enc[i]);
							if(!s.equals(c2)) continue;
							SJIS = ary_sjis_enc[i];
							break;
						}catch(UnsupportedEncodingException e){
						}
					}
				}
				SJIS=null;
				if(SJIS!=null) logger.info("sjis encoding="+SJIS);
			}

			// decorder for ISO-2022-JP
			{
				String[] ary_jis_enc={
					"JIS",
					"ISO2022JP",
				};
				for(int i=0;i<ary_jis_enc.length;++i){
					try{
						s="日本語";
						String c2 = new String(s.getBytes(ary_jis_enc[i]),ary_jis_enc[i]);
						JIS=ary_jis_enc[i];
						break;
					}catch(UnsupportedEncodingException e){}
				}
				if(JIS!=null) logger.info("jis encoding="+JIS);
				else JIS="UTF-8";
			}


			// decorder for other
			try{base_decorder = Charset.forName("ISO-8859-1").newDecoder();}catch(Throwable e){}
			try{x201_decorder = Charset.forName("JIS0201").newDecoder();}catch(Throwable e){}
	//このコンバータは出力専用らしい try{jis212_decorder = Charset.forName("JIS0212").newDecoder();}catch(Throwable e){}

			// aryToJISを組み立て、各文字について変換の要不要を検討する
			StringBuffer log_sb = new StringBuffer();
			log_sb.append("ToJIS:");

			aryToJIS = new char[ary_ToDisplay.length][2];
			for(int i=0;i<aryToJIS.length;++i){
				aryToJIS[i][0] =ary_ToDisplay[i][1];
				aryToJIS[i][1] =ary_ToDisplay[i][0];
				a1=a2=false;
				// in 
				try{
					c[0]=aryToJIS[i][0];
					s = new String(c);
					a1= s.equals(new String(s.getBytes(JIS),JIS));
				}catch(UnsupportedEncodingException e){}
				// out 
				try{
					c[0]=aryToJIS[i][1];
					s = new String(c);
					a2= s.equals(new String(s.getBytes(JIS),JIS));
				}catch(UnsupportedEncodingException e){}
				if(a1 == a2 ){
					// 両方とも使えるか両方とも使えない
					if(!a1) log_sb.append(" 0x"
						+Integer.toHexString(aryToJIS[i][0])
						+"&0x" 
						+Integer.toHexString(aryToJIS[i][1])
						+"=>X"
					);
					// 変換しない
					aryToJIS[i][0] =0;
				}else{
					if(!a2){
						// 逆方向に変換するべき
						c[0]=aryToJIS[i][0];
						aryToJIS[i][0]=aryToJIS[i][1];
						aryToJIS[i][1]=c[0];
					}
					log_sb.append(" 0x"
						+Integer.toHexString(aryToJIS[i][0])
						+"=>0x" 
						+Integer.toHexString(aryToJIS[i][1])
					);
				}
			}
			logger.info(log_sb.toString());

			// ary_ToDisplay の各文字について、
			// 有効かどうか確認する
			log_sb = new StringBuffer();
			log_sb.append("ToDisplay:");
			for(int i=0;i<ary_ToDisplay.length;++i){
				a1=a2=false;
				// in 
				{
					c[0]=ary_ToDisplay[i][0];
					s = new String(c);
					a1= s.equals(new String(s.getBytes()));
				}
				// out 
				{
					c[0]=ary_ToDisplay[i][1];
					s = new String(c);
					a2= s.equals(new String(s.getBytes()));
				}

				if(a1 == a2 ){
					// 両方とも使えるか両方とも使えないなら、変換しない
					ary_ToDisplay[i][0] =0;
				}else{
					if(!a2){
						// 逆方向に変換するべき
						c[0]=ary_ToDisplay[i][0];
						ary_ToDisplay[i][0]=ary_ToDisplay[i][1];
						ary_ToDisplay[i][1]=c[0];
					}
					log_sb.append(" 0x"
						+Integer.toHexString(ary_ToDisplay[i][0])
						+"=>0x"
						+Integer.toHexString(ary_ToDisplay[i][1])
					);
				}
			}
			logger.info(log_sb.toString());
		}
	//	System.err.println(JISToUnicode(UnicodeToJIS("のえ仨のえぷう",0,-1)));
	//	System.exit(0);
	}

	// 画面に表示できる状態にする
	public static String convertForDisplay(String from){
		StringBuffer dst = new StringBuffer();
		FromLoop: for(int i=0;i<from.length();++i){
			char f=from.charAt(i);
			if(f!=0){
				// JISをそのまま表示できない環境(たぶんMS932)のための変換
				// (表示できる文字は表のキーが0になっててスキップされるはず)
				for(int j=0;j<ary_ToDisplay.length;++j){
					if(f!=ary_ToDisplay[j][0]) continue;
					dst.append(ary_ToDisplay[j][1]);
					continue FromLoop;
				}
			}
			// 機種依存文字を無害化する
			String s = FindNode(kigou,f);
			if(s!=null) dst.append(s);
			else dst.append(f);
		}
		return new String(dst);
	}

	public static String HexDump(byte[] ba,int start,int end){
		StringBuffer tmp = new StringBuffer();
		char[] a=new char[1];
		for(int i=start;i<end;++i){
			byte c= ba[i];
			if(c >=0x20 && c <=0x7e && c != '%' ){
				tmp.append((char)c);
			}else{
				tmp.append('%');
				String Hex=Integer.toHexString(c+256);
				tmp.append(Hex.substring(Hex.length()-2));
			}
		}
		return new String(tmp);
	}

	// JIS1面の変換
	private static final byte[][] jis1_escapes= new byte[][]{
		new byte[]{ 0x1b,'$','B'}, // JIS X0208('83) 
		new byte[]{ 0x1b,'$','@'},// ESC $@ JIS X 0213の1面(include JIS X0208('78)) 
		new byte[]{ 0x1b,'$','(','O'}, // JIS X 0213の1面(include JIS X0212?)
	};
	private final static void decodeJIS1(StringBuffer sb,byte b1,byte b2){
		String badsjis=null;
		// sjisでの変換を試みる
		if(SJIS!=null){
			int k = (b1&255)-0x20;
			int t = (b2&255)-0x20;

			decodejis_bb.clear();
			decodejis_bb.put((byte)( (k + ((k<63)?0x101:0x181))>>1 ));
			decodejis_bb.put((byte)( (0==(k&1))?(t+0x9e):((t>=64)?(t+0x40):(t+0x3f)) ));
			decodejis_bb.flip();
			try{
				sb.append(new String(decodejis_bb.array(),0,decodejis_bb.remaining(),SJIS));
				return;
			}catch(Throwable e){
				String hex = "0000"+Integer.toString( (b1&255)*256+(b2&255));
				badsjis = hex.substring(hex.length()-4);
			}
		}
		if(JIS !=null){
			// JISで変換
			for(int index=0;index<jis1_escapes.length;++index){
				decodejis_bb.clear();
				decodejis_bb.put(jis1_escapes[index]);
				decodejis_bb.put(b1);
				decodejis_bb.put(b2);
				decodejis_bb.flip();
				try{
					String s = new String(decodejis_bb.array(),0,decodejis_bb.remaining(),JIS);
					if(s!=null && s.length()>0){
						sb.append(s);
						return;
					}
				}catch(Throwable e){}
			}
		}
		if(badsjis!=null) sb.append( "(sjis:%"+badsjis+")");
		else sb.append( "(jis1-%"+Integer.toHexString(b1&255)+",%"+Integer.toHexString(b2&255)+")");
	}

	private static final byte[] jis2_escape=new byte[]{ 0x1b,'$','(','P'};
	private final static void decodeJIS2(StringBuffer sb,byte b1,byte b2){
		String badsjis=null;
		// sjisでの変換を試みる
		if(SJIS!=null){
			int k = (b1&255)-0x20;
			int t = (b2&255)-0x20;

			decodejis_bb.clear();
			decodejis_bb.put((byte)( (k>=78)?((k + 0x19b)>>1):(((k+0x1df)>>1)-(k>>3)*3) ));
			decodejis_bb.put((byte)( (0==(k&1))?(t+0x9e):((t>=64)?(t+0x40):(t+0x3f))    ));
			decodejis_bb.flip();

			try{
				sb.append(new String(decodejis_bb.array(),0,decodejis_bb.remaining(),SJIS));
				return;
			}catch(Throwable e){
				String hex = "0000"+Integer.toString( (b1&255)*256+(b2&255));
				badsjis = hex.substring(hex.length()-4);
			}
		}
		// JISで変換
		if(JIS !=null){
			decodejis_bb.clear();
			decodejis_bb.put(jis2_escape);
			decodejis_bb.put(b1);
			decodejis_bb.put(b2);
			decodejis_bb.flip();
			try{
				String s = new String(decodejis_bb.array(),0,decodejis_bb.remaining(),JIS);
				if(s!=null && s.length()>0){
					sb.append(s);
					return;
				}
			}catch(Throwable e){}
		}
		if(badsjis!=null) sb.append( "(sjis:%"+badsjis+")");
		else sb.append( "(jis2-%"+Integer.toHexString(b1&255)+"%"+Integer.toHexString(b2&255)+")");
	}

	private final static void decodeX201( StringBuffer sb,byte b,char mode,boolean shift_out ){
		// まじめにmodeとshift_outを見て判定するべきなんだけど、
		// 微妙に動作が変だったので適当です

		if(b<0){
			// 8bit目が立っているのでX201カナだとわかる
		}else{
			if( (mode=='J' && shift_out ) // ESC (J + shiftout +7bit
			||  mode=='I'                // ESC (I + 7bit
			){
				// これは -128して問題ない
			}else{
				// 知らないタイプなのでログに記録する
				logger.finer("mode="+mode+" shiftout="+shift_out+" code=0x"+Integer.toHexString(b&255));
			}
			// いずれにせよ -128しておく。 IRCでX201の前半を使う人っているのか？
			b-=128;
		}

		if(x201_decorder!=null){
			decodejis_bb.clear();
			decodejis_bb.put( (byte)(b<0?b:b-128) );
			decodejis_bb.flip();
			try{
				CharBuffer cb = x201_decorder.decode(decodejis_bb);
				sb.append( cb.array(),cb.position(),cb.remaining());
				return;
			}catch(Throwable e){}
		}

		String hex = "00"+Integer.toHexString(b&255);
		hex=hex.substring(hex.length()-2);
		sb.append( "(x201%"+hex+" )" );
	}

	private static byte[] jis212_escape=new byte[]{ 0x1b,'$','D'};
	static  byte[] src212 = new byte[2];
	private final static void decodeJIS212(StringBuffer sb,byte b1,byte b2){
		// JISで変換
		if(jis212_decorder !=null){
			decodejis_bb.clear();
			decodejis_bb.put(jis212_escape);
			decodejis_bb.put(b1);
			decodejis_bb.put(b2);
			decodejis_bb.flip();
			try{
				CharBuffer cb = jis212_decorder.decode(decodejis_bb);
				String s = new String( cb.array(),cb.position(),cb.remaining());
				if(s!=null && s.length()>0){
					// System.err.println("x212 "+s);
					sb.append(s);
					return;
				}
			}catch(Throwable e){}
		}
		sb.append( "(jis212%"+Integer.toHexString(b1&255)+"%"+Integer.toHexString(b2&255)+")");
	}

	public static String JISToUnicode(byte[] ba){ return JISToUnicode(ba,0,ba.length);}

	public static String JISToUnicode(byte[] ba,int arg_start,int end){
		int i=arg_start;
		StringBuffer sb = new StringBuffer();
		LOOP: while(i<end){
			// ESC が登場するまではbase_decorder でデコード
			int start=i; while( i<end && ba[i]!=0x1B ) ++i;
			if(i>start){
				try{
					CharBuffer cb = base_decorder.decode(ByteBuffer.wrap(ba,start,i-start));
					sb.append( cb.array(),cb.position(),cb.remaining());
				}catch(Throwable e){
					logger.log(Level.SEVERE,HexDump(ba,start,i),e);
					for(int j=start;j<i;++j) sb.append( ba[j]&255 );
				}
			}

			int maxescape = end - 3;
			int page = 0;
			for(;i<end;++i){
				switch(ba[i]){
				// 空白なら必ずASCIIに戻す
				case 0x20: page= 0; continue LOOP;

				case 0x0E: // shift out
					if(page==72010 ){ page=72011; continue; }
					if(page==82010 ){ page=82011; continue; }
					page=0;
					continue LOOP;

				case 0x0F: // shift in
					if(page==72011){ page=72010; continue; }
					if(page==82011){ page=82010; continue; }
					page=0;
					continue LOOP;

				case 0x1B:
					if(i<= maxescape){
						switch(ba[i+1]){
						case '(':
							switch(ba[i+2]){
							case 'B':page=    0;i+=2; continue;	// ASCII ESC (B アスキー
							case 'I':page=72010;i+=2; continue;	// JIS X 0201 片仮名 7ビット半角カナの開始
							case 'J':page=82010;i+=2; continue;	// JIS X0201(LH) ESC (J 半角カナ
							}
							break;
						case '$':
							switch(ba[i+2]){
							case '@': page=2131; i+=2; continue;	// ESC $@ JIS X 0213の1面(include JIS X0208('78)) 
							case 'B': page= 208; i+=2; continue;	// JIS X0208('83) 
							case 'I': page= 201; i+=2; continue;	// 8ビット半角カナの開始
							case '(':
								if(i<maxescape)
								switch(ba[i+3]){
								case 'O': page=2131; i+=3; continue;	// JIS X 0213の1面(include JIS X0212?)
								case 'P': page=2132; i+=3; continue;	// JIS X 0213の2面
								case 'D': page=212;  i+=3; continue;	// JIS X0212  補助漢字
								}
								break;
							}
							break;
						}
						logger.warning(HexDump(ba,i,i+3));
					}
					sb.append(ba[i++]);
					// どれでもないなら一旦エスケープを解除する
					continue LOOP;
				}
				switch(page){
				default: continue LOOP;
				// 半角カナ
				case 201  : decodeX201(sb,ba[i],' ',false); continue;
				// roman ESC (J
				case 82010: decodeX201(sb,ba[i],'J',false); continue;
				case 82011: decodeX201(sb,ba[i],'J',true ); continue;
				// junet ESC (I
				case 72010: decodeX201(sb,ba[i],'I',false); continue;
				case 72011: decodeX201(sb,ba[i],'I',true ); continue;

				case 2131: // 0213-1 いわゆるJIS漢字
				case  208: // 0213-1 は 0208の上位互換
					if(i==end-1 || ba[i] <= 0x20 || ba[i] >= 0x7f ) continue LOOP;
					decodeJIS1(sb,ba[i],ba[i+1]);
					++i;continue;
				case 2132: // 0213-2 補助漢字
					if(i==end-1 || ba[i] <= 0x20 || ba[i] >= 0x7f ) continue LOOP;
					decodeJIS2(sb,ba[i],ba[i+1]);
					++i;continue;
				case 212: // sjis には 0212 はない
					if(i==end-1 || ba[i] <= 0x20 || ba[i] >= 0x7f ) continue LOOP;
					decodeJIS212(sb,ba[i],ba[i+1]);
					++i;continue;
				}
			}
		}
		String src = sb.toString();
		if(decode_NumericCharacterReferences && -1== src.indexOf("&#") ){
			sb= new StringBuffer();
			// &#H; &#HHHH; をunicodeに変換する
			char entity=0;
			int e_start;
			int e_end=0;
			for(int start=0;start<src.length();){
				e_start=start;
				CHECK: while( e_start<src.length() ){
					e_start=src.indexOf("&#",e_start);
					if(e_start==-1){ e_end=e_start=src.length(); break; }
					int n_start= e_start+2;
					int radix=10;
					int maxlen = 5;
					e_end = src.indexOf(';',n_start);
					if(e_end==-1 ){ e_end=e_start=src.length(); break; }

					if(-1!= "xX".indexOf(src.charAt(n_start))){
						++n_start;
						radix=16;
						maxlen = 4;
					}

					if( e_end-n_start<1 || e_end-n_start >maxlen)
					{ e_start=e_end+1; continue CHECK; }

					// check
					boolean flag=true;
					for(i=n_start;i<e_end;++i){
						char c = src.charAt(i);
						if( !( c>='a' && c<='f')
						&&  !( c>='A' && c<='F')
						&&  !( c>='0' && c<='9')
						){ e_end=e_start=i; continue CHECK; }
					}
					try{
						logger.finer("decode entity "+src.substring(n_start,e_end)+" radix="+radix);
						entity = (char)Integer.parseInt(src.substring(n_start,e_end),radix);
					}catch(Throwable e){ e_start=e_end+1; continue CHECK; }
					break;
				}
				if(e_start > start){
					sb.append(src.substring(start,e_start));
					start=e_start;
				}
				if(e_end>e_start){
					start=e_end+1;
					// entity を処理する…
					// ms932の間違った割り当てを直す
					// (表示できる文字は表のキーが0になっててスキップされるはず)
					if(entity!=0){
						for(i=0;i<aryToJIS.length;++i){
							if(entity!=aryToJIS[i][0]) continue;
							entity= aryToJIS[i][1];
							break;
						}
					}
				//	// 機種依存文字を無害化する
				//	String s=FindNode(kigou,entity);
				//	if(s!=null){ sb.append(s); }else

					{
						// 表示できる文字なのか知りたい
						if(!font.canDisplay(entity))
						{
							// 現在のデフォルトエンコーディングでは表現できない
							logger.fine(src.substring(e_start,start)+"は現在の環境では表示できない");
							sb.append(src.substring(e_start,start));
						}else{
							sb.append(entity);
						}
					}
				}
			}
			src= sb.toString();
		}
		if(src.indexOf(0xa4) !=-1){
			logger.info( HexDump(ba,arg_start,end));
		}
		return src;
	}

	public static byte[] UnicodeToJIS(String src,int start,int end){
		if(end<0) end=src.length();
		char[] tmp = new char[1];
		ByteArrayOutputStream bao = UnicodeToJIS_bao;
		int mode = 0;
		boolean need_shift_in=false;
		try{
			if(bao==null) bao = UnicodeToJIS_bao = new ByteArrayOutputStream();
			bao.reset();
			for(int i=start;i<end;++i){
				char c = src.charAt(i);
				String s=null;
				// ms932の間違った割り当てを直す
				// (表示できる文字は表のキーが0になっててスキップされるはず)
				if(c!=0){
					for(int j=0;j<aryToJIS.length;++j){
						if(c!=aryToJIS[j][0]) continue;
						c= aryToJIS[j][1];
						break;
					}
				}
				// 機種依存文字を無害化する
				s = FindNode(kigou,c);
				int max =(s!=null?s.length():1);
				for(int j=0;j<max;++j){
					if(s!=null) c = s.charAt(j);
					// x201の混乱を回避する
					switch(c){
					case 0xa5  : c= 0x5c; break;
					case 0x203e: c= 0x7e; break;
					}
					// ASCIIの範囲は変換不要
					if( c < 0x7f ){
						if(mode != 0 ){
							if(need_shift_in){need_shift_in=false;bao.write(0x0f);}
							bao.write((byte[]) encode_escapes[mode=0][1]);
						}
						bao.write((byte)c);
						continue;
					}
					// JISに変換する
					tmp[0]=c;
					int new_mode=-1;
					byte[] encode_result=nojis_result;
					for(int ie=0;ie<encode_escapes.length;++ie){
						if(encode_escapes[ie][0]==null) continue;
						try{
							byte[] b = new String(tmp).getBytes((String)encode_escapes[ie][0]);
							if(b==null||(b.length==1 && b[0]=='?')) continue;
							new_mode =ie;
							encode_result=b;
							break;
						}catch(UnsupportedEncodingException e){}
					}
					// 変換できなかったら数値文字参照にする
					if( new_mode==-1 ){
						new_mode=0;
						String hex = "&#"+Integer.toString((int)c)+";";
						try{ encode_result = (hex).getBytes("UTF-8"); }
						catch(UnsupportedEncodingException e){
							logger.severe("encode entity failed");
							encode_result=nojis_result;
						}
					}
					byte[] new_escape=(byte[])encode_escapes[new_mode][1];
					boolean new_need_shift_in = false;
					// 半角カナの場合
					if(new_escape==null){
						switch(kana_escapes_type){
						case 0: new_mode=2010;new_escape=kana_escapesJ;new_need_shift_in=true;break;
						case 1: new_mode=2011;new_escape=kana_escapesJ;break;
						case 2: new_mode=2012;new_escape=kana_escapesI;break;
						case 3: new_mode=2013;new_escape=kana_escapesI;break;
						}
					}
					// エスケープが異なるならエスケープを出力
					if(mode!=new_mode){
						mode=new_mode;
						if(need_shift_in){need_shift_in=false;bao.write(0x0f);}
						bao.write(new_escape);
						need_shift_in = new_need_shift_in;
						if(need_shift_in) bao.write(0x0e);
					}
					// 文字を出力
					switch(new_mode){
					case 2010:case 2012:
						bao.write(encode_result[0]&127);break;
					case 2011:case 2013:
					default:
						bao.write(encode_result);break;
					}
				}
			}
			// エスケープを閉じる
			if(mode != 0 ){
				if(need_shift_in){need_shift_in=false;bao.write(0x0f);}
				bao.write((byte[]) encode_escapes[0][1]);
			}
			return bao.toByteArray();
		}catch(IOException e){
			logger.log(java.util.logging.Level.WARNING,"UnicodeToJIS",e);
			return new byte[0];
		}
	}
}

/*

JIS X 0201 については
http://www2d.biglobe.ne.jp/~msyk/charcode/jisx0201kana/
を参照

CTCP文字装飾の例
	%03 01  // color
	%1b$B?';XDj%1b(B
	%02 //bold
	%1b$B %25\!<%25k%25I %1b(B
	%16 // 反転
	%1b$B H?E> %1b(B
	%1f // アンダーライン
	%1b$B %25"%25s%25@!<%25i%25$%25s%1b(B
	%0f //CTCP 打ち消し
	%1b$B BG$A>C$7 %1b(B

CTCP文字装飾とSIの絡み
ﾊﾝｶｸ<O>ABC日本語ABC<O>ﾊﾝｶｸ
roman 7bit
	%1b(J %0e J]68 %0f
	%1b(B %0f ABC
	%1b$B F|K\8l
	%1b(B ABC %0f 
	%1b(J %0eJ]68%0f
	%1b(B

roman 8bit
	%1b(J %ca%dd%b6%b8
	%1b(B %0f ABC
	%1b$B F|K\8l
	%1b(B ABC %0f 
	%1b(J %ca%dd%b6%b8
	%1b(B

junet 7bit
	%1b(I J]68
	%1b(B %0f ABC
	%1b$B F|K\8l 
	%1b(B ABC %0f 
	%1b(I J]68
	%1b(B

junet 8bit
	%1b(I %ca%dd%b6%b8 
	%1b(B %0f ABC
	%1b$B F|K\8l 
	%1b(B ABC %0f 
	%1b(I %ca%dd%b6%b8 
	%1b(B

*/
