// UTF-8 ☀☁☂☃
package bluntirc;
import base.*;
import gui.*;
import buffer.*;
import irc.*;

import java.awt.*;
import java.awt.event.*;
import java.util.*;

/*

バッファテキスト書式 

-------------------------------------------------------
メッセージはまずIRCMessageで要素に分解され、
次に PrintFormat.confで設定した書式にあわせてフォーマットされる。

		%hoge  スタイルとオーバライド許可がある
		%#fff; 以後の色の指定。%hogeをオーバライドする。
		%#;    色の指定を解除する
		%b     bold       二度目の登場で解除 Font.BOLD =1
		%i     イタリック 二度目の登場で解除 Font.ITALIC=2
		%w     折り返し指定
		その他普通の文字

このとき書式文字列中のマクロごとにフォントや色の指定を行える。
これはStyle.conf で定義される。

次にマクロの種類によってはCTCPの文字装飾をパースする。
それ以外の場合もキーワード判定は行われる。
次にURLを取り出す。
最後にキーワード判定を行い、マッチしたキーワードにあわせたスタイルが乗る。

これらの複数のスタイルは以下の優先順位で処理される。
	キーワード強調のスタイル
	CTCPの装飾（色を除く)
	URLのスタイル
	CTCPの色指定
	PFの %#fff; で指定した色
	PFのスタイル

-------------------------------------------------------
ログ表示モジュールには制限がある。
フォント名などの属性を記録するハッシュは256エントリまでしか記録できない。
足りなくなったらLRUの古いものから消去されて、
その結果ふるいキーに対して全く異なる値が返ってくる。
普段から256種類もの属性が使用されるとは思えないが。
*/


class DecorationStack{
	/////////////////////////////
	// 現在かかっている装飾
	String nPFStyle;
	String nURLStyle;
	String nKWStyle;
	boolean nCTCPBold    ;
	boolean nCTCPReverse ;
	boolean nCTCPUnder   ;
	Color  nPFColor;
	int nCTCPColor1 =-1;
	int nCTCPColor2 =-1;

	boolean[] pfm_extra = new boolean[5];
	/////////////////////////////
	// 装飾を優先度の低いものから辿る

	boolean allowOverride;
	String base_style;
	int option_style;
	int ctcp_color1;
	int ctcp_color2;
	Color pfm_color;
	FontInfo2 last_base;

	private void applyBaseStyle(String name){
		if(!allowOverride || name==null) return;
		base_style=name;
		ctcp_color1=-1;
		ctcp_color2=-1;
		pfm_color=null;
		last_base= App.style_manager.findTextStyle(base_style);
		option_style = last_base.getFontStyle();
		allowOverride= last_base.canOverride() ;
	}
	PackedStyle solve(){
		allowOverride=true;
		applyBaseStyle("base");
		// PFM macro style
		applyBaseStyle(nPFStyle);
		// PrintFormat color macro
		if(allowOverride && nPFColor!=null ) pfm_color = nPFColor;
		if(allowOverride && pfm_extra[0] ) option_style |=FontInfo2.BOLD     ;
		if(allowOverride && pfm_extra[1] ) option_style |=FontInfo2.ITALIC   ;
		if(allowOverride && pfm_extra[2] ) option_style |=FontInfo2.UNDERLINE;
		if(allowOverride && pfm_extra[3] ) option_style |=FontInfo2.REVERSE  ;
		if(allowOverride && pfm_extra[4] ) option_style |=FontInfo2.STRIKE   ;
		// PrintFormat color macro
		if(allowOverride && nCTCPColor1 != -1 ){
			ctcp_color1=nCTCPColor1;
			ctcp_color2=nCTCPColor2;
			pfm_color=null;
		}
		// clickable URL
		applyBaseStyle(nURLStyle);
		// CTCP文字装飾
		if(allowOverride && nCTCPBold    ) option_style |=FontInfo2.BOLD;
		if(allowOverride && nCTCPReverse ) option_style |=FontInfo2.REVERSE;
		if(allowOverride && nCTCPUnder   ) option_style |=FontInfo2.UNDERLINE;
		// keyword 強調
		applyBaseStyle(nKWStyle);
		/////////////
		// ここまでの結果を StyleID で圧縮して PackedSpanStyleに格納する
		PackedStyle ps =new PackedStyle(sid,base_style);
		if(last_base.getFontStyle() !=  option_style){
			ps.optionid[StyleID.OPTION_STYLE] = sid.setFontStyle(option_style);
		}
		if(ctcp_color1 !=-1 ){
			if(ctcp_color2<0){
				ps.optionid[StyleID.OPTION_CTCPCOLOR1] = sid.setCTCPColor1(ctcp_color1);
			}else{
				ps.optionid[StyleID.OPTION_CTCPCOLOR2] = sid.setCTCPColor2(ctcp_color1,ctcp_color2);
			}
		//	App.logger.finer("packedstyle c1="+c1+" c2="+c2);
		}
		if(pfm_color !=null){
			ps.optionid[StyleID.OPTION_PFMCOLOR] = sid.setPFMColor(pfm_color);
		}
		return ps;
	}

	/////////////////////////////////////////////
	StyleID sid;
	PackedLineWriter tsw;
	Vector string=new Vector();
	void flush_string(){
		if(string.size()==0) return;
		tsw.appendText(string,0,string.size()-1,solve());
		string.clear();
	}
	DecorationStack(StyleID sid){
		this.sid=sid;
		tsw=new PackedLineWriter(sid,"base");
	}
	DecorationStack(int dummy){}
	PackedLineWriter getTextLine(){
		if(string.size()!=0) flush_string();
		return tsw;
	}
	//////////////////////////////////////////////////
	// 順にadd して最後にgetTextLine する
	void add(String o){ string.add(o); }
	void addPFStyle (String s){ flush_string();nPFStyle =(s==null?null:s); }
	void addURLStyle(String s){ flush_string();nURLStyle=(s==null?null:s); }
	void addKWStyle (String s){ flush_string();nKWStyle =(s==null?null:s); }
	void addCTCPBold   (boolean b){flush_string();nCTCPBold   =b;}
	void addCTCPReverse(boolean b){flush_string();nCTCPReverse=b;}
	void addCTCPUnder  (boolean b){flush_string();nCTCPUnder  =b;}
	void addCTCPColor(int c1,int c2){

		 flush_string();nCTCPColor1=c1;nCTCPColor2=c2;}
	void addPFExtra(int id,Object o){
		flush_string();
		switch(id){
		case PrintFormatMacro.SAVE_WRAP:
			tsw.appendSaveIndent(solve());
			break;
		case PrintFormatMacro.COLOR:
			nPFColor    =(Color)o;
			break;
		case FontInfo2.BOLD     : pfm_extra[0]=pfm_extra[0];break;
		case FontInfo2.ITALIC   : pfm_extra[1]=pfm_extra[1];break;
		case FontInfo2.UNDERLINE: pfm_extra[2]=pfm_extra[2];break;
		case FontInfo2.REVERSE  : pfm_extra[3]=pfm_extra[3];break;
		case FontInfo2.STRIKE   : pfm_extra[4]=pfm_extra[4];break;
		}
	}
}

// 文字のみを取り出したい場合
class DecorationStack2 extends DecorationStack{
	DecorationStack2(){ super(-1); }
	void flush_string(){}
	String getText(){
		StringBuffer sb=new StringBuffer();
		for(Iterator it=string.iterator();it.hasNext();){
			sb.append((String)it.next());
		}
		return sb.toString();
	}
	// 順にadd して最後にgetTextLine する
	void add(String o){ string.add(o); }
	void addPFStyleBase(){}
	void addPFStyle (String s){}
	void addURLStyle(String s){}
	void addKWStyle (String s){}
	void addCTCPBold   (boolean b){}
	void addCTCPReverse(boolean b){}
	void addCTCPUnder  (boolean b){}
	void addCTCPColor(int c1,int c2){}
	void addPFExtra(int id,Object o){}
}

public class PrintFormatToTextSpan{
	// メッセージと出力先とPrintFormatからテキストを得る
	public static String getText(
		IRCMessage m,Object to
		,PrintFormat pf
		,boolean unshiftTime
	){
		if(pf.NoDisp) return null;
		DecorationStack2 ary = new DecorationStack2();
		read(ary,m,to,pf,unshiftTime,true);
		return ary.getText();
	}
	// メッセージと出力先とPrintFormatから
	// TextBlockに書き込める状態のPackedLineWriterを用意する
	public static PackedLineWriter getTextLine(
		IRCMessage m,Object to
		,PrintFormat pf
		,boolean unshiftTime
		,StyleID sid
	){
		if(pf.NoDisp) return null;
		DecorationStack ary = new DecorationStack(sid);
		read(ary,m,to,pf,unshiftTime,false);
		return ary.getTextLine();
	}

	///////////////////////////////////////////////////////////
	public static void read(
		DecorationStack ary
		,IRCMessage m,Object to
		,PrintFormat pf
		,boolean unshiftTime
		,boolean toFile
	){
		if(unshiftTime){
			if(!toFile) ary.addPFStyle("time");
			ary.add(m.getTimeStr());
			if(!toFile) ary.addPFStyle(null);
			ary.add(" ");
		}
		for(int i=0;i<pf.macro.size();++i){
			Object o = pf.macro.get(i);
			if(o instanceof String){ ary.add((String)o); continue; }
			PrintFormatMacro pfm = (PrintFormatMacro)o;
			// スタイル指定
			int s= pfm.getStyle();
			if(s!=0){
				if(!toFile) ary.addPFExtra(s,pfm.extra());
				continue;
			}
			// 文字列展開
			String text=pfm.getText(m,to);
			if(text==null) continue;
			if(toFile){
				ary.add(text);
			}else{
				ary.addPFStyle( pfm.name());
				if( pfm.allowCTCP() ){ solveCTCP(m,to,ary,text);}
					else          { solveKeyword(m,to,ary,text); }
				ary.addPFStyle(null);
			}
		}
	}

	///////////////////////////////////////////////////////////
	// CTCP文字装飾のパース

	public static String[] controlToText =new String[]{
		"%00","%01","%02","%03","%04","%05","%06","%07",
		"%08","\t" ,"%0a","%0b","%0c","%0d","%0e","%0f",
		"%10","%11","%12","%13","%14","%15","%16","%17",
		"%18","%19","%1a","%1b","%1c","%1d","%1e","%1f",
	};

	protected static void solveCTCP(IRCMessage m,Object to,DecorationStack ary,String text){
		int lastColor=-1;
		boolean in_b=false;
		boolean in_r=false;
		boolean in_u=false;
		for(int i=0;i<text.length();++i){
			char c= text.charAt(i);
			// 普通のテキスト
			if(c>=0x20){
				int start=i++;
				while(i<text.length() && text.charAt(i)>=0x20)++i;
				solveURL(m,to,ary,text.substring(start,i) );
				--i;continue;
			}
			if(c==   2){ ary.addCTCPBold   (in_b=!in_b); continue; }
			if(c==0x16){ ary.addCTCPReverse(in_r=!in_r); continue; }
			if(c==0x1f){ ary.addCTCPUnder  (in_u=!in_u); continue; }
			if(c==0x0f){
				if(lastColor!=-1) ary.addCTCPColor(-1,(lastColor=-1));
				if(in_b) ary.addCTCPBold   (in_b=!in_b);
				if(in_r) ary.addCTCPReverse(in_r=!in_r);
				if(in_u) ary.addCTCPUnder  (in_u=!in_u);
				continue;
			}
			if(c==3){
				int fc,bc;
				// %03 num{2} ---- 2文字の色
				if( text.length()-i >= 3
				&& Util.isURIDigit(text.charAt(i+1))
				&& Util.isURIDigit(text.charAt(i+2))
				&& 0<= (fc=Integer.parseInt(text.substring(i+1,i+3)))
				&& fc<16
				){
					;
					// %03 num{2} , num{2} 
					if(text.length()-i >= 6
					&&  ','==text.charAt(i+3)
					&& Util.isURIDigit(text.charAt(i+4))
					&& Util.isURIDigit(text.charAt(i+5))
					&& 0<= (bc=Integer.parseInt(text.substring(i+4,i+6)))
					&& bc<16
					){
						ary.addCTCPColor(lastColor=fc,bc);
						i+=5;
						continue;
					}
					ary.addCTCPColor(lastColor=fc,-1);
					i+=2;
					continue;
				}
				// 分からなかったので解除扱い
				ary.addCTCPColor(lastColor=-1,-1);
				continue;
			}
			// その他の制御コードは表示可能な文字列に変換する
			ary.add(controlToText[c]);
		}
		// 末尾で解除
		if(lastColor!=-1) ary.addCTCPColor(-1,-1);
		if(in_b) ary.addCTCPBold   (false);
		if(in_r) ary.addCTCPReverse(false);
		if(in_u) ary.addCTCPUnder  (false);
	}

	// URLの分解
	protected static void solveURL(IRCMessage m,Object to,DecorationStack ary,String text){
		for(Iterator it=Util.URISplitter(text).iterator();it.hasNext();){
			Object o = it.next();
			if( o instanceof MyURL ){
				ary.addURLStyle("url");
				ary.add( ((MyURL)o).toString());
				ary.addURLStyle(null);
			}else{
				solveKeyword(m,to,ary,(String)o);
			}
		}
	}

	// キーワード反応
	protected static void solveKeyword(IRCMessage m,Object to,DecorationStack ary,String text){
		// 除外キーワード
		for(Iterator it=KeywordMatching.getExcludeList().iterator();it.hasNext();){
			if(((KeywordMatching)it.next()).check(text,0)){
				ary.add(text);
				return;
			}
		}
		// キーワードの一覧
		java.util.List key_list=KeywordMatching.getList();
		KLIST:for(int i=0;i<text.length();){
			int checkstart=i;
			for(Iterator it=key_list.iterator();it.hasNext();){
				KeywordMatching km = (KeywordMatching)it.next();
				if(!km.check(text,checkstart))continue;
				int matchstart = km.getLastMatchStart();
				i = matchstart + km.getLastMatchLength();
				if(matchstart>checkstart) ary.add(text.substring(checkstart,matchstart));
				ary.addKWStyle(km.getStyleName());
				ary.add(text.substring(matchstart,i));
				ary.addKWStyle(null);
				continue KLIST;
			}
			ary.add(text.substring(checkstart));
			break;
		}
	}
}
