// UTF-8 ☀☁☂☃
package irc;
import base.*;
import java.util.zip.Adler32;
import java.io.ByteArrayOutputStream;
import java.io.IOException;

/*
	IRCのチャンネル名。
	HashMap,TreeMapのキーに指定するために独立したクラスにしている。
	JISで書かれたチャンネル名には変な特性がいっぱいあるので、
	そのあたりも考慮したい。
	・バイト配列のままの名前
	・バイト配列のまま小文字化した名前
*/

public class IRCChannelName
implements Comparable,IRCChannelNameOrPrefix
{
	// エスケープされた名前の開始を示す記号
	public static char config_EscapedNameStart='∴';

	protected byte[] RawBytes;      // バイト配列のままの名前
	protected byte[] LowBytes;      // 小文字化した名前。サーバ上でのキー。
	protected String PrintableName; // 表示できる名前。
	protected String ShortName;     // 表示できる短い名前。!チャンネルの場合のみ意味がある

	public boolean isChannelName(){ return isChannelName(RawBytes[0]);}
	public static boolean isChannelName(String s){ return isChannelName(s.charAt(0)); }
	public static boolean isChannelName(byte[] s){ return s.length==0?false:isChannelName(s[0]); }
	public static boolean isChannelName(int c){
		if(-1 != "#!+&".indexOf(c)) return true;
		return false;
	}
	//---------------------------------------------------------
	// コンストラクタと設定と取得
	public IRCChannelName(byte[] a){set(a);}
	public IRCChannelName(String a){set(a);}

	public void set(byte[] src){
		int s=0;while(s<src.length && Character.isWhitespace((char)src[s]))++s;
		int start=s;
		while(s<src.length && isValidChannelChar((char)src[s]))++s;
		if(start==0 && s==src.length){
			RawBytes = src;
		}else{
			RawBytes = new byte[s-start];
			for(s=0;s<RawBytes.length;++s) RawBytes[s]=src[start+s];
		}
		LowBytes = RawToLow(RawBytes);
		updateHashCode();
		PrintableName = RawToPrintable(RawBytes);
		ShortName = PrintableToShort(PrintableName);
	}
	public void set(String src){
		this.RawBytes = unescape(TrimSpace(src));
		LowBytes = RawToLow(RawBytes);
		updateHashCode();
		PrintableName = RawToPrintable(RawBytes);
		ShortName = PrintableToShort(PrintableName);
	}

	public  byte[] getRawBytes(){return RawBytes;}
	public  byte[] getLowBytes(){return LowBytes;}
	public  String getPrintableName(){return PrintableName;}
	public  String getShortName(){return ShortName;}

	public String getName(){ return getPrintableName();}
	public String getPrefix(){ return getPrintableName();}

	public byte[] getJoinNameBytes(){
		if( RawBytes.length <= 6
		||  RawBytes[0] !='!'
		) return getRawBytes();
		for(int i=1;i<=5;++i){
			if(-1== "ABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890".indexOf((char)RawBytes[i])
			) return getRawBytes();
		}
		// !xxxxxほげ として扱う
		byte[] r = new byte[RawBytes.length-5];
		r[0]='!';
		for(int i=6;i<RawBytes.length;++i) r[i-5]=RawBytes[i];
		return r;
	}

	//------------------------------------------------------
	// 正確でなおかつ表示できる名前を作成する。
	// この名前は 先頭が config_EscapedNameStart で始まり、
	// そこから空白までの間にはURIエスケープされた文字が連なる。
	static byte[] cvt_buf = new byte[5];
	public String getEscapedName(){
		StringBuffer sb =new StringBuffer();
		sb.append(config_EscapedNameStart);
		boolean escaped =false;
		boolean isInJIS =false;
		cvt_buf[0]=0x1b;
		cvt_buf[1]='$';
		cvt_buf[2]='B';

		for(int i=0;i<RawBytes.length;++i){
			int c = RawBytes[i];
			if( c >0x20 && c<0x7f ){
				if(i>=3 && RawBytes[i-3] == 0x1b){
					if( RawBytes[i-2] =='$' && RawBytes[i-1] =='B' ){
						isInJIS =true;
					}
				}
				// 日本語のまま書けるか試みる
				if( isInJIS ){
					if( i+1 < RawBytes.length 
					&&  RawBytes[i+1] > 0x20 && RawBytes[i+1]<0x7f
					){
						cvt_buf[3]=(byte)c;
						cvt_buf[4]=RawBytes[i+1];
						String s = Util.fromJIS(cvt_buf);
						if(s.length()>0){
							byte[] rev=Util.toJIS(s);
							if(rev.length >=5
							&& rev[3] == cvt_buf[3]
							&& rev[4] == cvt_buf[4]
							){
								sb.append(s.charAt(0));
								++i;
								continue;
							}
						}
					}else{
						// 一度でも条件があわなくなったらやめる
						isInJIS=false;
					}
				}
				sb.append((char)c);
			}else{
				isInJIS =false;
				escaped =true;
				sb.append('%');
				String Hex = Integer.toHexString(c+256);
				sb.append(Hex.substring(Hex.length()-2));
			}
		}
		if(!escaped) return Util.fromJIS(RawBytes);
		return new String(sb);
	}

	// 文字列をunescape しつつunicode=>JIS変換を行いつつbyte[]にする
	// エスケープされた名前の前後は空白か始端終端でないといけない
	// これはユーザの入力がコマンドだった場合に使われることを想定している
	static String alnum="0123456789ABCDEFabcdef";
	public static byte[] unescape(String s){
		int i=0;
		ByteArrayOutputStream ba= new ByteArrayOutputStream();
		char c;
		ConvertLoop: while(i<s.length()){
			// 普通の部分を処理する
			int start=i;
			while(i<s.length() && s.charAt(i)!=config_EscapedNameStart) ++i;
			if(i>start){
				try{ 
					ba.write(Util.toJIS(s,start,i));
				}catch(IOException e){
				}
			}
			++i;
			if(i>=s.length()) break;
			// 次に空白が出るまでがエスケープ範囲
			start =i;while(i<s.length()&& s.charAt(i) !=' ') ++i;
			// start ～ i までをunescape してしまう
			for(int pos=start;pos<i;++pos){
				c=s.charAt(pos);
				if(c>0x7f){
					byte[] rev=Util.toJIS(""+c);
					if(rev.length >=5  ){
						ba.write((byte) rev[3]);
						ba.write((byte) rev[4]);
						continue;
					}
				}
				if( c=='%' &&  i-pos >=3 ){
					int h=alnum.indexOf(s.charAt(pos+1));
					int l=alnum.indexOf(s.charAt(pos+2));
					if(h!=-1&&l!=-1){
						if(h>16)h-=6;
						if(l>16)l-=6;
						ba.write(h*16+l);
						pos+=2;
						continue;
					}
				}
				ba.write(c);
			}
		}
		return ba.toByteArray();
	}

	//---------------------------------------------------------
	// バイト列から表示できる名前を作る。しかしこれは必ずしも正確ではない。
	private static String RawToPrintable(byte[] src){
		String a= Util.fromJIS(src);
		StringBuffer sb=new StringBuffer();
		for(int i=0;i<a.length();++i){
			char c=a.charAt(i);
			if(c<=0x20 || c==0x7f){
				String hex =Integer.toHexString(256+c);
				sb.append('%');
				sb.append(hex.substring(hex.length()-2));
			}else{
				sb.append(c);
			}
		}
		return new String(sb);
	}

	public static String PrintableToShort(String src){
		if(src.charAt(0)!='!') return src;
		return "!"+src.substring(6,src.length());
	}

	//---------------------------------------------------------

	// comparator
	public int compare(Object o1, Object o2){
		if(o1==o2) return 0;
		byte[] a = ((IRCChannelName)o1).LowBytes;
		byte[] b = ((IRCChannelName)o2).LowBytes;
		for(int i=0;;++i){
			if(i >= a.length || i >= b.length ){
				return a.length - b.length;
			}
			int r = ((int)a[i]) - ((int)b[i]);
			if(r!=0) return r;
		}
	}

	// comparable 
	public int  compareTo(Object b){ return (this==b)?0:compare(this,b); }
	public boolean equals(Object b){ return (hashCode()!=b.hashCode())?false:(this==b)?true:(0==compare(this,b)); }

	//---------------------------------------------------------
	// ハッシュコード
	private int hashcode;
	public int hashCode(){ return hashcode;}
	private void updateHashCode(){
		Adler32 check = new Adler32();
		check.update(LowBytes);
		long value = check.getValue();
		if(value > Integer.MAX_VALUE) value -= 0x100000000L;
		hashcode =(int)value;
	}

	//---------------------------------------------------------
	// 大文字小文字を同一視したバイト列を作る
	static String ary_from="ABCDEFGHIJKLMNOPQRSTUVWXYZ[]\\";
	static String ary_to  ="abcdefghijklmnopqrstuvwxyz{}|";
	public static byte[] RawToLow(byte[] src){
		byte[] dst = new byte[src.length];
		for(int i=0;i<dst.length;++i){
			byte c = src[i];
			if(c>=0x20 && c<=0x7f ){
				int offset = ary_from.indexOf(c);
				if(offset!=-1){
					dst[i] = (byte)ary_to.charAt(offset);
					continue;
				}
			}
			dst[i]=c;
		}
		return dst;
	}

	// チャンネル名に使える文字かどうか
	public static boolean isValidChannelChar(char ch){
		switch(ch){
		case 0x07: //これは古いサーバでチャンネル名とモードの区切りに使われているようだ
	//	case 0x2c: // ,
	//  case 0x3A: // : これはクライアント上では考慮しない
			return false;
		}
		return true;
	}

	// チャンネル名の指定から空白などを取り除く
	public static String TrimSpace(String src){
		int s=0;while(s<src.length() && Character.isWhitespace(src.charAt(s)))++s;
		int start=s;
		while(s<src.length() && isValidChannelChar(src.charAt(s)))++s;
		return src.substring(start,s);
	}
}
