package bodybuilder.inspector;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import bodybuilder.exception.BodyBuilderException;
import bodybuilder.util.Config;
import bodybuilder.util.ObjectUtils;

/**
 * インスペクターマッピング
 */
class InspectorMapping {

    /**
     * インスペクターのマップ
     */
    private static Map inspectors = new HashMap();

    static {
        try {
            init();
        } catch (Throwable e) {
            e.printStackTrace();
            throw new BodyBuilderException("failed to initialize '"
                    + InspectorMapping.class.getName() + "'.", e);
        }
    }

    /**
     * 初期化する。
     */
    private static void init() {
        // インスペクターのマップを取得。
        Map inspectorMap = Config.getInspectorMap();
        Iterator names = inspectorMap.keySet().iterator();
        // キャッシュを生成。
        Map cache = new HashMap();

        // ターゲットクラスとインスペクターのマップを作成。
        // インスペクターはシングルトン。
        while (names.hasNext()) {
            String name = (String) names.next();
            String className = (String) inspectorMap.get(name);
            Inspector inspector = (Inspector) cache.get(className);

            if (inspector == null) {
                inspector = (Inspector) ObjectUtils.getObject(className);
                cache.put(className, inspector);
            }

            // インスペクターにターゲットクラスを追加。
            inspector.addTarget(name);
            inspectors.put(name, inspector);
        }
    }

    /**
     * インスペクターを取得する。
     * 
     * @param object オブジェクト
     * @return インスペクター
     */
    static Inspector getInspector(Object object) {
        // オブジェクトがnullの場合はnullを返す。
        if (object == null) {
            return null;
        }

        Inspector inspector = null;
        // 実装クラスのリストを取得。
        List classNames = ObjectUtils.getClassNames(object);
        Iterator names = classNames.iterator();

        // 下位の実装クラスから順に、マップされたインスペクターを探索。
        while (names.hasNext() && inspector == null) {
            String name = (String) names.next();
            inspector = (Inspector) inspectors.get(name);
        }

        // インスペクターを返す。
        return inspector;
    }

}