package bodybuilder.util;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import bodybuilder.exception.BodyBuilderException;

/**
 * 設定
 */
public class Config {

    /**
     * プロパティ
     */
    private static Properties config = new Properties();

    /////////////////////////////////////////////////////////////////
    // constant field

    /**
     * デフォルト設定ファイル名
     */
    private static final String DEFAULT_CONFIG_FILE_NAME = "bodybuilder-default.properties";

    /**
     * 設定ファイル名
     */
    private static final String CONFIG_FILE_NAME = "bodybuilder.properties";

    /**
     * キーのプリフィックス
     */
    private static final String PROP_KEY_PREFIX = "bodybuilder.";

    /////////////////////////////////////////////////////////////////
    // property

    /**
     * XMLの妥当性を検査するかどうかを取得する。
     * 
     * @return XMLの妥当性を検査する場合はtrue
     */
    public static boolean isValidate() {
        return getBooleanProperty(PROP_KEY_PREFIX + "xml.validate", false);
    }

    /**
     * 出力ハンドラ名を取得する。
     * 
     * @return 出力ハンドラ名
     */
    public static String getOutputHandler() {
        return getProperty(PROP_KEY_PREFIX + "output.handler");
    }

    /////////////////////////////////////////////////////////////////
    // inspection property

    /**
     * 検査のプロパティのプリフィックス
     */
    private static final String INSPECTION_PROP_KEY_PRIFIX = PROP_KEY_PREFIX
            + "inspection.";

    /**
     * サイズを検査するかどうかを取得する。
     * 
     * @return サイズを検査する場合はtrue
     */
    public static boolean isInspectionSize() {
        return getBooleanProperty(INSPECTION_PROP_KEY_PRIFIX + "size", true);
    }

    /**
     * キーセットを厳密に検査するかどうかを取得する。
     * 
     * @return キーセットを厳密に検査する場合はtrue
     */
    public static boolean isInspectionKeySetStrictly() {
        return getBooleanProperty(INSPECTION_PROP_KEY_PRIFIX
                + "keyset.strictly", false);
    }

    /**
     * クラスを検査するかどうかを取得する。
     * 
     * @return クラスを検査する場合はtrue
     */
    public static boolean isInspectionClass() {
        return getBooleanProperty(INSPECTION_PROP_KEY_PRIFIX + "class", true);
    }

    /**
     * 例外のメッセージの検査に正規表現を使用するかどうかを返す。
     * 
     * @return 正規表現を使用する場合はtrue
     */
    public static boolean isInspectionExceptionRegex() {
        return getBooleanProperty(INSPECTION_PROP_KEY_PRIFIX
                + "exception.regex", true);
    }

    /**
     * 例外の原因を検査するかどうかを返す。
     * 
     * @return 検査する場合はtrue
     */
    public static boolean isInspectionExceptionCause() {
        return getBooleanProperty(INSPECTION_PROP_KEY_PRIFIX
                + "exception.cause", false);
    }

    /////////////////////////////////////////////////////////////////
    // view property

    /**
     * ビューのプロパティのプリフィックス
     */
    private static final String VIEW_PROP_KEY_PRIFIX = PROP_KEY_PREFIX
            + "view.";

    /**
     * スタックトレースを出力するかどうかを返す。
     * 
     * @return スタックトレースを出力する場合はtrue
     */
    public static boolean isViewPrintStackTrace() {
        return getBooleanProperty(VIEW_PROP_KEY_PRIFIX + "stacktrace", false);
    }

    /////////////////////////////////////////////////////////////////
    // builder property

    /**
     * ビルダーのプロパティのプリフィックス
     */
    private static final String BUILDER_PROP_KEY_PRIFIX = PROP_KEY_PREFIX
            + "builder.";

    /**
     * ビルダーのマップを取得する。
     * 
     * @return ビルダーのマップ
     */
    public static Map getBuilderMap() {
        return subMap(BUILDER_PROP_KEY_PRIFIX);
    }

    /////////////////////////////////////////////////////////////////
    // viewer property

    /**
     * ビューアのプロパティのプリフィックス
     */
    private static final String VIEWER_PROP_KEY_PRIFIX = PROP_KEY_PREFIX
            + "viewer.";

    /**
     * ビューアのマップを取得する。
     * 
     * @return ビューアのマップ
     */
    public static Map getViewerMap() {
        return newMap(VIEWER_PROP_KEY_PRIFIX, ".type", ".class");
    }

    /////////////////////////////////////////////////////////////////
    // inspector property

    /**
     * インスペクターのプロパティのプリフィックス
     */
    private static final String INSPECTOR_PROP_KEY_PRIFIX = PROP_KEY_PREFIX
            + "inspector.";

    /**
     * インスペクターのマップを取得する。
     * 
     * @return インスペクターのマップ
     */
    public static Map getInspectorMap() {
        return newMap(INSPECTOR_PROP_KEY_PRIFIX, ".type", ".class");
    }

    /////////////////////////////////////////////////////////////////
    // extended value property

    /**
     * 拡張値のプロパティのプリフィックス
     */
    private static final String VALUE_PROP_KEY_PRIFIX = PROP_KEY_PREFIX
            + "value.";

    /**
     * 拡張値のマップを取得する。
     * 
     * @return 拡張値のマップ
     */
    public static Map getValueMap() {
        return newMap(VALUE_PROP_KEY_PRIFIX, ".format", ".class");
    }

    /////////////////////////////////////////////////////////////////
    // test property

    /**
     * テストのプロパティのプリフィックス
     */
    private static final String TEST_PROP_KEY_PREFIX = PROP_KEY_PREFIX
            + "test.";

    /**
     * テストのルートディレクトリのリストを取得する。
     * 
     * @return テストのルートディレクトリのリスト
     */
    public static String[] getTestRootDirs() {
        // ディレクトリのリストを取得。
        String dirs = getRequiredProperty(TEST_PROP_KEY_PREFIX + "root.dirs");
        String[] roots = getList(dirs);

        // パスの末尾からパス区切り文字を削除。
        for (int i = 0; i < roots.length; i++) {
            roots[i] = FileUtils.removeEndSeparator(roots[i]);
        }

        return roots;
    }

    /**
     * 無視するディレクトリのリストを取得する。
     * 
     * @return 無視するディレクトリのリスト
     */
    public static List getTestIgnoreDirs() {
        // ディレクトリのリストを取得。
        String dirs = getProperty(TEST_PROP_KEY_PREFIX + "ignore.dirs");

        // 未定義の場合は空のリストを返す。
        if (dirs == null) {
            return new ArrayList();
        }

        String[] ignores = getList(dirs);

        // パスの末尾からパス区切り文字を削除。
        for (int i = 0; i < ignores.length; i++) {
            ignores[i] = FileUtils.removeEndSeparator(ignores[i]);
        }

        return Arrays.asList(ignores);
    }

    /**
     * テストケースのマップを取得する。
     * 
     * @return テストケースのマップ
     */
    public static Map getTestCaseMap() {
        return subMap(TEST_PROP_KEY_PREFIX + "type.");
    }

    /////////////////////////////////////////////////////////////////
    // database property

    /**
     * データベースのプロパティのプリフィックス
     */
    private static final String DATABASE_PROP_KEY_PREFIX = PROP_KEY_PREFIX
            + "database.";

    /**
     * データベースのドライバ名を取得する。
     * 
     * @return データベースのドライバ名
     */
    public static String getDatabaseDriver() {
        return getRequiredProperty(DATABASE_PROP_KEY_PREFIX + "driver");
    }

    /**
     * データベースのURLを取得する。
     * 
     * @return データベースのURL
     */
    public static String getDatabaseUrl() {
        return getRequiredProperty(DATABASE_PROP_KEY_PREFIX + "url");
    }

    /**
     * データベースのユーザ名を取得する。
     * 
     * @return データベースのユーザ名
     */
    public static String getDatabaseUser() {
        return getRequiredProperty(DATABASE_PROP_KEY_PREFIX + "user");
    }

    /**
     * データベースのパスワードを取得する。
     * 
     * @return データベースのパスワード
     */
    public static String getDatabasePassword() {
        return getRequiredProperty(DATABASE_PROP_KEY_PREFIX + "password");
    }

    /**
     * データセットのマップをを取得する。
     * 
     * @return データセットのマップ
     */
    public static Map getDataSetMap() {
        return subMap(DATABASE_PROP_KEY_PREFIX + "dataset.");
    }

    /////////////////////////////////////////////////////////////////
    // property utility

    /**
     * プロパティを取得する。
     * 
     * @param key キー
     * @return プロパティ
     */
    public static String getProperty(String key) {
        // プロパティを取得。
        String value = config.getProperty(key);
        // 念のため値をトリム。
        return (value != null) ? value.trim() : value;
    }

    /**
     * プロパティを取得する。未定義の場合はエラー。
     * 
     * @param key キー
     * @return プロパティ
     */
    public static String getRequiredProperty(String key) {
        // プロパティを取得。
        String value = getProperty(key);

        // 未定義の場合はエラー。
        if (value == null) {
            throw new BodyBuilderException("undefined property '" + key + "'.");
        }

        return value;
    }

    /**
     * ブール値のプロパティを取得する。
     * 
     * @param key キー
     * @param defaultValue デフォルト値
     * @return ブール値のプロパティ
     */
    private static boolean getBooleanProperty(String key, boolean defaultValue) {
        // プロパティを取得。
        String value = getProperty(key);

        if ("true".equalsIgnoreCase(value)) {
            return true;
        } else if ("false".equalsIgnoreCase(value)) {
            return false;
        } else {
            // "true""false"以外の場合はデフォルト値。
            return defaultValue;
        }
    }

    /////////////////////////////////////////////////////////////////
    // property map utility

    /**
     * サブマップを取得する。
     * 
     * @param プリフィックス
     * @return サブマップ
     */
    private static Map subMap(String prefix) {
        Map map = new HashMap();
        Iterator keys = config.keySet().iterator();

        // 指定されたプリフィックスのキーと値のマップを作成。
        while (keys.hasNext()) {
            String key = (String) keys.next();

            if (key.startsWith(prefix)) {
                String value = getProperty(key);
                key = key.substring(prefix.length());
                map.put(key, value);
            }
        }

        return map;
    }

    /**
     * 二つのプロパティを対応させたマップを作成する。
     * 
     * @param prefix プリフィックス
     * @param keySuffix キーのサフィックス
     * @param valSuffix 値のサフィックス
     * @return 二つのプロパティを対応させたマップ
     */
    private static Map newMap(String prefix, String keySuffix, String valSuffix) {
        Map map = new HashMap();
        // 指定されたプリフィックスのサブマップを取得する。
        Map submap = subMap(prefix);
        Iterator keys = submap.keySet().iterator();
        Set nameSet = new HashSet();

        // 指定されたサフィックスでキーのリストを取得。
        while (keys.hasNext()) {
            String key = (String) keys.next();
            String name = key.substring(0, key.indexOf('.'));
            nameSet.add(name);
        }

        Iterator names = nameSet.iterator();

        // キーに紐付く値を取得。
        while (names.hasNext()) {
            // キーを取得して、カンマで分割。
            String name = (String) names.next();
            String nameList[] = getList((String) submap.get(name + keySuffix));
            // キーに紐付く値を取得。
            String value = (String) submap.get(name + valSuffix);

            // 値が未定義の場合はエラー。
            if (value == null) {
                throw new BodyBuilderException(
                        "undefined value corresponding to key '" + prefix
                                + name + keySuffix + "'.");
            }

            // キーと値をマップに追加。
            for (int i = 0; i < nameList.length; i++) {
                map.put(nameList[i], value);
            }
        }

        return map;
    }

    /////////////////////////////////////////////////////////////////
    // other utility

    /**
     * 文字列をカンマで分割したリストを取得する。
     * 
     * @param 文字列
     * @return カンマで分割したリスト
     */
    private static String[] getList(String str) {
        // nullの場合は空リストを返す。
        if (str == null) {
            return new String[0];
        }

        // カンマで分割。
        String[] strs = str.split(",");

        // 分割した値はトリム。
        for (int i = 0; i < strs.length; i++) {
            strs[i] = strs[i].trim();
        }

        return strs;
    }

    /////////////////////////////////////////////////////////////////
    // initialize

    static {
        try {
            init();
        } catch (Throwable e) {
            e.printStackTrace();
            throw new BodyBuilderException("failed to initialize '"
                    + Config.class.getName() + "'.", e);
        }
    }

    /**
     * 初期化する。
     */
    private static void init() {
        // プロパティをクリア。
        config.clear();
        // 設定ファイルを読み込む。
        load(DEFAULT_CONFIG_FILE_NAME, true);
        load(CONFIG_FILE_NAME, false);
        // システムプロパティを読み込む。
        config.putAll(System.getProperties());
    }

    /**
     * 設定ファイルを読み込む。
     * 
     * @param filename ファイル名
     * @param required 必須の場合はtrue
     */
    private static void load(String filename, boolean required) {
        InputStream in = null;

        try {
            // クラスローダを取得。
            ClassLoader cloader = Thread.currentThread()
                    .getContextClassLoader();

            if (cloader == null) {
                cloader = Config.class.getClass().getClassLoader();
            }

            // クラスパスから設定ファイルを取得。
            in = cloader.getResourceAsStream(filename);

            // 必須なのに取得できない場合はエラー。
            if (required && in == null) {
                throw new BodyBuilderException("cannot load '" + filename
                        + "' in CLASSPATH.");
            }

            // 設定を読み込む。
            if (in != null) {
                config.load(in);
            }
        } catch (IOException e) {
            throw new BodyBuilderException("cannot load '" + filename + "'.", e);
        } finally {
            if (in != null) {
                try {
                    in.close();
                } catch (IOException e) {
                    throw new BodyBuilderException("cannot close '" + filename
                            + "'.", e);
                }
            }
        }
    }

}