package bodybuilder.builder;

import org.jdom.Element;

import bodybuilder.builder.argument.Argument;
import bodybuilder.builder.argument.ArgumentBuilder;
import bodybuilder.builder.value.ExtendedValue;
import bodybuilder.exception.BodyBuilderException;
import bodybuilder.util.ObjectUtils;
import bodybuilder.util.jdom.JDOMUtils;

/**
 * ビルダー
 */
public abstract class Builder {

    /////////////////////////////////////////////////////////////////
    // constant field

    /**
     * name属性
     */
    protected static final String ATTR_NAME = "name";

    /**
     * value属性
     */
    protected static final String ATTR_VALUE = "value";

    /**
     * type属性
     */
    protected static final String ATTR_TYPE = "type";

    /**
     * コンストラクタ要素
     */
    protected static final String ELEM_CONSTRUCTOR = "constructor";

    /////////////////////////////////////////////////////////////////
    // variable field

    /**
     * ビルダーがコンストラクタを持っているかどうかのフラグ
     */
    protected boolean hasConstructor = true;

    /**
     * ビルダーが子要素を持っているかどうかのフラグ
     */
    protected boolean hasChild = true;

    /////////////////////////////////////////////////////////////////
    // abstract method

    /**
     * オブジェクトを取得する。
     * 
     * @param element XML要素
     * @param constructor コンストラクタ
     */
    protected abstract Object getMuscle(Element element, Argument constructor);

    /////////////////////////////////////////////////////////////////
    // public method

    /**
     * XML要素から値を取得する。
     * 
     * @param element XML要素
     */
    public static Object getValue(Element element) {
        return getValue(element, null);
    }

    /**
     * XML要素から値を取得する。
     * 
     * @param element XML要素
     * @param type 値の型
     */
    public static Object getValue(Element element, String type) {
        // XML要素からvalue属性を取得。
        Object value = element.getAttributeValue(ATTR_VALUE);

        if (value == null) {
            // value属性がない場合は子要素を取得。
            Element child = JDOMUtils.getRequiredChild(element);
            return build(child);
        } else if (ExtendedValue.isExtended((String) value)) {
            // value属性が拡張値の場合は、拡張値を取得。
            return ExtendedValue.getValue((String) value);
        }

        // 値の型が指定されない場合、XML要素のtype属性を取得。
        if (type == null) {
            type = getType(element);
        }

        // value属性の値をコンストラクタの引数として、値を生成。
        value = ObjectUtils.getObject((String) value, type);

        // 値を返す。
        return value;
    }

    /**
     * XML要素から値を生成する。
     * 
     * @param element XML要素
     */
    public static Object build(Element element) {
        // 要素名にマップされたビルダーを取得。
        Builder builder = getBuilder(element.getName());
        // コンストラクタを取得。
        Argument constructor = getConstructor(element, builder.hasConstructor);

        // 子要素が不要なのに子要素がある場合はエラー。
        if (!builder.hasChild && element.getChildren().size() > 0) {
            throw new BodyBuilderException("cannot define child '"
                    + element.getName() + "'.");
        }

        // ビルダーからオブジェクトを取得。
        return builder.getMuscle(element, constructor);
    }

    /////////////////////////////////////////////////////////////////
    // private method

    /**
     * ビルダーを取得する。
     * 
     * @param name XML要素名
     */
    private static Builder getBuilder(String name) {
        // 要素名にマップされたビルダーを取得。
        Builder builder = BuilderMapping.getBuilder(name);

        // ビルダーを取得できない場合はエラー。
        if (builder == null) {
            throw new BodyBuilderException("cannot find builder '" + name
                    + "'.");
        }

        // ビルダーを返す。
        return builder;
    }

    /**
     * コンストラクタを取得する。
     * 
     * @param element XML要素
     * @param hasConstructor コンストラクタを持っているかどうかのフラグ
     * @return コンストラクタ
     */
    private static Argument getConstructor(Element element,
            boolean hasConstructor) {
        Argument argument = null;
        // コンストラクタ要素を取得。
        Element constructor = element.getChild(ELEM_CONSTRUCTOR);

        if (constructor != null) {
            // コンストラクタを持っていないのにコンストラクタが定義されている場合はエラー。
            if (!hasConstructor) {
                throw new BodyBuilderException("cannot pass constructor '"
                        + element.getName() + "'.");
            }

            /// コンストラクタを取得。
            argument = ArgumentBuilder.getArgument(constructor);
            element.removeChild(ELEM_CONSTRUCTOR);
        }

        // コンストラクタを返す。
        return argument;
    }

    /////////////////////////////////////////////////////////////////
    // utility method

    /**
     * オブジェクトを生成する。
     * 
     * @param type 型
     * @param constructor コンストラクタ
     */
    protected static Object newObject(String type, Argument constructor) {
        if (constructor != null) {
            // 指定されたコンストラクタでインスタンスを生成。
            return ObjectUtils.getInstance(type, constructor.getClasses(),
                    constructor.getArguments());
        } else {
            // デフォルトコンストラクタでインスタンスを生成。
            return ObjectUtils.getObject(type);
        }
    }

    /**
     * オブジェクトを生成する。
     * 
     * @param type 型
     * @param constructor コンストラクタ
     */
    protected static Object newObject(Class type, Argument constructor) {
        if (constructor != null) {
            // 指定されたコンストラクタでインスタンスを生成。
            return ObjectUtils.getInstance(type, constructor.getClasses(),
                    constructor.getArguments());
        } else {
            // デフォルトコンストラクタでインスタンスを生成。
            return ObjectUtils.getObject(type);
        }
    }

    /**
     * name属性を取得する。未定義の場合はエラー。
     * 
     * @param element XML要素
     * @return name属性
     */
    protected static String getRequiredName(Element element) {
        return JDOMUtils.getRequiredAttrValue(element, ATTR_NAME);
    }

    /**
     * value属性を取得する。
     * 
     * @param element XML要素
     * @return value属性
     */
    protected static String getValueAttr(Element element) {
        return element.getAttributeValue(ATTR_VALUE);
    }

    /**
     * value属性を取得する。未定義の場合はエラー。
     * 
     * @param element XML要素
     * @return value属性
     */
    protected static String getRequiredValueAttr(Element element) {
        return JDOMUtils.getRequiredAttrValue(element, ATTR_VALUE);
    }

    /**
     * type属性を取得する。
     * 
     * @param element XML要素
     * @return type属性
     */
    protected static String getType(Element element) {
        return element.getAttributeValue(ATTR_TYPE);
    }

    /**
     * type属性を取得する。
     * 
     * @param element XML要素
     * @param defaultType デフォルトタイプ
     * @return type属性
     */
    protected static String getType(Element element, String defaultType) {
        return JDOMUtils.getAttrValue(element, ATTR_TYPE, defaultType);
    }

    /**
     * type属性を取得する。未定義の場合はエラー。
     * 
     * @param element XML要素
     * @return type属性
     */
    protected static String getRequiredType(Element element) {
        return JDOMUtils.getRequiredAttrValue(element, ATTR_TYPE);
    }

}