package bodybuilder.viewer;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import bodybuilder.exception.BodyBuilderException;
import bodybuilder.util.Config;
import bodybuilder.util.ObjectUtils;
import bodybuilder.viewer.Viewer;

/**
 * ビューアマッピング
 */
class ViewerMapping {

    /**
     * ビューアのマップ
     */
    private static Map viewers = new HashMap();

    static {
        try {
            init();
        } catch (Throwable e) {
            e.printStackTrace();
            throw new BodyBuilderException("failed to initialize '"
                    + ViewerMapping.class.getName() + "'.", e);
        }
    }

    /**
     * 初期化する。
     */
    private static void init() {
        // ビューアのマップを取得。
        Map viewerMap = Config.getViewerMap();
        Iterator names = viewerMap.keySet().iterator();
        // キャッシュを生成。
        Map cache = new HashMap();

        // ターゲットクラスとビューアのマップを作成。
        // ビューアはシングルトン。
        while (names.hasNext()) {
            String name = (String) names.next();
            String className = (String) viewerMap.get(name);
            Viewer viewer = (Viewer) cache.get(className);

            if (viewer == null) {
                viewer = (Viewer) ObjectUtils.getObject(className);
                cache.put(className, viewer);
            }

            viewers.put(name, viewer);
        }
    }

    /**
     * ビューアを取得する。
     * 
     * @param object オブジェクト
     * @return ビューア
     */
    static Viewer getViewer(Object object) {
        // オブジェクトがnullの場合はnullを返す。
        if (object == null) {
            return null;
        }

        Viewer viewer = null;
        // 実装クラスのリストを取得。
        List classNames = ObjectUtils.getClassNames(object);
        Iterator names = classNames.iterator();

        // 下位の実装クラスから順に、マップされたビューアを探索。
        while (names.hasNext() && viewer == null) {
            String name = (String) names.next();
            viewer = (Viewer) viewers.get(name);

            // クラス名で取得できない場合はパッケージ名で取得。
            if (viewer == null) {
                String pkg = ObjectUtils.getPackage(name, true);
                viewer = (Viewer) viewers.get(pkg);
            }
        }

        // ビューアを返す。
        return viewer;
    }

}