package bodybuilder.util.jdom;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.jdom.Attribute;
import org.jdom.CDATA;
import org.jdom.DefaultJDOMFactory;
import org.jdom.Namespace;
import org.jdom.Text;

import bodybuilder.util.Config;
import bodybuilder.util.Utils;

/**
 * 拡張JDOMファクトリ
 */
public class ExtendedJDOMFactory extends DefaultJDOMFactory {

    /**
     * 属性を生成する。
     * 
     * @param name 属性名
     * @param value 属性値
     * @param namespace 名前空間
     */
    public Attribute attribute(String name, String value, Namespace namespace) {
        return super.attribute(name, replace(value), namespace);
    }

    /**
     * 属性を生成する。
     * 
     * @param name 属性名
     * @param value 属性値
     * @param type タイプ
     * @param namespace 名前空間
     */
    public Attribute attribute(String name, String value, int type,
            Namespace namespace) {
        return super.attribute(name, replace(value), type, namespace);
    }

    /**
     * 属性を生成する。
     * 
     * @param name 属性名
     * @param value 属性値
     */
    public Attribute attribute(String name, String value) {
        return super.attribute(name, replace(value));
    }

    /**
     * 属性を生成する。
     * 
     * @param name 属性名
     * @param value 属性値
     * @param type タイプ
     */
    public Attribute attribute(String name, String value, int type) {
        return super.attribute(name, replace(value), type);
    }

    /**
     * CDATAセクションを生成する。
     * 
     * @param text テキスト
     */
    public CDATA cdata(String text) {
        return super.cdata(replace(text));
    }

    /**
     * 文字データを生成する。
     * 
     * @param text テキスト
     */
    public Text text(String text) {
        return super.text(replace(text));
    }

    /**
     * プロパティ参照を置換する。
     * 
     * @param str 文字列
     * @return プロパティ参照を置換した文字列
     */
    private String replace(String str) {
        Pattern pattern = Pattern.compile("(\\$+)\\{([^}]+)\\}");
        Matcher matcher = pattern.matcher(str);
        StringBuffer buffer = new StringBuffer();
        boolean matches = matcher.find();

        // プロパティ参照を置換する。
        while (matches) {
            String dollars = matcher.group(1);
            String key = matcher.group(2);
            int len = dollars.length();
            String value = null;

            if (len < 2) {
                value = Config.getRequiredProperty(key);
            } else {
                int mod = len % 2;
                dollars = dollars.substring(0, (len - mod) / 2);

                if (mod == 0) {
                    value = dollars + "{" + key + "}";
                } else {
                    value = dollars + Config.getRequiredProperty(key);
                }
            }

            matcher.appendReplacement(buffer, Utils.escapeReplaceString(value));
            matches = matcher.find();
        }

        matcher.appendTail(buffer);
        return buffer.toString();
    }

}