package bodybuilder.inspector;

import java.util.Stack;

import bodybuilder.util.Config;
import bodybuilder.util.ObjectUtils;

/**
 * オブジェクトバックトレース
 */
public class ObjectBackTrace {

    /**
     * タブ
     */
    private static final String TAB = "  ";

    /**
     * バッファ
     */
    private StringBuffer buffer = new StringBuffer();

    /**
     * ネストのレベル
     */
    private int level = 0;

    /**
     * インデントポイント
     */
    private Stack indentPoints = new Stack();

    /**
     * インデックスポイント
     */
    private Stack indexPoints = new Stack();

    /**
     * インデントする。
     */
    public void indent() {
        // ネストのレベルを上げる。
        level++;

        // 末尾の改行を削除。
        if (buffer.toString().endsWith(Config.NL)) {
            int len = buffer.length();
            buffer.delete(len - Config.NL.length(), len);
        }

        // インデントする。
        buffer.append(" {");
        setIndentPosition();
        buffer.append(Config.NL);
        setIndexPosition();
    }

    /**
     * アンインデントする。
     */
    public void unindent() {
        // ネストのレベルを下げる。
        level--;
        // インデントの位置を元に戻す。
        rollbackIndentPosition();
        // インデックスを元に戻す。
        indexPoints.pop();
    }

    /**
     * オブジェクトの情報を追加する。
     * 
     * @param object オブジェクト
     */
    public void append(Object object) {
        // ネストしてオブジェクトの情報をバッファに追加。
        nest();
        buffer.append(ObjectUtils.getInfo(object));
        buffer.append(Config.NL);
    }

    /**
     * インデックスを追加する。
     * 
     * @param index インデックス
     */
    public void appendIndex(int index) {
        // インデックスの位置を元に戻す。
        rollbackIndexPosition();
        // ネストしてインデックスを追加。
        nest();
        buffer.append("[" + index + "]=>");
        buffer.append(Config.NL);
    }

    /**
     * キーインデックスを追加する。
     * 
     * @param key キー
     */
    public void appendKey(Object key) {
        appendKey(key, null);
    }

    /**
     * キーインデックスを追加する。
     * 
     * @param key キー
     * @param suffix サフィックス
     */
    public void appendKey(Object key, String suffix) {
        // インデックスの位置を元に戻す。
        rollbackIndexPosition();

        // ネストしてキーインデックスを追加。
        nest();
        buffer.append("[\"" + key + "\"]");

        if (suffix != null) {
            buffer.append(suffix);
        }

        buffer.append("=>");
        buffer.append(Config.NL);
    }

    /**
     * バックトレースのダンプを出力する。
     * 
     * @return バックトレースのダンプ
     */
    String dump() {
        // バッファからバックトレースを取得。
        String trace = buffer.toString();
        // バッファをクリア。
        clear();
        buffer = null;
        // バックトレースを返す。
        return trace;
    }

    /**
     * バッファをクリアする。
     */
    void clear() {
        buffer.delete(0, buffer.length());
    }

    /**
     * ネストする。
     */
    private void nest() {
        // ネストレベル分、タブをバッファに追加。
        for (int i = 0; i < level; i++) {
            buffer.append(TAB);
        }
    }

    /**
     * インデント位置をセットする。
     */
    private void setIndentPosition() {
        // 最後の改行の後ろをインデント位置としてセット。
        int pos = buffer.lastIndexOf(Config.NL);

        if (pos >= 0) {
            indentPoints.push(new Integer(pos + Config.NL.length()));
        }
    }

    /**
     * インデント位置を元に戻す。
     */
    private void rollbackIndentPosition() {
        // スタックが空の時は処理を抜ける。
        if (indentPoints.size() < 1) {
            return;
        }

        // スタックからインデント位置を取得して、バッファをそこまで削除。
        Integer pos = (Integer) indentPoints.pop();

        if (pos != null) {
            buffer.delete(pos.intValue(), buffer.length());
        }
    }

    /**
     * インデックス位置をセットする。
     */
    private void setIndexPosition() {
        // 最後の改行の後ろをインデックス位置としてセット。
        int pos = buffer.lastIndexOf(Config.NL);

        if (pos >= 0) {
            indexPoints.push(new Integer(pos + Config.NL.length()));
        }
    }

    /**
     * インデックス位置を元に戻す。
     */
    private void rollbackIndexPosition() {
        // スタックからインデックス位置を取得。
        Integer pos = (Integer) indexPoints.lastElement();

        if (pos != null) {
            // バッファをインデックス位置まで削除。
            if (pos.intValue() < buffer.length()) {
                buffer.delete(pos.intValue(), buffer.length());
            }
        }
    }

}