package bodybuilder.util.dbunit;

import java.math.BigDecimal;
import jxl.BooleanCell;
import jxl.Cell;
import jxl.CellType;
import jxl.DateCell;
import jxl.LabelCell;
import jxl.NumberCell;
import jxl.Sheet;
import org.dbunit.dataset.AbstractTable;
import org.dbunit.dataset.Column;
import org.dbunit.dataset.DataSetException;
import org.dbunit.dataset.DefaultTableMetaData;
import org.dbunit.dataset.ITableMetaData;
import org.dbunit.dataset.datatype.DataType;
import org.dbunit.dataset.datatype.DataTypeException;

/**
 * Java Excel APIテーブル
 */
public class JxlTable extends AbstractTable {

    /**
     * シート
     */
    private Sheet sheet = null;

    /**
     * メタデータ
     */
    private ITableMetaData metadata = null;

    /**
     * テーブルを生成する。
     * 
     * @param sheet シート
     */
    public JxlTable(Sheet sheet) throws DataSetException {
        // シートをフィールドにセット。
        this.sheet = sheet;
        // シート名を取得。
        String sname = sheet.getName();

        // カラムを取得。
        Column[] columns = null;

        if (sheet.getRows() > 0) {
            Cell[] head = sheet.getRow(0);
            columns = new Column[head.length];

            for (int i = 0; i < head.length; i++) {
                String cname = head[i].getContents();
                columns[i] = new Column(cname, DataType.UNKNOWN);
            }
        } else {
            columns = new Column[0];
        }

        // メタデータを生成。
        metadata = new DefaultTableMetaData(sname, columns);
    }

    /**
     * メタデータを取得する。
     * 
     * @return メタデータ
     */
    public ITableMetaData getTableMetaData() {
        return metadata;
    }

    /**
     * 行数を取得する。
     * 
     * @return 行数
     */
    public int getRowCount() {
        return sheet.getRows() - 1;
    }

    /**
     * カラムの値を取得する。
     * 
     * @param rows 行数
     * @param cname カラム名
     * @return カラムの値
     */
    public Object getValue(int rows, String cname) throws DataSetException {
        // 行数を検査。
        assertValidRowIndex(rows);

        // カラム名を精査。
        assertValidColumn(cname);

        // セルを取得。
        Cell[] cells = sheet.getRow(rows + 1);
        Cell cell = cells[getColumnIndex(cname)];
        CellType type = cell.getType();

        // セルの値を取得。
        Object value = null;

        if (type == CellType.LABEL || type == CellType.STRING_FORMULA) {
            // 文字列の場合
            value = ((LabelCell) cell).getString();
        } else if (type == CellType.NUMBER || type == CellType.NUMBER_FORMULA) {
            // 数値の場合
            value = new BigDecimal(((NumberCell) cell).getValue());
        } else if (type == CellType.DATE || type == CellType.DATE_FORMULA) {
            // 日付の場合
            value = ((DateCell) cell).getDate();
        } else if (type == CellType.BOOLEAN || type == CellType.BOOLEAN_FORMULA) {
            // ブール値の場合
            value = new Boolean(((BooleanCell) cell).getValue());
        } else if (type == CellType.EMPTY) {
            // 空の場合
            value = null;
        } else {
            throw new DataTypeException("Unsupported type '" + type
                    + "' at row=" + rows + ", column=" + cname);
        }

        // 値を返す。
        return value;
    }

}