package bodybuilder.util;

import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import bodybuilder.exception.BodyBuilderException;

/**
 * オブジェクトユーティリティ
 */
public class ObjectUtils {

    /////////////////////////////////////////////////////////////////
    // object getter

    /**
     * オブジェクトを取得する。
     * 
     * @param type オブジェクトの型
     * @return オブジェクト
     */
    public static Object getObject(String type) {
        return getObject(null, type);
    }

    /**
     * オブジェクトを取得する。
     * 
     * @param value コンストラクタに渡す文字列
     * @param type オブジェクトの型
     * @return オブジェクト
     */
    public static Object getObject(String value, String type) {
        // 型が指定されていない、またはStringクラスの場合、文字列をそのまま返す。
        if (type == null || type.equals(String.class.getName())) {
            return value;
        }

        // プリミティブ型の場合はラッパーを返す。
        if (isPrimitive(type)) {
            return getPrimitiveWrapperInstance(value, type);
        }

        // オブジェクトを生成。
        if (value != null) {
            return getInstance(type, new Class[] { String.class },
                    new Object[] { value }, false);
        } else {
            return getInstance(type, new Class[0], new Object[0], false);
        }
    }

    /**
     * オブジェクトを取得する。
     * 
     * @param type オブジェクトの型
     * @return オブジェクト
     */
    public static Object getObject(Class type) {
        return getObject(null, type);
    }

    /**
     * オブジェクトを取得する。
     * 
     * @param value コンストラクタに渡す文字列
     * @param type オブジェクトの型
     * @return オブジェクト
     */
    public static Object getObject(String value, Class type) {
        // 型が指定されていない、またはStringクラスの場合、文字列をそのまま返す。
        if (type == null || type.equals(String.class.getName())) {
            return value;
        }

        // オブジェクトを生成。
        if (value != null) {
            return getInstance(type, new Class[] { String.class },
                    new Object[] { value }, false);
        } else {
            return getInstance(type, new Class[0], new Object[0], false);
        }
    }

    /**
     * プリミティブ型かどうかを取得する。
     * 
     * @param type 型
     * @return プリミティブ型の場合はtrue
     */
    public static boolean isPrimitive(String type) {
        boolean primitive = false;

        // いずれかの文字列と等しい場合はプリミティブ型。
        if ("bool".equals(type) || "boolean".equals(type)) {
            primitive = true;
        } else if ("byte".equals(type)) {
            primitive = true;
        } else if ("char".equals(type) || "character".equals(type)) {
            primitive = true;
        } else if ("short".equals(type)) {
            primitive = true;
        } else if ("int".equals(type) || "integer".equals(type)) {
            primitive = true;
        } else if ("long".equals(type)) {
            primitive = true;
        } else if ("float".equals(type)) {
            primitive = true;
        } else if ("double".equals(type)) {
            primitive = true;
        }

        return primitive;
    }

    /**
     * プリミティブラッパーを取得する。
     * 
     * @param value 値
     * @param type 型
     * @return プリミティブラッパー
     */
    public static Object getPrimitiveWrapperInstance(String value, String type) {
        Object obj = null;

        // ラッパーを取得する。
        if ("bool".equals(type) || "boolean".equals(type)) {
            obj = Boolean.valueOf(value);
        } else if ("byte".equals(type)) {
            obj = Byte.valueOf(value);
        } else if ("char".equals(type) || "character".equals(type)) {
            obj = new Character((char) Integer.parseInt(value));
        } else if ("short".equals(type)) {
            obj = Short.valueOf(value);
        } else if ("int".equals(type) || "integer".equals(type)) {
            obj = Integer.valueOf(value);
        } else if ("long".equals(type)) {
            obj = Long.valueOf(value);
        } else if ("float".equals(type)) {
            obj = Float.valueOf(value);
        } else if ("double".equals(type)) {
            obj = Double.valueOf(value);
        }

        return obj;
    }

    /**
     * プリミティブラッパーを取得する。
     * 
     * @param type 型
     * @return プリミティブラッパー
     */
    public static Class getPrimitiveWrapperClass(String type) {
        Class clazz = null;

        // ラッパーを取得する。
        if ("bool".equals(type) || "boolean".equals(type)) {
            clazz = Boolean.class;
        } else if ("byte".equals(type)) {
            clazz = Byte.class;
        } else if ("char".equals(type) || "character".equals(type)) {
            clazz = Character.class;
        } else if ("short".equals(type)) {
            clazz = Short.class;
        } else if ("int".equals(type) || "integer".equals(type)) {
            clazz = Integer.class;
        } else if ("long".equals(type)) {
            clazz = Long.class;
        } else if ("float".equals(type)) {
            clazz = Float.class;
        } else if ("double".equals(type)) {
            clazz = Double.class;
        }

        return clazz;
    }

    /**
     * インスタンスを取得する。
     * 
     * @param type クラス名
     * @param classes コンストラクタの引数の型
     * @param arguments コンストラクタの引数
     * @param search コンストラクタの検索をする場合はtrue
     * @return インスタンス
     */
    public static Object getInstance(String type, Class[] classes,
            Object[] arguments, boolean search) {
        Class clazz = getClass(type);
        return getInstance(clazz, classes, arguments, search);
    }

    /**
     * インスタンスを取得する。
     * 
     * @param clazz クラス
     * @param classes コンストラクタの引数の型
     * @param arguments コンストラクタの引数
     * @param search コンストラクタの検索をする場合はtrue
     * @return インスタンス
     */
    public static Object getInstance(Class clazz, Class[] classes,
            Object[] arguments, boolean search) {
        try {
            Constructor constructor = getConstructor(clazz, classes, search);
            constructor.setAccessible(true);
            return constructor.newInstance(arguments);
        } catch (SecurityException e) {
            throw new BodyBuilderException(e);
        } catch (IllegalArgumentException e) {
            throw new BodyBuilderException(e);
        } catch (InstantiationException e) {
            throw new BodyBuilderException(e);
        } catch (IllegalAccessException e) {
            throw new BodyBuilderException(e);
        } catch (InvocationTargetException e) {
            throw new BodyBuilderException(e);
        }
    }

    /**
     * コンストラクタを取得する。
     * 
     * @param clazz クラス
     * @param types パラメータの型
     * @param search コンストラクタの検索をする場合はtrue
     * @return コンストラクタ
     */
    public static Constructor getConstructor(Class clazz, Class[] types,
            boolean search) {
        try {
            if (types == null || types.length < 1) {
                return clazz.getConstructor(new Class[0]);
            } else if (!search) {
                return clazz.getConstructor(types);
            }

            Constructor constructor = null;
            Constructor[] constructors = clazz.getDeclaredConstructors();

            for (int i = 0; i < constructors.length; i++) {
                Class[] classes = constructors[i].getParameterTypes();

                if (classes.length != types.length) {
                    continue;
                }

                boolean equalsTypes = true;

                for (int j = 0; j < types.length; j++) {
                    if (!instance_of(types[j], classes[j])) {
                        equalsTypes = false;
                        break;
                    }
                }

                if (equalsTypes) {
                    constructor = constructors[i];
                    break;
                }
            }

            if (constructor == null) {
                throw new BodyBuilderException("cannot find constructor "
                        + Arrays.asList(types) + ".");
            }

            return constructor;
        } catch (SecurityException e) {
            throw new BodyBuilderException("cannot find constructor "
                    + Arrays.asList(types) + ".", e);
        } catch (NoSuchMethodException e) {
            throw new BodyBuilderException("cannot find constructor "
                    + Arrays.asList(types) + ".", e);
        }
    }

    /////////////////////////////////////////////////////////////////
    // array method

    /**
     * 配列かどうか判定する。
     * 
     * @param type 型
     * @return 配列の場合はtrue
     */
    public static boolean isArray(String type) {
        return type.matches("^[^\\[]+(\\[\\])+$");
    }

    /**
     * 配列のクラス名を取得する。
     * 
     * @param type 型
     * @return 配列のクラス名
     */
    public static String getArrayClassName(String type) {
        Pattern pattern = Pattern.compile("^([^\\[]+)((\\[\\])+)$");
        Matcher matcher = pattern.matcher(type);

        if (!matcher.matches()) {
            return null;
        }

        String componentType = matcher.group(1);
        String dimension = matcher.group(2);
        StringBuffer arrayType = new StringBuffer();
        int plane = dimension.length() / 2;

        for (int i = 0; i < plane; i++) {
            arrayType.append("[");
        }

        if ("bool".equals(componentType) || "boolean".equals(componentType)) {
            arrayType.append("Z");
        } else if ("byte".equals(componentType)) {
            arrayType.append("B");
        } else if ("char".equals(componentType)
                || "character".equals(componentType)) {
            arrayType.append("C");
        } else if ("short".equals(componentType)) {
            arrayType.append("S");
        } else if ("int".equals(componentType)
                || "integer".equals(componentType)) {
            arrayType.append("I");
        } else if ("long".equals(componentType)) {
            arrayType.append("J");
        } else if ("float".equals(componentType)) {
            arrayType.append("F");
        } else if ("double".equals(componentType)) {
            arrayType.append("D");
        } else {
            arrayType.append("L");
            arrayType.append(componentType);
            arrayType.append(";");
        }

        return arrayType.toString();
    }

    /////////////////////////////////////////////////////////////////
    // class getter

    /**
     * クラスを取得する。
     * 
     * @param type クラス名
     * @return クラス
     */
    public static Class getClass(String type) {
        try {
            if (isPrimitive(type)) {
                return getPrimitiveClass(type);
            }

            return Class.forName(type);
        } catch (ClassNotFoundException e) {
            throw new BodyBuilderException(e);
        }
    }

    /**
     * プリミティブクラスを取得する。
     * 
     * @param type クラス名
     * @return プリミティブクラス
     */
    public static Class getPrimitiveClass(String type) {
        Class clazz = null;

        // プリミティブクラスを取得。
        if ("bool".equals(type) || "boolean".equals(type)) {
            clazz = boolean.class;
        } else if ("byte".equals(type)) {
            clazz = byte.class;
        } else if ("char".equals(type) || "character".equals(type)) {
            clazz = char.class;
        } else if ("short".equals(type)) {
            clazz = short.class;
        } else if ("int".equals(type) || "integer".equals(type)) {
            clazz = int.class;
        } else if ("long".equals(type)) {
            clazz = long.class;
        } else if ("float".equals(type)) {
            clazz = float.class;
        } else if ("double".equals(type)) {
            clazz = double.class;
        }

        return clazz;
    }

    /////////////////////////////////////////////////////////////////
    // object info utility

    /**
     * クラス名を取得する。
     * 
     * @param object オブジェクト
     * @param omit 省略する場合はtrue
     * @return クラス名
     */
    public static String getName(Object object, boolean omit) {
        if (object == null) {
            return "null";
        }

        return getName(object.getClass(), omit);
    }

    /**
     * クラス名を取得する。
     * 
     * @param clazz クラス
     * @param omit 省略する場合はtrue
     * @return クラス名
     */
    public static String getName(Class clazz, boolean omit) {
        String className = null;

        if (clazz.isArray()) {
            // 配列の場合
            className = clazz.getComponentType().getName() + "[]";
        } else {
            className = clazz.getName();
        }

        // クラス名を省略する場合、パッケージ名を削除する。
        if (omit) {
            className = omit(className);
        }

        if (Proxy.isProxyClass(clazz)) {
            StringBuffer proxyName = new StringBuffer();
            Class[] interfaces = clazz.getInterfaces();
            proxyName.append(className);
            proxyName.append("(");

            for (int i = 0; i < interfaces.length; i++) {
                String interfaceName = interfaces[i].getName();
                proxyName.append(omit(interfaceName));

                if (i > 0) {
                    proxyName.append(",");
                }
            }

            proxyName.append(")");
            className = proxyName.toString();
        }

        return className;
    }

    /**
     * 完全修飾クラス名からパッケージ名を削除する。
     * 
     * @param className 完全修飾クラス名
     * @return パッケージ名を削除したクラス名
     */
    private static String omit(String className) {
        int pos = className.lastIndexOf('.');

        if (pos > 0) {
            className = className.substring(pos + 1);
        }

        return className;
    }

    /**
     * サイズを取得する。
     * 
     * @param object オブジェクト
     * @return サイズ
     */
    public static String getSize(Object object) {
        if (object == null) {
            return null;
        }

        Class clazz = object.getClass();

        // 配列の場合
        if (clazz.isArray()) {
            return String.valueOf(Array.getLength(object));
        }

        // 配列以外の場合はsize()、またはlength()の値を返す。
        Method method = null;

        // size()
        try {
            method = clazz.getMethod("size", new Class[0]);
        } catch (NoSuchMethodException e) {
            method = null;
        }

        // length()
        try {
            if (method == null) {
                method = clazz.getMethod("length", new Class[0]);
            }
        } catch (NoSuchMethodException e) {
            method = null;
        }

        if (method == null) {
            return null;
        }

        String size = null;

        // サイズを取得。
        try {
            Object ret = method.invoke(object, new Object[0]);

            if (ret != null) {
                size = ret.toString();
            }
        } catch (NumberFormatException e) {
            size = null;
        } catch (IllegalAccessException e) {
            size = null;
        } catch (InvocationTargetException e) {
            size = null;
        }

        return size;
    }

    /**
     * オブジェクトの情報を取得する。
     * 
     * @param object オブジェクト
     * @return オブジェクトの情報
     */
    public static String getInfo(Object object) {
        return getInfo(object, false);
    }

    /**
     * オブジェクトの情報を取得する。
     * 
     * @param object オブジェクト
     * @param appendValue オブジェクトの値を出力する場合はtrue
     * @return オブジェクトの情報
     */
    public static String getInfo(Object object, boolean appendValue) {
        if (object == null) {
            return "null";
        }

        // クラス名を取得。
        StringBuffer buffer = new StringBuffer();
        buffer.append(ObjectUtils.getName(object, true));

        // サイズを取得。
        String size = ObjectUtils.getSize(object);

        if (size != null) {
            buffer.append("(" + size + ")");
        }

        // 値を取得。
        if (appendValue) {
            buffer.append(" \"");
            buffer.append(object.toString());
            buffer.append("\"");
        }

        return buffer.toString();
    }

    /////////////////////////////////////////////////////////////////
    // class list getter

    /**
     * 実装するクラスのリストを取得する。
     * 
     * @param object オブジェクト
     * @return 実装するクラスのリスト
     */
    public static List getClassNames(Object object) {
        return getClassNames(object, false);
    }

    /**
     * 実装するクラスのリストを取得する。
     * 
     * @param object オブジェクト
     * @param isReverse 逆順の場合はtrue
     * @return 実装するクラスのリスト
     */
    public static List getClassNames(Object object, boolean isReverse) {
        if (object == null) {
            List classNames = new ArrayList();
            classNames.add("null");
            return classNames;
        }

        return getClassNames(object.getClass(), isReverse);
    }

    /**
     * 実装するクラスのリストを取得する。
     * 
     * @param clazz クラス
     * @return 実装するクラスのリスト
     */
    public static List getClassNames(Class clazz) {
        return getClassNames(clazz, false);
    }

    /**
     * 実装するクラスのリストを取得する。
     * 
     * @param clazz クラス
     * @param isReverse 逆順の場合はtrue
     * @return 実装するクラスのリスト
     */
    public static List getClassNames(Class clazz, boolean isReverse) {
        List classNames = new ArrayList();

        if (clazz.isPrimitive()) {
            // プリミティブ型の場合
            classNames.add(clazz.getName());
            classNames.add("primitive");
        } else if (clazz.isArray()) {
            // 配列の場合
            String name = clazz.getComponentType().getName() + "[]";
            classNames.add(getArrayClassName(name));
            classNames.add(name);
            classNames.add("array");
        } else {
            // それ以外の場合
            final String objectClassName = Object.class.getName();
            List interfaceNames = new ArrayList();
            Class cls = clazz;

            // 親クラス名を取得。
            while (cls != null && !cls.getName().equals(objectClassName)) {
                classNames.add(cls.getName());
                // インターフェイス名を取得。
                addInterfaceNames(cls, interfaceNames);
                cls = cls.getSuperclass();
            }

            // インターフェース名をリストに追加。
            classNames.addAll(interfaceNames);
        }

        if (isReverse) {
            Collections.reverse(classNames);
        }

        return classNames;
    }

    /**
     * インターフェース名をクラス名リストに追加する。
     * 
     * @param clazz クラス
     * @param names クラス名リスト
     */
    private static void addInterfaceNames(Class clazz, List names) {
        // インターフェースを取得。
        Class[] interfaces = clazz.getInterfaces();

        // インターフェース名を再帰的に取得。
        for (int i = 0; i < interfaces.length; i++) {
            String interfaceName = interfaces[i].getName();

            if (names.contains(interfaceName)) {
                continue;
            }

            names.add(interfaceName);
            addInterfaceNames(interfaces[i], names);
        }
    }

    /////////////////////////////////////////////////////////////////
    // method utility

    /**
     * メソッドを取得する。
     * 
     * @param object オブジェクト
     * @param name メソッド名
     * @param classes 引数のクラス
     * @param search メソッドの検索をする場合はtrue
     * @return メソッド
     */
    public static Method getMethod(Object object, String name, Class[] classes,
            boolean search) {
        if (object == null) {
            throw new BodyBuilderException("object that '" + name
                    + "' defined is null.");
        }

        return getMethod(object.getClass(), name, classes, search);
    }

    /**
     * メソッドを取得する。
     * 
     * @param clazz クラス
     * @param types パラメータの型
     * @param search メソッドの検索をする場合はtrue
     * @return メソッド
     */
    public static Method getMethod(Class clazz, String name, Class[] types,
            boolean search) {
        try {
            if (types == null || types.length < 1) {
                return clazz.getMethod(name, new Class[0]);
            } else if (!search) {
                return clazz.getMethod(name, types);
            }

            Method method = null;
            Method[] pmethods = clazz.getMethods();
            Method[] dmethods = clazz.getDeclaredMethods();
            Method[] methods = new Method[pmethods.length + dmethods.length];
            System.arraycopy(pmethods, 0, methods, 0, pmethods.length);
            System.arraycopy(dmethods, 0, methods, pmethods.length,
                    dmethods.length);

            for (int i = 0; i < methods.length; i++) {
                if (!methods[i].getName().equals(name)) {
                    continue;
                }

                Class[] classes = methods[i].getParameterTypes();

                if (classes.length != types.length) {
                    continue;
                }

                boolean equalsTypes = true;

                for (int j = 0; j < types.length; j++) {
                    if (!instance_of(types[j], classes[j])) {
                        equalsTypes = false;
                        break;
                    }
                }

                if (equalsTypes) {
                    method = methods[i];
                    break;
                }
            }

            if (method == null) {
                throw new BodyBuilderException("cannot find method '" + name
                        + "' as '" + Arrays.asList(types) + "'");
            }

            return method;
        } catch (SecurityException e) {
            throw new BodyBuilderException("cannot get method '" + name + " "
                    + Arrays.asList(types) + "' at '" + clazz.getName() + "'.",
                    e);
        } catch (NoSuchMethodException e) {
            throw new BodyBuilderException("cannot get method '" + name + " "
                    + Arrays.asList(types) + "' at '" + clazz.getName() + "'.",
                    e);
        }
    }

    /**
     * メソッドを実行する。
     * 
     * @param type クラス名
     * @param name メソッド名
     * @param classes 引数のクラス
     * @param arguments 引数
     * @param search メソッドの検索をする場合はtrue
     * @return メソッドの戻り値
     */
    public static Object invokeMethod(String type, String name,
            Class[] classes, Object[] arguments, boolean search) {
        return invokeMethod(getClass(type), null, name, classes, arguments,
                search);
    }

    /**
     * メソッドを実行する。
     * 
     * @param object オブジェクト
     * @param name メソッド名
     * @param classes 引数のクラス
     * @param arguments 引数
     * @param search メソッドの検索をする場合はtrue
     * @return メソッドの戻り値
     */
    public static Object invokeMethod(Object object, String name,
            Class[] classes, Object[] arguments, boolean search) {
        if (object == null) {
            throw new BodyBuilderException("object that '" + name
                    + "' defined is null.");
        }

        return invokeMethod(object.getClass(), object, name, classes,
                arguments, search);
    }

    /**
     * メソッドを実行する。
     * 
     * @param clazz クラス
     * @param object オブジェクト
     * @param name メソッド名
     * @param classes 引数のクラス
     * @param arguments 引数
     * @param search メソッドの検索をする場合はtrue
     * @return メソッドの戻り値
     */
    public static Object invokeMethod(Class clazz, Object object, String name,
            Class[] classes, Object[] arguments, boolean search) {
        Method method = getMethod(clazz, name, classes, search);
        return invokeMethod(clazz, object, method, arguments);
    }

    /**
     * メソッドを実行する。
     * 
     * @param clazz クラス
     * @param object オブジェクト
     * @param method メソッド
     * @param arguments 引数
     * @return メソッドの戻り値
     */
    public static Object invokeMethod(Class clazz, Object object,
            Method method, Object[] arguments) {
        try {
            // privateメソッドも実行する。
            method.setAccessible(true);
            return method.invoke(object, arguments);
        } catch (SecurityException e) {
            throw new BodyBuilderException("cannot invole method '"
                    + method.getName() + "' at '" + object + "'.", e);
        } catch (IllegalArgumentException e) {
            throw new BodyBuilderException("cannot invole method '"
                    + method.getName() + "' at '" + object + "'.", e);
        } catch (IllegalAccessException e) {
            throw new BodyBuilderException("cannot invole method '"
                    + method.getName() + "' at '" + object + "'.", e);
        } catch (InvocationTargetException e) {
            throw new BodyBuilderException("cannot invole method '"
                    + method.getName() + "' at '" + object + "'.", e);
        }
    }

    /**
     * スタティックメソッドかどうかを取得する。
     * 
     * @param method メソッド
     * @return スタティックメソッドの場合はtrue
     */
    public static boolean isStaticMethod(Method method) {
        return Modifier.isStatic(method.getModifiers());
    }

    /////////////////////////////////////////////////////////////////
    // field utility

    /**
     * フィールドの値を取得する。
     * 
     * @param type クラス名
     * @param name フィールド名
     * @return フィールドの値
     */
    public static Object getFiledValue(String type, String name) {
        return getFiledValue(getClass(type), null, name);
    }

    /**
     * フィールドの値を取得する。
     * 
     * @param object オブジェクト
     * @param name フィールド名
     * @return フィールドの値
     */
    public static Object getFiledValue(Object object, String name) {
        if (object == null) {
            throw new BodyBuilderException("object that '" + name
                    + "' defined is null.");
        }

        return getFiledValue(object.getClass(), object, name);
    }

    /**
     * フィールドの値を取得する。
     * 
     * @param clazz クラス
     * @param object オブジェクト
     * @param name フィールド名
     * @return フィールドの値
     */
    public static Object getFiledValue(Class clazz, Object object, String name) {
        try {
            Field field = null;

            try {
                // publicフィールド(スーパークラス含む)を取得。
                field = clazz.getField(name);
            } catch (NoSuchFieldException e) {
                // public以外のフィールドを取得。
                field = clazz.getDeclaredField(name);
                field.setAccessible(true);
            }

            return field.get(object);
        } catch (SecurityException e) {
            throw new BodyBuilderException("cannot get field '" + name
                    + "' as '" + object + "(" + clazz + ")'.", e);
        } catch (NoSuchFieldException e) {
            throw new BodyBuilderException("cannot get field '" + name
                    + "' as '" + object + "(" + clazz + ")'.", e);
        } catch (IllegalArgumentException e) {
            throw new BodyBuilderException("cannot get field '" + name
                    + "' as '" + object + "(" + clazz + ")'.", e);
        } catch (IllegalAccessException e) {
            throw new BodyBuilderException("cannot get field '" + name
                    + "' as '" + object + "(" + clazz + ")'.", e);
        }
    }

    /////////////////////////////////////////////////////////////////
    // instanceof

    /**
     * クラスが指定された型のインスタンスかどうかを返す。
     * 
     * @param clazz クラス
     * @param type 型
     * @return インスタンスの場合はtrue
     */
    public static boolean instance_of(Class clazz, String type) {
        if (isPrimitive(type)) {
            if (clazz.isPrimitive()) {
                return clazz.getName().equals(type);
            } else {
                Class wrapper = getPrimitiveWrapperClass(type);
                return wrapper.getName().equals(clazz.getName());
            }
        } else if (Object.class.getName().equals(type)) {
            return true;
        } else {
            List classes = getClassNames(clazz);
            return classes.contains(type);
        }
    }

    /**
     * クラスが指定された型のインスタンスかどうかを返す。
     * 
     * @param clazz クラス
     * @param type 型
     * @return インスタンスの場合はtrue
     */
    public static boolean instance_of(Class clazz, Class type) {
        return instance_of(clazz, type.getName());
    }

    /**
     * オブジェクトが指定された型のインスタンスかどうかを返す。
     * 
     * @param obj オブジェクト
     * @param type 型
     * @return インスタンスの場合はtrue
     */
    public static boolean instance_of(Object obj, String type) {
        if (obj == null) {
            return false;
        } else {
            return instance_of(obj.getClass(), type);
        }
    }

    /**
     * オブジェクトが指定された型のインスタンスかどうかを返す。
     * 
     * @param obj オブジェクト
     * @param type 型
     * @return インスタンスの場合はtrue
     */
    public static boolean instance_of(Object obj, Class type) {
        return instance_of(obj, type.getName());
    }

    /////////////////////////////////////////////////////////////////
    // package getter

    /**
     * 指定されたオブジェクトのパッケージ名を取得する。
     * 
     * @param clazz クラス
     * @param asterisk アスタリスクをつけるかどうかのフラグ
     * @return パッケージ名
     */
    public static String getPackage(String clazz, boolean asterisk) {
        int index = clazz.lastIndexOf('.');

        if (index < 0) {
            return "";
        }

        String pkg = clazz.substring(0, index);

        return asterisk ? pkg + ".*" : pkg;
    }

    /**
     * 指定されたクラスのパッケージ名を取得する。
     * 
     * @param clazz クラス
     * @return パッケージ名
     */
    public static String getPackage(String clazz) {
        return getPackage(clazz, false);
    }

    /**
     * 指定されたクラスのパッケージ名を取得する。
     * 
     * @param clazz クラス
     * @return パッケージ名
     */
    public static String getPackage(Class clazz) {
        return getPackage(clazz.getName());
    }

    /**
     * 指定されたオブジェクトのパッケージ名を取得する。
     * 
     * @param obj オブジェクト
     * @return パッケージ名
     */
    public static String getPackage(Object obj) {
        if (obj == null) {
            return "";
        }

        return getPackage(obj.getClass().getName());
    }

}