package bodybuilder.test;

import java.io.File;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import bodybuilder.exception.BodyBuilderException;
import bodybuilder.util.Config;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestResult;
import junit.framework.TestSuite;

/**
 * XMLテストランナー
 */
public class XMLTestRunner extends TestCase {

    private static final Log logger = LogFactory.getLog(XMLTestRunner.class);

    /////////////////////////////////////////////////////////////////
    // constant field

    /**
     * 成功終了
     */
    public static final int SUCCESS_EXIT = 0;

    /**
     * 失敗終了
     */
    public static final int FAILURE_EXIT = 1;

    /**
     * 例外終了
     */
    public static final int EXCEPTION_EXIT = 2;

    /////////////////////////////////////////////////////////////////
    // private constant field

    /**
     * テストケースの拡張子
     */
    private static final String TESTCASE_EXT = ".xml";

    /////////////////////////////////////////////////////////////////
    // running method

    /**
     * メインメソッド
     * 
     * @param args 引数
     */
    public static void main(String[] args) {
        try {
            // テストを実行。
            TestResult result = doRun(suite(args));

            if (!result.wasSuccessful()) {
                // 失敗終了
                System.exit(FAILURE_EXIT);
            }

            // 成功終了
            System.exit(SUCCESS_EXIT);
        } catch (Throwable e) {
            // 例外終了
            System.exit(EXCEPTION_EXIT);
        }
    }

    /**
     * テストを実行する。
     * 
     * @param suite テストスイート
     * @return テスト結果
     */
    public static TestResult doRun(Test suite) {
        // テスト結果オブジェクトを生成。
        // TODO テスト結果の出力先はSystem.outに固定。
        XMLResultPrinter printer = new XMLResultPrinter(System.out);
        TestResult result = new TestResult();
        result.addListener(printer);

        // テストを実行する。
        long startTime = System.currentTimeMillis();
        suite.run(result);
        long endTime = System.currentTimeMillis();
        long runTime = endTime - startTime;

        // 結果を出力する。
        printer.print_(result, runTime);
        // 結果を返す。
        return result;
    }

    /////////////////////////////////////////////////////////////////
    // build method

    /**
     * テストスイートを取得する。
     * 
     * @return テストスイート
     */
    public static Test suite() throws Throwable {
        return suite(null);
    }

    /**
     * テストスイートを取得する。
     * 
     * @param roots ルートディレクトリ
     * @return テストスイート
     */
    public static Test suite(String[] roots) throws Throwable {
        // テストスイートを生成。
        TestSuite suite = new TestSuite();

        // ルートディレクトリが渡されなかった場合は、設定ファイルから取得。
        if (roots == null || roots.length < 1) {
            roots = Config.getTestRootDirs();
        }

        // テストスイートを構築。
        for (int i = 0; i < roots.length; i++) {
            buildSuite(roots[i], suite);
        }

        // テストスイートを返す。
        return suite;
    }

    /**
     * テストスイートを構築する。
     * 
     * @param path パス
     * @param suite テストスイート
     */
    static void buildSuite(String path, TestSuite suite) {
        File file = new File(path);

        // ファイルが存在しない場合はエラー。
        if (!file.exists()) {
            warn("cannot find file '" + path + "'.");
            addErrorTest(file, new BodyBuilderException("cannot find file '"
                    + path + "'."), suite);
            return;
        }

        // 無視するリソースを取得。
        List ignores = Config.getTestIgnores();

        // 無視するリソースの場合は処理をスキップ。
        if (ignores.contains(file.getName())) {
            return;
        }

        if (file.isDirectory()) {
            // ディレクトリの場合

            String[] children = file.list();

            // 再帰的にbuildSuite()を呼び出す。
            for (int i = 0; i < children.length; i++) {
                File child = new File(file.getPath(), children[i]);
                buildSuite(child.getAbsolutePath(), suite);
            }
        } else if (path.toLowerCase().endsWith(TESTCASE_EXT)) {
            // XMLファイルの場合
            debug("add test '" + path + "'.");

            // テストを追加。
            try {
                addTest(file, suite);
            } catch (Exception e) {
                if (logger.isDebugEnabled()) {
                    debug("cannot add test '" + path + "'", e);
                } else if (logger.isWarnEnabled()) {
                    warn("cannot add test '" + path + "': " + e.getMessage());
                }

                addErrorTest(file, e, suite);
            }
        }
    }

    /**
     * テストを追加する。
     * 
     * @param xmlfile XMLファイル
     * @param e 例外
     * @param suite テストスイート
     */
    private static void addTest(File xmlfile, TestSuite suite) {
        // XMLを生成して、テストを追加。
        XML xml = XML.newXML(xmlfile);
        suite.addTest(xml.getTest());
    }

    /**
     * エラーテストを追加する。
     * 
     * @param xmlfile XMLファイル
     * @param e 例外
     * @param suite テストスイート
     */
    private static void addErrorTest(File xmlfile, Throwable e, TestSuite suite) {
        suite.addTest(new ErrorTestCase(xmlfile, e));
    }

    /////////////////////////////////////////////////////////////////
    // logging method

    /**
     * デバッグログを出力する。
     * 
     * @param message メッセージ
     */
    protected static void debug(String message) {
        logger.debug(message);
    }

    /**
     * デバッグログを出力する。
     * 
     * @param message メッセージ
     * @param e 例外
     */
    protected static void debug(String message, Throwable e) {
        logger.debug(message, e);
    }

    /**
     * 情報ログを出力する。
     * 
     * @param message メッセージ
     */
    protected static void info(String message) {
        logger.info(message);
    }

    /**
     * 情報ログを出力する。
     * 
     * @param message メッセージ
     * @param e 例外
     */
    protected static void info(String message, Throwable e) {
        logger.info(message, e);
    }

    /**
     * 警告ログを出力する。
     * 
     * @param message メッセージ
     */
    protected static void warn(String message) {
        logger.warn(message);
    }

    /**
     * 警告ログを出力する。
     * 
     * @param message メッセージ
     * @param e 例外
     */
    protected static void warn(String message, Throwable e) {
        logger.warn(message, e);
    }

}