package bodybuilder.util;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.beanutils.PropertyUtils;

import bodybuilder.exception.BodyBuilderException;

/**
 * 拡張プロパティユーティリティ
 */
public class ExtendedPropertyUtils extends PropertyUtils {

    /**
     * プロパティをマップとして取得する。
     * 
     * @param bean ビーン
     * @return プロパティのマップ
     */
    public static Map getProperties(Object bean) {
        Map map = new HashMap();
        // プロパティを取得。
        PropertyDescriptor[] props = PropertyUtils.getPropertyDescriptors(bean);

        // プロパティをマップに格納。
        for (int i = 0; i < props.length; i++) {
            // getterのないプロパティは無視。
            if (props[i].getReadMethod() == null) {
                continue;
            }

            String name = props[i].getName();

            // Object#getClass()は無視。
            if ("class".equals(name)) {
                continue;
            }

            Object value = getProperty(bean, name);
            map.put(name, value);
        }

        return map;
    }

    /**
     * プロパティを取得する。
     * 
     * @param bean ビーン
     * @param name プロパティ名
     * @return プロパティ値
     */
    public static Object getProperty(Object bean, String name) {
        try {
            return PropertyUtils.getProperty(bean, name);
        } catch (IllegalAccessException e) {
            throw new BodyBuilderException(e);
        } catch (InvocationTargetException e) {
            throw new BodyBuilderException(e);
        } catch (NoSuchMethodException e) {
            throw new BodyBuilderException(e);
        }
    }

    /**
     * プロパティを取得する。
     * 
     * @param bean ビーン
     * @param name プロパティ名
     * @param value プロパティ値
     */
    public static void setProperty(Object bean, String name, Object value) {
        try {
            PropertyDescriptor prop = PropertyUtils.getPropertyDescriptor(bean,
                    name);
            Class clazz = prop.getPropertyType();

            if (value != null && !clazz.isArray()
                    && !ObjectUtils.instance_of(value, clazz)
                    && value instanceof String) {
                String valStr = (String) value;

                if (clazz.isPrimitive()) {
                    value = ObjectUtils.getPrimitiveWrapperInstance(valStr,
                            clazz.getName());
                } else {
                    Constructor constructor = clazz
                            .getConstructor(new Class[] { String.class });
                    value = constructor.newInstance(new Object[] { valStr });
                }
            }

            PropertyUtils.setProperty(bean, name, value);
        } catch (Exception e) {
            throw new BodyBuilderException("cannot set value '" + value
                    + "' to property '" + name + "'.", e);
        }
    }

    /**
     * プロパティをコピーする。
     * 
     * @param dest コピー先
     * @param orig コピー元
     */
    public static void copyProperties(Object dest, Object orig) {
        try {
            PropertyUtils.copyProperties(dest, orig);
        } catch (IllegalAccessException e) {
            throw new BodyBuilderException(e);
        } catch (InvocationTargetException e) {
            throw new BodyBuilderException(e);
        } catch (NoSuchMethodException e) {
            throw new BodyBuilderException(e);
        }
    }

}