;
; rcsID=$Id: PrePostProcess.lisp,v 1.16 2008/04/29 16:25:15 tmatsugaki Exp $
;
(proclaim '(inline
                   linePreProcessInEncoding
                   linePostProcessInEncoding
                   linePreProcessInDecoding
                   linePostProcessInDecoding
                   wordPreProcess
                   wordPreProcessInJp
                   wordPreProcessInUs
                   wordPreProcessInFq
                   wordPreProcessInAcbc
                   wordPreProcessInDigit
                   wordPostProcess
                   wordPostProcessInJp
                   wordPostProcessInUs
                   wordPostProcessInFq
                   wordPostProcessInAcbc
                   wordPostProcessInDigit
                   decideCombination
                   maintainSubstrCount
                   suppressSigns
                   wholeWordPostProcess
                   maintainBrackets
))
(proclaim '(optimize speed))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 linePreProcessInEncoding
;; 【機能　】 墨字文字列→点字文字列の変換の前処理をする。
;; 【入力　】 istr  : 入力墨字文字列
;;            grade : グレード
;; 【出力　】 墨字文字列
;; 【例外　】 なし
;; 【使用例】 (linePreProcessInEncoding ";,,NHK　ノ　テレビ　プログラムデ、")
;;            → "；，，ＮＨＫ ノ テレビ プログラムデ、"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun linePreProcessInEncoding (istr putCapSign dicGen grade)
  (let ((str istr))
    ; 冒頭にコントロールコード（改ページなど）がある場合は削除する。
    (if (char< (char str 0) #\Space)
      (setf str (subseq str 1))
      nil)
    ; 大文字符を付与する。
    (if (and putCapSign (= grade 2))
      (setf str (emit-capital-sign str *nonAlphaList*))
      nil)
    ; 全角文字列に変換する。
    (setf str (wide-string str grade))
    (if (/= grade 2)
      (progn
        ; 下がり数字を漢数字（内部形式）に変換する。
        (setf str (lowerNumber2KanjiNumber str))
        ; 全角のスペースを半角のスペースに変換する。
        (setf str (substitute-string str _blank_w_ _blank_s_))
        ; "。" に続く2つの半角のスペースを2つの全角のスペースに変換する。
        (setf str (substitute-string str "。  " _kutenSumiji_))
        ; "、" に続く1つの半角のスペースを1つの全角のスペースに変換する。
        (setf str (substitute-string str "、 " _toutenSumiji_)))
      (progn
        ; 全角大文字を全角小文字に変換する。
        (if str
          (setf str (string-downcase str))
          nil)
        ; 全角のスペースを半角のスペースに変換する。
        (setf str (substitute-string str _blank_w_ _blank_s_))
        (if (not dicGen)
            ; 通常のエンコード時には、末端に半角のスペースを1つ追加する。
            (setf str (concatenate 'string str _blank_s_))
            nil)))
    str))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 linePostProcessInEncoding
;; 【機能　】 墨字文字列→点字文字列の変換の後処理をする。
;; 【入力　】 istr  : 入力墨字文字列
;;            ostr  : 処理中の点字文字列
;;            grade : グレード
;; 【出力　】 出力用点字文字列
;; 【例外　】 なし
;; 【使用例】 (linePostProcessInEncoding "")
;;            → ""
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun linePostProcessInEncoding (istr ostr grade)
  (let ((ret nil) (str ostr))
    (if (/= grade 2)
      (progn
        ; 入力行末が "、" の場合は辞書には無いので、セミコロンを追加する。
        (if (char= (char istr (1- (length istr))) #\IDEOGRAPHIC_COMMA)
          (setf str (format nil "~A;" str))
          nil)
        ; 入力行末が "。" の場合は辞書には無いので、"．" を追加する。
        (if (char= (char istr (1- (length istr))) #\IDEOGRAPHIC_FULL_STOP)
          (setf str (format nil "~A4" str))
          nil))
      nil)
    ; 半角文字列に変換する。
    (setf ret (narrow-string (string-right-trim _blank_s_ str) grade))
    ; '¥' を外部形式の '\' に変換する。
    (setf ret (substitute-string ret "¥" "\\"))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 linePreProcessInDecoding
;; 【機能　】 点字文字列→墨字文字列の変換の前処理をする。
;; 【入力　】 istr : 入力点字文字列
;;            grade : グレード
;; 【出力　】 点字文字列
;; 【例外　】 なし
;; 【使用例】 (linePreProcessInDecoding "ＡＢＣＤ")
;;            → "abcd"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun linePreProcessInDecoding (istr grade)
  (let ((str istr))
    ; 冒頭にコントロールコード（改ページなど）がある場合は削除する。
    (if (char< (char str 0) #\Space)
      (setf str (subseq str 1))
      nil)
    ; 半角文字列に変換する。（全角の筈が無いので本来は不要）
    (setf str (narrow-string str grade))
    ; 半角小文字の文字列に変換する。
    (setf str (string-downcase str))
    ; '\' を内部形式の '¥' に変換する。
    (setf str (substitute-string str "\\" "¥"))
    ; 英語グレード２の場合は、行末に半角空白を 1つ追加する。
    ; 【目的】行末に Whole Word が位置した場合、*uk_grade2*/*us_grade2* 中の空白を従えた
    ;  Whole Word へ適合させる。
    (if (= grade 2)
      (setf str (concatenate 'string str _blank_s_))
      nil)
    str))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 linePostProcessInDecoding
;; 【機能　】 点字文字列→墨字文字列の変換の後処理をする。
;; 【入力　】 ostr    : 処理中の墨字文字列
;;            hankaku : 出力を半角文字列にするか否か
;;            grade   : グレード
;; 【出力　】 出力用墨字文字列
;; 【例外　】 なし
;; 【使用例】 (linePostProcessInDecoding "ッッッッッッ")
;;            → "‥‥‥‥‥‥"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun linePostProcessInDecoding (ostr hankaku grade)
  (let ((ret ostr))
    (if (/= grade 2)
      (progn
        ; 連続した "ッ" を "‥" に変換する。
        (setf ret (substitute-string (substitute-string ret "ッッ" "‥‥") "‥ッ" "‥‥"))
        ; 連続した "ーー" を "――" に変換する。
        (setf ret (substitute-string ret "ーー" "――"))
        ; 出力行末のセミコロンは、"、" に変換する。
        (if (char= (char ret (1- (length ret))) #\;)
          (setf ret (concatenate 'string (subseq ret 0 (1- (length ret))) _touten_w_))
          nil)
        ; 出力行末の "．" は、"。" に変換する。
        (if (char= (char ret (1- (length ret))) #\FULLWIDTH_FULL_STOP)
          (setf ret (concatenate 'string (subseq ret 0 (1- (length ret))) _kuten_w_))
          nil))
      ; 英語グレード２の場合
      (progn
        ; 前置詞・不定詞・接続詞が連なっていない場合に不正な全角空白+半角空白を全角空白に変換する。
        (setf ret (substitute-string ret (concatenate 'string (string #\Space) (string #\IDEOGRAPHIC_SPACE)) (string #\IDEOGRAPHIC_SPACE)))
        ; 余分な末端の空白を削除する。
        (setf ret (string-right-trim '(#\Space #\IDEOGRAPHIC_SPACE) ret))
        ; 内部形式でシングルクォートの開始として使用しているバッククォートをシングルクォートに置換する。
        (if *dontInOutBackQuote*
          (setf ret (substitute-string ret "`" "'"))
          nil)))
    (if hankaku
      (setf ret (narrow-string (string-right-trim _blank_w_ ret) grade))
      (setf ret (wide-string (string-right-trim _blank_w_ ret) grade)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 wordPreProcess
;; 【機能　】 変換の事前に点字／墨字の文字列を評価して、新点字モードと新言語を
;;          決定する。
;; 【入力　】 o_mode     : 前回の変換時の言語モード
;;            token      : 墨字(全角)・点字(半角)の変換元の候補
;;            last_token : 前回の変換結果(墨字(全角)・点字(半角))
;;            grade      : グレード
;;            encode     : t の場合 token を墨字として処理、nil の場合 token
;;                         を点字として処理する。
;; 【出力　】 新点字モードと新言語
;;            mode : 
;;            lang : 
;; 【例外　】 なし
;; 【使用例】 (wordPreProcess "Us" "," ";" nil) → "US" ;
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun wordPreProcess (o_mode token last_token grade encode)
  (let (mode (upcaseM0 (string-upcase (subseq o_mode 0 1))))
    (cond
      ; （日本モード）
      ((string= upcaseM0 "J")
        (setf mode (wordPreProcessInJp o_mode token last_token encode)))
      ; （英語モード）
      ((string= upcaseM0 "U")
        (setf mode (wordPreProcessInUs o_mode token last_token grade encode)))
      ; （外語引用モード）
      ((string= upcaseM0 "F")
        (setf mode (wordPreProcessInFq o_mode token grade encode)))
      ; （米国点字符号モード）
      ((string= upcaseM0 "A")
        (setf mode (wordPreProcessInAcbc o_mode token grade encode)))
      ; （数字モード）
      ((string= upcaseM0 "D")
        (setf mode (wordPreProcessInDigit o_mode token last_token grade encode)))
      ; （エラー）
      (t (format t "~%Error: wordPreProcess")))
    (getLang mode t)
    mode))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 wordPreProcessInJp
;; 【機能　】 日本語モードの場合の点字文字列を評価して、点字のモードを取得する。
;; 【入力　】 o_mode     : 前回の変換時の言語モード
;;            token      : 墨字(全角)・点字(半角)の変換元の候補
;;            last_token : 前回の変換結果(墨字(全角)・点字(半角))
;;            encode     : t の場合 token を墨字として処理、nil の場合 token
;;                         を点字として処理する。
;; 【出力　】 点字モードの文字列
;; 【例外　】 なし
;; 【使用例】 (wordPreProcessInJp "JP" "；" "" t) → "us"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun wordPreProcessInJp (o_mode n_token last_token encode)
  (let ((ret o_mode))
    (if encode
      ; エンコード
      (cond
        ((string= n_token _letterSign_w_) (setf ret *defaultEnglish*))
        ((string= n_token _numeralSign_w_) (setf ret "DI"))
        ((and (string/= last_token _hyphen_s_) (zero-nine-wcharp (char n_token 0)))
          (setf ret "DI")))
      ; デコード
      (cond
        ((string= n_token _numeralSign_s_) (setf ret "DI"))))
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 wordPreProcessInUs
;; 【機能　】 us/Us/US モードの場合の点字文字列を評価して、点字のモードを取得する。
;; 【入力　】 o_mode     : 前回の変換時の言語モード
;;            token      : 墨字(全角)・点字(半角)の変換元の候補
;;            last_token : 前回の変換結果(墨字(全角)・点字(半角))
;;            grade      : グレード
;;            encode     : t の場合 token を墨字として処理、nil の場合 token
;;                         を点字として処理する。
;; 【出力　】 点字モードの文字列
;; 【例外　】 なし
;; 【使用例】 (wordPreProcessInUs "us" "，" "" 1 t) → "Us"
;;           (wordPreProcessInUs "us" "#" "" 2 nil) → "DI"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun wordPreProcessInUs (o_mode n_token last_token grade encode)
  (let ((ret nil) (upcaseM0 "") c)
    (if encode
      ; エンコード
      (progn
        (if (> (length o_mode) 0)
          (setf upcaseM0 (string-upcase (subseq o_mode 0 1)))
          nil)
        (setf c (char (narrow-string n_token grade) 0))
        (cond
          ; 【ルール3a】英語モードのハイフン(エンコード時プリ処理)
          ;  "-" は日本語モードで "「" または "」" として処理したいところだが、
          ; "Ｉ－サン" などを正しく処理するためにプリ処理で英語モードを維持させるが、
          ; グレード１の場合 "－"（ダッシュ）出力後にポスト処理で日本語モードに遷移させる。
          ; よって、a-j-etc-charp で数字と同列には扱わない。
          ((string= n_token _hyphen_w_) (setf ret o_mode))

          ; "；" を検出した。
          ((string= n_token _letterSign_w_)
            (progn
              (setf ret *defaultEnglish*)
              ; 英語グレード２で " ;" を検出したので、強制英語グレード１モードにする。
              (if (and (= grade 2)
                       (or (string= last_token "")
                           (us_ignoreableLeft last_token nil encode)))
                (setf *forceGrade1* t)
                nil)))
          ((and (string= o_mode "uk")
                (string= n_token _doubleCapitalSign_w_))
            (setf ret "UK"))
          ((and (string= o_mode "us")
                (string= n_token _doubleCapitalSign_w_))
            (setf ret "US"))
          ; 大文字符は、冒頭に位置するかクォーテーション等に連なる。
          ((and (string= o_mode "uk")
                (string= n_token _capitalSign_w_))
            (setf ret "Uk"))
          ((and (string= o_mode "UK")
                (alpha-etc-charp c))
            (setf ret "UK"))
          ((and (string= o_mode "us")
                (string= n_token _capitalSign_w_))
            (setf ret "Us"))
          ((and (string= o_mode "US")
                (alpha-etc-charp c))
            (setf ret "US"))
          ((string= n_token _acbcStartSign_w_)
            (setf ret "ac"))
          ((and (string= n_token _numeralSign_w_)
                (or (string= last_token "")        ; 数値モードは、空白やダッシュを超える。
                    (string= last_token _blank_w_) ; 数値モードは、空白やダッシュを超える。
                    (string= last_token _dash_w_)  ; 数値モードは、空白やダッシュを超える。
                    (and (/= grade 2)              ; Jan 2, 2008 コメントアウトを検討したが、NG (ISSUE-3)
                         (both-case-p (char (narrow-string last_token grade) 0)))))
            (setf ret "DI"))
          ((and (string= upcaseM0 "U")
                (> (length n_token) 0)
                (not (both-case-p c)))
            (setf ret *defaultMode*))
          (t (setf ret o_mode))))
      ; デコード
      (progn
        (if (> (length o_mode) 0)
          (setf upcaseM0 (string-upcase (subseq o_mode 0 1)))
          nil)
        (setf c (char n_token 0))
        (cond
          ; 【ルール3a】英語モードのハイフン(デコード時プリ処理)
          ;  "-" は日本語モードで "「" または "」" として処理したいところだが、
          ; "Ｉ－サン" などを正しく処理するためにプリ処理で英語モードを維持させるが、
          ; グレード１の場合 "－"（ダッシュ）出力後にポスト処理で日本語モードに遷移させる。
          ; よって、a-j-etc-charp で数字と同列には扱わない。
          ((string= n_token _hyphen_s_) (setf ret o_mode))

          ; ";" を検出した。
          ((string= n_token _letterSign_s_)
            (progn
              (setf ret *defaultEnglish*)
              ; 英語グレード２で " ;" を検出したので、強制英語グレード１モードにする。
              (if (and (= grade 2)
                       (or (string= last_token "")
                           (us_ignoreableLeft last_token nil encode)))
                (setf *forceGrade1* t)
                nil)))
          ((and (string= o_mode "uk")
                (string= n_token _doubleCapitalSign_s_))
            (setf ret "UK"))
          ((and (string= o_mode "uk")
                (string= n_token _capitalSign_s_))
            (setf ret "Uk"))
          ((and (string= o_mode "UK")
                (alpha-etc-charp c))
            (setf ret "UK"))
          ((and (string= o_mode "us")
                (string= n_token _doubleCapitalSign_s_))
            (setf ret "US"))
          ((and (string= o_mode "us")
                (string= n_token _capitalSign_s_))
            (setf ret "Us"))
          ((and (string= o_mode "US")
                (alpha-etc-charp c))
            (setf ret "US"))
          ((string= n_token _acbcStartSign_s_)
            (setf ret "ac"))
          ((and (string= n_token _numeralSign_s_)
                (or (string= last_token "")         ; 数値モードは、空白やダッシュを超える。
                    (string= last_token _blank_s_)  ; 数値モードは、空白やダッシュを超える。
                    (string= last_token _hyphen_s_) ; 数値モードは、空白やダッシュを超える。
                    (and (/= grade 2)               ; Jan 2, 2008 コメントアウトを検討したが、NG (ISSUE-3)
                         (both-case-p (char last_token 0)))))
            (setf ret "DI"))
          ((and (string= upcaseM0 "U")
                (/= grade 2)
                (> (length n_token) 0)
                (not (both-case-p c)))
            (setf ret *defaultMode*))
          (t (setf ret o_mode)))))
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 wordPreProcessInFq
;; 【機能　】 外語引用モードの場合の点字文字列を評価して、点字のモードを取得する。
;; 【入力　】 o_mode     : 前回の変換時の言語モード
;;            token      : 墨字(全角)・点字(半角)の変換元の候補
;;            grade      : グレード
;;            encode     : t の場合 token を墨字として処理、nil の場合 token
;;                         を点字として処理する。
;; 【出力　】 点字モードの文字列
;; 【例外　】 なし
;; 【使用例】 (wordPreProcessInFq "fq" "，" t) → "Fq"
;;           (wordPreProcessInFq "Fq" "," nil) → "FQ"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun wordPreProcessInFq (o_mode n_token grade encode)
  (let ((ret nil) (upcaseM0 "") c)
    (if encode
      ; エンコード
      (progn
        (if (> (length o_mode) 0)
          (setf upcaseM0 (string-upcase (subseq o_mode 0 1)))
          nil)
        (setf c (char (narrow-string n_token grade) 0))
        (cond
          ((and (string= o_mode "fq") (string= n_token _doubleCapitalSign_w_)) (setf ret "FQ"))
          ((and (string= o_mode "fq") (string= n_token _capitalSign_w_)) (setf ret "Fq"))
          ((and (string= o_mode "Fq") (string= n_token _capitalSign_w_)) (setf ret "FQ"))
          ; 外語引用モードからの復帰では、日本語の "ん" が出力されるのを防ぐため
          ; ここではなく、ポスト処理で日本語モードに遷移させる。
          (t (setf ret o_mode))))
      ; デコード
      (progn
        (if (> (length o_mode) 0)
          (setf upcaseM0 (string-upcase (subseq o_mode 0 1)))
          nil)
        (setf c (char n_token 0))
        (cond
          ((and (string= o_mode "fq") (string= n_token _doubleCapitalSign_s_)) (setf ret "FQ"))
          ((and (string= o_mode "fq") (string= n_token _capitalSign_s_)) (setf ret "Fq"))
          ((and (string= o_mode "Fq") (string= n_token _capitalSign_s_)) (setf ret "FQ"))
          ; 外語引用モードからの復帰では、日本語の "ん" が出力されるのを防ぐため
          ; ここではなく、ポスト処理で日本語モードに遷移させる。
          (t (setf ret o_mode)))))
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 wordPreProcessInAcbc
;; 【機能　】 外語引用モードの場合の点字文字列を評価して、点字のモードを取得する。
;; 【入力　】 o_mode     : 前回の変換時の言語モード
;;            token      : 墨字(全角)・点字(半角)の変換元の候補
;;            grade      : グレード
;;            encode     : t の場合 token を墨字として処理、nil の場合 token
;;                         を点字として処理する。
;; 【出力　】 点字モードの文字列
;; 【例外　】 なし
;; 【使用例】 (wordPreProcessInAcbc "ac" "，" t) → "Ac"
;;            (wordPreProcessInAcbc "ac" "," nil) → "Ac"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun wordPreProcessInAcbc (o_mode n_token grade encode)
  (let ((ret nil) (upcaseM0 "") c)
    (if encode
      ; エンコード
      (progn
        (if (> (length o_mode) 0)
          (setf upcaseM0 (string-upcase (subseq o_mode 0 1)))
          nil)
        (setf c (char (narrow-string n_token grade) 0))
        (cond
          ((and (string= o_mode "ac") (string= n_token _doubleCapitalSign_w_)) (setf ret "AC"))
          ((and (string= o_mode "ac") (string= n_token _capitalSign_w_)) (setf ret "Ac"))
          ((and (string= o_mode "Ac") (string= n_token _capitalSign_w_)) (setf ret "AC"))
          ; 外語引用モードからの復帰では、日本語の "ん" が出力されるのを防ぐため
          ; ここではなく、ポスト処理で日本語モードに遷移させる。
          (t (setf ret o_mode))))
      ; デコード
      (progn
        (if (> (length o_mode) 0)
          (setf upcaseM0 (string-upcase (subseq o_mode 0 1)))
          nil)
        (setf c (char n_token 0))
        (cond
          ((and (string= o_mode "ac") (string= n_token _doubleCapitalSign_s_)) (setf ret "AC"))
          ((and (string= o_mode "ac") (string= n_token _capitalSign_s_)) (setf ret "Ac"))
          ((and (string= o_mode "Ac") (string= n_token _capitalSign_s_)) (setf ret "AC"))
          ; 外語引用モードからの復帰では、日本語の "ん" が出力されるのを防ぐため
          ; ここではなく、ポスト処理で日本語モードに遷移させる。
          (t (setf ret o_mode)))))
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 wordPreProcessInDigit
;; 【機能　】 数字モードの場合の点字文字列を評価して、点字のモードを取得する。
;; 【入力　】 o_mode     : 前回の変換時の言語モード
;;            token      : 墨字(全角)・点字(半角)の変換元の候補
;;            last_token : 前回の変換結果(墨字(全角)・点字(半角))
;;            grade      : グレード
;;            encode     : t の場合 token を墨字として処理、nil の場合 token
;;                         を点字として処理する。
;; 【出力　】 点字モードの文字列
;; 【例外　】 なし
;; 【使用例】 (wordPreProcessInDigit "DI" "０" "" 1 t) → "DI"
;;            (wordPreProcessInDigit "DI" "-" "0" 1 nil) → "DI"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun wordPreProcessInDigit (o_mode n_token last_token grade encode)
  (let ((ret nil))
    (if encode
      ; エンコード
      (cond
        ; 【ルール2a】数字モードのハイフン(エンコード時プリ処理)
        ;  "-" は日本語モードで "「" または "」" として処理したいところだが、
        ; "＃１０００－＃２０００" などを正しく処理するためにプリ処理で数字モードを
        ; 維持させるが、グレード１の場合 "－"（ダッシュ）出力後にポスト処理で日本語モードに
        ; 遷移させる。よって、zero-nine-etc-charp で数字と同列には扱わない。
        ((string= n_token _dash_w_) (setf ret o_mode))

        ((zero-nine-etc-charp (char (narrow-string n_token grade) 0)) (setf ret o_mode))
        ((string= n_token "’") (setf ret o_mode)) ; Mar 17, 2007
;        ((string= n_token _comma_w_) (setf ret o_mode)) ; Jan 2, 2008 導入を検討したが、NG (ISSUE-3)
        ((string= n_token _numeralSign_w_) (setf ret "DI"))
        (t (setf ret *defaultMode*)))
      ; デコード
      (cond
        ; 【ルール1】数字モードの数符の直後でない下がり数字(デコード時プリ処理)
        ;  数字モードでデコード時に数符の直後でない下がり数字を検出したら、デフォルトモード
        ;  にする。eg. "＃１ーゐ" でなく "＃１』" を出力する。
        ((and (zero-nine-charp (char n_token 0))
              (string/= last_token _numeralSign_s_))
          (setf ret *defaultMode*))

        ; 【ルール2a】数字モードのハイフン(デコード時プリ処理)
        ;  "-" は日本語モードで "「" または "」" として処理したいところだが、
        ; "＃１０００－＃２０００" などを正しく処理するためにプリ処理で数字モードを
        ; 維持させるが、グレード１の場合 "－"（ダッシュ）出力後にポスト処理で日本語モードに
        ; 遷移させる。よって、a-j-etc-charp で数字と同列には扱わない。
        ((string= n_token _hyphen_s_) (setf ret o_mode))

        ((a-j-etc-charp (char n_token 0)) (setf ret o_mode))
        ((string= n_token "'") (setf ret o_mode)) ; Mar 17, 2007
;        ((string= n_token _comma_s_) (setf ret o_mode)) ; Jan 2, 2008 導入を検討したが、NG (ISSUE-3)
        ((string= n_token _numeralSign_s_) (setf ret "DI"))
        (t (setf ret *defaultMode*))))
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 wordPostProcess
;; 【機能　】 変換後に点字／墨字の文字列を評価して、新点字モードと新言語を決定する。
;; 【入力　】 mode   : 現在の言語モード
;;            brl    : 点字の変換元の候補(全角)
;;            txt    : 墨字の変換元の候補(半角)
;;            jpPrep : 濁音、半濁音を前置きするか否か
;;            grade  : グレード
;;            encode : t の場合 token を墨字として処理、nil の場合 token
;;                     を点字として処理する。
;; 【出力　】 新点字モードと新言語
;;            m      : 点字モードの文字列
;;            l      : 言語
;;            s      : 必要に応じて加工された文字列
;; 【例外　】 なし
;; 【使用例】 (wordPostProcess "Us" "," "，" nil nil)
;;            → "Us" ;
;;               ((| | . 0)...
;;               ","
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun wordPostProcess (mode brl txt jpPrep grade encode)
  (let (m s (upcaseM0 (string-upcase (subseq mode 0 1))))
    (cond
      ; （日本語モード）
      ((string= upcaseM0 "J")
        (multiple-value-setq (m s) (wordPostProcessInJp mode brl txt jpPrep grade encode)))
      ; （英語モード）
      ((string= upcaseM0 "U")
        (multiple-value-setq (m s) (wordPostProcessInUs mode brl txt grade encode)))
      ; （外語引用モード）
      ((string= upcaseM0 "F")
        (multiple-value-setq (m s) (wordPostProcessInFq mode brl txt encode)))
      ; （米国点字符号モード）
      ((string= upcaseM0 "A")
        (multiple-value-setq (m s) (wordPostProcessInAcbc mode brl txt encode)))
      ; （数字モード）
      ((string= upcaseM0 "D")
        (multiple-value-setq (m s) (wordPostProcessInDigit mode brl txt grade encode)))
      ; （エラー）
      (t (format t "~%Error: wordPostProcess")))
    (getLang m t)
    (values m s)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 wordPostProcessInJp
;; 【機能　】 変換後に点字／墨字の文字列を評価して、新点字モードと新言語を決定する。
;; 【入力　】 mode   : 現在の言語モード
;;            brl    : 点字の変換元の候補(全角)
;;            txt    : 墨字の変換元の候補(半角)
;;            jpPrep : 濁音、半濁音を前置きするか否か
;;            grade  : グレード
;;            encode : t の場合 token を墨字として処理、nil の場合 token
;;                     を点字として処理する。
;; 【出力　】 新点字モードと新言語
;;            m      : 点字モードの文字列
;;            s      : 必要に応じて加工された文字列
;; 【例外　】 なし
;; 【使用例】 (wordPostProcessInJp "us" "8" "" nil t)
;;            → "fq" ;
;;               "8"
;;           (wordPostProcessInJp "us" "" "{" nil nil)
;;            → "fq" ;
;;               "{"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun wordPostProcessInJp (mode brl txt jpPrep grade encode)
  (let ((m mode) s)
    (if encode
      (setf s brl)
      (setf s txt))
    ; 外字符をフェッチした。
    (if (string= s _letterSign_s_)
      (setf m *defaultEnglish*)
      nil)
    (if encode
      ; エンコード（日本語モード）
      ; 日本語モードで "8"('｛') (外国語引用符開始)を出力したら、外語引用モードに遷移させる。
      (if (string= brl _foreignQuotationStartSign_s_)
        (setf m "fq")
        nil)
      ; デコード（日本語モード）
      (progn
        ; 【括弧モード管理】状態遷移があった場合は、適宜調整する。
        (if (maintainBrackets brl txt grade encode)
          ; 括弧や鍵括弧は開始・終了に同じ点字コードを使用しているので、墨字をユニークにする。
          (cond
            ((and (not (inUsBracket)) (string= txt "（")) (setf s "）"))
            ((and (not (inUsSquareBracket)) (string= txt "「")) (setf s "」")))
          nil)
        ; '{' (外国語引用符開始)を出力したら、外語引用モードに遷移させる。
        (if (string= txt "{")
          (setf m "fq")
          nil)
        ; 濁音・半濁音に '"' や ',' を前置きする。
        (if jpPrep
          (setf s (voicedSound-or-pSound s))
          nil)))
    (values m s)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 wordPostProcessInUs
;; 【機能　】 変換後に点字／墨字の文字列を評価して、新点字モードと新言語を決定する。
;; 【入力　】 mode   : 現在の言語モード
;;            brl    : 点字の変換元の候補(全角)
;;            txt    : 墨字の変換元の候補(半角)
;;            grade  : グレード
;;            encode : t の場合 token を墨字として処理、nil の場合 token
;;                     を点字として処理する。
;; 【出力　】 新点字モードと新言語
;;            m      : 点字モードの文字列
;;            s      : 必要に応じて加工された文字列
;; 【例外　】 なし
;; 【使用例】 (wordPostProcessInUs "US" "_+" "" t)
;;            → "ac" ;
;;               "_+"
;;            (wordPostProcessInUs "Us" "" "a" nil)
;;            → "us" ;
;;               "A"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun wordPostProcessInUs (mode brl txt grade encode)
  (let ((m mode) s
        (rawM0 (subseq mode 0 1))
        (rawM1 (subseq mode 1 2)))
    (if encode
      (setf s brl)
      (setf s txt))
    ; 外字符をフェッチした。
    (if (string= s _letterSign_s_)
      (setf m *defaultEnglish*)
      nil)
    (if encode
      ; エンコード（英語モード）
      (progn
        ; 英語モードで "_+" (ACBC開始)を出力したら、米国点字符号モードに遷移させる。
        (if (string= brl _acbcStartSign_s_)
          (setf m "ac")
          nil)
        ; 【括弧モード管理】状態遷移があった場合は、適宜調整する。
        (if (maintainBrackets brl txt grade encode)
          ; 必要があれば、調整する。
          ; シングルクォートの開始・終了およびアポストロフィーは同じ墨字コードを使用している
          ; ので、それぞれユニークな点字に振り分ける。
          ;
          ; 【語の冒頭】:シングルクォートの開始、アポストロフィー
          ; 【語中】　　:アポストロフィー
          ; 【語の末端】:シングルクォートの終了、アポストロフィー
          ()
          nil)
        (cond
          ((= grade 1)
            (progn
              ; 【ルール3b】英語モードのハイフン(エンコード時ポスト処理)
              ; 英語モードで "-" を出力したら、ポスト処理で "JP"モードに遷移させる。
              ;
              ;  "-" は日本語モードで "「" または "」" として処理したいところだが、
              ; "Ｉ－サン" などを正しく処理するため例外的にプリ処理で英語モードを
              ; 維持させるが、グレード１の場合 "－"（ダッシュ）出力後にポスト処理で日本語
              ; モードに遷移させる。よって、zero-nine-etc-charp で数字と同列には扱わない。
              (if (string= brl _hyphen_s_)
                (setf m *defaultMode*)
                nil)))
          ((= grade 2)
            (progn
              ; 英語グレード２で " ;" 検出による強制英語グレード１モードを解除する。
              (if (and (string/= brl _letterSign_s_)
                       (string/= brl _capitalSign_s_)
                       (string/= brl _hyphen_s_)
                       (not (both-case-p (char brl 0))))
                (setf *forceGrade1* nil)
                nil)
              ; 英語グレード２で "4#" 検出したので、数値モードに遷移させる。
              (if (string= brl _dollar_brl_)
                (setf m "DI")
                nil)))))
      ; デコード（英語モード）
      (progn
        (cond
          ((string= rawM0 "U")
            (progn
              ; 大文字変換
              (if (and (= grade 2) (> (length txt) 1) (string= rawM1 "s"))
                ; 英語グレード２で "Us" の場合
                (let (fore rest)
                  (setf fore (subseq txt 0 1))
                  (setf rest (subseq txt 1 (length txt)))
                  (setf s (format nil "~A~A" (string-upcase fore) (string-downcase rest))))
                ; 英語グレード２で "Us" 以外の場合
                (setf s (string-upcase txt)))
              (cond
                ; "Uk" モードで 1文字出力したら、"uk"モードに降格させる。
                ((and (string= m "Uk") (both-case-p (char txt 0)))
                  (progn (setf m "uk")))
                ; 二重大文字符モードで "－"（ダッシュ）"　" を出力したので、"uk"モードに降格させる。
                ((and (string= m "UK") (or (string= txt _dash_w_) (string= txt _blank_w_)))
                  (setf m "uk"))
                ; "Us" モードで 1文字出力したら、"us"モードに降格させる。
                ((and (string= m "Us") (both-case-p (char txt 0)))
                  (progn (setf m "us")))
                ; 二重大文字符モードで "－"（ダッシュ）"　" を出力したので、"us"モードに降格させる。
                ((and (string= m "US") (or (string= txt _dash_w_) (string= txt _blank_w_)))
                  (setf m "us"))))))
        ; 【括弧モード管理】状態遷移があった場合は、適宜調整する。
        (if (maintainBrackets brl txt grade encode)
          ; 括弧や角括弧や大括弧は開始・終了に同じ点字コードを使用しているので、墨字をユニークにする。
          (cond
            ((and (not (inUsBracket)) (string= txt "(")) (setf s ")"))
            ((and (not (inUsAngleBracket)) (string= txt "<")) (setf s ">"))
            ((and (not (inUsBraceBracket)) (string= txt "{")) (setf s "}")))
          nil)
        (cond
          ((= grade 1)
            (progn
              ; 【ルール3b】英語モードのハイフン(デコード時ポスト処理)
              ; 英語モードで "－"（ダッシュ）を出力したら、ポスト処理で "JP"モードに遷移させる。
              ;
              ;  "-" は日本語モードで "「" または "」" として処理したいところだが、
              ; "Ｉ－サン" などを正しく処理するため例外的にプリ処理で英語モードを
              ; 維持させるが、グレード１の場合 "－"（ダッシュ）出力後にポスト処理で日本語
              ; モードに遷移させる。よって、a-j-etc-charp で数字と同列には扱わない。
              (if (string= txt _dash_w_)
                (setf m *defaultMode*)
                nil)))
          ((= grade 2)
            (progn
              ; 英語グレード２で " ;" 検出による強制英語グレード１モードを解除する。
              (if (and (string/= txt _letterSign_s_)
                       (string/= txt _capitalSign_s_)
                       (string/= txt _hyphen_s_)
                       (not (both-case-p (char txt 0))))
                (setf *forceGrade1* nil)
                nil)
              ; 英語グレード２で "4#" 検出したので、数値モードに遷移させる。
              (if (string= brl _dollar_brl_)
                (setf m "DI")
                nil))))))
    (values m s)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 wordPostProcessInFq
;; 【機能　】 変換後に点字／墨字の文字列を評価して、新点字モードと新言語を決定する。
;; 【入力　】 mode   : 現在の言語モード
;;            brl    : 点字の変換元の候補(全角)
;;            txt    : 墨字の変換元の候補(半角)
;;            encode : t の場合 token を墨字として処理、nil の場合 token
;;                     を点字として処理する。
;; 【出力　】 新点字モードと新言語
;;            m      : 点字モードの文字列
;;            s      : 必要に応じて加工された文字列
;; 【例外　】 なし
;; 【使用例】 (wordPostProcessInFq "Fq" "0" "" t)
;;            → "JP" ;
;;               "0"
;;            (wordPostProcessInFq "Fq" "," "}" nil)
;;            → "JP" ;
;;               "}"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun wordPostProcessInFq (mode brl txt encode)
  (let ((m mode) s
        (rawM0 (subseq mode 0 1)))
    (if encode
      (setf s brl)
      (setf s txt))
    ; 外字符をフェッチした。
    (if (string= s _letterSign_s_)
      (setf m *defaultEnglish*)
      nil)
    (if encode
      ; エンコード（外語引用モード）
      ; 外語引用モードで "0"('}') (外国語引用符終了)を出力したら、デフォルト（日本語モード）に遷移させる。
      (if (string= brl _foreignQuotationStopSign_s_)
        (setf m *defaultMode*)
        nil)
      ; デコード（外語引用モード）
      (progn
        (cond
          ((string= rawM0 "F")
            (progn
              ; 大文字変換
              (setf s (string-upcase txt))
              ; "Fq" モードで 1文字出力したら、"fq"モードに降格させる。
              (cond
                ((and (string= m "Fq") (both-case-p (char txt 0)))
                  (setf m "fq")))
              ; "FQ" モードで "." "/" を出力したので、"fq"モードに降格させる。
              (cond
                ((and (string= m "FQ") (or (char= (char txt 0) #\.) (char= (char txt 0) #\/)))
                  (setf m "fq"))))))
        ; '}' (外国語引用符終了)を出力したら、デフォルト（日本語モード）に遷移させる。
        (if (string= txt "}")
          (setf m *defaultMode*)
          nil)))
    (values m s)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 wordPostProcessInAcbc
;; 【機能　】 変換後に点字／墨字の文字列を評価して、新点字モードと新言語を決定する。
;; 【入力　】 mode   : 現在の言語モード
;;            brl    : 点字の変換元の候補(全角)
;;            txt    : 墨字の変換元の候補(半角)
;;            encode : t の場合 token を墨字として処理、nil の場合 token
;;                     を点字として処理する。
;; 【出力　】 新点字モードと新言語
;;            m      : 点字モードの文字列
;;            s      : 必要に応じて加工された文字列
;; 【例外　】 なし
;; 【使用例】 (wordPostProcessInAcbc "ac" "_:" "" t)
;;            → "JP" ;
;;               "_:"
;;            (wordPostProcessInAcbc "Ac" "z" "ｚ" nil)
;;            → "ac" ;
;;               "Ｚ"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun wordPostProcessInAcbc (mode brl txt encode)
  (let ((m mode) s
        (rawM0 (subseq mode 0 1)))
    (if encode
      (setf s brl)
      (setf s txt))
    ; 外字符をフェッチした。
    (if (string= s _letterSign_s_)
      (setf m *defaultEnglish*)
      nil)
    (if encode
      ; エンコード（米国点字符号モード）
      ; 米国点字符号モードで "_:" (ACBC終了)を出力したら、デフォルト（英語モード）に遷移させる。
      (if (string= brl _acbcStopSign_s_)
        (setf m *defaultMode*)
        nil)
      ; デコード（米国点字符号モード）
      (progn
        (cond
          ((string= rawM0 "A")
            (progn
              ; 大文字変換
              (setf s (string-upcase txt))
              ; "Ac" モードで 1文字出力したら、"ac"モードに降格させる。
              (cond
                ((and (string= m "Ac") (both-case-p (char txt 0)))
                  (setf m "ac"))))))
        ; (ACBC終了)を出力したので、デフォルト（英語モード）に遷移させる。
        (if (string= txt _acbcStopSign_s_)
          (setf m *defaultMode*)
          nil)))
    (values m s)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 wordPostProcessInDigit
;; 【機能　】 変換後に点字／墨字の文字列を評価して、新点字モードと新言語を決定する。
;; 【入力　】 mode   : 現在の言語モード
;;            brl    : 点字の変換元の候補(全角)
;;            txt    : 墨字の変換元の候補(半角)
;;            grade  : グレード
;;            encode : t の場合 token を墨字として処理、nil の場合 token
;;                     を点字として処理する。
;; 【出力　】 新点字モードと新言語
;;            m      : 点字モードの文字列
;;            s      : 必要に応じて加工された文字列
;; 【例外　】 なし
;; 【使用例】 (wordPostProcessInDigit "DI" "-" "－" nil)
;;            → "JP" ;
;;               "－"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun wordPostProcessInDigit (mode brl txt grade encode)
  (let ((m mode) s)
    (if encode
      (setf s brl)
      (setf s txt))
    ; 外字符をフェッチした。
    (if (string= s _letterSign_s_)
      (setf m *defaultEnglish*)
      nil)
    (if encode
      ; エンコード（数字モード）
      (progn
        (cond
          ; 【ルール2b】数字モードのハイフン(エンコード時ポスト処理)
          ; 数字モードで "-" を出力したら、ポスト処理で "JP"モードに遷移させる。
          ;
          ;  "-" は日本語モードで "「" または "」" として処理したいところだが、
          ; "＃１０００－＃２０００" などを正しく処理するため例外的にプリ処理で
          ; 数字モードを維持させるが、グレード１の場合 "－"（ダッシュ）出力後にポスト処理で
          ; 日本語モードに遷移させる。よって、zero-nine-etc-charp で数字と同列には扱わない。
          ((= grade 1)
            (if (string= brl _hyphen_s_)
              (setf m *defaultMode*)
              nil))))
      ; デコード（数字モード）
      (progn
        (cond
          ; 【ルール2b】数字モードのハイフン(デコード時ポスト処理)
          ; 数字モードで"－"（ダッシュ）を出力したら、ポスト処理で "JP"モードに遷移させる。
          ;
          ;  "-" は日本語モードで "「" または "」" として処理したいところだが、
          ; "＃１０００－＃２０００" などを正しく処理するため例外的にプリ処理で
          ; 数字モードを維持させるが、グレード１の場合 "－"（ダッシュ）出力後にポスト処理で
          ; 日本語モードに遷移させる。よって、a-j-etc-charp で数字と同列には扱わない。
          ((= grade 1)
            (if (string= txt _dash_w_)
              (setf m *defaultMode*)
              nil)))))
    (values m s)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 decideCombination
;; 【機能　】 英語グレード２でのエンコード・デコード時の結合モードを管理する。
;; 【入力　】 left             : チャンクの冒頭から、変換対象の部分文字列までのオフセット
;;            past             : 変換済みの点字／墨字
;;            next             : 未変換の点字／墨字
;;            i_b              : 点字
;;            i_s              : 墨字
;;            hyphenated       : ハイフンに連なっている
;;            substr_cnt       : チャンクの部分文字列カウンタ
;;            inhibitMedialDot : "." の中間結合をインヒビットするか否か
;;            grade            : グレード
;;            encode           : t の場合 token を墨字として処理、nil の場合 token
;;                               を点字として処理する。
;; 【出力　】 返す。
;; 【例外　】 なし
;; 【使用例】 (decideCombination ) → 
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun decideCombination (left past next i_b i_s hyphenated substr_cnt inhibitMedialDot grade encode)
  (let ((b i_b) (s i_s) (whole nil) (init nil) (mid nil) (fin nil) (g2info nil))
    (if (= *debug* 8)
      (debug8-decideCombination 8 1 left past next i_b i_s hyphenated substr_cnt inhibitMedialDot _dbgContentMask_)
      nil)
    (if encode
      ; 【エンコード】
      (let ((hankakuStr (narrow-string s grade)) ww)
        (setf ww (us_wholeWordCandidate left past s next t encode))
        ; ハイフネーションの場合は、続く語を無条件で末尾結合とする。
        ; ハイフンで連結された語の場合は、続く語が構成符号である場合を除き末尾結合とする。
        (setf fin hyphenated)
        (if (not fin)
          (progn
            (setf whole ww)
            (if whole
              nil
              (if (setf mid (us_medialWordCandidate left past s next t encode))
                nil
                (if (setf fin (us_terminalWordCandidate left past s next t encode))
                  nil
                  (setf init (us_initialWordCandidate left past s next t encode))))))
          nil)
        ; 2文字以上の (whole word/initial word) の候補は、英語グレード２エンコード用の辞書を検索する。
        (if (and (= grade 2)
                 *lookup_dict*
                 (>  (length hankakuStr) 1)
                 (not (us_prep-inf-conj-p hankakuStr nil t))
                 ; 辞書検索は、アポストロフィーの前部なども wholeWord として検索させる。
                 ; トークンのデリミタは " " "--" と 1文字版の us_ignoreableRight が t となる文字列の 3種類である。
                 (or whole init fin
                     (and (= substr_cnt 0) (us_ignoreableRight next t encode nil))))
          (let (mask)
            (setf g2info (assoc (intern hankakuStr) *grade2_enc_dict*))
            ; 英語グレード２辞書の結合も、LiteraryUK/LiteraryUS の *grade2* に準拠する。
            (setf mask (nth _dic_flags_index_ g2info))
            (if (null (checkMask mask whole init mid fin))
              (setf g2info nil)
              nil))
          nil)
        (if g2info
          ; 英語グレード２のエンコード用辞書にある語を採用する。
          (progn
            (setf b (nth _dic_string_index_ g2info))
            (if (and (= *debug* 4) (null (nth _dic_mode_index_ g2info)))
              (format t "~%エンコード用辞書参照[~S]→[~S]" s b)
              nil))
          nil)
        (values whole init mid fin b nil nil g2info))
      ;
      ; 【デコード】
      (let ((modified nil) ww modifiedStr)
        (multiple-value-setq (ww modifiedStr)
          (us_wholeWordCandidate left past b next nil encode))
        ; us_wholeWordCandidate でチャンクの末端（デリミタ部分）が空白化された文字列。

        ; ハイフネーションの場合は、続く語を無条件で末尾結合とする。
        ; ハイフンで連結された語の場合は、続く語が構成符号である場合を除き末尾結合とする。
        (setf fin hyphenated)
        (if (not fin)
          (progn
            (setf whole ww)
            (if whole
              nil
              (progn
                (if (and inhibitMedialDot (string= b _dot_brl_))
                  (setf mid nil)
                  (setf mid (us_medialWordCandidate left past b next nil encode)))
                (if mid
                  nil
                  (if (setf fin (us_terminalWordCandidate left past b next nil encode))
                    nil
                    (setf init (us_initialWordCandidate left past b next nil encode)))))))
          nil)
        ; 2文字以上の (whole word/initial word) の候補は、英語グレード２デコード用の辞書を検索する。
        (if (and (= grade 2)
                 *lookup_dict*
                 (>  (length b) 1)
                 ; 辞書検索は、アポストロフィーの前部なども wholeWord として検索させる。
                 ; トークンのデリミタは " " "--" と 1文字版の us_ignoreableRight が t となる文字列の 3種類である。
                 (or whole init fin
                     (and (= substr_cnt 0) (us_ignoreableRight next nil encode nil))))
          (let (tmp mask)
            (setf tmp (string-upcase (substitute-string b "¥" "\\")))
            (setf g2info (assoc (intern tmp) *us_grade2_dec_dict*))
            ; 英語グレード２辞書の結合も、LiteraryUK/LiteraryUS の *grade2* に準拠する。
            (setf mask (nth _dic_flags_index_ g2info))
            (if (null (checkMask mask whole init mid fin))
              (setf g2info nil)
              nil))
          nil)
        (if g2info
          ; 英語グレード２デコード用の辞書にある語を採用する。
          (progn
            (setf s (nth _dic_string_index_ g2info))
            (if (and (= *debug* 4) (null (nth _dic_mode_index_ g2info)))
              (format t "~%デコード用辞書参照[~S]←[~S]" s b)
              nil))
          ; us_wholeWordCandidate でチャンクの末端（デリミタ部分）が空白化された場合は変換後に後処理が必要。
          (progn
            (setf modified (string/= b modifiedStr))
            (setf b modifiedStr)))
        (values whole init mid fin b s modified g2info)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 maintainSubstrCount
;; 【機能　】 連続した前置詞・不定詞・接続詞に対して部分文字列の数を制御する処理。（不要？）
;;          encodeBrailleChunk/decodeBrailleChunk/wordList2String に影響あり。
;; 【入力　】 b                : 点字
;;            s                : 墨字
;;            effective_last_s : エンコード時：前回エンコード元の墨字
;;                               デコード時　：前回エンコード結果の有効な墨字
;;            i_substr_cnt     : チャンクの部分文字列カウンタ
;;            encode           : t の場合 token を墨字として処理、nil の場合 token
;;                               を点字として処理する。
;; 【出力　】 部分文字列の数
;; 【例外　】 チャンクの部分文字列カウンタ
;; 【使用例】 (maintainSubstrCount "0" "by" "to" 2 nil) → 1
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun maintainSubstrCount (b s effective_last_s i_substr_cnt encode)
  (let ((substr_cnt i_substr_cnt))
    (if (= *debug* 5)
      (debug5-maintainSubstrCount 5 1 substr_cnt effective_last_s s b _dbgContentMask_)
      nil)
    (if encode
      ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
      ; 英語グレード２のエンコード時で前置詞・不定詞・接続詞を連結
      ; するモードで直前の部分文字列が前置詞・不定詞・接続詞であった
      ; 場合は、以下を行う。
      ;-----------------------------------------------------------------------
      ; 当該の部分文字列が空白である場合は、前置詞・不定詞・接続詞が連なっている場合に備えて
      ; チャンクの部分文字列カウンタを 1 に設定する。（連結）
      ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
      ; 空白を隔てた先行の語が前置詞・不定詞・接続詞である。
      ; 当該の語を前置詞・不定詞・接続詞の候補と見なす。
      (if
        (and (= substr_cnt 2)
             (us_prep-inf-conj-p effective_last_s nil t)
             (constant-char-stringp b #\Space))
        (setf substr_cnt 1)
        nil)
      ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
      ; 英語グレード２のデコード時　で前置詞・不定詞・接続詞を分割するモードで
      ; 当該の部分文字列（墨字）が前置詞・不定詞・接続詞である場合は、以下を
      ; 行う。（分割）
      ;-----------------------------------------------------------------------
      ; 　前置詞・不定詞・接続詞が連なっている場合に備えてチャンク
      ;   の部分文字列カウンタを 0 に設定する。
      ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
      (if
        (and (= substr_cnt 2)
             (string= s "by")
             (us_prep-inf-conj-p effective_last_s nil t))
        (setf substr_cnt 1)
        nil))
    (values substr_cnt)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 suppressSigns
;; 【機能　】 英語グレード２デコード時に適宜、数符・外字符・ACBC符・大文字符の出力を抑止する。
;; 【入力　】 s      : 墨字
;;          b      : 点字
;; 【出力　】 適宜加工された墨字文字列を返す。
;; 【例外　】 なし
;; 【使用例】 (suppressSigns ) → 
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun suppressSigns (s b)
  (let ((ret s))
    (if (or (string= s _numeralSign_s_)
            (and (string= s _letterSign_s_)
                 (string= b _letterSign_s_))
            (string= s _acbcStartSign_s_)
            (string= s _acbcStopSign_s_)
            (string= s _tripleCapitalSign_s_)
            (string= s _doubleCapitalSign_s_)
            (and (string= s _capitalSign_s_)
                 (string= b _capitalSign_s_)))
      (setf ret "")
      nil)
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 wholeWordPostProcess
;; 【機能　】 us_wholeWordCandidate でチャンクの末端（デリミタ部分）が空白化された場合は
;;          デコードされた語の末端の空白を削除し、空白化された語（デリミタ部分）を新たに、
;;          デコード対象に繰り入れる。
;; 【入力　】 i_s      : 
;;            i_right  : 
;; 【出力　】 を返す。
;; 【例外　】 なし
;; 【使用例】 (wholeWordPostProcess ) → 
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun wholeWordPostProcess (i_s i_right)
  (let ((s i_s) (right i_right))
    ;---------------------------------------------------------------------------
    ; us_wholeWordCandidate でチャンクの末端（デリミタ部分）が空白化された場合は、以下を実施する。
    ; 1.デコードされた語の末端の空白を削除し、
    ; 2.空白化された語（デリミタ部分）を新たに、デコード対象に繰り入れる。
    ;---------------------------------------------------------------------------
    (setf s (subseq s 0 (1- (length s))))
    (setf right (1- right))
    (values s right)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 maintainBrackets
;; 【機能　】 エンコード・デコード時のカッコのモードを管理する。
;;           尚、括弧の状態（*bracketStats*）を初期化するタイミングは以下の３箇所。
;;           1.プログラム起動時
;;           2.ファイルの変換開始時
;;           3.空白行を検出したタイミング（英語グレード２のみ）
;; 【入力　】 brl       : 点字文字列
;;            txt       : 墨字文字列
;;            grade     : グレード
;;            encode    : t の場合 token を墨字として処理、nil の場合 token
;;                        を点字として処理する。
;; 【出力　】 括弧の状態遷移の有無を t/nil で返す。
;; 【例外　】 なし
;; 【使用例】 (maintainBrackets "7" "（" 2 t) → t
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 以下の修正は右括弧のフラグの変更により、")" なども取得可能になったことによる変更。
; *bracketStats* による管理から外すことも可能であるが、正確に括弧の状態を知ることを
; 優先し依然管理する。
; ネスト対応のため *bracketStats* でなくスタックを使用するほど凝る予定はない。
; (on May 2 2007 by T.M.)
;
(defun maintainBrackets (brl txt grade encode)
  (let ((lastBracketStats *bracketStats*))
    (if encode
      ; エンコード
      (if (/= grade 2)
        nil
        (cond
          ((and (string= brl "7") (or (string= txt "（") (string= txt "）")) (not (inUsBracket)))
            (setf *bracketStats* (logior *bracketStats* _bracMask_)))
          ((and (string= brl "7") (or (string= txt "（") (string= txt "）"))      (inUsBracket))
            (setf *bracketStats* (logand *bracketStats* (lognot _bracMask_))))
          ((and (string= brl ".7") (or (string= txt "｛") (string= txt "｝")) (not (inUsBraceBracket)))
            (setf *bracketStats* (logior *bracketStats* _braceBracMask_)))
          ((and (string= brl ".7") (or (string= txt "｛") (string= txt "｝"))      (inUsBraceBracket))
            (setf *bracketStats* (logand *bracketStats* (lognot _braceBracMask_))))
          ((and (string= brl "^7") (string= txt "［") (not (inUsPhonemicBracket)))
            (setf *bracketStats* (logior *bracketStats* _phomicBracMask_)))
          ((and (string= brl _caret_s_) (string= txt "］")       (inUsPhonemicBracket))
            (setf *bracketStats* (logand *bracketStats* (lognot _phomicBracMask_))))
          ((and (string= brl ",7") (string= txt "［") (not (inUsSquareBracket)))
            (setf *bracketStats* (logior *bracketStats* _sqrBracMask_)))
          ((and (string= brl "7'") (string= txt "］")      (inUsSquareBracket))
            (setf *bracketStats* (logand *bracketStats* (lognot _sqrBracMask_))))
          ((and (string= brl "_7") (or (string= txt "＜") (string= txt "＞")) (not (inUsAngleBracket)))
            (setf *bracketStats* (logior *bracketStats* _anglBracMask_)))
          ((and (string= brl "_7") (or (string= txt "＜") (string= txt "＞"))      (inUsAngleBracket))
            (setf *bracketStats* (logand *bracketStats* (lognot _anglBracMask_))))
          ((and (string= brl "8") (string= txt "”") (not (inUsDoubleQuote)))
            (setf *bracketStats* (logior *bracketStats* _dQMask_)))
          ((and (string= brl "0") (string= txt "”")      (inUsDoubleQuote))
            (setf *bracketStats* (logand *bracketStats* (lognot _dQMask_))))
          ((and (string= brl _backQuote_brl_) (string= txt "’") (not (inUsSingleInvertedComma)))
            (setf *bracketStats* (logior *bracketStats* _sngInvCommMask_)))
          ((and (string= brl "J'") (string= txt "’")      (inUsSingleInvertedComma))
            (setf *bracketStats* (logand *bracketStats* (lognot _sngInvCommMask_))))))
      ; デコード
      (if (/= grade 2)
        (cond
          ((and (string= brl "-") (string= txt "「") (not (inUsSquareBracket)))
            (setf *bracketStats* (logior *bracketStats* _sqrBracMask_)))
          ((and (string= brl "-") (string= txt "「")      (inUsSquareBracket))
            (setf *bracketStats* (logand *bracketStats* (lognot _sqrBracMask_))))
          ((and (string= brl "7") (string= txt "（") (not (inUsBracket)))
            (setf *bracketStats* (logior *bracketStats* _bracMask_)))
          ((and (string= brl "7") (string= txt "（")      (inUsBracket))
            (setf *bracketStats* (logand *bracketStats* (lognot _bracMask_)))))
        ; 英語グレード２のデコードでは、シングルクォートは開始・終了にそれぞれ異なる
        ; 内部コードを使用している。ここでのメンテは不要で後で [`]→['] の置換をしている。
        (cond
          ((and (string= brl "7") (or (string= txt "（") (string= txt "）")) (not (inUsBracket)))
            (setf *bracketStats* (logior *bracketStats* _bracMask_)))
          ((and (string= brl "7") (or (string= txt "（") (string= txt "）"))      (inUsBracket))
            (setf *bracketStats* (logand *bracketStats* (lognot _bracMask_))))
          ((and (string= brl ".7") (or (string= txt "｛") (string= txt "｝")) (not (inUsBraceBracket)))
            (setf *bracketStats* (logior *bracketStats* _braceBracMask_)))
          ((and (string= brl ".7") (or (string= txt "｛") (string= txt "｝"))      (inUsBraceBracket))
            (setf *bracketStats* (logand *bracketStats* (lognot _braceBracMask_))))
          ((and (string= brl "^7") (string= txt "［") (not (inUsPhonemicBracket)))
            (setf *bracketStats* (logior *bracketStats* _phomicBracMask_)))
          ((and (string= brl _caret_s_) (string= txt "］")       (inUsPhonemicBracket))
            (setf *bracketStats* (logand *bracketStats* (lognot _phomicBracMask_))))
          ((and (string= brl ",7") (string= txt "［") (not (inUsSquareBracket)))
            (setf *bracketStats* (logior *bracketStats* _sqrBracMask_)))
          ((and (string= brl "7'") (string= txt "］")      (inUsSquareBracket))
            (setf *bracketStats* (logand *bracketStats* (lognot _sqrBracMask_))))
          ((and (string= brl "_7") (or (string= txt "＜") (string= txt "＞")) (not (inUsAngleBracket)))
            (setf *bracketStats* (logior *bracketStats* _anglBracMask_)))
          ((and (string= brl "_7") (or (string= txt "＜") (string= txt "＞"))      (inUsAngleBracket))
            (setf *bracketStats* (logand *bracketStats* (lognot _anglBracMask_))))
          ((and (string= brl "8") (string= txt "”") (not (inUsDoubleQuote)))
            (setf *bracketStats* (logior *bracketStats* _dQMask_)))
          ((and (string= brl "0") (string= txt "”")      (inUsDoubleQuote))
            (setf *bracketStats* (logand *bracketStats* (lognot _dQMask_))))
          ((and (string= brl _backQuote_brl_) (string= txt "’") (not (inUsSingleInvertedComma)))
            (setf *bracketStats* (logior *bracketStats* _sngInvCommMask_)))
          ((and (string= brl "J'") (string= txt "’")      (inUsSingleInvertedComma))
            (setf *bracketStats* (logand *bracketStats* (lognot _sngInvCommMask_)))))))
    (/= lastBracketStats *bracketStats*)))
