;
; rcsID=$Id: Utils.lisp,v 1.16 2008/04/29 16:25:15 tmatsugaki Exp $
;
(proclaim '(inline substitute-string
                   split-string
                   split-except
                   join-string
                   explode-string
                   explode-delims
                   constant-char-stringp
                   formatter-decode-p
                   prohibit-hankaku-p
                   divide-string
                   search-string
                   search-chars
                   subtract-strings
                   string-head=
                   string-tail=
                   string-all=
                   narrow-string
                   wide-string
                   lowerNumber2KanjiNumber
                   voicedSound-or-pSound
                   zero-nine-etc-charp
                   a-j-etc-charp
;                   zero-nine-charp
                   strip-header
                   non-alphaList
                   emit-capital-sign
                   braille-bytes
                   win
                   compiler-version
))
(proclaim '(optimize speed))

; _wide_alphas1_/_wide_alphas2_ 中の ー－ はそれぞれハイフン（ー）、ダッシュ（－）
(defconstant _narrow_alphas_ "#;:,.?!_(){}[]<>\"'+-*/abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ ")
(defconstant _wide_alphas1_ "＃；：，．？！＿（）｛｝［］＜＞”’＋ー＊／ａｂｃｄｅｆｇｈｉｊｋｌｍｎｏｐｑｒｓｔｕｖｗｘｙｚＡＢＣＤＥＦＧＨＩＪＫＬＭＮＯＰＱＲＳＴＵＶＷＸＹＺ　")
(defconstant _wide_alphas2_ "＃；：，．？！＿（）｛｝［］＜＞”’＋－＊／ａｂｃｄｅｆｇｈｉｊｋｌｍｎｏｐｑｒｓｔｕｖｗｘｙｚＡＢＣＤＥＦＧＨＩＪＫＬＭＮＯＰＱＲＳＴＵＶＷＸＹＺ　")
(defconstant _narrow_digits_ "0123456789")
(defconstant _wide_digits_ "０１２３４５６７８９")
(defconstant _voiced_sound_ "ガギグゲゴザジズゼゾダヂヅデドバビブベボ")
(defconstant _p_sound_ "パピプペポ")

(defparameter *g1_narrow_wide_char_list* nil)
(defparameter *g2_narrow_wide_char_list* nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 substitute-string
;; 【機能　】 文字列で指定した部分文字列を置換した文字列を生成する。
;; 【入力　】 str : 文字列
;;            src : 文字列
;;            dst : 文字列
;; 【出力　】 文字列
;; 【例外　】 なし
;; 【使用例】 (substitute-string "ab,bc,de" "," "_") → "ab_bc_de"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun substitute-string (str src dst)
  (let ((n (search src str)))
    (if n
      (concatenate 'string (concatenate 'string (subseq str 0 n) dst)
        (substitute-string (subseq str (+ n (length src))) src dst))
      str)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 split-string
;; 【機能　】 文字列を指定した文字列を区切り文字列として分割し、文字列のリスト
;;            を生成する。
;;           regexp:regexp-split の簡易版。
;; 【入力　】 str : 文字列
;;            sep : 文字列（1文字）
;; 【出力　】 文字列のリスト
;; 【例外　】 なし
;; 【使用例】 (split-string "ab cd ef") → ("ab" "cd" "ef")
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun split-string (str &optional (sep " "))
  (let ((n (search sep str)))
    (if n
      (cons (subseq str 0 n) (split-string (subseq str (+ n (length sep))) sep))
      (cons str nil))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 split-except
;; 【機能　】 文字列が禁則文字列を含まなければ分割し、文字列のリストを生成する。
;; 【入力　】 str   : 文字列
;;            delim : 区切り文字列
;;            except : 禁則文字列
;; 【出力　】 文字列のリスト
;; 【例外　】 なし
;; 【使用例】 (split-except "ab-cd" "-" "--") → ("ab-" "cd")
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun split-except (str delim except beg)
  (let (lst (delimLen (length delim)))
;    (format t "~%str=~S delim=~S except=~S" str delim except)
    (if (null (search-string except str (+ delimLen beg) t))
      (progn
        (setf lst (explode-string (subseq str (+ delimLen beg)) (list delim) nil))
        (cons (concatenate 'string (subseq str 0 (+ delimLen beg)) (car lst)) (cdr lst)))
      (list str))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 join-string
;; 【機能　】 文字列のリストを連結した文字列を生成する。
;; 【入力　】 strlst : 文字列のリスト
;;            sep    : 文字列
;; 【出力　】 文字列
;; 【例外　】 なし
;; 【使用例】 (join-string '("ab" "cd" "ef")) → "ab cd ef"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun join-string (strlst &optional (sep " "))
  (let ((ret nil))
    (dolist (str strlst)
      (setf ret (concatenate 'string ret str))
      (setf ret (concatenate 'string ret sep)))
    (string-right-trim sep ret)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 explode-string
;; 【機能　】 文字列を指定した区切り文字列リストで分割し、文字列のリストを生成
;;            する。
;; 【入力　】 str     : 文字列
;;            marklst : 文字列リスト
;; 【出力　】 文字列のリスト
;; 【例外　】 なし
;; 【使用例】 (explode-string "a+b,,c-d,,e*f,,g/h" '(",," "+" "-" "*" "/"))
;;           → ("a+" "b,," "c-" "d,," "e*" "f,," "g/" "h")
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun explode-string (str marklst &optional (permitMarkOnly t))                           
  (let (ret tlst)
    (if (and (= (length marklst) 1) (string= (nth 0 marklst) ""))
      (let ((strlen (length str)))
        (do ((i 0 (1+ i)))
          ((>= i strlen) nil)
          (setf ret (append ret (list (subseq str i (1+ i)))))))
      (progn
        (setf tlst (list str))
        (dolist (m marklst)
          (setf ret nil)
          (dolist (x tlst)
            (setf ret (append ret (divide-string x m permitMarkOnly))))
          (setf tlst ret))))
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 explode-delims
;; 【機能　】 文字列が全て指定した区切り文字列であれば分割し、文字列のリストを
;;            生成する。
;; 【入力　】 str   : 文字列
;;            delim : 区切り文字列
;; 【出力　】 文字列のリスト
;; 【例外　】 なし
;; 【使用例】 (explode-delims "    " #\Space) → (" " " " " " " ")
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun explode-delims (m_str m_chr)
  (if (every #'(lambda(x) (char= x m_chr)) m_str)
    (explode-string m_str (list (string m_chr)))
    (list m_str)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 constant-char-stringp
;; 【機能　】 文字列が、全て指定した文字で構成されているか否かを返す。
;; 【入力　】 str : 文字列
;;            chr : 文字
;; 【出力　】 t/nil
;; 【例外　】 なし
;; 【使用例】 (constant-char-stringp "        " #\Space) → t
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun constant-char-stringp (m_str m_chr)
  (every #'(lambda(x) (char= x m_chr)) m_str))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 digit-stringp
;; 【機能　】 文字列が、全て指定した数字で構成されているか否かを返す。（全角も可）
;; 【入力　】 str : 文字列
;; 【出力　】 t/nil
;; 【例外　】 なし
;; 【使用例】 (digit-stringp "0123") → t
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun digit-stringp (m_str)
  (and (>= (length m_str) 1) (every #'(lambda (x) (digit-char-p x)) m_str)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 formatter-decode-p
;; 【機能　】 構成符号を表す文字列か否かを返す。
;; 【入力　】 brl : 点字文字列
;; 【出力　】 t/nil
;; 【例外　】 なし
;; 【使用例】 (formatter-decode-p "=") → t
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun formatter-decode-p (m_brl)
  (and m_brl
       (find (intern m_brl) _formatter-decode-list_)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 formatter-encode-p
;; 【機能　】 構成符号を表す文字列か否かを返す。
;; 【入力　】 brl : 点字文字列
;; 【出力　】 t/nil
;; 【例外　】 なし
;; 【使用例】 (formatter-encode-p "；") → t
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun formatter-encode-p (m_str)
  (and m_str
       (find (intern m_str) _formatter-encode-list_)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 prohibit-hankaku-p
;; 【機能　】 エンコード時に墨字を半角化できないか否かを返す。
;; 【入力　】 str : 文字列
;; 【出力　】 t/nil
;; 【例外　】 なし
;; 【使用例】 (prohibit-hankaku-p "＜") → nil
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun prohibit-hankaku-p (m_str m_mode)
  (or
    (and (string= m_mode "JP")
         m_str
         (find (intern m_str) _prohibit-hankaku-jp-list_))
    (and (string= m_mode "DI")
         m_str
         (find (intern m_str) _prohibit-hankaku-di-list_))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 divide-string
;; 【機能　】 文字列を指定した文字列を区切り文字列として分割し、文字列のリストを
;;            生成する。
;; 【入力　】 str  : 文字列
;;            mark : 文字列
;; 【出力　】 文字列のリスト
;; 【例外　】 なし
;; 【使用例】 (divide-string "a+b,,c-d,,e*f,,g/h" ",,")
;;           → ("a+b,," "c-d,," "e*f,," "g/h")
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun divide-string (str mark &optional (permitMarkOnly t))
  (let ((n (search-string mark str 0 (not permitMarkOnly))))
    (if n
      (cons (concatenate 'string (subseq str 0 n) mark)
            (divide-string (subseq str (+ n (length mark))) mark permitMarkOnly))
      (if (string/= str "") (cons str nil) nil))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 search-string
;; 【機能　】 文字列に対して最初に見つかった部分文字列を検索してインデクスを返す。
;;          部分文字列が複数連結されていて、lastOne が t の場合は、最後の
;;          部分文字列のインデクスを返す。
;; 【入力　】 mark : 文字列
;;            str  : 文字列
;;            beg  : 開始インデックス
;;            lastOne  : 部分文字列が複数連結されている場合、後端のものを取得する
;; 【出力　】 インデクス
;; 【例外　】 なし
;; 【使用例】 (search-string ",," "ab,,cd,,ef" 0) → 2
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun search-string (mark str beg &optional (lastOne nil))
  (let ((ret (search mark str :start2 beg)))
    (if (and lastOne ret)
      (let ((marklen (length mark)) (strlen (length str)) (theLast ret))
        (do ((i (+ ret marklen) (+ i marklen)))
          ((or (null theLast) (> (+ i marklen) strlen) (>= i strlen)) nil)
          (setf theLast (search mark str :start2 i :end2 (+ i marklen)))
          (if theLast
            (setf ret theLast)
            nil)))
      nil)
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 search-chars
;; 【機能　】 文字列から、指定した文字列群を削除した結果を返す。
;; 【入力　】 str : 文字列
;;          strlst : 文字列のリスト
;; 【出力　】 t/nil
;; 【例外　】 なし
;; 【使用例】 (search-chars "axbycz" '("a" "b" "c")) → t
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun search-chars (str strlst)
  (let ((ret nil))
    (dolist (s strlst)
      (if (and (null ret) (some #'(lambda(x) (string= x s)) str))
        (setf ret t)
        nil))
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 subtract-strings
;; 【機能　】 文字列から、指定した文字列群を削除した結果を返す。
;; 【入力　】 str : 文字列
;;          strlst : 文字列のリスト
;;          repl   : 置換文字列
;; 【出力　】 t/nil
;; 【例外　】 なし
;; 【使用例】 (subtract-strings "axbycz" '("a" "b" "c")) → "xyz"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun subtract-strings (str strlst &optional (repl ""))
  (let ((ret str))
    (dolist (x strlst)
      (if (>= (length ret) (length x))
        (setf ret (substitute-string ret x repl))
        nil))
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 string-head=
;; 【機能　】 文字列の冒頭が指定した文字列群であるか否かを返す。
;; 【入力　】 str : 文字列
;;          strlst : 文字列のリスト
;; 【出力　】 t/nil
;; 【例外　】 なし
;; 【使用例】 (string-head= "-a" '("-")) → t
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun string-head= (str strlst)
  (let (index)
    (dolist (x strlst nil)
      (setf index (length x))
      (if (and (<= index (length str)) (string= (subseq str 0 index) x))
        (return t)
        nil))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 string-tail=
;; 【機能　】 文字列の末端が指定した文字列群であるか否かを返す。
;; 【入力　】 str : 文字列
;;          strlst : 文字列のリスト
;; 【出力　】 t/nil
;; 【例外　】 なし
;; 【使用例】 (string-tail= "a-" '("-")) → t
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun string-tail= (str strlst)
  (let (index)
    (dolist (x strlst nil)
      (setf index (- (length str) (length x)))
      (if (and (>= index 0) (string= (subseq str index) x))
        (return t)
        nil))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 string-all=
;; 【機能　】 文字列が指定した文字列群に含まれるか否かを返す。
;; 【入力　】 str : 文字列
;;          strlst : 文字列のリスト
;; 【出力　】 t/nil
;; 【例外　】 なし
;; 【使用例】 (string-all= "-a" '("-a")) → t
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun string-all= (str strlst)
  (progn
    (dolist (x strlst nil)
      (if (string= str x)
        (return t)
        nil))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 narrow-string
;; 【機能　】 文字列中の半角文字を全角文字に変換する。
;; 【入力　】 str : 文字列
;; 【出力　】 文字列
;; 【例外　】 なし
;; 【使用例】 (narrow-string "Ａｂｃ０１２あいう" 1) → "Abc012あいう"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun narrow-string (str grade &optional (downing nil))
  (let ((ret nil) (len (length str)) narrow_wide_char_list)
    (if (= grade 1)
      (setf narrow_wide_char_list *g1_narrow_wide_char_list*)
      (setf narrow_wide_char_list *g2_narrow_wide_char_list*))
    (do ((i 0 (1+ i)))
      ((= i len) nil)
      (let (cell s)
        (setf s (string (char str i)))
        (setf cell (rassoc (intern s) narrow_wide_char_list))
        (if cell
          (setf ret (concatenate 'string ret (string (car cell))))
          (setf ret (concatenate 'string ret s)))))
    (if downing (string-downcase ret) ret)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 wide-string
;; 【機能　】 文字列中の半角文字を全角文字に変換する。
;; 【入力　】 str : 文字列
;; 【出力　】 文字列
;; 【例外　】 なし
;; 【使用例】 (wide-string "Abc012あいう" 1) → "Ａｂｃ０１２あいう"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun wide-string (str grade)
  (let ((ret nil) (len (length str)) narrow_wide_char_list)
    (if (= grade 1)
      (setf narrow_wide_char_list *g1_narrow_wide_char_list*)
      (setf narrow_wide_char_list *g2_narrow_wide_char_list*))
    (do ((i 0 (1+ i)))
      ((= i len) nil)
      (let (cell s)
        (setf s (string (char str i)))
        (setf cell (assoc (intern s) narrow_wide_char_list))
        (if cell
          (setf ret (concatenate 'string ret (string (cdr cell))))
          (setf ret (concatenate 'string ret s)))))
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 lowerNumber2KanjiNumber
;; 【機能　】 文字列中の下がり数字を漢数字に変換する。
;; 【入力　】 str : 文字列
;; 【出力　】 文字列
;; 【例外　】 なし
;; 【使用例】 (lowerNumber2KanjiNumber "Ａ＿０１＿２あいう") → "Ａ零１二あいう"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun lowerNumber2KanjiNumber (str)
  (let ((ret str))
    (setf ret (substitute-string ret _sagari0Sumiji_ "零"))
    (setf ret (substitute-string ret _sagari1Sumiji_ "一"))
    (setf ret (substitute-string ret _sagari2Sumiji_ "二"))
    (setf ret (substitute-string ret _sagari3Sumiji_ "三"))
    (setf ret (substitute-string ret _sagari4Sumiji_ "四"))
    (setf ret (substitute-string ret _sagari5Sumiji_ "五"))
    (setf ret (substitute-string ret _sagari6Sumiji_ "六"))
    (setf ret (substitute-string ret _sagari7Sumiji_ "七"))
    (setf ret (substitute-string ret _sagari8Sumiji_ "八"))
    (setf ret (substitute-string ret _sagari9Sumiji_ "九"))
  ret))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 voicedSound-or-pSound
;; 【機能　】 濁音・半濁音の前に、” や ， を追加する。
;; 【入力　】 str : 文字列
;; 【出力　】 文字列
;; 【例外　】 なし
;; 【使用例】 (voicedSound-or-pSound "だ") → "”だ"
;;            (voicedSound-or-pSound "ぱ") → "，ぱ"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun voicedSound-or-pSound (s)
  (let ((ret s) ix)
  ; 濁音
  (setf ix (search s _voiced_sound_))
  (if (and (numberp ix) (>= ix 0))
   (setf ret (concatenate 'string "”" ret))
    (progn
      ; 半濁音
      (setf ix (search s _p_sound_))
      (if (and (numberp ix) (>= ix 0))
        (setf ret (concatenate 'string _capitalSign_w_ ret))
        nil)))
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 zero-nine-etc-charp
;; 【機能　】 エンコード対象文字が半角数字・漢数字か否かを返す。
;; 【入力　】 ch : 文字
;; 【出力　】 t/nil
;; 【例外　】 なし
;; 【使用例】 (zero-nine-etc-charp (char "0" 0)) → t
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun zero-nine-etc-charp (ch &optional (math nil))
  (let (ret)
    (if ch
      (if math
        (cond
          ((and (char>= ch #\0) (char<= ch #\9)) (setf ret t))
          ((char= ch #\-) (setf ret t))
          ((char= ch #\.) (setf ret t))
          ((char= ch #\,) (setf ret t))
          ((char= ch #\+) (setf ret t))
          ((char= ch #\*) (setf ret t))
          ((char= ch #\/) (setf ret t))
          ((char= ch #\:) (setf ret t))
          ((char= ch #\=) (setf ret t))
          ((char= ch #\>) (setf ret t))
          ((char= ch #\<) (setf ret t))
          ((char= ch #\() (setf ret t))
          ((char= ch #\)) (setf ret t))
          ((char= ch #\{) (setf ret t))
          ((char= ch #\}) (setf ret t))
          ((char= ch #\[) (setf ret t))
          ((char= ch #\]) (setf ret t)))
        (let (ix)
          ; ※下がり数字は事前に漢数字に変換しておく。
          (setf ix (search (string ch) "一二三四五六七八九壱零"))
          (if (and (numberp ix) (>= ix 0))
            (setf ret t)
            (cond
              ((char= ch #\FULLWIDTH_HYPHEN-MINUS) (setf ret t))
              ; 数字か否か？
              (t (setf ret (zero-nine-charp ch)))))))
      nil)
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 a-j-etc-charp
;; 【機能　】 デコード対象文字が数値か否かを返す。
;; 【入力　】 ch : 半角文字
;; 【出力　】 t/nil
;; 【例外　】 なし
;; 【使用例】 (a-j-etc-charp (char "A" 0)) → t
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun a-j-etc-charp (ch &optional (math nil))
  (let (ret)
    (if ch
      (if math
        (cond
          ((or
             (and (char>= ch #\a) (char<= ch #\j))
             (and (char>= ch #\A) (char<= ch #\J))) (setf ret t))
          ((char= ch #\-) (setf ret t))
          ((char= ch #\.) (setf ret t))
          ((char= ch #\,) (setf ret t))
          ((char= ch #\+) (setf ret t))
          ((char= ch #\*) (setf ret t))
          ((char= ch #\/) (setf ret t))
          ((char= ch #\:) (setf ret t))
          ((char= ch #\=) (setf ret t))
          ((char= ch #\>) (setf ret t))
          ((char= ch #\<) (setf ret t))
          ((char= ch #\() (setf ret t))
          ((char= ch #\)) (setf ret t))
          ((char= ch #\{) (setf ret t))
          ((char= ch #\}) (setf ret t))
          ((char= ch #\[) (setf ret t))
          ((char= ch #\]) (setf ret t)))
        (cond
          ; 数字
          ((or
             (and (char>= ch #\a) (char<= ch #\j))
             (and (char>= ch #\A) (char<= ch #\J))) (setf ret t))
          ; 下がり数字か否か？
          (t (setf ret (zero-nine-charp ch)))))
      nil)
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 strip-header
;; 【機能　】 ヘッダの
;; 【入力　】 s : 文字列
;; 【出力　】 t/nil
;; 【例外　】 なし
;; 【使用例】 (strip-header "coffin, and sign it with blood.\"            353")
;;            → "coffin, and sign it with blood.\""
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun strip-header (s)
  (if (> (length s) 0)
    (let (lst)
      (setf lst (remove-if #'(lambda (x) (string= x "")) (split-string s)))
      (join-string (ldiff lst (last lst))))
    s))

; アルファベット以外の 7Bit Ascii コードのリストを生成する。
(defun non-alphaList ()
  (let ((ret nil) c)
    (do ((i 32 (1+ i)))
      ((= i 127) nil)
      (setf c (code-char i))
      (if (not (both-case-p c))
        (pushnew (string c) ret)
        nil))
    ret))

(defun emit-capital-sign (str nonAlphaLst)
  (let ((ret "") paddedStr chunkStr lst)
    (setf paddedStr (concatenate 'string str _blank_s_))
    ; アルファベット以外の 7Bit Ascii コードで分割する。
    (setf lst (explode-string paddedStr nonAlphaLst))
    (dolist (x lst)
      (setf chunkStr x)
      ; アルファベットと 1文字のデリミタからなる文字列
      (if (both-case-p (char x 0))
        ; 大文字の個数をカウントする。
        (let (s1 (cnt 0) (chunkLen (length chunkStr)))
          (do ((i 0 (1+ i)))
            ((= i (1- chunkLen)) nil)
            (setf s1 (subseq chunkStr i (1+ i)))
            (if (and (string>= s1 "A") (string<= s1 "Z"))
              (incf cnt)
              nil))
          ; 大文字がある。
          (if (> cnt 0)
            ; 大文字は 1文字以上
            (if (and (/= cnt 1) (= cnt (1- chunkLen)))
              ; 大文字は 1文字以上で、全てが大文字の場合は連続大文字符。
              (setf chunkStr (concatenate 'string _doubleCapitalSign_s_ chunkStr))
              ; 大文字は 1文字か、あっても離散しているので、全て確認して大文字符を前に連結する。
              (let ((wrkChunk "") s2)
                (do ((j 0 (1+ j)))
                  ((= j chunkLen) nil)
                  (setf s2 (subseq chunkStr j (1+ j)))
                  (if (and (string>= s2 "A") (string<= s2 "Z"))
                    (setf wrkChunk (concatenate 'string wrkChunk _capitalSign_s_ s2))
                    (setf wrkChunk (concatenate 'string wrkChunk s2))))
                (setf chunkStr wrkChunk)))
            ; 大文字は 1文字以上で、全てが大文字の場合は連続大文字符。
            nil))
        nil)
      (setf ret (concatenate 'string ret chunkStr)))
    (subseq ret 0 (1- (length ret)))))

(defun braille-bytes (val)
  (let ((ret 0) (v val))
    (do ((i v (1- i)))
      ((< v 1) nil)
      (setf v (/ v 64))
      (incf ret))
    ret))

(defun timeStamp ()
  (black)
  (bold)
  (let (dttm)
    (setf dttm (multiple-value-list (get-decoded-time)))
    (format t "~4A/~2,'0D/~2,'0D ~2,'0D:~2,'0D"
            (nth 5 dttm) (nth 4 dttm) (nth 3 dttm) (nth 2 dttm) (nth 1 dttm)))
  (plain))

(defun win ()
  (cd "c:/home/BrailleConverter")
  (load "Braille.lisp"))

(defun compiler-version ()
  #+:lispworks (concatenate 'string
		"lispworks" " " (lisp-implementation-version))
  #+excl      (concatenate 'string
		"excl" " " excl::*common-lisp-version-number*)
  #+sbcl      (concatenate 'string
			   "sbcl" " " (lisp-implementation-version))
  #+cmu       (concatenate 'string
		"cmu" " " (lisp-implementation-version))
  #+scl       (concatenate 'string
		"scl" " " (lisp-implementation-version))

  #+kcl       "kcl"
  #+IBCL      "ibcl"
  #+akcl      "akcl"
  #+gcl       "gcl"
  #+ecl       "ecl"
  #+lucid     "lucid"
  #+ACLPC     "aclpc"
  #+CLISP     "clisp"
  #+Xerox     "xerox"
  #+symbolics "symbolics"
  #+mcl       "mcl"
  #+coral     "coral"
  #+gclisp    "gclisp")

(defparameter *cell* nil)
(defparameter *len* nil)

(setf *cell* nil i nil)
(setf *len* (length _narrow_alphas_))
(do ((i 0 (1+ i)))
  ((>= i *len*) nil)
  (setf *cell* (cons (intern (subseq _narrow_alphas_ i (1+ i))) (intern (subseq _wide_alphas1_ i (1+ i)))))
  (push *cell* *g1_narrow_wide_char_list*))
(do ((i 0 (1+ i)))
  ((>= i *len*) nil)
  (setf *cell* (cons (intern (subseq _narrow_alphas_ i (1+ i))) (intern (subseq _wide_alphas2_ i (1+ i)))))
  (push *cell* *g2_narrow_wide_char_list*))

(setf *len* (length _narrow_digits_))
(do ((i 0 (1+ i)))
  ((>= i *len*) nil)
  (setf *cell* (cons (intern (subseq _narrow_digits_ i (1+ i))) (intern (subseq _wide_digits_ i (1+ i)))))
  (push *cell* *g1_narrow_wide_char_list*)
  (push *cell* *g2_narrow_wide_char_list*))
