// $Id: BRbReadNode.cpp,v 1.22 2002/11/22 02:15:38 yuya Exp $

#include "BRbReadNode.h"

////////////////////////////////////////////////////////////////////////////////

extern "C" {
	NODE *logop(enum node_type type, NODE* left, NODE* right); // parse.c
	void top_local_init();                                     // parse.c
	void top_local_setup();                                    // parse.c
	void local_push();                                         // parse.c
	void local_pop();                                          // parse.c
	int local_cnt(ID id);                                      // parse.c
	ID* local_tbl();                                           // parse.c
	NODE* block_append(NODE *head, NODE *tail);                // parse.c
}

////////////////////////////////////////////////////////////////////////////////

BRbReadNode::BRbReadNode(BRbInputBuffer& buffer, bool debug)
	: BRbNode(), m_buffer(buffer), m_debug(debug)
{
}

BRbReadNode::~BRbReadNode()
{
}

////////////////////////////////////////////////////////////////////////////////

NODE*
BRbReadNode::read()
{
	BRbHeader header = read_header();

	if ( header.signature != BRB_FILE_SIGNATURE ) {
		throw "invalid signature";
	}

	if ( header.file_version != BRB_FILE_VERSION ) {
		throw "invalid version";
	}

	m_strtbl.load(m_buffer);

	m_debug.printf("reading node.\n");

	top_local_init();
	NODE* node = read_node();
	top_local_setup();

	m_debug.printf("read node.\n");

	m_debug.printf("============================================================\n");
	m_debug.dumpnode(node);
	m_debug.printf("============================================================\n");

	return node;
}

////////////////////////////////////////////////////////////////////////////////

BRbHeader
BRbReadNode::read_header()
{
	BRbHeader header;
	m_buffer.read_bytes((void*)&header, sizeof(header));
	return header;
}

NODE*
BRbReadNode::read_node()
{
	byte type = read_byte();

	m_debug.printf("R:%02X:%s\n", type, BRbNode::get_node_name(type));

	switch ( type ) {
		case NODE_ALIAS:
		{
			ID new_id = read_id();
			ID old_id = read_id();
			return NEW_ALIAS(new_id, old_id);
		}
		case NODE_VALIAS:
		{
			ID new_id = read_id();
			ID old_id = read_id();
			return NEW_VALIAS(new_id, old_id);
		}
		case NODE_AND:
		{
			NODE* n1st = read_node();
			NODE* n2nd = read_node();
			return logop(NODE_AND, n1st, n2nd);
		}
		case NODE_OR:
		{
			NODE* n1st = read_node();
			NODE* n2nd = read_node();
			return logop(NODE_OR, n1st, n2nd);
		}
		case NODE_ARGS:
		{
			NODE* node = NEW_ARGS(0, 0, 0);
			node->nd_cnt  = read_word();
			node->nd_rest = read_word();
			node->nd_opt  = read_node_opt();
			return node;
		}
	//	case NODE_ARGSCAT:
	//	case NODE_ARGSPUSH:
	//		break;  
		case NODE_ARRAY:
		{
			/**/
			word  n    = read_word();
			word  len  = read_word();
			NODE* node = NEW_ARRAY(read_node());
			NODE* cur  = node;
			node->nd_alen = len;
			for ( word i = 1; i < n; i++ ) {
				cur->nd_next = NEW_ARRAY(read_node());
				cur = cur->nd_next;
			}
			cur->nd_next = NULL;
			/**/
			/*
			NODE* node = NEW_ARRAY(NULL);
			node->nd_alen = read_word();
			node->nd_head = read_node();
			node->nd_next = read_node_opt();
			*/
			return node;
		}
	//	case NODE_ATTRSET:
	//		break;
		case NODE_BACK_REF:
		{
			byte nth = read_byte();

			local_push();
			NODE* node = NEW_BACK_REF(nth);
			local_pop();

			return node;
		}
		case NODE_BEGIN:
		{
			NODE* body = read_node();
			return NEW_BEGIN(body);
		}
		case NODE_BLOCK:
		{
			word  len  = read_word();
			NODE* node = NEW_BLOCK(read_node());
			NODE* prev = node;
			for ( word i = 1; i < len; i++ ) {
				prev->nd_next = NEW_BLOCK(read_node());
				prev = prev->nd_next;
			}
			prev->nd_next = NULL;
			return node;
		}
		case NODE_BLOCK_ARG:
		{
			ID name = read_id();

			local_push();
			NODE* node = NEW_BLOCK_ARG(name);
			local_pop();

			return node;
		}
		case NODE_BLOCK_PASS:
		{
			NODE* body = read_node();
			return NEW_BLOCK_PASS(body);
		}
	//	case NODE_BMETHOD:
	//		break;
		case NODE_BREAK:
		{
			return NEW_BREAK();
		}
		case NODE_CALL:
		{
			ID    name = read_id();
			NODE* recv = read_node();
			NODE* args = read_node_opt();
			return NEW_CALL(recv, name, args);
		}
		case NODE_FCALL:
		{
			ID    name = read_id();
			NODE* args = read_node_opt();
			return NEW_FCALL(name, args);
		}
		case NODE_CASE:
		{
			NODE* head = read_node();
			NODE* body = read_node();
			return NEW_CASE(head, body);
		}
		case NODE_WHEN:
		{
			NODE* head = read_node();
			NODE* body = read_node();
			NODE* next = read_node_opt();
			return NEW_WHEN(head, body, next);
		}
		case NODE_CDECL:
		{
			ID    id    = read_id();
			NODE* value = read_node();
			return NEW_CDECL(id, value);
		}
		case NODE_CLASS:
		{
			ID name = read_id();

			local_push();
			NODE* node = NEW_CLASS(name, 0, 0);
			local_pop();

			node->nd_super = read_node_opt();
			node->nd_body = read_node_opt();

			return node;
		}
		case NODE_COLON2:
		{
			ID    name = read_id();
			NODE* head = read_node();
			return NEW_COLON2(head, name);
		}
		case NODE_COLON3:
		{
			ID name = read_id();
			return NEW_COLON3(name);
		}
		case NODE_CONST:
		{
			ID name = read_id();
			return NEW_CONST(name);
		}
	//	case NODE_CREF:
	//		break;
		case NODE_CVAR:
		{
			ID name = read_id();
			return NEW_CVAR(name);
		}
	//	case NODE_CVAR2:
	//		break;
		case NODE_CVDECL:
		{
			ID    name  = read_id();
			NODE* value = read_node();
			return NEW_CVDECL(name, value);
		}
		case NODE_CVASGN:
		{
			ID    name  = read_id();
			NODE* value = read_node();
			return NEW_CVASGN(name, value);
		}
	//	case NODE_DASGN:
	//		break;
		case NODE_DASGN_CURR:
		{
			ID    name  = read_id();
			NODE* value = read_node_opt();
			return NEW_DASGN_CURR(name, value);
		}
		case NODE_DEFINED:
		{
			NODE* head = read_node();
			return NEW_DEFINED(head);
		}
		case NODE_DEFN:
		{
			//NODE* node = NEW_DEFN(...);
			NODE* node = rb_node_newnode(NODE_DEFN, 0, 0, 0);
			node->nd_mid  = read_id();
			node->nd_defn = read_node();
			return node;
		}
		case NODE_DEFS:
		{
			//NODE* node = NEW_DEFS(...);
			NODE* node = rb_node_newnode(NODE_DEFS, 0, 0, 0);
			node->nd_mid  = read_id();
			node->nd_recv = read_node();
			node->nd_defn = read_node();
			return node;
		}
	//	case NODE_DMETHOD:
	//		break;
		case NODE_DVAR:
		{
			ID name = read_id();
			return NEW_DVAR(name);
		}
		case NODE_DSTR:
		{
			NODE* node = NEW_DSTR(NULL);
			node->nd_lit  = read_literal();
			node->nd_next = read_node_opt();
			return node;
		}
	//	case NODE_DXSTR:
	//		break;
		case NODE_DREGX:
		case NODE_DREGX_ONCE:
		{
			NODE* node = NEW_STR(NULL);
			nd_set_type(node, type);
			node->nd_lit  = read_literal();
			node->nd_next = read_node();
			return node;
		}
	//		break;
		case NODE_ENSURE:
		{
			NODE* head = read_node();
			NODE* ensr = read_node();
			return NEW_ENSURE(head, ensr);
		}
		case NODE_FALSE:
		{
			return NEW_FALSE();
		}
		case NODE_DOT2:
		{
			NODE* from = read_node();
			NODE* to   = read_node();
			return NEW_DOT2(from, to);
		}
		case NODE_DOT3:
		{
			NODE* from = read_node();
			NODE* to   = read_node();
			return NEW_DOT3(from, to);
		}
		case NODE_FLIP2:
		{
			NODE* from = read_node();
			NODE* to   = read_node();
			return rb_node_newnode(NODE_FLIP2, from, to, 0);
		}
		case NODE_FLIP3:
		{
			NODE* from = read_node();
			NODE* to   = read_node();
			return rb_node_newnode(NODE_FLIP3, from, to, 0);
		}
	//	case NODE_FBODY:
	//		break;
		case NODE_FOR:
		{
			NODE* iter = read_node();
			NODE* body = read_node();
			NODE* var  = read_node();
			return NEW_FOR(var, iter, body);
		}
		case NODE_ITER:
		{
			NODE* iter = read_node();
			NODE* body = read_node();
			NODE* var  = read_node_opt();
			return NEW_ITER(var, iter, body);
		}
		case NODE_GASGN:
		{
			ID    name  = read_id();
			NODE* value = read_node();
			return NEW_GASGN(name, value);
		}
		case NODE_GVAR:
		{
			ID name = read_id();
			return NEW_GVAR(name);
		}
		case NODE_HASH:
		{
			NODE* head = read_node_opt();
			return NEW_HASH(head);
		}
		case NODE_IASGN:
		{
			ID    name  = read_id();
			NODE* value = read_node();
			return NEW_IASGN(name, value);
		}
		case NODE_IF:
		{
			NODE* cond = read_node();
			NODE* body = read_node_opt();
			NODE* els  = read_node_opt();
			return NEW_IF(cond, body, els);
		}
		case NODE_IVAR:
		{
			ID name = read_id();
			return NEW_IVAR(name);
		}
		case NODE_LASGN:
		{
			ID    name  = read_id();
			NODE* value = read_node_opt();

			local_push();
			NODE* node = NEW_LASGN(name, value);
			local_pop();

			return node;
		}
		case NODE_EVSTR:
		{
			VALUE lit = read_literal();
			return rb_node_newnode(NODE_EVSTR, lit, 0, 0);
		}
		case NODE_XSTR:
		{
			VALUE lit = read_literal();
			return NEW_XSTR(lit);
		}
		case NODE_LIT:
		{
			VALUE lit = read_literal();
			return NEW_LIT(lit);
		}
		case NODE_LVAR:
		{
			local_push();
			ID    name = read_id();
			NODE* node = NEW_LVAR(name);
			node->nd_cnt = read_word();
			local_pop();
			return node;
		}
		case NODE_MASGN:
		{
			NODE* node = NEW_MASGN(NULL, NULL);
			node->nd_head  = read_node();
			node->nd_value = read_node();
			node->nd_args  = read_node_opt();
			return node;
		}
		case NODE_MATCH:
		{
			NODE* head = read_node();
			return NEW_MATCH(head);
		}
		case NODE_MATCH2:
		{
			NODE* recv  = read_node();
			NODE* value = read_node();
			return NEW_MATCH2(recv, value);
		}
		case NODE_MATCH3:
		{
			NODE* recv  = read_node();
			NODE* value = read_node();
			return NEW_MATCH3(recv, value);
		}
	//	case NODE_METHOD:
	//		break;
		case NODE_MODULE:
		{
			ID    name = read_id();
			NODE* body = read_node_opt();

			local_push();
			NODE* node = NEW_MODULE(name, body);
			local_pop();

			return node;
		}
		case NODE_NEWLINE:
		{
			word  lineno = read_word();
			NODE* node = NEW_NEWLINE(read_node());
			node->nd_nth = lineno;
			return node;
		}
		case NODE_NEXT:
		{
			return NEW_NEXT();
		}
		case NODE_NIL:
		{
			return NEW_NIL();
		}
		case NODE_NOT:
		{
			NODE* body = read_node();
			return NEW_NOT(body);
		}
		case NODE_NTH_REF:
		{
			byte nth = read_byte();

			local_push();
			NODE* node = NEW_NTH_REF(nth);
			local_pop();

			return node;
		}
	//	case NODE_OP_ASGN1:
	//		break;
	//	case NODE_OP_ASGN2:
	//		break;
		case NODE_OP_ASGN_AND:
		{
			NODE* head  = read_node();
			NODE* value = read_node();
			return NEW_OP_ASGN_AND(head, value);
		}
		case NODE_OP_ASGN_OR:
		{
			NODE* head  = read_node();
			NODE* value = read_node();
			return NEW_OP_ASGN_OR(head, value);
		}
	//	case NODE_OPT_N:
	//		break;
		case NODE_POSTEXE:
		{
			return NEW_POSTEXE();
		}
		case NODE_REDO:
		{
			return NEW_REDO();
		}
		case NODE_RESCUE:
		{
			NODE* head = read_node();
			NODE* resq = read_node();
			NODE* els  = read_node_opt();
			return NEW_RESCUE(head, resq, els);
		}
		case NODE_RESBODY:
		{
			NODE* body = read_node();
			NODE* args = read_node_opt();
			NODE* head = read_node_opt();
			return NEW_RESBODY(args, body, head);
		}
		case NODE_RESTARGS:
		{
			NODE* head = read_node();
			return NEW_RESTARGS(head);
		}
		case NODE_RETRY:
		{
			return NEW_RETRY();
		}
		case NODE_RETURN:
		{
			NODE* stts = read_node_opt();
			return NEW_RETURN(stts);
		}
		case NODE_SCLASS:
		{
			NODE* recv = read_node();
			NODE* body = read_node();

			local_push();
			NODE* node = NEW_SCLASS(recv, body);
			local_pop();

			return node;
		}
		case NODE_SCOPE:
		{
			NODE* next = read_node_opt();

			local_push();
			NODE* node = NEW_SCOPE(next);
			local_pop();

			return node;
		}
		case NODE_SELF:
		{
			return NEW_SELF();
		}
		case NODE_STR:
		{
			VALUE str = read_string();
			return NEW_STR(str);
		}
		case NODE_SUPER:
		{
			NODE* args = read_node();
			return NEW_SUPER(args);
		}
		case NODE_TRUE:
		{
			return NEW_TRUE();
		}
		case NODE_UNDEF:
		{
			ID name = read_id();
			return NEW_UNDEF(name);
		}
		case NODE_UNTIL:
		{
			NODE* cond  = read_node();
			NODE* body  = read_node();
			byte  state = read_byte();
			return NEW_UNTIL(cond, body, state);
		}
		case NODE_WHILE:
		{
			NODE* cond  = read_node();
			NODE* body  = read_node();
			byte  state = read_byte();
			return NEW_WHILE(cond, body, state);
		}
		case NODE_VCALL:
		{
			ID name = read_id();
			return NEW_VCALL(name);
		}
		case NODE_YIELD:
		{
			NODE* stts = read_node_opt();
			return NEW_YIELD(stts);
		}
		case NODE_ZARRAY:
		{
			return NEW_ZARRAY();
		}
		case NODE_ZSUPER:
		{
			return NEW_ZSUPER();
		}
		default:
		{
			::printf("Unknown Node:0x%02X(%i)\n", type, type);
			//char buffer[512] = "";
			//sprintf(buffer, "Unknown Node:0x%02X(%i)", type, type);
			//throw &buffer;
			throw "Unknown Node Type!!";
			break;
		}
	}

	return NULL;
}


inline NODE*
BRbReadNode::read_node_opt()
{
	byte has_node = read_byte();
	if ( has_node ) {
		return read_node();
	} else {
		return NULL;
	}
}

inline byte
BRbReadNode::read_byte()
{
	return m_buffer.read_byte();
}

inline word
BRbReadNode::read_word()
{
	return m_buffer.read_word();
}

inline dword
BRbReadNode::read_dword()
{
	return m_buffer.read_dword();
}

inline const char*
BRbReadNode::read_cstr()
{
	return m_strtbl.get(read_word());
}

inline const char*
BRbReadNode::read_cstr(word* len)
{
	return m_strtbl.get(read_word(), len);
}

inline VALUE
BRbReadNode::read_string()
{
	word len = 0;
	const char* cstr = read_cstr(&len);
	VALUE rstr = ::rb_str_new(cstr, len);
	return rstr;
}

inline ID
BRbReadNode::read_id()
{
	return ::rb_intern(read_cstr());
}

inline VALUE
BRbReadNode::read_literal()
{
	byte klass = read_byte();
	switch ( klass ) {
		case 0: return read_dword();
		case 1: return ID2SYM(read_id());
		case 2: return read_string();
		case 3: return read_object();
		default: throw "Unknown Literal Class";
	}
}

inline VALUE
BRbReadNode::read_object()
{
	static VALUE mod = ::rb_eval_string("::Marshal");
	static ID    mid = ::rb_intern("load");
	return ::rb_funcall(mod, mid, 1, read_string());
}

////////////////////////////////////////////////////////////////////////////////
