// $Id: BRb.cpp,v 1.23 2002/12/23 06:59:30 yuya Exp $

#include "BRbBase.h"
#include "BRbInputBuffer.h"
#include "BRbOutputBuffer.h"
#include "BRbReadNode.h"
#include "BRbWriteNode.h"
#include "BRbDebug.h"

////////////////////////////////////////////////////////////////////////////////

extern "C" {
	VALUE ruby_top_self;         // object.c
	VALUE rb_eval(VALUE, NODE*); // eval.c
}

////////////////////////////////////////////////////////////////////////////////

static void  bruby_exception(const char* msg);
static VALUE bruby_read_file(VALUE filename);
#ifdef BRB_WRITER
static NODE* bruby_compile(VALUE source, VALUE filename);
static VALUE bruby_dumpnode(NODE* node);
#endif
#ifdef BRB_READER
static NODE* bruby_loadnode(VALUE bytecode);
#endif

#ifdef BRB_WRITER
static VALUE rb_brb_s_dump(VALUE obj, VALUE source, VALUE filename);
static VALUE rb_brb_s_dumpfile(VALUE obj, VALUE filename);
#endif
#ifdef BRB_READER
static VALUE rb_brb_s_load(VALUE obj, VALUE bytecode);
static VALUE rb_brb_s_loadfile(VALUE obj, VALUE filename);
#endif
#if defined(BRB_WRITER) && defined(BRB_READER)
static VALUE rb_brb_s_verify(VALUE obj, VALUE source, VALUE filename);
static VALUE rb_brb_s_verifyfile(VALUE obj, VALUE filename);
#endif
static VALUE rb_brb_s_debug_get(VALUE obj);
static VALUE rb_brb_s_debug_set(VALUE obj, VALUE debug);

////////////////////////////////////////////////////////////////////////////////

extern "C" void Init_bruby();

////////////////////////////////////////////////////////////////////////////////

static bool  g_debug          = false;
static VALUE rb_mBRb          = 0;
static VALUE rb_mBRbException = 0;

////////////////////////////////////////////////////////////////////////////////

static void
bruby_exception(const char* msg)
{
	::rb_raise(rb_mBRbException, msg);
}

static VALUE
bruby_read_file(VALUE filename)
{
	static ID id_open  = ::rb_intern("open");
	static ID id_read  = ::rb_intern("read");
	static ID id_close = ::rb_intern("close");

	VALUE mode = ::rb_str_new2("rb");
	VALUE file = ::rb_funcall(rb_cFile, id_open, 2, filename, mode);
	VALUE text = ::rb_funcall(file, id_read, 0);
	::rb_funcall(file, id_close, 0);

	return text;
}

#ifdef BRB_WRITER

static NODE*
bruby_compile(VALUE source, VALUE filename)
{
	static ID id_basename = ::rb_intern("basename");

	const char* basename = STR2CSTR(::rb_funcall(rb_cFile, id_basename, 1, filename));
	const int   lineno   = 1;

	return ::rb_compile_string(basename, source, lineno);
}

static VALUE
bruby_dumpnode(NODE* node)
{
	BRbDebug        debug(g_debug, stderr, bruby_exception);
	BRbOutputBuffer buffer;
	BRbWriteNode    writer(debug);

	writer.write(node, buffer);

	return buffer.to_string();
}

#endif

#ifdef BRB_READER

static NODE*
bruby_loadnode(VALUE bytecode)
{
	BRbDebug       debug(g_debug, stderr, bruby_exception);
	BRbInputBuffer buffer(bytecode);
	BRbReadNode    reader(buffer, debug);

	char* sourcefile = ruby_sourcefile;
	ruby_sourcefile = "bruby";

	NODE* node = reader.read();

	ruby_sourcefile = sourcefile;

	return node;
}

#endif

////////////////////////////////////////////////////////////////////////////////

#ifdef BRB_WRITER

static VALUE
rb_brb_s_dump(VALUE obj, VALUE source, VALUE filename)
{
	Check_Type(source,   T_STRING);
	Check_Type(filename, T_STRING);

	NODE* root     = ::bruby_compile(source, filename);
	VALUE bytecode = ::bruby_dumpnode(root);

	return bytecode;
}

static VALUE
rb_brb_s_dumpfile(VALUE obj, VALUE filename)
{
	Check_Type(filename, T_STRING);

	VALUE source   = ::bruby_read_file(filename);
	VALUE bytecode = ::rb_brb_s_dump(obj, source, filename);

	return bytecode;
}

#endif

#ifdef BRB_READER

static VALUE
rb_brb_s_load(VALUE obj, VALUE bytecode)
{
	Check_Type(bytecode, T_STRING);

	NODE* node = ::bruby_loadnode(bytecode);
	VALUE self = ::rb_eval_string("self");
	::rb_eval(self, node);

	return Qnil;
}

static VALUE
rb_brb_s_loadfile(VALUE obj, VALUE filename)
{
	Check_Type(filename, T_STRING);

	VALUE bytecode = ::bruby_read_file(filename);
	::rb_brb_s_load(obj, bytecode);

	return Qnil;
}

#endif

#if defined(BRB_WRITER) && defined(BRB_READER)

static VALUE
rb_brb_s_verify(VALUE obj, VALUE source, VALUE filename)
{
	Check_Type(source,   T_STRING);
	Check_Type(filename, T_STRING);

	VALUE bytecode1 = ::rb_brb_s_dump(obj, source, filename);
	NODE* root      = ::bruby_loadnode(bytecode1);
	VALUE bytecode2 = ::bruby_dumpnode(root);

	return ::rb_funcall(bytecode1, ::rb_intern("=="), 1, bytecode2);
}

static VALUE
rb_brb_s_verifyfile(VALUE obj, VALUE filename)
{
	Check_Type(filename, T_STRING);

	VALUE source = ::bruby_read_file(filename);

	return ::rb_brb_s_verify(obj, source, filename);
}

#endif

static VALUE
rb_brb_s_debug_get(VALUE obj)
{
	return (g_debug ? Qtrue : Qfalse);
}

static VALUE
rb_brb_s_debug_set(VALUE obj, VALUE debug)
{
	g_debug = (RTEST(debug) ? true : false);
	return debug;
}

////////////////////////////////////////////////////////////////////////////////

extern "C" void
Init_bruby()
{
	rb_mBRb = ::rb_define_module("BRb");
#ifdef BRB_WRITER
	::rb_define_singleton_method(rb_mBRb, "dump",       (VALUE(*)(...))rb_brb_s_dump,       2);
	::rb_define_singleton_method(rb_mBRb, "dumpfile",   (VALUE(*)(...))rb_brb_s_dumpfile,   1);
#endif
#ifdef BRB_READER
	::rb_define_singleton_method(rb_mBRb, "load",       (VALUE(*)(...))rb_brb_s_load,       1);
	::rb_define_singleton_method(rb_mBRb, "loadfile",   (VALUE(*)(...))rb_brb_s_loadfile,   1);
#endif
#if defined(BRB_WRITER) && defined(BRB_READER)
	::rb_define_singleton_method(rb_mBRb, "verify",     (VALUE(*)(...))rb_brb_s_verify,     2);
	::rb_define_singleton_method(rb_mBRb, "verifyfile", (VALUE(*)(...))rb_brb_s_verifyfile, 1);
#endif
	::rb_define_singleton_method(rb_mBRb, "debug",      (VALUE(*)(...))rb_brb_s_debug_get,  0);
	::rb_define_singleton_method(rb_mBRb, "debug=",     (VALUE(*)(...))rb_brb_s_debug_set,  1);

	rb_mBRbException = ::rb_define_class("BRbException", rb_eException);
}

////////////////////////////////////////////////////////////////////////////////
