/* -*-tab-width:8;indent-tabs-mode:t;c-file-style:"bsd"-*- */
/*
 * rndis: USB Remote ndis driver for Windows Mobile
 *
 * Copyright 2007 Hashi,Hiroaki <hashiz@meridiani.jp>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE. KEEP THE LOVE.
 *
 * $Id: if_rndis.c 352M 2007-08-31 01:30:42Z (ローカル) $
 */

/*
 * Supported devices are #defined below. 
 * These would normally be added into src/sys/dev/usb/usbdevs.
 */

#include <sys/cdefs.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/lock.h>
#include <sys/mbuf.h>
#include <sys/kernel.h>
#include <sys/module.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/lockmgr.h>
#include <sys/sockio.h>
#include <sys/endian.h>

#include <net/if.h>
#include <net/if_arp.h>
#include <net/if_dl.h>
#include <net/if_media.h>
#include <net/ethernet.h>
#include <net/if_types.h>

#include <net/bpf.h>

#include <netinet/in.h>
#include <netinet/if_ether.h>

#include <sys/bus.h>
#include <machine/bus.h>

#if __FreeBSD_version >= 700000
#include <dev/usb/usb_port.h>
#endif
#include <dev/usb/usb.h>
#include <dev/usb/usbdi.h>
#include <dev/usb/usbdi_util.h>
#include "usbdevs.h"
#include <dev/usb/usbdivar.h>
#include <dev/usb/usb_ethersubr.h>
#include <dev/usb/usbcdc.h>

#include <compat/ndis/cfg_var.h>
#include <compat/ndis/resource_var.h>
#include <compat/ndis/pe_var.h>
#include <compat/ndis/ntoskrnl_var.h>
#include <compat/ndis/ndis_var.h>

#include <sys/sysctl.h>

#include "if_rndisreg.h"

MODULE_DEPEND(rndis, usb, 1, 1, 1);
MODULE_DEPEND(rndis, ether, 1, 1, 1);

#ifndef USB_PRODUCT_SHARP_WZERO3ADES_RNDIS
#define USB_PRODUCT_SHARP_WZERO3ADES_RNDIS	0x91ad
#endif
#ifndef UICLASS_MISC
#define UICLASS_MISC		0xef
#endif
#ifndef UIPROTO_CDC_VENDOR
#define UIPROTO_CDC_VENDOR	0xff
#endif

typedef u_char etheraddr_t[ETHER_ADDR_LEN] ;
/*
 * sysctl
 */

static int rndisrtvendor = 0;
static int rndisrtproduct = 0;

SYSCTL_DECL(_hw_usb);

SYSCTL_NODE(_hw_usb, OID_AUTO, rndis, CTLFLAG_RW, 0, "USB rndis");

/* Rather than the user having to compile in their device id,
 * let them set a single dynamic runtime id via sysctl.
 * If user has multiple device types they will need to be
 * compiled in, but this should suit most people.
 */
SYSCTL_INT(_hw_usb_rndis, OID_AUTO, vendor, CTLFLAG_RW,
	   &rndisrtvendor, 0, "rndis runtime vendor ID");
SYSCTL_INT(_hw_usb_rndis, OID_AUTO, product, CTLFLAG_RW,
	   &rndisrtproduct, 0, "rndis runtime product ID");

#ifdef RNDIS_DEBUG
static int rndisdebug = 1;

SYSCTL_INT(_hw_usb_rndis, OID_AUTO, debug, CTLFLAG_RW,
	   &rndisdebug, 0, "rndis debug level");
#define DPRINTF(x)	do { \
				if (rndisdebug) \
					logprintf x; \
			} while (0)

#define DPRINTFN(n, x)	do { \
				if (rndisdebug > (n)) \
					logprintf x; \
			} while (0)

#else
#define DPRINTF(x)
#define DPRINTFN(n,x)
#endif

#define RNDIS_IOBUF_SIZE   16384

#define RNDIS_CONFIG_NO	   1

struct rndis_type {
	struct usb_devno	uv_dev;
	u_int16_t		uv_flags;
};

/* New supported products go in here */

static const struct rndis_type rndis_devs[] = {
	{{ USB_VENDOR_SHARP, USB_PRODUCT_SHARP_WZERO3ADES_RNDIS }, 0},
};

/* attach/detach */
Static int rndis_match(device_ptr_t);
Static int rndis_attach(device_ptr_t);
Static int rndis_detach(device_ptr_t);
Static void rndis_shutdown(device_ptr_t);

/* usb */
Static int  rndis_open_rx_pipe(struct rndis_softc *sc);
Static int  rndis_open_tx_pipe(struct rndis_softc *sc);
Static int  rndis_close_rx_pipe(struct rndis_softc *sc);
Static int  rndis_close_tx_pipe(struct rndis_softc *sc);
Static void rndis_rxeof(usbd_xfer_handle xfer, usbd_private_handle priv, usbd_status status);
Static void rndis_txeof(usbd_xfer_handle, usbd_private_handle, usbd_status);

/* device control */
Static int rndis_init_device(struct rndis_softc *);
//Static int rndis_reset_device(struct rndis_softc *);
//Static int rndis_halt_device(struct rndis_softc *);
Static int rndis_get_uint32(struct rndis_softc *sc, u_int32_t oid, u_int32_t *val, int *cnt);
Static int rndis_set_uint32(struct rndis_softc *sc, u_int32_t oid, u_int32_t *val, int cnt);
Static int rndis_get_etheraddr(struct rndis_softc *sc, u_int32_t oid, etheraddr_t *eaddr, int *listcnt);
Static int rndis_set_etheraddr(struct rndis_softc *sc, u_int32_t oid, etheraddr_t *eaddr, int listcnt);
Static int rndis_setmulti(struct rndis_softc *sc);

/* ifnet */
Static void rndis_start(struct ifnet *);
Static void rndis_init(void *);
Static int  rndis_ioctl(struct ifnet *, u_long, caddr_t);
Static void rndis_rxstart(struct ifnet *);
Static int  rndis_send(struct rndis_softc *, struct mbuf *, int );
Static void rndis_stop(struct ifnet *, int);
Static void rndis_watchdog(struct ifnet *);
Static int  rndis_ifmedia_change(struct ifnet *);
Static void rndis_ifmedia_status(struct ifnet *, struct ifmediareq *);

/* rndis */
Static int  rndis_send_message(struct rndis_softc *, struct rndis_msg_snd *, int);
Static int  rndis_recv_message(struct rndis_softc *, struct rndis_msg_rcv *, int, u_int32_t, u_int32_t);
Static int  rndis_msg_query(struct rndis_softc *, u_int32_t, u_char *, int *);
Static int  rndis_msg_set(struct rndis_softc *, u_int32_t, u_char *, int);

Static const char *rndis_status_str(u_int32_t);

#define REQID (htole32((++sc->sc_req_id)?(sc->sc_req_id):(++sc->sc_req_id)))

#ifdef RNDIS_DEBUG
#define RNDIS_DEBUG_DUMP(b,l) dump_data((u_char *)(b), (int)(l))
Static void
dump_data(u_char *datap, int datalen)
{
	int bcnt;
	for ( bcnt = 0 ; bcnt < datalen ; bcnt++ ) {
		if ( bcnt % 8 == 0) {
			printf("    %04x : ", bcnt);
		}
		printf("%02x", datap[bcnt]) ;
		if ( bcnt%8 == 7 ) {
			printf("\n") ;
		}
		else {
			printf(" ") ;
		}
	}
	if ( bcnt%8 > 0 ) {
		printf("\n") ;
	}
}
#else
#define RNDIS_DEBUG_DUMP(b,l)
#endif

Static device_method_t rndis_methods[] = {
        /* Device interface */
        DEVMETHOD(device_probe,         rndis_match),
        DEVMETHOD(device_attach,        rndis_attach),
        DEVMETHOD(device_detach,        rndis_detach),
        DEVMETHOD(device_shutdown,      rndis_shutdown),

        /* bus interface */
        DEVMETHOD(bus_print_child,      bus_generic_print_child),
        DEVMETHOD(bus_driver_added,     bus_generic_driver_added),

        { 0, 0 }
};

Static driver_t rndis_driver = {
        "rndis",
        rndis_methods,
        sizeof(struct rndis_softc)
};

Static devclass_t rndis_devclass;

DRIVER_MODULE(rndis, uhub, rndis_driver, rndis_devclass, usbd_driver_load, 0);
MODULE_VERSION(rndis, 0);

#define rndis_lookup(v, p) ((const struct rndis_type *)usb_lookup(rndis_devs, v, p))

USB_MATCH(rndis)
{
	USB_MATCH_START(rndis, uaa);
	
	if (uaa->iface != NULL)
		return (UMATCH_NONE);

	/* Check the runtime device before the table */
	if (uaa->vendor == rndisrtvendor && uaa->product == rndisrtproduct)
		return UMATCH_VENDOR_PRODUCT;

	return (rndis_lookup(uaa->vendor, uaa->product) != NULL ?
		UMATCH_VENDOR_PRODUCT : UMATCH_NONE);
}

/* device attach */
USB_ATTACH(rndis)
{
	/* This gives us a valid struct rndis_softc *sc */
	USB_ATTACH_START(rndis, sc, uaa);
	usbd_device_handle dev = uaa->device;
	usbd_interface_handle iface;
	usb_interface_descriptor_t *id;
	usb_endpoint_descriptor_t *ed;
	char *devinfo;
	const char *devname;
	int i,j;
	usbd_status err;
	struct ifnet *ifp;

	etheraddr_t eaddr;
	int         eaddrcnt;

	DPRINTFN(10,("\nrndis_attach: sc=%p\n", sc));

	bzero(sc, sizeof(struct rndis_softc));

	devinfo = malloc(1024, M_USBDEV, M_WAITOK);

	usbd_devinfo(dev, 0, devinfo);
	USB_ATTACH_SETUP;

	devname = USBDEVNAME(sc->sc_dev);
	printf("%s: %s\n", devname, devinfo);

	/* Move the device into the configured state. */
	err = usbd_set_config_no(dev, RNDIS_CONFIG_NO, 1);
	if (err) {
		printf("%s: setting config no failed\n", devname);
		goto attachfail;
	}

	sc->sc_udev = dev;

	sc->sc_max_rx_size = RNDIS_IOBUF_SIZE;

	/* search interface */
	sc->sc_iface_ctrl  = sc->sc_iface_data = NULL;
	sc->sc_epaddr_rx = sc->sc_epaddr_tx = -1 ;

	DPRINTF(("%s: interface count=%d\n", devname, dev->cdesc->bNumInterface));
	for ( i = 0; i < dev->cdesc->bNumInterface; i++ ) {
		int ctrl_iface = 0;
		err = usbd_device2interface_handle(dev, i, &iface);
		if (err) {
			printf("\n%s: failed to get interface, err=%s\n",
			       devname, usbd_errstr(err));
			goto attachfail;
		}

		id = usbd_get_interface_descriptor(iface);

		DPRINTF(("%s: iface class=0x%x, subclass=0x%x, protocol=0x%x\n",
			 devname,
			 id->bInterfaceClass,
			 id->bInterfaceSubClass,
			 id->bInterfaceProtocol));

		if ( id->bInterfaceClass == UICLASS_CDC &&
		     id->bInterfaceSubClass == UISUBCLASS_ABSTRACT_CONTROL_MODEL &&
		     id->bInterfaceProtocol == UIPROTO_CDC_VENDOR ) {
			DPRINTF(("%s: found CDC communication interface\n",
				 devname));
			sc->sc_iface_ctrl = iface ;
			ctrl_iface = 1;
		}
		else if ( id->bInterfaceClass == UICLASS_MISC &&
			  id->bInterfaceSubClass == 1 &&
			  id->bInterfaceProtocol == 1 ) {
			DPRINTF(("%s: found ActiveSync interface\n",
				 devname));
			sc->sc_iface_ctrl = iface ;
			ctrl_iface = 1;
		}
		else if ( id->bInterfaceClass == UICLASS_CDC_DATA &&
			  id->bInterfaceSubClass == UISUBCLASS_DATA ) {
			DPRINTF(("%s: found CDC Data interface\n",
				 devname));
			sc->sc_iface_data = iface ;
		}
		else {
			printf("%s: found unexpected interface: class=0x%x,subclass=0x%x,protocol=0x%x\n",
			       devname,
			       id->bInterfaceClass,
			       id->bInterfaceSubClass,
			       id->bInterfaceProtocol);
			continue;
		}

		DPRINTF(("%s: endpoint count=%d\n", devname, id->bNumEndpoints));

		if (ctrl_iface) {
			continue;
		}

		for (j = 0; j < id->bNumEndpoints; j++) {
			int addr, dir, attr;
			ed = usbd_interface2endpoint_descriptor(iface, j);
			if (ed == NULL) {
				printf("%s: could not read endpoint descriptor"
				       ": %s\n", devname, usbd_errstr(err));
				goto attachfail;
			}
			addr = ed->bEndpointAddress;
			dir = UE_GET_DIR(ed->bEndpointAddress);
			attr = ed->bmAttributes & UE_XFERTYPE;

			DPRINTF(("%s: endpoint=0x%x dir=0x%x attr=0x%x\n",
				 devname, addr, dir, attr));

			if (!ctrl_iface &&
			    dir == UE_DIR_IN && attr == UE_BULK)
				sc->sc_epaddr_rx = addr;
			else if (!ctrl_iface &&
				 dir == UE_DIR_OUT && attr == UE_BULK)
				sc->sc_epaddr_tx = addr;
			else {
				printf("%s: unexpected endpoint 0x%x dir=0x%x attr=0x%x\n",
				       devname, addr, dir, attr);
				goto attachfail;
			}
		}
	}

	if ( sc->sc_iface_ctrl == NULL ) {
		printf("%s: Could not find control interface\n",
		       devname);
		goto attachfail;
	}
	if ( sc->sc_iface_data == NULL ) {
		printf("%s: Could not find data interface\n",
		       devname);
		goto attachfail;
	}

	if (sc->sc_epaddr_rx == -1) {
		printf("%s: Could not find input data channel endpoint\n",
		       devname);
		goto attachfail;
	}
	if (sc->sc_epaddr_tx == -1) {
		printf("%s: Could not find output data channel endpoint\n",
		       devname);
		goto attachfail;
	}

	sc->sc_flags = rndis_lookup(uaa->vendor, uaa->product)->uv_flags;

	if (uaa->vendor == rndisrtvendor && uaa->product == rndisrtproduct) 
		printf("%s: <Windows Mobile Device> %s (rt)\n", devname, devinfo);
	else
		printf("%s: <Windows Mobile Device> %s\n", devname, devinfo);

	DPRINTF(("%s: in=0x%x out=0x%x\n", devname,
		 sc->sc_epaddr_rx, sc->sc_epaddr_tx));

	mtx_init(&sc->sc_mtx, device_get_nameunit(self), MTX_NETWORK_LOCK,
		 MTX_DEF | MTX_RECURSE);

	/* initialize device */
	err = rndis_init_device(sc);
	if (err) {
		printf("%s: device initialize failed\n", devname);
		mtx_destroy(&sc->sc_mtx);
		goto attachfail;
	}

	/* Get Ethernet Address */
	eaddrcnt = 1 ;
	err = rndis_get_etheraddr(sc, OID_802_3_PERMANENT_ADDRESS, &eaddr, &eaddrcnt);
	if (err) {
		printf("%s: read MAC address failed\n", devname);
		mtx_destroy(&sc->sc_mtx);
		goto attachfail;
	}

	/* Print Ethernet Address */
	printf("%s: Ethernet address %s\n", devname, ether_sprintf(eaddr));

	/* Initialize Ethernet Interface Infomation */
	ifp = sc->sc_ifp = if_alloc(IFT_ETHER);
	if (ifp == NULL) {
		printf("%s: can not if_alloc\n", devname);
		mtx_destroy(&sc->sc_mtx);
		goto attachfail;
	}
	ifp->if_softc = sc;
	ifp->if_mtu = ETHERMTU;
	if_initname(ifp, "rndis",  device_get_unit(self));
	ifp->if_flags = IFF_BROADCAST | IFF_SIMPLEX | IFF_MULTICAST |
		IFF_NEEDSGIANT;
	ifp->if_start = rndis_start;
	ifp->if_ioctl = rndis_ioctl;
	ifp->if_watchdog = rndis_watchdog;
	ifp->if_init = rndis_init;
	ifp->if_snd.ifq_maxlen = IFQ_MAXLEN;

	sc->sc_qdat.ifp = ifp;
	sc->sc_qdat.if_rxstart = rndis_rxstart;

	ifmedia_init(&sc->sc_ifmedia, 0, rndis_ifmedia_change, rndis_ifmedia_status);

	/* No IFM type for 11Mbps USB, so go with 10baseT */
	ifmedia_add(&sc->sc_ifmedia, IFM_ETHER | IFM_10_T, 0, 0);
	ifmedia_set(&sc->sc_ifmedia, IFM_ETHER | IFM_10_T);

	/*
	 * Call MI attach routine.
	 */

	ether_ifattach(ifp, eaddr);

	usb_register_netisr();
	sc->sc_attached = 1;

	usbd_add_drv_event(USB_EVENT_DRIVER_ATTACH, dev, USBDEV(sc->sc_dev));

	free(devinfo, M_USBDEV);
	USB_ATTACH_SUCCESS_RETURN;

attachfail:
	DPRINTF(("%s: %s: attach failed\n", devname, __func__));
	sc->sc_dying = 1;
	free(devinfo, M_USBDEV);
	USB_ATTACH_ERROR_RETURN;
}

/* device detach */
USB_DETACH(rndis)
{
	USB_DETACH_START(rndis, sc);
 
	sc->sc_dying = 1;

	struct ifnet *ifp = sc->sc_ifp;

	DPRINTF(("%s: rndis_detach: sc=%p\n", USBDEVNAME(sc->sc_dev), sc));

	/* Detached before attached finished */
	if (!sc->sc_attached)
		return 0;

	if (--sc->sc_refcnt >= 0) {
		/* Wait for processes to go away */
		usb_detach_wait(USBDEV(sc->sc_dev));
	}

	if (ifp->if_drv_flags & IFF_DRV_RUNNING)
		rndis_stop(ifp, 1);

	ether_ifdetach(ifp);

	if_free(ifp);

	ifmedia_removeall(&sc->sc_ifmedia);

	if (sc->sc_pipe_tx != NULL)
		printf("%s: detach has active tx endpoint.\n",
		       USBDEVNAME(sc->sc_dev));
	if (sc->sc_pipe_rx != NULL)
		printf("%s: detach has active rx endpoint.\n",
		       USBDEVNAME(sc->sc_dev));

	sc->sc_attached = 0;

	mtx_destroy(&sc->sc_mtx);

	usbd_add_drv_event(USB_EVENT_DRIVER_DETACH, sc->sc_udev,
		USBDEV(sc->sc_dev));

	return 0;
}

/*
 * Stop all chip I/O so that the kernel's probe routines don't
 * get confused by errant DMAs when rebooting.
 */
Static void
rndis_shutdown(device_ptr_t dev)
{
	struct rndis_softc *sc;
	sc = device_get_softc(dev);

	DPRINTF(("%s: %s enter\n", USBDEVNAME(sc->sc_dev), __func__)) ;

	rndis_stop(sc->sc_ifp, 1);

	return;
}

Static int
rndis_open_rx_pipe(struct rndis_softc *sc)
{
	usbd_status err;
	int error = 0;

	DPRINTF(("%s: %s enter\n", USBDEVNAME(sc->sc_dev), __func__)) ;

	if (sc->sc_dying)
		return EIO;

	/* Initialize receive buffer */
	sc->sc_rx_xfer = usbd_alloc_xfer(sc->sc_udev);
	sc->sc_rx_buf  = usbd_alloc_buffer(sc->sc_rx_xfer, sc->sc_max_rx_size);
	if (sc->sc_rx_buf == NULL) {
		printf("%s: cannot allocate rx buffer\n", USBDEVNAME(sc->sc_dev));
		return ENOBUFS;
	}
	sc->sc_rx_mbuf = usb_ether_newbuf();
	if (sc->sc_rx_mbuf == NULL) {
		printf("%s: cannot allocate rx mbuf\n", USBDEVNAME(sc->sc_dev));
		return ENOBUFS;
	}
	sc->sc_rx_buf_len = sc->sc_max_rx_size;

	/* open rx data channel pipe */
	sc->sc_refcnt++ ;
	err = usbd_open_pipe(sc->sc_iface_data, sc->sc_epaddr_rx,
			     USBD_EXCLUSIVE_USE, &sc->sc_pipe_rx);
	if (--sc->sc_refcnt < 0)
		usb_detach_wakeup(USBDEV(sc->sc_dev));
	if (err) {
		printf("%s: open rx pipe failed: %s\n",
		       USBDEVNAME(sc->sc_dev), usbd_errstr(err));
		error = EIO;
		goto done;
	}

	/* setup rx pipe transfer */
	if (sc->sc_rx_xfer_cnt == 0) {
		usbd_setup_xfer(sc->sc_rx_xfer, sc->sc_pipe_rx,
				sc, sc->sc_rx_buf, sc->sc_rx_buf_len,
				USBD_SHORT_XFER_OK | USBD_NO_COPY,
				USBD_NO_TIMEOUT, rndis_rxeof);
		sc->sc_refcnt++ ;
		usbd_transfer(sc->sc_rx_xfer);
		if (--sc->sc_refcnt < 0)
			usb_detach_wakeup(USBDEV(sc->sc_dev));
		sc->sc_rx_xfer_cnt++;
		DPRINTF(("%s: %s: start read\n", USBDEVNAME(sc->sc_dev),
			 __func__));
	}

done:
	return error;
}

Static int
rndis_open_tx_pipe(struct rndis_softc *sc)
{
	usbd_status err;
	int error = 0;

	DPRINTF(("%s: %s enter\n", USBDEVNAME(sc->sc_dev), __func__)) ;

	if (sc->sc_dying)
		return EIO;

	/* Initialize transmit buffer */
	sc->sc_tx_xfer = usbd_alloc_xfer(sc->sc_udev);
	sc->sc_tx_buf  = usbd_alloc_buffer(sc->sc_tx_xfer, sc->sc_max_tx_size);
	if (sc->sc_tx_buf == NULL) {
		printf("%s: cannot allocate tx buffer\n", USBDEVNAME(sc->sc_dev));
		return ENOBUFS;
	}
	sc->sc_tx_buf_len = sc->sc_max_tx_size;

	/* open tx data channel pipe */
	sc->sc_refcnt++ ;
	err = usbd_open_pipe(sc->sc_iface_data, sc->sc_epaddr_tx,
			     USBD_EXCLUSIVE_USE, &sc->sc_pipe_tx);
	if (--sc->sc_refcnt < 0)
		usb_detach_wakeup(USBDEV(sc->sc_dev));
	if (err) {
		printf("%s: open tx pipe failed: %s\n",
		       USBDEVNAME(sc->sc_dev), usbd_errstr(err)) ;
		error = EIO;
		goto done;
	}

done:
	return error;
}

Static int
rndis_close_rx_pipe(struct rndis_softc *sc)
{
	usbd_status err;
	int error = 0;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	/* RX endpoint */
	if (sc->sc_pipe_rx != NULL) {
		err = usbd_abort_pipe(sc->sc_pipe_rx);
		if (err)
			printf("%s: abort rx pipe failed: %s\n",
			       USBDEVNAME(sc->sc_dev), usbd_errstr(err));
		err = usbd_close_pipe(sc->sc_pipe_rx);
		if (err)
			printf("%s: close rx pipe failed: %s\n",
			       USBDEVNAME(sc->sc_dev), usbd_errstr(err));
		sc->sc_pipe_rx = NULL;
	}

	/* Free RX resources. */
	if (sc->sc_rx_xfer != NULL) {
		usbd_free_xfer(sc->sc_rx_xfer);
		sc->sc_rx_xfer = NULL;
		sc->sc_rx_buf  = NULL;
		sc->sc_rx_buf_len = 0;
	}
	if (sc->sc_rx_mbuf != NULL) {
		m_freem(sc->sc_rx_mbuf);
		sc->sc_rx_mbuf = NULL;
	}
	return error ;
}

Static int
rndis_close_tx_pipe(struct rndis_softc *sc)
{
	usbd_status err;
	int error = 0;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	/* TX endpoint */
	if (sc->sc_pipe_tx != NULL) {
		err = usbd_abort_pipe(sc->sc_pipe_tx);
		if (err)
			printf("%s: abort tx pipe failed: %s\n",
			       USBDEVNAME(sc->sc_dev), usbd_errstr(err));
		err = usbd_close_pipe(sc->sc_pipe_tx);
		if (err)
			printf("%s: close tx pipe failed: %s\n",
			       USBDEVNAME(sc->sc_dev), usbd_errstr(err));
		sc->sc_pipe_tx = NULL;
	}
	/* Free TX resources. */
	if (sc->sc_tx_xfer != NULL) {
		usbd_free_xfer(sc->sc_tx_xfer);
		sc->sc_tx_xfer = NULL;
		sc->sc_tx_buf  = NULL;
		sc->sc_tx_buf_len = 0;
	}
	if (sc->sc_tx_mbuf != NULL) {
		m_freem(sc->sc_tx_mbuf);
		sc->sc_tx_mbuf = NULL;
	}
	return error ;
}

Static void
rndis_init(void *xsc)
{
	struct rndis_softc *sc = (struct rndis_softc *)xsc;
	struct ifnet	*ifp = sc->sc_ifp;
	etheraddr_t *eaddr;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	if (sc->sc_dying)
		return;

	/* Cancel pending I/O and free all TX/RX buffers */
	rndis_stop(ifp, 1);

#if __FreeBSD_version >= 700000
	eaddr = (etheraddr_t *)IF_LLADDR(ifp);
#else
	eaddr = (etheraddr_t *)IFP2ENADDR(ifp);
#endif

	if (rndis_set_etheraddr(sc, OID_802_3_CURRENT_ADDRESS, eaddr, 1))
		printf("%s: set MAC addr failed\n", USBDEVNAME(sc->sc_dev));

	/* Initialize packet filter */
	sc->sc_ndis_filter = NDIS_PACKET_TYPE_DIRECTED;
	if (ifp->if_flags & IFF_BROADCAST)
		sc->sc_ndis_filter |= NDIS_PACKET_TYPE_BROADCAST;
	if (ifp->if_flags & IFF_PROMISC)
		sc->sc_ndis_filter |= NDIS_PACKET_TYPE_PROMISCUOUS;
	if (rndis_set_uint32(sc, OID_GEN_CURRENT_PACKET_FILTER, &sc->sc_ndis_filter, 1)) {
		printf("%s: set filter failed\n",
		       USBDEVNAME(sc->sc_dev));
	}

	/* Set lookahead */
	u_int32_t val = ifp->if_mtu ;
	if (rndis_set_uint32(sc, OID_GEN_CURRENT_LOOKAHEAD, &val, 1))
		printf("%s: set lookahead failed\n", USBDEVNAME(sc->sc_dev));

	/* Load the multicast filter */
	if (rndis_setmulti(sc))
		printf("%s: set multicast addr failed\n", USBDEVNAME(sc->sc_dev));

	if (sc->sc_pipe_rx == NULL) {
		if (rndis_open_rx_pipe(sc)) {
			printf("%s: open rx pipe failed\n",
			       USBDEVNAME(sc->sc_dev));
			return;
		}
	}

	if (sc->sc_pipe_tx == NULL ) {
		if (rndis_open_tx_pipe(sc)) {
			printf("%s: open tx pipe failed\n",
			       USBDEVNAME(sc->sc_dev));
                        return;
		}
	}

	ifp->if_drv_flags |= IFF_DRV_RUNNING;
	ifp->if_drv_flags &= ~IFF_DRV_OACTIVE;
	sc->sc_link = 1;

	return ;
}

/* Stop the adapter and free any mbufs allocated to the RX and TX lists. */
Static void
rndis_stop(struct ifnet *ifp, int disable)
{
	struct rndis_softc *sc = ifp->if_softc;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	ifp->if_timer = 0;

	/* Stop transfers */
	rndis_close_rx_pipe(sc);
	rndis_close_tx_pipe(sc);

	sc->sc_link = 0;
	ifp->if_drv_flags &= ~(IFF_DRV_RUNNING | IFF_DRV_OACTIVE);
	return;
}


Static int
rndis_init_device(struct rndis_softc *sc)
{
	struct rndis_msg_snd *msg;
	struct rndis_msg_rcv *res;
	int fact;
	int err = 0;
	
	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	if (sc->sc_dying)
		return 0;

	msg = malloc(RNDIS_MSG_INITIALIZE_SIZE, M_USBDEV, M_WAITOK | M_ZERO);
	res = malloc(RNDIS_MSG_BUFFER_SIZE, M_USBDEV, M_WAITOK | M_ZERO);

	msg->h.type   = RNDIS_MSG_INITIALIZE;
	msg->h.length = htole32(RNDIS_MSG_INITIALIZE_SIZE);
	msg->s.initialize.req_id    = REQID;
	msg->s.initialize.ver_major = htole32(1);
	msg->s.initialize.ver_minor = htole32(0);
	msg->s.initialize.max_xfer_size = htole32(sc->sc_max_rx_size);

	err = rndis_send_message(sc, msg, RNDIS_MSG_INITIALIZE_SIZE);
	if (err) {
		printf("%s: rndis_send_message failed\n", USBDEVNAME(sc->sc_dev));
		goto done;
	}

	err = rndis_recv_message(sc, res, RNDIS_MSG_BUFFER_SIZE,
				 msg->h.type | RNDIS_MSG_COMPLETION, msg->s.initialize.req_id);
	if (err) {
		printf("%s: rndis_recv_message failed\n", USBDEVNAME(sc->sc_dev));
		goto done;
	}
	if (res->r.initialize_cmplt.status != RNDIS_STATUS_SUCCESS) {
		printf("%s: device initialize failed %s\n",
		       USBDEVNAME(sc->sc_dev), rndis_status_str(res->r.initialize_cmplt.status));
		err = EIO;
		goto done;
	}
	if (res->r.initialize_cmplt.medium != RNDIS_MEDIUM_802_3) {
		printf("%s: unsupported rndis media 0x%x\n",
		       USBDEVNAME(sc->sc_dev), le32toh(res->r.initialize_cmplt.medium));
		err = EIO;
		goto done;
	}
	sc->sc_max_tx_size      = le32toh(res->r.initialize_cmplt.max_xfer_size);
	sc->sc_max_pkts_per_msg = le32toh(res->r.initialize_cmplt.max_pkts_per_msg);
	fact = le32toh(res->r.initialize_cmplt.pkt_align_factor);
	if (fact > 3) {
		printf("%s: unsupported packet align factor %d\n",
		       USBDEVNAME(sc->sc_dev), fact);
		err = EIO;
		goto done;
	}
	sc->sc_pkt_align_bytes = (1 << fact);

 done:
        free(msg, M_USBDEV);
        free(res, M_USBDEV);

	return err;

}

#if 0
Static int
rndis_reset_device(struct rndis_softc *sc)
{
	struct rndis_msg_snd *msg;
	struct rndis_msg_rcv *res;
	int err = 0;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	if (sc->sc_dying)
		return 0;

	msg = malloc(RNDIS_MSG_RESET_SIZE, M_USBDEV, M_WAITOK | M_ZERO);
	res = malloc(RNDIS_MSG_BUFFER_SIZE, M_USBDEV, M_WAITOK | M_ZERO);

	msg->h.type   = RNDIS_MSG_RESET;
	msg->h.length = htole32(RNDIS_MSG_RESET_SIZE);

	err = rndis_send_message(sc, msg, RNDIS_MSG_RESET_SIZE);
	if (err) {
		printf("%s: rndis_send_message failed\n", USBDEVNAME(sc->sc_dev));
		goto done;
	}

	err = rndis_recv_message(sc, res, RNDIS_MSG_BUFFER_SIZE,
				 msg->h.type | RNDIS_MSG_COMPLETION, 0);
	if (err) {
		printf("%s: rndis_recv_message failed\n", USBDEVNAME(sc->sc_dev));
		goto done;
	}
	if (res->r.reset_cmplt.status != RNDIS_STATUS_SUCCESS) {
		printf("%s: device reset failed %s\n",
		       USBDEVNAME(sc->sc_dev), rndis_status_str(res->r.reset_cmplt.status));
		err = EIO;
		goto done;
	}
	if (le32toh(res->r.reset_cmplt.addressing_reset)) {
		rndis_setmulti(sc);
	}

 done:
        free(msg, M_USBDEV);
        free(res, M_USBDEV);

	return err;
}
#endif

#if 0
Static int
rndis_halt_device(struct rndis_softc *sc)
{
	struct rndis_msg_snd *msg;
	struct rndis_msg_rcv *res;
	int err = 0;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	if (sc->sc_dying)
		return 0;

	msg = malloc(RNDIS_MSG_RESET_SIZE, M_USBDEV, M_WAITOK | M_ZERO);
	res = malloc(RNDIS_MSG_BUFFER_SIZE, M_USBDEV, M_WAITOK | M_ZERO);

	msg->h.type   = RNDIS_MSG_HALT;
	msg->h.length = htole32(RNDIS_MSG_HALT_SIZE);
	msg->s.halt.req_id = REQID;

	err = rndis_send_message(sc, msg, RNDIS_MSG_HALT_SIZE);
	if (err) {
		printf("%s: rndis_send_message failed\n", USBDEVNAME(sc->sc_dev));
		goto done;
	}
	/* this message has no completion message */

 done:
        free(msg, M_USBDEV);
        free(res, M_USBDEV);

	return err;
}
#endif

Static int
rndis_msg_query(struct rndis_softc *sc, u_int32_t oid, u_char *datap, int *datalen)
{
	struct rndis_msg_snd *msg;
	struct rndis_msg_rcv *res;
	u_char *infp;
	int msglen = 0;
	int reslen = 0;
	int inflen = 0;
	int infoff = sizeof(msg->s.query);
	int err = 0;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	if (sc->sc_dying)
		return 0;

	/* dummy? */
	switch (oid) {
	case OID_GEN_SUPPORTED_LIST:
		inflen = 0x00;
		break;
	case OID_802_3_MAXIMUM_LIST_SIZE:
	case OID_GEN_CURRENT_LOOKAHEAD:
	case OID_GEN_CURRENT_PACKET_FILTER:
	case OID_GEN_LINK_SPEED:
	case OID_GEN_MAC_OPTIONS:
	case OID_GEN_MAXIMUM_FRAME_SIZE:
	case OID_GEN_MAXIMUM_LOOKAHEAD:
	case OID_GEN_MAXIMUM_SEND_PACKETS:
	case OID_GEN_VENDOR_DRIVER_VERSION:
		inflen = 0x04;
		break;
	case OID_802_3_CURRENT_ADDRESS:
	case OID_802_3_PERMANENT_ADDRESS:
		inflen = 0x06;
		break;
	default:
		inflen = 0;
		break;
	}

	msglen = RNDIS_MSG_QUERY_SIZE + inflen;

	msg = malloc(msglen, M_USBDEV, M_WAITOK | M_ZERO);
	res = malloc(RNDIS_MSG_BUFFER_SIZE, M_USBDEV, M_WAITOK | M_ZERO);

	msg->h.type   = RNDIS_MSG_QUERY;
	msg->h.length = htole32(msglen);
	msg->s.query.req_id    = REQID;
	msg->s.query.oid       = htole32(oid);
	msg->s.query.inf_buf_length = htole32(inflen);
	msg->s.query.inf_buf_offset = htole32(infoff);

	err = rndis_send_message(sc, msg, msglen);
	if (err) {
		printf("%s: rndis_send_message failed\n", USBDEVNAME(sc->sc_dev));
		goto done;
	}
	err = rndis_recv_message(sc, res, RNDIS_MSG_BUFFER_SIZE,
				 msg->h.type | RNDIS_MSG_COMPLETION, msg->s.query.req_id);
	if (err) {
		printf("%s: rndis_recv_message failed\n", USBDEVNAME(sc->sc_dev));
		goto done;
	}
	if (res->r.query_cmplt.status != RNDIS_STATUS_SUCCESS) {
		printf("%s: query message failed %s\n",
		       USBDEVNAME(sc->sc_dev), rndis_status_str(res->r.reset_cmplt.status));
		err = EIO;
		goto done;
	}
	reslen = le32toh(res->h.length) - sizeof(res->h);
	inflen = le32toh(res->r.query_cmplt.inf_buf_length);
	infoff = le32toh(res->r.query_cmplt.inf_buf_offset);
	if (reslen < infoff + inflen) {
		printf("%s: query completion size mismatch reslen(%d) < infoff(%d) + inflen(%d)\n",
		       USBDEVNAME(sc->sc_dev), reslen, infoff, inflen) ;
		err = EIO;
		goto done;
	}
	if (inflen > *datalen) {
		printf("%s: data buffer size mismatch need=%d, present=%d\n",
		       USBDEVNAME(sc->sc_dev), inflen, *datalen);
		err = EIO;
		goto done;
	}
	infp = (u_char *)&res->r + infoff;
	bcopy(infp, datap, inflen);
	*datalen = inflen;

 done:
        free(msg, M_USBDEV);
        free(res, M_USBDEV);

	return err;
}

Static int
rndis_msg_set(struct rndis_softc *sc, u_int32_t oid, u_char *datap, int datalen)
{
	struct rndis_msg_snd *msg;
	struct rndis_msg_rcv *res;
	int msglen = 0;
	u_char *infp;
	int err = 0;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	if (sc->sc_dying)
		return 0;

	msglen = RNDIS_MSG_SET_SIZE + datalen;

	msg = malloc(msglen, M_USBDEV, M_WAITOK | M_ZERO);
	res = malloc(RNDIS_MSG_BUFFER_SIZE, M_USBDEV, M_WAITOK | M_ZERO);

	msg->h.type   = RNDIS_MSG_SET;
	msg->h.length = htole32(msglen);
 	msg->s.set.req_id    = REQID;
	msg->s.set.oid       = htole32(oid);
	msg->s.set.inf_buf_length = htole32(datalen);
	msg->s.set.inf_buf_offset = htole32(sizeof(msg->s.set));

	infp = (u_char *)&msg->s + sizeof(msg->s.set);
	bcopy(datap, infp, datalen);

	err = rndis_send_message(sc, msg, msglen);
	if (err) {
		printf("%s: rndis_send_message failed\n", USBDEVNAME(sc->sc_dev));
		goto done;
	}
	err = rndis_recv_message(sc, res, RNDIS_MSG_BUFFER_SIZE,
				 msg->h.type | RNDIS_MSG_COMPLETION, msg->s.set.req_id);
	if (err) {
		printf("%s: rndis_recv_message failed\n", USBDEVNAME(sc->sc_dev));
		goto done;
	}
	if (res->r.set_cmplt.status != RNDIS_STATUS_SUCCESS) {
		printf("%s: set message failed: %s\n",
		       USBDEVNAME(sc->sc_dev),
		       rndis_status_str(res->r.set_cmplt.status));
		err = EIO;
		goto done;
	}

 done:
        free(msg, M_USBDEV);
        free(res, M_USBDEV);

	return err;
}

Static int
rndis_get_uint32(struct rndis_softc *sc, u_int32_t oid, u_int32_t *val, int *cnt)
{
	int err = 0;
	int len = 0;
	int i;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	if (sc->sc_dying)
		return 0;

	len = sizeof(*val) * (*cnt);
	err = rndis_msg_query(sc, oid, (u_char *)val, &len);
	if (err) {
		printf("%s: rndis_msg_query failed\n", USBDEVNAME(sc->sc_dev));
		goto done;
	}
	len = len/sizeof(*val);
	for (i = 0; i < len; i++)
		*(val + 1) = le32toh(*(val + 1));
	*cnt = len;
 done:
	return err;
}

Static int
rndis_set_uint32(struct rndis_softc *sc, u_int32_t oid, u_int32_t *val, int cnt)
{
	int err = 0;
	int i = 0;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	if (sc->sc_dying)
		return 0;

	for (i = 0; i < cnt; i++)
		*(val + i) = htole32(*(val + i));

	err = rndis_msg_set(sc, oid, (u_char *)val, sizeof(*val)*cnt);
	if (err) {
		printf("%s: rndis_msg_set failed\n", USBDEVNAME(sc->sc_dev));
		goto done;
	}

done:
	return err;
}

Static int
rndis_get_etheraddr(struct rndis_softc *sc, u_int32_t oid, etheraddr_t *eaddr, int *cnt)
{
	int err = 0;
	int len = 0;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	if (sc->sc_dying)
		return 0;

	len = ETHER_ADDR_LEN * (*cnt);
	err = rndis_msg_query(sc, oid, (u_char *)eaddr, &len);
	if (err) {
		printf("%s: rndis_msg_query failed\n", USBDEVNAME(sc->sc_dev));
		goto done;
	}

 done:
	return err;
}

Static int
rndis_set_etheraddr(struct rndis_softc *sc, u_int32_t oid, etheraddr_t *eaddr, int cnt)
{
	int err = 0;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	if (sc->sc_dying)
		return 0;

	err = rndis_msg_set(sc, oid, (u_char *)eaddr, sizeof(*eaddr)*cnt);
	if (err) {
		printf("%s: rndis_msg_set failed\n", USBDEVNAME(sc->sc_dev));
		goto done;
	}

done:
	return err;
}

Static int
rndis_send_message(struct rndis_softc *sc, struct rndis_msg_snd *msg, int msglen)
{
	usbd_status err;
	int error = 0;
	usb_device_request_t req;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	if (sc->sc_dying)
		return 0;

	msg->h.length = htole32(msglen);

	req.bmRequestType = UT_WRITE_CLASS_INTERFACE;
	req.bRequest = UCDC_SEND_ENCAPSULATED_COMMAND;
	USETW(req.wValue, 0x0000);
	USETW(req.wIndex, sc->sc_iface_ctrl->index);
	USETW(req.wLength, msglen);

	DPRINTF(("%s: write request send\n", USBDEVNAME(sc->sc_dev)));

	RNDIS_DEBUG_DUMP(msg, msglen);

	sc->sc_refcnt++;
	err = usbd_do_request(sc->sc_udev,
			      &req,
			      msg);
	if (--sc->sc_refcnt < 0)
		usb_detach_wakeup(USBDEV(sc->sc_dev));
	if (err) {
		printf("%s: usbd_do_request failed: %s\n",
		       USBDEVNAME(sc->sc_dev), usbd_errstr(err));
		error = EIO;
		goto done;
	}
	DPRINTF(("%s: write request done\n", USBDEVNAME(sc->sc_dev)));

done:
	return error;
}

Static int
rndis_recv_message(struct rndis_softc *sc, struct rndis_msg_rcv *res, int buflen,
		   u_int32_t msg_type, u_int32_t msg_req_id)
{
	int i;
	usb_device_request_t req;
	int reslen;
	int total_len;
	int error = 0;
	usbd_status err;
	int flag;
	u_int32_t req_id = 0, status = 0;
	struct rndis_msg_snd *msg;

#define RNDIS_MSG_HAS_NONE   0
#define RNDIS_MSG_HAS_REQID  1
#define RNDIS_MSG_HAS_STATUS 2
#define RNDIS_MSG_HAS_CMPLT  4
#define RNDIS_MSG_HAS_ALL    7

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	for ( i = 0; i < 10; i++ ) {
		req.bmRequestType = UT_READ_CLASS_INTERFACE;
		req.bRequest = UCDC_GET_ENCAPSULATED_RESPONSE;
		USETW(req.wValue, 0x0000);
		USETW(req.wIndex, sc->sc_iface_ctrl->index);
		USETW(req.wLength, buflen);

		DPRINTF(("%s: read request send\n", USBDEVNAME(sc->sc_dev)));

		total_len = 0;

		sc->sc_refcnt++;
		err = usbd_do_request_flags(sc->sc_udev,
					    &req,
					    res,
					    USBD_SHORT_XFER_OK,
					    &total_len,
					    USBD_DEFAULT_TIMEOUT);
		if (--sc->sc_refcnt < 0)
			usb_detach_wakeup(USBDEV(sc->sc_dev));
		if (err == USBD_TIMEOUT) {
			printf("%s: usbd_do_request timeout, retry\n",
			       USBDEVNAME(sc->sc_dev));
			tsleep(sc, PWAIT, "rndis", 2);
			continue;
		}
		if (err) {
			printf("%s: usbd_do_request failed: %s\n",
			       USBDEVNAME(sc->sc_dev), usbd_errstr(err));
			error = EIO;
			goto done;
		}
		DPRINTF(("%s: read request done, actlen = %d, req.wLength=%d\n",
			 USBDEVNAME(sc->sc_dev), total_len, UGETW(req.wLength)));

		if (total_len < RNDIS_MSG_HEADER_SIZE) {
			printf("%s: too short message(no header size) require %d, get %d\n",
			       USBDEVNAME(sc->sc_dev), RNDIS_MSG_HEADER_SIZE, total_len);
			tsleep(sc, PWAIT, "rndis", 2);
			continue;
		}

		DPRINTF(("%s: type=0x%x\n",
			 USBDEVNAME(sc->sc_dev), htole32(res->h.type)));

		reslen = le32toh(res->h.length);
		if (total_len < reslen) {
			printf("%s: size mismatch total_len(%d) < message len(%d)\n",
			       USBDEVNAME(sc->sc_dev), total_len, reslen);
			error = EIO;
			goto done;
		}

		RNDIS_DEBUG_DUMP(res, total_len);

		flag = RNDIS_MSG_HAS_NONE;
		switch (res->h.type) {
		case RNDIS_MSG_INITIALIZE_CMPLT:
			if (reslen < RNDIS_MSG_INITIALIZE_CMPLT_SIZE) {
				DPRINTF(("%s: %s too short message\n",
					 USBDEVNAME(sc->sc_dev), __func__));
				goto done;
			}
			req_id = res->r.initialize_cmplt.req_id;
			status = res->r.initialize_cmplt.status;
			flag = RNDIS_MSG_HAS_ALL;
			break;
		case RNDIS_MSG_QUERY_CMPLT:
			if (reslen < RNDIS_MSG_QUERY_CMPLT_SIZE) {
				DPRINTF(("%s: %s too short message\n",
					 USBDEVNAME(sc->sc_dev), __func__));
				goto done;
			}
			req_id = res->r.query_cmplt.req_id;
			status = res->r.query_cmplt.status;
			flag = RNDIS_MSG_HAS_ALL;
			break;
		case RNDIS_MSG_SET_CMPLT:
			if (reslen < RNDIS_MSG_SET_CMPLT_SIZE) {
				DPRINTF(("%s: %s too short message\n",
					 USBDEVNAME(sc->sc_dev), __func__));
				goto done;
			}
			req_id = res->r.set_cmplt.req_id;
			status = res->r.set_cmplt.status;
			flag = RNDIS_MSG_HAS_ALL;
			break;
		case RNDIS_MSG_RESET_CMPLT:
			if (reslen < RNDIS_MSG_RESET_CMPLT_SIZE) {
				DPRINTF(("%s: %s too short message\n",
					 USBDEVNAME(sc->sc_dev), __func__));
				goto done;
			}
			status = res->r.reset_cmplt.status;
			flag = RNDIS_MSG_HAS_STATUS|RNDIS_MSG_HAS_CMPLT;
			break;
		case RNDIS_MSG_INDICATE_STATUS:
			if (reslen < RNDIS_MSG_INDICATE_STATUS_SIZE) {
				DPRINTF(("%s: %s too short message\n",
					 USBDEVNAME(sc->sc_dev), __func__));
				goto done;
			}
			status = res->r.indicate_status.status;
			flag = RNDIS_MSG_HAS_STATUS;
			break;
		case RNDIS_MSG_KEEPALIVE:
			if (reslen < RNDIS_MSG_KEEPALIVE_SIZE) {
				printf("%s: %s: too short\n",
				       USBDEVNAME(sc->sc_dev), __func__);
				break;
				
			}
			msg = malloc(RNDIS_MSG_KEEPALIVE_SIZE, M_USBDEV, M_WAITOK | M_ZERO);
			msg->h.type   = RNDIS_MSG_KEEPALIVE_CMPLT;
			msg->h.length = htole32(RNDIS_MSG_KEEPALIVE_CMPLT_SIZE);
			msg->s.keepalive_cmplt.req_id = res->r.keepalive.req_id;
			msg->s.keepalive_cmplt.status = RNDIS_STATUS_SUCCESS;
			rndis_send_message(sc, msg, RNDIS_MSG_KEEPALIVE_CMPLT_SIZE);
			free(msg, M_USBDEV);
			break;
		case RNDIS_MSG_KEEPALIVE_CMPLT:
			if (reslen < RNDIS_MSG_KEEPALIVE_CMPLT_SIZE) {
				DPRINTF(("%s: %s too short message\n",
					 USBDEVNAME(sc->sc_dev), __func__));
				goto done;
			}
			req_id = res->r.keepalive_cmplt.req_id;
			status = res->r.keepalive_cmplt.status;
			flag = RNDIS_MSG_HAS_ALL;
			break;
		default:
			printf("%s: %s: unknown/unexpected message recv %x\n",
			       USBDEVNAME(sc->sc_dev), __func__, le32toh(res->h.type));
			break;
		}

		if (res->h.type != msg_type) {
			DPRINTF(("%s: %s: message type mismatch, retry\n",
				 USBDEVNAME(sc->sc_dev), __func__));
			tsleep(sc, PWAIT, "rndis", 2);
			continue;
		}
		if ((flag & RNDIS_MSG_HAS_REQID) &&
		    (req_id != msg_req_id)) {
			DPRINTF(("%s: %s: request id mismatch, retry\n",
				 USBDEVNAME(sc->sc_dev), __func__));
			tsleep(sc, PWAIT, "rndis", 2);
			continue;
		}
		if ((flag & RNDIS_MSG_HAS_STATUS) &&
		    (status != RNDIS_STATUS_SUCCESS)) {
			DPRINTF(("%s: %s: status is not a success\n",
				 USBDEVNAME(sc->sc_dev), __func__));
		}
		/* read success */
		error = 0;
		goto done;
	}
	/* retry timeout */
	DPRINTF(("%s: %s retry time out\n",
		 USBDEVNAME(sc->sc_dev), __func__));
	error = EIO;
done:
	return error;
}

Static void
rndis_rxeof(usbd_xfer_handle xfer, usbd_private_handle priv, usbd_status status)
{
	struct rndis_softc *sc = priv;
	struct ifnet *ifp = sc->sc_ifp;
	struct mbuf *m;
	int    total_len, msglen, framelen;
	struct rndis_msg_packet *msg;
	u_char *frame;
	u_char *rx_buf = sc->sc_rx_buf;
	int    rx_buf_len = sc->sc_rx_buf_len;

	DPRINTF(("%s: %s: enter, status = %s\n",
		 USBDEVNAME(sc->sc_dev),__func__, usbd_errstr(status)));

	if (sc->sc_dying)
		return;

	if (!sc->sc_link)
		return;

	sc->sc_rx_xfer_cnt--;

	if (status != USBD_NORMAL_COMPLETION) {
		if (status == USBD_NOT_STARTED || status == USBD_CANCELLED)
			return;
		sc->sc_rx_errs++;
		if (usbd_ratecheck(&sc->sc_rx_notice)) {
			printf("%s: %u usb errors on rx: %s\n",
			       USBDEVNAME(sc->sc_dev), sc->sc_rx_errs,
			       usbd_errstr(status));
			sc->sc_rx_errs = 0;
		}
		if (status == USBD_STALLED) {
			sc->sc_refcnt++;
			usbd_clear_endpoint_stall(sc->sc_pipe_rx);
			if (--sc->sc_refcnt < 0)
				usb_detach_wakeup(USBDEV(sc->sc_dev));
		}
		goto done;
	}

	usbd_get_xfer_status(xfer, NULL, NULL, &total_len, NULL);


	msg = (struct rndis_msg_packet *)(sc->sc_rx_buf) ;
	RNDIS_DEBUG_DUMP(msg, total_len);

	if (sc->sc_rx_remain) {
		total_len += sc->sc_rx_remain;
		sc->sc_rx_remain = 0;
	}

	for ( ; total_len > 0; total_len -= msglen,
		      msg = (struct rndis_msg_packet *)(((u_char *)msg) + msglen)) {
		msglen = total_len;
		if (total_len < sizeof(msg->h)) {
			/* tooooo short */
			printf("%s: usb packet length(%d) < rndis message header size(%d)\n",
			       USBDEVNAME(sc->sc_dev), total_len, sizeof(msg->h));
			goto done;
		}
		if (le32toh(msg->h.length) == 0) {
			DPRINTF(("%s: %s: recieved zero size message\n",
				 USBDEVNAME(sc->sc_dev), __func__));
			continue;
		}
		msglen = sizeof(msg->h);
		if (msg->h.type != RNDIS_MSG_PACKET) {
			/* unknown message */
			printf("%s: unknown rndis message type(0x%x)\n",
			       USBDEVNAME(sc->sc_dev), le32toh(msg->h.type));
			goto done;
		}
		msglen = le32toh(msg->h.length);
		if (msglen < RNDIS_MSG_PACKET_SIZE) {
			/* too short */
			printf("%s: message length(%d) < rndis packet messege size(%d)\n",
			       USBDEVNAME(sc->sc_dev), msglen, RNDIS_MSG_PACKET_SIZE);
			continue;
		}
		if (total_len < msglen) {
			/* continue next transfer */
			DPRINTF(("%s: %s: usb packet length(%d) < message length(%d) continue next transfer\n",
				 USBDEVNAME(sc->sc_dev), __func__, total_len, msglen));
			int remain = msglen - total_len;
			bcopy(msg, sc->sc_rx_buf, total_len);
			rx_buf += total_len;
			rx_buf_len -= total_len;
			sc->sc_rx_remain = remain;
			goto done;
		}

		framelen = le32toh(msg->d.length);

		frame = ((u_char *)&msg->d) + le32toh(msg->d.offset);

		DPRINTF(("%s: packaet data\n"
			 "    msg     = %p\n"
			 "    &msg->d = %p\n"
			 "    framelen= 0x%x\n"
			 "    frame   = %p\n",
			 USBDEVNAME(sc->sc_dev),
			 msg,
			 &msg->d,
			 framelen,
			 frame));

		if (framelen < sizeof(struct ether_header)) {
			printf("%s: frame length(%d) < ether header(%d)",
			       USBDEVNAME(sc->sc_dev), framelen, sizeof(struct ether_header));
			ifp->if_ierrors++;
			continue;
		}

		/* copy data to mbuf */
		m = sc->sc_rx_mbuf;

		if (m == NULL) {
			ifp->if_ierrors++;
			printf("%s: cannot allocate rx mbuf"
			       " -- packet dropped!\n", USBDEVNAME(sc->sc_dev));
			return;
		}

		bcopy(frame, mtod(m, char *), framelen);

		ifp->if_ipackets++;
		m->m_pkthdr.len   = m->m_len = framelen;
		m->m_pkthdr.rcvif = (struct ifnet *)&sc->sc_qdat;

		DPRINTF(("%s: %s: recieved %d\n", USBDEVNAME(sc->sc_dev),
			 __func__, m->m_len));

		usb_ether_input(m);

		m = usb_ether_newbuf();
		if (m == NULL)
			printf("%s: cannot allocate rx mbuf\n",
			       USBDEVNAME(sc->sc_dev));

		sc->sc_rx_mbuf = m;
	}
	return;

 done:
	/* Setup new transfer */
	usbd_setup_xfer(xfer, sc->sc_pipe_rx, sc, rx_buf, rx_buf_len,
			USBD_SHORT_XFER_OK | USBD_NO_COPY,
			USBD_NO_TIMEOUT, rndis_rxeof);
	sc->sc_refcnt++;
	usbd_transfer(xfer);
	if (--sc->sc_refcnt < 0)
		usb_detach_wakeup(USBDEV(sc->sc_dev));
	sc->sc_rx_xfer_cnt++;
	DPRINTF(("%s: %s: start read\n", USBDEVNAME(sc->sc_dev), __func__));
	return ;
}

Static void
rndis_txeof(usbd_xfer_handle xfer, usbd_private_handle priv, usbd_status status)
{
	struct rndis_softc *sc = (struct rndis_softc *)priv;
	struct ifnet *ifp = sc->sc_ifp;

	if (sc->sc_dying)
		return;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	if (status != USBD_NORMAL_COMPLETION) {
		if (status == USBD_NOT_STARTED || status == USBD_CANCELLED) {
			return;
		}
		ifp->if_oerrors++;
		printf("%s: usb error on tx: %s\n", USBDEVNAME(sc->sc_dev),
		       usbd_errstr(status));
		if (status == USBD_STALLED) {
			sc->sc_refcnt++;
			usbd_clear_endpoint_stall(sc->sc_pipe_tx);
			if (--sc->sc_refcnt < 0)
				usb_detach_wakeup(USBDEV(sc->sc_dev));
		}
		return;
	}

	ifp->if_timer = 0;
	ifp->if_drv_flags &= ~IFF_DRV_OACTIVE;
	ifp->if_opackets++;

	if (sc->sc_tx_mbuf != NULL) {
		sc->sc_tx_mbuf->m_pkthdr.rcvif = ifp;
		usb_tx_done(sc->sc_tx_mbuf);
		sc->sc_tx_mbuf = NULL;
	}
}

Static void
rndis_start(struct ifnet *ifp)
{
	struct rndis_softc *sc = ifp->if_softc;
	struct mbuf *m_head = NULL;

	DPRINTF(("%s: %s: enter, link=%d\n", USBDEVNAME(sc->sc_dev),
		 __func__, sc->sc_link));

	if (sc->sc_dying)
		return;

	if (!sc->sc_link)
		return;

	if (ifp->if_drv_flags & IFF_DRV_OACTIVE)
		return;

	IF_DEQUEUE(&ifp->if_snd, m_head);
	if (m_head == NULL)
		return;

	if (rndis_send(sc, m_head, 0)) {
		IF_PREPEND(&ifp->if_snd, m_head);
		ifp->if_drv_flags |= IFF_DRV_OACTIVE;
		return;
	}

	BPF_MTAP(ifp, m_head);

	ifp->if_drv_flags |= IFF_DRV_OACTIVE;

	/* Set a timeout in case the chip goes out to lunch. */
	ifp->if_timer = 5;

	return;
}

Static int
rndis_ioctl(struct ifnet *ifp, u_long cmd, caddr_t data)
{
	struct rndis_softc *sc = ifp->if_softc;
	struct ifreq *ifr = (struct ifreq *)data;
	int error = 0;

	if (sc->sc_dying)
		return EIO;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	switch (cmd) {
	case SIOCSIFFLAGS:
		if (ifp->if_flags & IFF_UP) {
			if (ifp->if_drv_flags & IFF_DRV_RUNNING) {
				sc->sc_ndis_filter &= ~(NDIS_PACKET_TYPE_PROMISCUOUS|NDIS_PACKET_TYPE_ALL_MULTICAST);
				if (ifp->if_flags & IFF_PROMISC)
					sc->sc_ndis_filter |= NDIS_PACKET_TYPE_PROMISCUOUS;
				if (ifp->if_flags & IFF_ALLMULTI)
					sc->sc_ndis_filter |= NDIS_PACKET_TYPE_ALL_MULTICAST;
				if (rndis_set_uint32(sc, OID_GEN_CURRENT_PACKET_FILTER,
						     &sc->sc_ndis_filter, 1))
					printf("%s: set filter failed\n",
					       USBDEVNAME(sc->sc_dev));
			}
			else {
				DPRINTF(("%s: %s: if up\n", USBDEVNAME(sc->sc_dev), __func__));
 				rndis_init(sc);
			}
 		} else {
 			if (ifp->if_drv_flags & IFF_DRV_RUNNING) {
				DPRINTF(("%s: %s: if down\n", USBDEVNAME(sc->sc_dev), __func__));
 				rndis_stop(ifp, 1);
			}
 		}
 		error = 0;
 		break;
	case SIOCADDMULTI:
	case SIOCDELMULTI:
		DPRINTF(("%s: %s: add/del multicast address\n", USBDEVNAME(sc->sc_dev), __func__));
		rndis_setmulti(sc);
		error = 0;
		break;
	case SIOCGIFMEDIA:
	case SIOCSIFMEDIA:
		DPRINTF(("%s: %s: media control\n", USBDEVNAME(sc->sc_dev), __func__));
		error = ifmedia_ioctl(ifp, ifr, &sc->sc_ifmedia, cmd);
		break;
	default:
		DPRINTF(("%s: %s: any other ioctll\n", USBDEVNAME(sc->sc_dev), __func__));
		error = ether_ioctl(ifp, cmd, data);
		break;
	}

	DPRINTF(("%s: %s: leave\n", USBDEVNAME(sc->sc_dev), __func__));
	return (error);
}

Static int
rndis_setmulti(struct rndis_softc *sc)
{
	struct ifnet *ifp;
	struct ifmultiaddr *ifma;
	int mclistmax;
	int cnt;
	int mclistlen;
	int error = 0;
	u_char (*mclist)[ETHER_ADDR_LEN] = NULL;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	if (sc->sc_dying)
		return 0;

	ifp = sc->sc_ifp;

	if (ifp->if_flags & IFF_ALLMULTI || ifp->if_flags & IFF_PROMISC) {
		sc->sc_ndis_filter |= NDIS_PACKET_TYPE_ALL_MULTICAST;
		goto done;
	}

	if (TAILQ_EMPTY(&ifp->if_multiaddrs))
		return 0;

	cnt = 1;
	if (rndis_get_uint32(sc, OID_802_3_MAXIMUM_LIST_SIZE, &mclistmax, &cnt)) {
		sc->sc_ndis_filter |= NDIS_PACKET_TYPE_ALL_MULTICAST;
		printf("%s: get malticast list max failed\n", USBDEVNAME(sc->sc_dev));
		error = EIO;
		goto done;
	}

	mclist = malloc(sizeof(mclist[0]) * mclistmax, M_USBDEV, M_WAITOK | M_ZERO);

	sc->sc_ndis_filter |= NDIS_PACKET_TYPE_MULTICAST;

	mclistlen = 0;
	IF_ADDR_LOCK(ifp);
	TAILQ_FOREACH(ifma, &ifp->if_multiaddrs, ifma_link) {
		if (ifma->ifma_addr->sa_family != AF_LINK)
			continue;
		if (mclistlen >= mclistmax) {
			IF_ADDR_UNLOCK(ifp);
			sc->sc_ndis_filter |= NDIS_PACKET_TYPE_ALL_MULTICAST;
			sc->sc_ndis_filter &= ~NDIS_PACKET_TYPE_MULTICAST;
			goto done;
		}
		bcopy(LLADDR((struct sockaddr_dl *)ifma->ifma_addr),
		      mclist[mclistlen], sizeof(mclist[mclistlen]));
		mclistlen++;
	}
	IF_ADDR_UNLOCK(ifp);

	error = rndis_set_etheraddr(sc, OID_802_3_MULTICAST_LIST, mclist, mclistlen);
	if (error) {
		printf("%s: set mclist failed: %d\n", USBDEVNAME(sc->sc_dev),error);
		sc->sc_ndis_filter |= NDIS_PACKET_TYPE_ALL_MULTICAST;
		sc->sc_ndis_filter &= ~NDIS_PACKET_TYPE_MULTICAST;
	}

done:
	if (mclist)
		free(mclist, M_USBDEV);

	if (rndis_set_uint32(sc, OID_GEN_CURRENT_PACKET_FILTER, &sc->sc_ndis_filter, 1)) {
		printf("%s: set filter failed\n", USBDEVNAME(sc->sc_dev));
		error = EIO;
	}

	return error;

}

Static void rndis_watchdog(struct ifnet *ifp)
{
	struct rndis_softc *sc = ifp->if_softc;
	usbd_status stat;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	ifp->if_oerrors++;
	printf("%s: watchdog timeout\n", USBDEVNAME(sc->sc_dev));

	usbd_xfer_handle xfer = sc->sc_tx_xfer;
	void *priv = sc;

	usbd_get_xfer_status(xfer, NULL, NULL, NULL, &stat);
	rndis_txeof(xfer, priv, stat);

	if ( ifp->if_snd.ifq_head != NULL )
		rndis_start(ifp);

	return;
}

Static void
rndis_rxstart(struct ifnet *ifp)
{
	struct rndis_softc	*sc;
	usbd_xfer_handle xfer;

	sc = ifp->if_softc;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	if (sc->sc_dying)
		return;

	/* Setup new transfer. */
	if (sc->sc_rx_mbuf == NULL) {
		sc->sc_rx_mbuf = usb_ether_newbuf();
		if (sc->sc_rx_mbuf == NULL) {
			printf("%s: cannot allocate rx mbuf"
			       " -- packet dropped!\n", USBDEVNAME(sc->sc_dev));
			ifp->if_ierrors++;
			return;
		}
	}
	if (sc->sc_rx_xfer_cnt == 0) {
		xfer = sc->sc_rx_xfer;
		usbd_setup_xfer(xfer, sc->sc_pipe_rx,
				sc, sc->sc_rx_buf, sc->sc_rx_buf_len,
				USBD_SHORT_XFER_OK | USBD_NO_COPY,
				USBD_NO_TIMEOUT, rndis_rxeof);
		usbd_transfer(xfer);
		sc->sc_rx_xfer_cnt++;
		DPRINTF(("%s: %s: start read\n", USBDEVNAME(sc->sc_dev),
			 __func__));
	}
	return;
}

/* rndis status to string */
Static const char *
rndis_status_str(u_int32_t rndis_status)
{
	switch (rndis_status) {
	case RNDIS_STATUS_BUFFER_OVERFLOW:
		return "buffer overflow";
	case RNDIS_STATUS_FAILURE:
		return "failure";
	case RNDIS_STATUS_INVALID_DATA:
		return "invalid data";
	case RNDIS_STATUS_MEDIA_CONNECT:
		return "media connect";
	case RNDIS_STATUS_MEDIA_DISCONNECT:
		return "media disconnect";
	case RNDIS_STATUS_NOT_SUPPORTED:
		return "not supported";
	case RNDIS_STATUS_PENDING:
		return "pending";
	case RNDIS_STATUS_RESOURCES:
		return "resources";
	case RNDIS_STATUS_SUCCESS:
		return "success";
	default:
		return "unknown";
	}
}

Static int
rndis_send(struct rndis_softc *sc, struct mbuf *m, int idx)
{
	int total_len;
	usbd_status err;
	struct rndis_msg_packet *pkt;
	int align = sc->sc_pkt_align_bytes;
	u_char *datap;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev),__func__));

	if (sc->sc_dying)
		return 0;

	if (!sc->sc_link)
		return 0;

	total_len = sizeof(*pkt) + m->m_pkthdr.len;
	total_len = (total_len + align - 1)/align * align;

	if (total_len > sc->sc_tx_buf_len) {
		printf("%s: %s: buffer over run tx_buf_len < total_len\n",
		       USBDEVNAME(sc->sc_dev), __func__);
		sc->sc_ifp->if_oerrors++;
		return EIO;
	}

	/* Copy the mbuf data into a contiguous buffer */
	pkt = (struct rndis_msg_packet *)sc->sc_tx_buf;
	bzero(pkt, sizeof(*pkt));
	pkt->h.type   = RNDIS_MSG_PACKET;
	pkt->h.length = htole32(total_len);
	pkt->d.offset = htole32(sizeof(pkt->d));
	pkt->d.length = htole32(m->m_pkthdr.len);

	datap = (u_char *)pkt + sizeof(*pkt);

	m_copydata(m, 0, m->m_pkthdr.len, datap);

	sc->sc_tx_mbuf = m;

	RNDIS_DEBUG_DUMP(pkt, total_len);

	usbd_xfer_handle xfer;
	xfer = sc->sc_tx_xfer;
	usbd_setup_xfer(sc->sc_tx_xfer, sc->sc_pipe_tx, sc, sc->sc_tx_buf, total_len,
			USBD_FORCE_SHORT_XFER | USBD_NO_COPY,
			RNDIS_TX_TIMEOUT, rndis_txeof);

	/* Transmit */
	sc->sc_refcnt++;
	err = usbd_transfer(xfer);
	if (--sc->sc_refcnt < 0)
		usb_detach_wakeup(USBDEV(sc->sc_dev));
	if (err != USBD_IN_PROGRESS) {
		printf("%s: rndis_send error=%s\n", USBDEVNAME(sc->sc_dev),
		       usbd_errstr(err));
		rndis_stop(sc->sc_ifp, 1);
		return (EIO);
	}

	DPRINTF(("%s: %s: send %d bytes\n", USBDEVNAME(sc->sc_dev),
		 __func__, total_len));

	return 0;
}

Static int
rndis_ifmedia_change(struct ifnet *ifp)
{
	struct rndis_softc *sc = (struct rndis_softc*)ifp->if_softc;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));
	if (sc->sc_dying)
		return 0;

	rndis_init(sc);

	return 0;
}

Static void
rndis_ifmedia_status(struct ifnet * const ifp, struct ifmediareq *ifmr)
{
	struct rndis_softc *sc = (struct rndis_softc*)ifp->if_softc;
	
	u_int32_t		media_info;
	ndis_media_state	linkstate = nmc_connected;
	int			error, cnt;

	DPRINTF(("%s: %s: enter\n", USBDEVNAME(sc->sc_dev), __func__));

	if (sc->sc_dying)
		return;

	ifmr->ifm_status = IFM_AVALID ;
	ifmr->ifm_active = IFM_ETHER ;

#if 0
	cnt = 1;
	error = rndis_get_uint32(sc, OID_GEN_MEDIA_CONNECT_STATUS,
				 (u_int32_t *)&linkstate, &cnt);
#endif

	cnt = 1;
	error = rndis_get_uint32(sc, OID_GEN_LINK_SPEED,
				 &media_info, &cnt);
	if (linkstate == nmc_connected)
		ifmr->ifm_status |= IFM_ACTIVE;

	switch(media_info) {
	case 100000:
		ifmr->ifm_active |= IFM_10_T;
		break;
	case 1000000:
		ifmr->ifm_active |= IFM_100_TX;
		break;
	case 10000000:
		ifmr->ifm_active |= IFM_1000_T;
		break;
	default:
		DPRINTF(("%s: unknown speed: %d\n", USBDEVNAME(sc->sc_dev), media_info));
		break;
	}

	return;
}
