/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_hook.c - branch trace module (djprobe interface)                      */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2006                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <linux/kallsyms.h>
//#include <linux/interrupt.h> // for DEBUG (preemptible)
#include <linux/module.h>
#include "djprobe.h"
#include "ctredirect.h"
#include "bt.h"	// for DEBUG

extern unsigned long switch_to_addr;
extern int switch_to_size;

extern cpumask_t bt_enabled;
extern int pid_cnt;
extern struct pid_manage pid_manage[NR_CPUS];

void write_pid_record(pid_t);
void bts_log_write(void);
int is_trace_pid(pid_t);
void bts_facility_on(void);
void bts_facility_off(void);
void bt_start(void);
void bt_stop(void);

void check_and_wait_relfs_write(void);

static void context_switch_probe(struct djprobe *djp, struct pt_regs *regs)
{
	/* !!! do not printk in this routine. */
	struct task_struct *prev, *next;
	int trace_prev, trace_next;
	int cpu;
	struct pid_manage *pid_mng;

	/*
	serial_prints("context_switch_probe(cpu:%d pid:%d preemptible:%d)\n",
		      smp_processor_id(), current->pid, preemptible());
		      */
	if (!is_app_mode)	/* case of kernel trace, nothing to do. */
		return;

	cpu = get_cpu();
	prev = (struct task_struct*)regs->eax;
	next = (struct task_struct*)regs->edx;
	if (prev->pid == next->pid || !cpu_isset(cpu, bt_enabled))
		goto EXIT;

	trace_prev = is_trace_pid(prev->pid);
	trace_next = is_trace_pid(next->pid);
	if (!trace_prev && !trace_next)
		goto EXIT;

	if (trace_next) {
		if (trace_prev) {
			bts_facility_off();
			bts_log_write();
		}
		pid_mng = &pid_manage[cpu];
		pid_mng->pid = next->pid;
		pid_mng->is_wrote = 0;
		if (!trace_prev)
			bts_facility_on();
	} else {
		bts_facility_off();
		bts_log_write();
	}
EXIT:
	put_cpu();
}

static void sysenter_probe(struct pt_regs regs)
{
	int cpu;

	cpu = get_cpu();
	/*
	serial_prints("sysenter_probe(cpu:%d pid:%d preemptible:%d)\n",
		      smp_processor_id(), current->pid, preemptible());
		      */
	if (!cpu_isset(cpu, bt_enabled))
		goto EXIT;
	if (!is_app_mode || !is_trace_pid(current->pid))
		goto EXIT;
	bts_facility_off();
	//serial_prints("(%d)sysenter\n", cpu);
	check_and_wait_relfs_write();
EXIT:
	put_cpu();
}

static void sysexit_probe(struct pt_regs regs)
{
	int cpu;
	
	cpu = get_cpu();
	/*
	serial_prints("sysexit_probe(cpu:%d pid:%d preemptible:%d)\n",
		      smp_processor_id(), current->pid, preemptible());
		      */
	if (!cpu_isset(cpu, bt_enabled))
		goto EXIT;
	if (!is_app_mode || !is_trace_pid(current->pid))
		goto EXIT;
	bts_facility_on();
EXIT:
	put_cpu();
}

static struct djprobe djp;

int reg_probe(void)
{
	int rc;

	if (!switch_to_addr || !switch_to_size)
		return -EINVAL;

	djp.addr = (void*)switch_to_addr;
	djp.handler = context_switch_probe;
	djp.size = switch_to_size;
	if ((rc = register_djprobe(&djp)) < 0)
		return rc;

	if ((rc = register_ctr(sysenter_probe, sysexit_probe)) < 0)
		return rc;
	return 0;
}

void unreg_probe(void)
{
	unregister_djprobe(&djp);
	unregister_ctr();
	return;
}
