/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_relfs.c - branch trace module (proc file system interface)            */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2006                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <asm/apic.h>
#include <asm/msr.h>
#include <asm/processor.h>
#include <asm/uaccess.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/smp.h>
#include <linux/vmalloc.h>
#include <linux/mm.h>
#include "bt.h"

extern cpumask_t bt_enabled;

struct proc_dir_entry *proc_bts;
static struct proc_dir_entry *proc_enable;
static struct proc_dir_entry *proc_pid;
extern pid_t *pid_tbl;
extern int pid_max;
extern int pid_cnt;

extern void bt_enable(void);
extern void bt_disable(void);

static int proc_write_core(const char *user_buf, unsigned long count,
			   char* (*func)(char*))
{
	int len = 64;
	char buf[len + 1], *p, *end;

	if (len > count)
		len = count;
	if (copy_from_user(buf, user_buf, len))
		return -EFAULT;
	buf[len - 1] = '\0';

	serial_prints("proc_write_core: >%.*s<\n", len, buf);
	for (p = buf; p < buf + len; p = end) {
		if (!(end = func(p))) {
			if (p == buf)
				return -EINVAL;
			end = p;
			break;
		}
		serial_prints("  consumed-len: %d\n", end - p);
	}
	return p - buf;
}

/* write of the pids is limitted by one line, 4095 chars */

static int proc_pid_read(char *buf, char **start, off_t off,
			 int count, int *eof, void *data)
{
	int i, n, line_chars = 11;
	char *fmt;

	n = 0;
	for (i = off / line_chars; i < pid_cnt; i++) {
		if (n + line_chars > count)
			break;
		fmt = (i == pid_cnt - 1) ? "%010d\n" : "%010d,";
		n += sprintf(buf + n, fmt, pid_tbl[i]);
	}
	*(unsigned long*)start = (unsigned long)n;
	return n;
}

static int pid_write_in_process;

static char* func_pid_write(char *buf)
{
	char *p_end;
	long val;

	val = simple_strtol(buf, &p_end, 0);
	if ((*p_end != '\0' && *p_end != ',') || pid_cnt >= pid_max)
		return NULL;

	if (*p_end == '\0')
		pid_write_in_process = 0;
	pid_tbl[pid_cnt++] = val;
	return p_end + 1;
}

static int proc_pid_write(struct file *file, const char *user_buf,
			  unsigned long count, void *data)
{
	int rc;

	if (!cpus_empty(bt_enabled))
		return -EFAULT;

	if (!pid_write_in_process)
		pid_cnt = 0;
	pid_write_in_process = 1;
	if ((rc = proc_write_core(user_buf, count, func_pid_write)) < 0)
		pid_cnt = 0;
	return rc;
}

#if 0
static void otameshi_print(void)
{
	struct task_struct *t;
	struct vm_area_struct *v;

	serial_prints("------ otameshi print all maps (from) ------\n");
	write_lock_irq(&tasklist_lock);
	t = &init_task;
	for (t = next_task(t); t != &init_task; t = next_task(t)) {
		if (!t->mm)
			continue;
		serial_prints("(%d)\n", t->pid);
		for (v = t->mm->mmap; v; v = v->vm_next) {
			if (v->vm_flags & VM_EXEC)
				serial_prints("%08lx-%08lx\n",
					      v->vm_start, v->vm_end);
		}
	}
	write_unlock_irq(&tasklist_lock);
	serial_prints("------ otameshi print all maps (to) ------\n");
}
#endif

static int proc_enable_read(char *buf, char **start, off_t off,
			    int count, int *eof, void *data)
{
	int n;

	n = snprintf(buf, count, "%d\n", !cpus_empty(bt_enabled));

	return n;
}

static int proc_enable_write(struct file *file, const char *user_buf,
			     unsigned long count, void *data)
{
	char *p_end;
	long val, old_val;
	char buf[16];
	unsigned long size;

	size = (sizeof(buf) < count) ? sizeof(buf) : count;

	if (copy_from_user(buf, user_buf, size))
		return -EFAULT;

	buf[size-1] = '\0';
	val = simple_strtol(buf, &p_end, 0);
	if (*p_end != '\0')
		return -EINVAL;

	old_val = cpus_empty(bt_enabled) ? 0 : 1;
	if (!(val ^ old_val))
		return count;
	if (val) {
		//otameshi_print();
		bt_enable();
	} else
		bt_disable();
	return count;
}

int proc_init(void)
{
	proc_bts = proc_mkdir("bts", &proc_root);
	if (!proc_bts)
		return -ENOMEM;
	pid_tbl = vmalloc(sizeof(pid_t) * pid_max);
	if (!pid_tbl)
		return -ENOMEM;
	proc_pid = create_proc_entry("pid", 0600, proc_bts);
	if (!proc_pid) {
		return -ENOMEM;
	}
	proc_pid->read_proc = proc_pid_read;
	proc_pid->write_proc = proc_pid_write;
	proc_enable = create_proc_entry("enable", 0600, proc_bts);
	if (!proc_enable) {
		return -ENOMEM;
	}
	proc_enable->read_proc = proc_enable_read;
	proc_enable->write_proc = proc_enable_write;

	return 0;
}

void proc_cleanup(void)
{
	if (proc_enable)
		remove_proc_entry(proc_enable->name, proc_bts);
	if (pid_tbl)
		vfree(pid_tbl);
	if (proc_pid)
		remove_proc_entry(proc_pid->name, proc_bts);
	if (proc_bts)
		remove_proc_entry(proc_bts->name, &proc_root);
}
