/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_hook.c - branch trace module (hook interface)                         */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2006                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

//#include <linux/interrupt.h> // for DEBUG (preemptible)
#include <linux/module.h>
#ifdef USE_KPROBES
#  include <linux/kprobes.h>
#else
#  include "djprobe.h"
#endif
#include "ctr.h"
#include "bt.h"	// for DEBUG

extern unsigned long switch_to_addr;
extern int switch_to_size;

extern cpumask_t enable;
extern int mode;

extern struct info_per_cpu bt_info_per_cpu[NR_CPUS];

void write_pid_record(pid_t);
void bts_log_write(void);
int is_trace_pid(pid_t);
void bts_facility_on(void);
void bts_facility_off(void);
void bts_facility_save_and_off(int*);
void bts_facility_restore(int);
void bt_start(void);
void bt_stop(void);

void check_and_wait_relfs_write(void);

static int is_kernel_thread(pid_t pid)
{
	struct task_struct *t;
	int rc = 0;

	/* write_lock_irq(&tasklist_lock); */
	t = &init_task;
	for (t = next_task(t); t != &init_task; t = next_task(t)) {
		if (t->pid < 2)
			continue;
		if (t->mm)
			break;
		if (t->pid == pid) {
			rc = 1;
			break;
		}
	}
	/* write_unlock_irq(&tasklist_lock); */
	//serial_prints("is_kt(%d):%d\n", pid, rc);
	return rc;
}

#define is_target_user_process(pid)	\
	(!is_kernel_thread(pid) && is_trace_pid(pid))

static void context_switch_probe(struct pt_regs *regs)
{
	/* !!! do not printk in this routine. */
	struct task_struct *prev, *next;
	int trace_prev, trace_next;
	int cpu, en, save;
	struct pid_manage *pid_mng;

	cpu = get_cpu();
	if (!(en = cpu_isset(cpu, enable)))
		goto EXIT;
	prev = (struct task_struct*)regs->eax;
	next = (struct task_struct*)regs->edx;
	if (prev->pid == next->pid)
		goto EXIT;

	//serial_prints("cs(%d) %d -> %d)\n", smp_processor_id(), prev, next);
	bts_facility_save_and_off(&save);
	if (is_kern_all_by_hook(mode)) {
		trace_prev = 1;
		trace_next = en;
	} else if (is_upid(mode)) {
		trace_prev = is_target_user_process(prev->pid);
		trace_next = en && is_target_user_process(next->pid);
	} else {
		trace_prev = is_trace_pid(prev->pid);
		trace_next = en && is_trace_pid(next->pid);
	}
	bts_facility_restore(save);

	if (trace_prev) {
		bts_facility_off();
		bts_log_write();
	}
	if (trace_next) {
		pid_mng = &bt_info_per_cpu[cpu].pid_manage;
		pid_mng->pid = next->pid;
		pid_mng->is_wrote = 0;
		bts_facility_on();
	}
EXIT:
	put_cpu();
}

static void sysenter_probe(const ctr_handler_t h)
{
	int cpu, save;

	cpu = smp_processor_id();
	if (!cpu_isset(cpu, enable))
		return;
	bts_facility_save_and_off(&save);
	if (!is_target_user_process(current->pid)) {
		bts_facility_restore(save);
		return;
	}
	//serial_prints("sent(%d) %d\n", smp_processor_id(), current->pid);
	check_and_wait_relfs_write();
	bts_log_write();
}

static void sysexit_probe(const ctr_handler_t h)
{
	int cpu, save;
	
	cpu = smp_processor_id();
	if (!cpu_isset(cpu, enable))
		return;
	bts_facility_save_and_off(&save);
	if (!is_target_user_process(current->pid)) {
		bts_facility_restore(save);
		return;
	}
	//serial_prints("sexit(%d) %d\n", smp_processor_id(), current->pid);
	bts_facility_on();
}

#ifdef USE_KPROBES
static struct kprobe kp;
static int context_switch_kp_wrapper(struct kprobe *kp, struct pt_regs *regs)
{
	context_switch_probe(regs);
	return 0;
}
#else
static struct djprobe djp;
static void context_switch_djp_wrapper(struct djprobe *djp,
				       struct pt_regs *regs)
{
	context_switch_probe(regs);
}
#endif

int reg_probe(void)
{
	int rc;

	if (!switch_to_addr || !switch_to_size)
		return -EINVAL;
#ifdef USE_KPROBES
	kp.addr = (kprobe_opcode_t*)switch_to_addr;
	kp.pre_handler = context_switch_kp_wrapper;
	if ((rc = register_kprobe(&kp)) < 0)
		return rc;
#else
	djp.addr = (void*)switch_to_addr;
	djp.handler = context_switch_djp_wrapper;
	djp.size = switch_to_size;
	if ((rc = register_djprobe(&djp)) < 0)
		return rc;
#endif
	if (is_user(mode)) {
		if ((rc = register_ctr(sysenter_probe, sysexit_probe)) < 0)
			return rc;
	}
	return 0;
}

void unreg_probe(void)
{
#ifdef USE_KPROBES
	unregister_kprobe(&kp);
#else
	unregister_djprobe(&djp);
#endif
	if (is_user(mode))
		unregister_ctr();
	return;
}
