/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_relfs.c - branch trace module (proc file system interface)            */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2006                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <asm/apic.h>
#include <asm/msr.h>
#include <asm/processor.h>
#include <asm/uaccess.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/smp.h>
#include <linux/vmalloc.h>
#include <linux/mm.h>
#include "bt.h"

struct proc_dir_entry *proc_btrax;
static struct proc_dir_entry *proc_enable;
static struct proc_dir_entry *proc_pid;
static struct proc_dir_entry *proc_btsbuf_sz;
static struct proc_dir_entry *proc_bts_mrecs;
extern struct info_per_cpu bt_info_per_cpu[NR_CPUS];
extern cpumask_t enable;
extern unsigned long *pid_tbl;
extern int pid_tbl_ulongs;
extern int pid_max;
extern int long_bits;
extern int mode;
extern size_t btsbuf_size;
extern size_t bts_int_margin_recs;

extern void bt_enable(void);
extern void bt_disable(void);
extern int add_pid_tbl(pid_t);

static int in_pid_write;

/* proc_pid_read is exists for debug purpose */
static int proc_pid_read(char *buf, char **start, off_t off,
			 int count, int *eof, void *data)
{
	int i, j, len, max = 10;
	unsigned long mask;
	pid_t pid, pid_max_in_tbl = 0;
	off_t loff;
	char *fmt, lbuf[max + 1], *p;

	loff = 0;
	*start = p = buf;
	
	/* first, check pid_max_in_tbl */
	for (i = 0; i < pid_tbl_ulongs; i++) {
		if (!pid_tbl[i])
			continue;
		for (j = 0, mask = 1; j < long_bits; j++, mask <<= 1) {
			if (!(pid_tbl[i] & mask))
				continue;
			pid = i * long_bits + j;
			if (pid > pid_max_in_tbl)
				pid_max_in_tbl = pid;
		}
	}

	for (i = 0; i < pid_tbl_ulongs; i++) {
		if (!pid_tbl[i])
			continue;
		for (j = 0, mask = 1; j < long_bits; j++, mask <<= 1) {
			if (!(pid_tbl[i] & mask))
				continue;
			pid = i * long_bits + j;
			fmt = (pid == pid_max_in_tbl) ? "%d\n" : "%d,";
			len = snprintf(lbuf, max, fmt, pid);
			if (len < 0 || len >= max)
				return -EFAULT;
			if (loff >= off && loff + len <= off + count) {
				memcpy(p, lbuf, len);
				p += len;
			} else if (loff + len > off + count)
				break;
			loff += len;
		}
	}
	return p - buf;
}

static int pid_write_from_buf(char *buf, unsigned long len,
			      unsigned long *consumed)
{
	long val;
	char *p, *p_max, *p_end;

	p_max = buf + len;
	for (p = buf; p < p_max; p = p_end + 1) {
		val = simple_strtol(p, &p_end, 0);
		switch (*p_end) {
		case ',':
			if (add_pid_tbl(val))
				return -EINVAL;
			break;
		case '\n':
			if (p_max != p_end + 1)
				return -EINVAL;
			if (add_pid_tbl(val))
				return -EINVAL;
			*consumed = p_max - p;
			return 1;
		case '\0':
			if (p_max != p_end)
				return -EINVAL;
			*consumed = p - buf;
			return 0;
			break;
		}
	}
	*consumed = p - buf;
	return 0;
}

/* The data length (count) might become shorter than the string length
 * of one pid. At this time, it is necessary to process the string
 * because it terminates abnormally when returns 0.
 * Therefore, it is necessary to preserve it in the static variable buf.
 */
#define PID_WRITE_BUF_SIZE	64
static int proc_pid_write(struct file *file, const char *user_buf,
			  unsigned long count, void *data)
{
	unsigned long max = PID_WRITE_BUF_SIZE, len, consumed;
	static unsigned long left;
	static char buf[PID_WRITE_BUF_SIZE + 1];
	char *p, *p_max;
	int rc;

	if (!is_enable_by_proc(mode) || !cpus_empty(enable))
		return -EFAULT;

	if (!in_pid_write) {
		memset(pid_tbl, 0,
		       pid_tbl_ulongs * (sizeof(long) / sizeof(char)));
		left = 0;
	}
	in_pid_write = 1;

	for (p = (char*)user_buf, p_max = (char*)user_buf + count;
	     p < p_max; p += len) {
		len = max - left;
		if (len > p_max - p)
			len = p_max - p;
		if (copy_from_user(buf + left, p, len))
			return -EFAULT;
		buf[left + len] = '\0';
		if ((rc = pid_write_from_buf(buf, left + len, &consumed)) < 0)
			return rc;
		//printk("BUF:%s:%ld:%d:%ld\n", buf, left + len, rc, consumed);
		if (rc)	/* finished ? */
			in_pid_write = 0;
		if (consumed) {
			len = consumed - left;
			left = 0;
		} else {
			left = len;
		}
	}
	return p - user_buf;
}

#if 0
static void otameshi_print(void)
{
	struct task_struct *t;
	struct vm_area_struct *v;

	serial_prints("------ otameshi print all maps (from) ------\n");
	write_lock_irq(&tasklist_lock);
	t = &init_task;
	for (t = next_task(t); t != &init_task; t = next_task(t)) {
		if (!t->mm)
			continue;
		serial_prints("(%d)\n", t->pid);
		for (v = t->mm->mmap; v; v = v->vm_next) {
			if (v->vm_flags & VM_EXEC)
				serial_prints("%08lx-%08lx\n",
					      v->vm_start, v->vm_end);
		}
	}
	write_unlock_irq(&tasklist_lock);
	serial_prints("------ otameshi print all maps (to) ------\n");
}
#endif

static int proc_enable_read(char *buf, char **start, off_t off,
			    int count, int *eof, void *data)
{
	int n;

	n = snprintf(buf, count, "%d\n", !cpus_empty(enable));

	return n;
}

static int proc_enable_write(struct file *file, const char *user_buf,
			     unsigned long count, void *data)
{
	char *p_end;
	long val;
	char buf[16];
	unsigned long size;

	size = (sizeof(buf) < count) ? sizeof(buf) : count;

	if (copy_from_user(buf, user_buf, size))
		return -EFAULT;

	buf[size-1] = '\0';
	val = simple_strtol(buf, &p_end, 0);
	if (*p_end != '\0')
		return -EINVAL;

	if (val) {
		//otameshi_print();
		bt_enable();
	} else
		bt_disable();
	return count;
}

static int proc_btsbuf_sz_read(char *buffer, char **start, off_t off,
			       int count, int *eof, void *data)
{
	return sprintf(buffer, "%d\n", btsbuf_size);
}

static int proc_bts_mrecs_read(char *buffer, char **start, off_t off,
			       int count, int *eof, void *data)
{
	return sprintf(buffer, "%d\n", bts_int_margin_recs);
}

static int proc_on_off_cnt_read(char *buf, char **start, off_t off,
				int count, int *eof, void *data)
{
	int n, cpu;

	cpu = (int)data;
	n = snprintf(buf, count, "%ld\n", bt_info_per_cpu[cpu].on_off_cnt);

	return n;
}

int proc_init(void)
{
	int i, bytes_of_long;
	char tmp[3 + 4 + 1]; /* cpuXXXX */
	struct info_per_cpu *info;

	proc_btrax = proc_mkdir("btrax", &proc_root);
	if (!proc_btrax)
		return -ENOMEM;

	bytes_of_long = sizeof(long) / sizeof(char);
	long_bits = BITS_PER_BYTE * bytes_of_long;
	pid_tbl_ulongs = pid_max / long_bits;
	if (pid_max % long_bits)
		pid_tbl_ulongs++;
	pid_tbl = vmalloc(pid_tbl_ulongs * bytes_of_long);
	if (!pid_tbl)
		return -ENOMEM;
	memset(pid_tbl, 0, pid_tbl_ulongs * bytes_of_long);

	proc_pid = create_proc_entry("pid", 0600, proc_btrax);
	if (!proc_pid)
		return -ENOMEM;
	proc_pid->read_proc = proc_pid_read;
	proc_pid->write_proc = proc_pid_write;

	proc_enable = create_proc_entry("enable", 0600, proc_btrax);
	if (!proc_enable)
		return -ENOMEM;
	proc_enable->read_proc = proc_enable_read;
	proc_enable->write_proc = proc_enable_write;

	proc_btsbuf_sz = create_proc_entry("btsbuf_size", 0400, proc_btrax);
	if (!proc_btsbuf_sz)
		return -ENOMEM;
	proc_btsbuf_sz->read_proc = proc_btsbuf_sz_read;

	proc_bts_mrecs = create_proc_entry("bts_mrecs", 0400, proc_btrax);
	if (!proc_bts_mrecs)
		return -ENOMEM;
	proc_bts_mrecs->read_proc = proc_bts_mrecs_read;

	for_each_online_cpu(i) {
		info = &bt_info_per_cpu[i];
		snprintf(tmp, sizeof(tmp), "cpu%d", i);
		info->p_cpuN = proc_mkdir(tmp, proc_btrax);
		if (!info->p_cpuN)
			return -ENOMEM;
		info->p_on_off_cnt = create_proc_entry("on_off_cnt", 0400,
						       info->p_cpuN);
		if (!info->p_on_off_cnt)
			return -ENOMEM;
		info->p_on_off_cnt->data = (void*)i;
		info->p_on_off_cnt->read_proc = proc_on_off_cnt_read;
	}
	return 0;
}

void proc_cleanup(void)
{
	int i;
	struct info_per_cpu *info;

	for (i = 0; i < NR_CPUS; i++) {
		info = &bt_info_per_cpu[i];
		if (info->p_on_off_cnt)
			remove_proc_entry(info->p_on_off_cnt->name,
					  info->p_cpuN);
		if (info->p_cpuN)
			remove_proc_entry(info->p_cpuN->name, proc_btrax);
	}
	if (proc_enable)
		remove_proc_entry(proc_enable->name, proc_btrax);
	if (pid_tbl)
		vfree(pid_tbl);
	if (proc_pid)
		remove_proc_entry(proc_pid->name, proc_btrax);
	if (proc_btsbuf_sz)
		remove_proc_entry(proc_btsbuf_sz->name, proc_btrax);
	if (proc_bts_mrecs)
		remove_proc_entry(proc_bts_mrecs->name, proc_btrax);
	if (proc_btrax)
		remove_proc_entry(proc_btrax->name, &proc_root);
}
