/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_execpath.c - execution path display program                           */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2006                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

//#define USE_OLD_REPEAT_CHK

#include "bfd_if.h"
#include "bt.h"
#include "chk_repeat.h"
#include <libgen.h>

#define BT_EXECPATH_VER	BT_VER
#define	COPYRIGHT	"Copyright (c) Hitachi, Ltd. 2005-2006"

#define	chk_next(argc, i)	if ((argc) <= (i)+1) { usage(); err_exit(); }

/* for summary */
#define MAX_BRI_CACHE	1024
struct br_info {
	unsigned int	type;
	unsigned long	from;
	unsigned long	next;
	unsigned long	branch;
	struct br_info	*link;
};

struct range_to_name {
	unsigned long	begin;
	unsigned long	end;
	unsigned long	offset;
	char		name[MAX_NAME_LEN];
	char		*dirname;
	char		*basename;

	struct bfd_if	bi;
	struct br_info	*bri_cache;
};

int output_summary = 0;
int verbose = 0;

/*----------------------------------------------------------------------------
 *  build range to name structure
 *----------------------------------------------------------------------------
 */
struct range_to_name **all_r2n;
int r2n_max = 128;

void dump_r2n(void)
{
	int i;
	struct range_to_name *p;

	if (!all_r2n)
		return;
	for (i = 0; i < r2n_max; i++) {
		p = all_r2n[i];
		if (p) {
			printf("====== 0x%08lx:0x%08lx\t%s/%s ======\n",
			       p->begin, p->end, p->dirname, p->basename);
			dump_bfd_symbols(&p->bi);
		}
	}
}

void free_r2n(void)
{
	int i;
	struct range_to_name *p;

	if (!all_r2n)
		return;
	for (i = 0; i < r2n_max; i++) {
		p = all_r2n[i];
		if (p) {
			//free(p->func_names);
			free(p);
		}
	}
	free(all_r2n);
}

struct range_to_name* addr_to_r2n(unsigned long addr)
{
	int i;
	struct range_to_name *p;

	for (i = 0; i < r2n_max; i++) {
		p = all_r2n[i];
		if (p && addr >= p->begin && addr <= p->end)
			return p;
	}
	return NULL;
}

int prepare_obj_file(struct range_to_name *r2n, char *kallsyms)
{
	char path[MAX_LINE_LEN];

	snprintf(path, MAX_LINE_LEN, "%s/%s", r2n->dirname, r2n->basename);
	printf("checking %s...\n", path);
	if (init_bfd_if(&r2n->bi, strdup(path), kallsyms) < 0)
		return -1;
	if (!r2n->begin && !r2n->end) {
		struct addr_range *range = get_pid_addr_range(ALL_PID);
		bfd_vma min, max;
		struct bt_record rec;

		get_min_max_addr(&r2n->bi, &min, &max);
		rec.from = min;
		rec.to = max;
		if (is_addr_match(&rec, range)) {
			r2n->begin = min;
			r2n->end = max;
		} else
			return 1;
	} else {
		r2n->offset = get_offset_addr(&r2n->bi, r2n->begin);
		/* for DEBUG
		printf("OFFSET: ");
		printf_bfd_vma(r2n->offset);
		printf("\n");
		*/
	}
	return 0;
}

int is_same_name_registered(char *name)
{
	int i, len;
	struct range_to_name *r2n;

	if (!all_r2n)
		return 0;
	for (i = 0; i < r2n_max; i++) {
		r2n = all_r2n[i];
		if (!r2n)
			continue;
		len = strlen(r2n->dirname);
		if (strstr(name, r2n->dirname) == name && name[len] == '/' &&
		    strstr(name, r2n->basename) == name + len + 1) {
			return 1;
		}
	}
	return 0;
}

int add_range_to_name(unsigned long begin, unsigned long end, char *name,
		      char *kallsyms, int is_module)
{
	int rc, i, step = 128;
	struct range_to_name *r2n;

	if (is_same_name_registered(name))
		return 0;
	r2n = malloc(sizeof(struct range_to_name));
	if (!r2n) {
		fprintf(stderr, "malloc failed.(%s)\n", strerror(errno));
		return -1;
	}
	memset(r2n, 0, sizeof(struct range_to_name));
	r2n->begin = begin;
	r2n->end = end;
	snprintf(r2n->name, MAX_NAME_LEN, "%s", name);
	r2n->basename = basename(r2n->name);
	r2n->dirname = dirname(r2n->name);

	for (i = 0; i < r2n_max && all_r2n[i]; i++);
	if (i >= r2n_max) {
		all_r2n = realloc(all_r2n, (r2n_max + step) * sizeof(r2n));
		if (!all_r2n) {
			free(r2n);
			fprintf(stderr, "realloc failed.(%s)\n",
				strerror(errno));
			return -1;
		}
		memset(&all_r2n[i], 0, step * sizeof(r2n));
		i = r2n_max;
		r2n_max += step;
	}
	all_r2n[i] = r2n;
	rc = prepare_obj_file(r2n, kallsyms);
	if (rc < 0)
		return 0;
		//return -1;
	else if (rc > 0) {
		free(r2n);
		all_r2n[i] = NULL;
	}
	return 0;
}

#ifdef USE_OLD_REPEAT_CHK
/*----------------------------------------------------------------------------
 *  repeat checking
 *----------------------------------------------------------------------------
 */
#define MAX_REPEAT_LEN	2048
static unsigned long addr_buf[MAX_REPEAT_LEN + 1];
static unsigned long *abp_top = addr_buf;
static unsigned long *abp_next = addr_buf;
#ifndef TEST
static unsigned long *abp_match;
static int repeat_cnt;
#endif

unsigned long* find_abuf(unsigned long addr)
{
	unsigned long *p;

	if (abp_top <= abp_next) {
		for (p = abp_top; p < abp_next; p++)
			if (*p == addr)
				return p;
	} else {
		for (p = abp_top;
		     p < addr_buf + MAX_REPEAT_LEN + 1; p++)
			if (*p == addr)
				return p;
		for (p = addr_buf; p < abp_next; p++)
			if (*p == addr)
				return p;
	}
	return NULL;
}

int get_repeat_checked_addrs(unsigned long *p, unsigned long *p_max,
			     unsigned long *p_dest)
{
	unsigned long *p_save = p_dest;

	if (p <= p_max) {
		for (; p < p_max; p++)
			*p_dest++ = *p;
	} else {
		for (; p < addr_buf + MAX_REPEAT_LEN + 1; p++)
			*p_dest++ = *p;
		for (p = addr_buf; p < p_max; p++)
			*p_dest++ = *p;
	}
	return (p_dest - p_save);
}

unsigned long* get_next_abuf_addr(unsigned long *p)
{
	return (p == addr_buf + MAX_REPEAT_LEN ? addr_buf : p + 1);
}

int proc_each_addr(unsigned long, int);

int flush_repeat_chk_buf(void)
{
	int i;

	for (i = 0; i < MAX_REPEAT_LEN; i++)
		if (proc_each_addr(0, 1) < 0)
			return -1;
	return 0;
}

#ifdef TEST
void chk_repeat(unsigned long addr, unsigned long *p, int *len, int *cnt,
		int is_flush)
{
	*p = addr;
	*len = 1;
	*cnt = 0;
}
#else
void chk_repeat(unsigned long addr, unsigned long *p, int *len, int *cnt,
		int is_flush)
{
	unsigned long *p_tmp = NULL;

	*len = 0;
	*cnt = 0;

	if (abp_match) {
		if (!is_flush && addr == *abp_match) {
			abp_match = get_next_abuf_addr(abp_match);
			if (abp_match == abp_next) {
				abp_match = abp_top;
				repeat_cnt++;
			}
		} else {
			if (repeat_cnt)
				*cnt = repeat_cnt + 1;
			repeat_cnt = 0;
			*len = get_repeat_checked_addrs(abp_top, abp_next, p);
			*abp_match = addr;
			abp_next = get_next_abuf_addr(abp_match);
			abp_match = NULL;
		}
	} else {
		if (!is_flush)
			p_tmp = find_abuf(addr);
		//printf("(F) %d\n", (p_tmp != NULL));
		if (p_tmp) {
			*len = get_repeat_checked_addrs(abp_top, p_tmp, p);
			abp_top = p_tmp;
			abp_match = get_next_abuf_addr(abp_top);
		} else {
			*abp_next = addr;
			abp_next = get_next_abuf_addr(abp_next);
			if (abp_top == abp_next) {
				abp_top = get_next_abuf_addr(abp_top);
				*p = *abp_next;
				*len = 1;
			}
		}
	}
	/* for debug */
	/*
	printf("(D)addr:%08lx ->top:%d, next:%d, match:%d, rcnt:%d\n",
	       addr, abp_top - addr_buf, abp_next - addr_buf,
	       abp_match ? abp_match - addr_buf : -1, repeat_cnt);
	       */
	/*
	{
		int i;
		printf("(B)                ");
		p = addr_buf;
		for (i = 0; i < MAX_REPEAT_LEN + 1; i++)
			printf("%08lx,", *p++);
		printf("\n");
	}
	*/
}
#endif
#endif

/*----------------------------------------------------------------------------
 *  get function name and mnemonic
 *----------------------------------------------------------------------------
 */
int print_exec_line(unsigned long addr, struct range_to_name *r2n)
{
	int rc, lno;
	size_t offset;
	const char *src_name, *func_name;
	struct bfd_if *bi = &r2n->bi;
	bfd_vma baddr;

	addr -= r2n->offset;
	printf("%s\t0x%08lx\t", r2n->basename, addr);
	rc = get_source_info(bi, addr, &src_name, &func_name, &lno);
	if (rc >= 0 && src_name) {
		addr_to_func_name_and_offset(bi, addr, &func_name, &offset);
		printf_func_name(func_name, offset);
		if (src_name)
			printf("\t%s:%d", src_name, lno);
		printf("\n");
	} else {
		if (!addr_to_func_name_and_offset(bi, addr, &func_name,
						  &offset)) {
			printf_func_name(func_name, offset);
			printf("\t");
		}
		rc = printf_mnemonic(&r2n->bi, addr, &baddr);
		//printf("\nBADDR: 0x%08lx\n", baddr);
		if (rc > 0 &&
		    !addr_to_func_name_and_offset(bi, baddr, &func_name,
						  &offset)) {
			printf(" ");
			printf_func_name(func_name, offset);
		}
		printf("\n");
	}
	return 0;
}

/*----------------------------------------------------------------------------
 *  normal output support
 *----------------------------------------------------------------------------
 */
#ifdef USE_OLD_REPEAT_CHK
int proc_each_addr(unsigned long addr, int is_flush)
{
	struct range_to_name *r2n;
	static unsigned long last_addr = 0;
	unsigned long abuf[MAX_REPEAT_LEN], *p;
	int i, len, cnt, indent;

	if (!is_flush && addr == last_addr)
		return 0;
	last_addr = addr;

	chk_repeat(addr, abuf, &len, &cnt, is_flush);
	if (cnt) {
		printf("===> repeat %d times\n", cnt);
		indent = 1;
	} else {
		indent = 0;
	}
	for (p = abuf; p < abuf + len; p++) {
		r2n = addr_to_r2n(*p);
		if (indent) {
			for (i = 0; i < indent; i++)
				printf("  ");
		}
		if (!r2n) {
			if (*p != 0)
				printf("--------\t0x%08lx\n", *p);
			continue;
		}
		if (print_exec_line(*p, r2n) < 0)
			return -1;
	}
	return 0;
}

int proc_each_record_for_normal(struct bt_record *p, struct addr_range *range)
{
	if (is_warn_record(p)) {
		struct warn_record *warn = (struct warn_record*)p;
		fprintf(stderr, "WARN: bts left only: %ld\n", warn->left);
		return 0;
	}
	if (is_pid_record(p) || !is_addr_match(p, range))
		return 0;

	if (proc_each_addr(p->from, 0) < 0)
		return -1;
	if (proc_each_addr(p->to, 0) < 0)
		return -1;
	return 0;
}
#else
int __comp_N(const void *p1, const void *p2)
{
	return ((unsigned long)p1 == (unsigned long)p2);
}

void __print_nest_space_N(int nest)
{
	int i;

	for (i = 0; i < nest; i++)
		printf("  ");
}

void __print_data_N(int nest, const void *p)
{
	struct range_to_name *r2n;
	unsigned long addr = (unsigned long)p;

	__print_nest_space_N(nest);
	r2n = addr_to_r2n(addr);
	if (!r2n) {
		if (addr != 0)
			printf("--------\t0x%08lx\n", addr);
	} else
		print_exec_line(addr, r2n);
}

void __print_start_repeat_N(int nest, unsigned long cnt)
{
	__print_nest_space_N(nest);
	printf("===> repeat %ld times\n", cnt);
}

struct chk_repeat_funcs chk_repeat_funcs_N =
{
	__comp_N,
	NULL,
	__print_data_N,
	__print_start_repeat_N,
	NULL,
};

int proc_each_addr(unsigned long addr)
{
	static unsigned long last_addr = 0;

	if (addr == last_addr)
		return 0;
	last_addr = addr;

	return chk_repeat_each((const void*)addr);
}

int proc_each_record_for_normal(struct bt_record *p, struct addr_range *range)
{
	if (is_warn_record(p)) {
		struct warn_record *warn = (struct warn_record*)p;
		fprintf(stderr, "WARN: bts left only: %ld\n", warn->left);
		return 0;
	}
	if (is_pid_record(p) || !is_addr_match(p, range))
		return 0;

	if (proc_each_addr(p->from) < 0)
		return -1;
	if (proc_each_addr(p->to) < 0)
		return -1;
	return 0;
}
#endif

/*----------------------------------------------------------------------------
 *  summary output support
 *----------------------------------------------------------------------------
 */
#define PRINTABLE_NEST_MAX	20
#define PRINTABLE_NEST_STEP	20
#define MAX_NEST	2048

static int min_nest;
static int nest;

#define RET_CHK
#ifdef RET_CHK
struct ret_chk {
	int		type;
	unsigned long	addr;
} ret_chk[MAX_NEST];

static struct ret_chk *p_ret_chk_last = ret_chk;

void dump_ret_chk(int n)
{
	struct ret_chk *p;

	for (p = ret_chk + n; p < p_ret_chk_last; p++)
		printf("%04d:0x%08lx(%d)\n",
		       n, ret_chk[n].addr, ret_chk[n].type);
}

void set_expect_ret_addr(int type, unsigned long addr)
{
	if (p_ret_chk_last - ret_chk >= MAX_NEST) {
		fprintf(stderr, "WARN: MAX_NEST is too small.\n");
		return;
	}
	p_ret_chk_last->type = type;
	p_ret_chk_last->addr = addr;
	p_ret_chk_last++;
	nest++;
}

int get_nest_from_ret_addr(unsigned long addr)
{
	struct ret_chk *p;

	for (p = p_ret_chk_last; p >= ret_chk; p--) {
		if (p->addr == addr) {
			p_ret_chk_last = p;
			return p - ret_chk;
		}
	}
	if (p_ret_chk_last > ret_chk)
		p_ret_chk_last--;
	return p_ret_chk_last - ret_chk;
}
#endif

struct br_info* chk_bri_cache(struct range_to_name *r2n, unsigned long from)
{
	int i;
	struct br_info *p, *p_prev;

	for (i = 0, p_prev = NULL, p = r2n->bri_cache; p;
	     i++, p_prev = p, p = p->link) {
		if (p->from == from) {
			if (p_prev) {
				p_prev->link = p->link;
				p->link = r2n->bri_cache;
				r2n->bri_cache = p;
			}
			/* else case means found element is already cache top.*/
			return p;
		}
	}
	if (i >= MAX_BRI_CACHE) {
		/* free last element */
		for (p = r2n->bri_cache; p && p->link && p->link->link;
		     p = p->link);
		free(p->link);
		p->link = NULL;
	}
	/* add new element */
	p = malloc(sizeof(struct br_info));
	if (!p) {
		fprintf(stderr, "malloc failed(%s).\n", strerror(errno));
		return NULL;
	}
	p->from = from;
	if (get_branch_info(&r2n->bi, p->from, &p->type, &p->next, &p->branch,
			    r2n->offset) < 0) {
		fprintf(stderr, "get_branch_info failed(0x%08lx).\n", p->from);
		return NULL;
	}
	p->link = r2n->bri_cache;
	r2n->bri_cache = p;
	return p;
}

struct summary {
	int type;
	int nest;
	unsigned long from;
	unsigned long to;
};

int __comp_S(const void *p1, const void *p2)
{
	return (memcmp(p1, p2, sizeof(struct summary)) == 0);
}

void __free_S(const void *p)
{
	free((struct summary*)p);
}

void __print_nest_space_S(int nest)
{
	int i;

	for (i = 0; i < nest; i++)
		printf("  ");
}

void __print_each_addr_S(unsigned long addr)
{
	size_t offset;
	const char *func_name;
	struct range_to_name *r2n;

	r2n = addr_to_r2n(addr);
	if (!r2n) {
		//printf("--------:0x%08lx", addr);
		printf("0x%08lx", addr);
		return;
	}
	//printf("%s:", r2n->basename);
	addr -= r2n->offset;
	if (addr_to_func_name_and_offset(&r2n->bi, addr, &func_name,
					 &offset) < 0) {
		printf("0x%08lx", addr);
		return;
	}
	printf_func_name(func_name, offset);
	return;
}

void __print_data_S(int nest, const void *p)
{
	struct summary *s = (struct summary*)p;
	int i;
	char c;

	if (s->nest > PRINTABLE_NEST_MAX) {
		int n;
		n = s->nest % PRINTABLE_NEST_STEP;
		if (n == 0)
			n += PRINTABLE_NEST_STEP;
		printf("(N:%4d)", s->nest - n);
		for (i = 0; i < n; i++)
			printf("+-");
	} else {
		for (i = 0; i < s->nest; i++)
			printf("+-");
	}
	switch (s->type) {
	case BTYPE_JMP:  c = 'J'; break;
	case BTYPE_CALL: c = 'C'; break;
	case BTYPE_INT:  c = 'I'; break;
	default:         c = '?'; break;
	}
	printf("%c ", c);
	__print_each_addr_S(s->to);
	printf(" (");
	//__print_each_addr_S(s->from);
	printf("0x%08lx", s->from);
	printf(")\n");
}

void __print_start_repeat_S(int nest, unsigned long cnt)
{
	__print_nest_space_S(nest);
	printf("===> repeat %ld times\n", cnt);
}

void __print_end_repeat_S(int nest)
{
	__print_nest_space_S(nest);
	printf("<===\n");
}

struct chk_repeat_funcs chk_repeat_funcs_S =
{
	__comp_S,
	__free_S,
	__print_data_S,
	__print_start_repeat_S,
	__print_end_repeat_S,
};

int proc_enter(int type, int nest_chk, struct bt_record *p, unsigned long ret)
{
	struct summary *s;

	switch (type) {
	case BTYPE_JMP:
	case BTYPE_CALL:
	case BTYPE_INT:
		if (!nest_chk) {
			s = calloc(1, sizeof(struct summary));
			if (!s) {
				fprintf(stderr, "calloc failed.(%s)\n",
					strerror(errno));
				return -1;
			}
			s->type = type;
			s->nest = nest;
			s->from = p->from;
			s->to = p->to;
			if (chk_repeat_each((const void*)s) < 0)
				return -1;
		}
		if (type != BTYPE_JMP) {
#ifdef RET_CHK
			/* !!! FIXME: Function call by jmp is not checked */
			set_expect_ret_addr(type, ret);
			/* for DEBUG
			if (!nest_chk)
				printf("enter(%d):0x%08lx(%d)\n",
				       nest, ret, type);
			*/
#else
			nest++;
#endif
		}
		break;
	}
	return 0;
}

void proc_leave(int type, int nest_chk, struct bt_record *p)
{
	if (type != BTYPE_CALL && type != BTYPE_INT)
		return;
#ifdef RET_CHK
	/* for DEBUG
	if (!nest_chk)
		printf("leave(%d):0x%08lx(%d)\n", nest, p->to, type);
	*/
	nest = get_nest_from_ret_addr(p->to);
#else
	nest--;
#endif
	if (nest_chk && nest < min_nest)
		min_nest = nest;
}

int proc_each_record_for_summary(struct bt_record *p, struct addr_range *range,
				 int nest_chk)
{
	struct range_to_name *r2n, *r2n_to;
	struct br_info *bri;

	if (is_warn_record(p)) {
		struct warn_record *warn = (struct warn_record*)p;
		fprintf(stderr, "WARN: bts left only: %ld\n", warn->left);
		return 0;
	}
	if (is_pid_record(p) || !is_addr_match(p, range))
		return 0;

	r2n = addr_to_r2n(p->from);
	r2n_to = addr_to_r2n(p->to);
	if (!r2n) {
		if (!r2n_to)
			return 0;
		/* iret */
		proc_leave(BTYPE_INT, nest_chk, p);
		return 0;
	}
	if (!r2n_to) {
		/* int */
		return proc_enter(BTYPE_INT, nest_chk, p, p->from);
	}
	if (!(bri = chk_bri_cache(r2n, p->from)))
		return -1;
	/* for DEBUG
	if (!nest_chk)
		printf("branch info(%d) from:%08lx branch:%08lx next:%08lx\n",
		       type, p->from, bri->branch, bri->next);
		       */
	switch (bri->type) {
	case BTYPE_BRANCH:
	case BTYPE_JMP:
	case BTYPE_CALL:
		if (bri->branch == UNKNOWN_BADDR || bri->branch == p->to) {
			/* branch, jmp, call */
			return proc_enter(bri->type, nest_chk, p, bri->next);
		} else {
			/* int */
			return proc_enter(BTYPE_INT, nest_chk, p, p->from);
		}
	case BTYPE_INT:
	case BTYPE_OTHER:
		/* int */
		return proc_enter(BTYPE_INT, nest_chk, p, p->from);
	case BTYPE_RET:
		proc_leave(BTYPE_CALL, nest_chk, p);
		break;
	case BTYPE_IRET:
		proc_leave(BTYPE_INT, nest_chk, p);
		break;
	}
	return 0;
}

int chk_nest(struct bt_record *p, size_t size, struct addr_range *range)
{
	struct bt_record *p_max;

	nest = 0;
	for (p_max = (struct bt_record*)((char*)p + size); p < p_max; p++) {
		if (proc_each_record_for_summary(p, range, 1) < 0)
			return -1;
	}
	////printf("checked nest1: %d\n", nest);
	nest = 0 - min_nest;
#ifdef RET_CHK
	p_ret_chk_last = ret_chk;
#endif
	////printf("checked nest2: %d\n", nest);
	return 0;
}

/*----------------------------------------------------------------------------
 *  program core
 *----------------------------------------------------------------------------
 */
int proc_block_records(struct bt_record *p, struct bt_record *p_max) {
	size_t size = (char*)p_max - (char*)p;
	struct addr_range *range;

	if (p_max < p) {
		fprintf(stderr, "from recnum is bigger than to recnum\n");
		return -1;
	}
	range = get_pid_addr_range(ALL_PID);
	if (output_summary) {
		if (chk_nest(p, size, range) < 0)
			return -1;
		chk_repeat_start(&chk_repeat_funcs_S);
		for (; p < p_max; p++) {
			if (proc_each_record_for_summary(p, range, 0) < 0)
				return -1;
		}
		chk_repeat_end();
	} else {
#ifdef USE_OLD_REPEAT_CHK
		for (; p < p_max; p++) {
			if (proc_each_record_for_normal(p, range) < 0)
				return -1;
		}
		if (flush_repeat_chk_buf() < 0)
			return -1;
#else
		chk_repeat_start(&chk_repeat_funcs_N);
		for (; p < p_max; p++) {
			if (proc_each_record_for_normal(p, range) < 0)
				return -1;
		}
		chk_repeat_end();
#endif
	}
	return 0;
}

int proc_logfile(char *logfile)
{
	int fd, rc = -1;
	size_t size;
	struct bt_record *p_top, *p, *p_max;
	char *dir, *func_name;

	if (open_mmapfile(logfile, &fd, (void**)&p_top, &size) < 0)
		goto EXIT;

	p = p_top;
	func_name = basename(logfile);
	dir = dirname(logfile);
	if (is_exists_maps(dir, func_name)) {
		if (parse_maps(add_range_to_name, dir, func_name) < 0)
			goto EXIT;
	}
	if (parse_modules(add_range_to_name, dir) < 0)
		goto EXIT;
	printf("start\n");
	if (verbose)
		dump_r2n();
	p_max = (struct bt_record*)((char*)p + size);
	if (proc_block_records(p, p_max) < 0)
		goto EXIT;
	rc = 0;
EXIT:
	close_mmapfile(fd, p_top, size);
	return rc;
}

void err_exit(void)
{
	free_ranges();
	free_r2n();
	exit(1);
}

void usage(void)
{
	fprintf(stderr, "bt_execpath %s\n", BT_EXECPATH_VER);
	fprintf(stderr, "    %s\n\n", COPYRIGHT);
	fprintf(stderr, "bt_execpath [-s] [-a top:end [...]] [-d top:end [...]] [--usr|--ker|--all]\n");
	fprintf(stderr, "            -f logfile\n");
	fprintf(stderr, "  -s: output execution path summary\n");
	fprintf(stderr, "  -a: add address range\n");
	fprintf(stderr, "  -d: delete address range\n");
	fprintf(stderr, "  --usr: alias of '-a 0:0xbfffffff'\n");
	fprintf(stderr, "  --ker: alias of '-a 0xc0000000:0xffffffff'\n");
	fprintf(stderr, "  --all: alias of '-a 0:0xffffffff'\n");
	fprintf(stderr, "  -f: logfile\n");
}

int main(int argc, char *argv[])
{
	int i;
	char *logfile = NULL;
	unsigned long begin, end;

	if (alloc_pid_range(ALL_PID) < 0)
		err_exit();
	for (i = 1; i < argc;) {
		if (strcmp(argv[i], "-s") == 0) {
			i++;
			output_summary = 1;
		} else if (strcmp(argv[i], "--usr") == 0) {
			add_range(ALL_PID, 0, 0xbfffffff);
			i++;
		} else if (strcmp(argv[i], "--ker") == 0) {
			add_range(ALL_PID, 0xc0000000, 0xffffffff);
			i++;
		} else if (strcmp(argv[i], "--all") == 0) {
			add_range(ALL_PID, 0, 0xffffffff);
			i++;
		} else if (strcmp(argv[i], "-a") == 0) {
			chk_next(argc, i);
			i++;
			while (i < argc && argv[i][0] != '-') {
				if (!range2ulongs(argv[i], &begin, &end))
					err_exit();
				add_range(ALL_PID, begin, end);
				i++;
			}
		} else if (strcmp(argv[i], "-d") == 0) {
			chk_next(argc, i);
			i++;
			while (i < argc && argv[i][0] != '-') {
				if (!range2ulongs(argv[i], &begin, &end))
					err_exit();
				printf("del_range\n");
				del_range(ALL_PID, begin, end);
				i++;
			}
		} else if (strcmp(argv[i], "-f") == 0) {
			chk_next(argc, i);
			i++;
			logfile = argv[i];
			i++;
		} else if (strcmp(argv[i], "-v") == 0) {
			verbose = 1;
			i++;
		} else {
			usage();
			err_exit();
		}
	}
	//dump_ranges();	/* for debug */
	if (!logfile) {
		usage();
		err_exit();
	}
	all_r2n = calloc(r2n_max, sizeof(struct range_to_name*));
	if (!all_r2n) {
		fprintf(stderr, "calloc failed.(%s)\n", strerror(errno));
		err_exit();
	}
	if (proc_logfile(logfile) < 0)
		err_exit();
	free_ranges();
	free_r2n();
	exit(0);
}
