/*
 djprobe -- Direct jmp-insertion probe
 Copyright (c) 2005 Hitachi,Ltd.,
 Created by Masami Hiramatsu<hiramatu@sdl.hitachi.co.jp>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */
#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <asm/system.h>
#include <asm/pgtable.h>
#include <linux/vmalloc.h>
#include <linux/slab.h>
#include <linux/list.h>
#include <linux/rcupdate.h>
#include "djprobe.h"

#define _DEBUG_(a) a

/* On pentium series, Unsynchronized cross-modifying code
 * operations can cause unexpected instruction execution results.
 * So after code modified, we should synchronize it on each processor.
 */
static void __local_serialize_cpu(void)
{
	__asm__ __volatile__ ("cpuid" : : : "ax", "bx", "cx", "dx");
}

/* jmp code manipulators */
struct __arch_jmp_op {
	char op;
	long raddr;
} __attribute__((packed));
/* insert jmp code */
static inline void __set_jmp_op(void *from, void *to)
{
	struct __arch_jmp_op *jop;
	jop = (struct __arch_jmp_op *)from;
	jop->raddr=(long)(to) - ((long)(from) + 5);
	mb();
	jop->op = RELATIVEJUMP_INSTRUCTION;
}
/* switch back to the kprobe */
static inline void __recover_ops(void *dest, void *orig)
{
	struct __arch_jmp_op *jop = (struct __arch_jmp_op *)dest, 
		*jop2 = (struct __arch_jmp_op *)orig;

	*jop = *jop2; /*TODO: use cmpxhg8*/
}

static inline int __check_stack_contents(void *addr)
{
	unsigned long esp = (unsigned long)current_stack_pointer;
	void **s_esp, **e_esp;
	s_esp = (void**)esp;
	e_esp = (void**)((unsigned long)current+8192);
	while (s_esp < e_esp) {
		if (*s_esp == addr) return -1;
		s_esp ++;
	}
	return 0;
}

#include <asm/segment.h>

/* djprobe call back function: called from stub code */
extern void asmlinkage djprobe_callback(struct djprobe_instance * djpi,
					struct pt_regs *regs)
{
	struct djprobe * djp;
	list_for_each_entry_rcu(djp, &djpi->plist, plist) {
		if (djp->handler)
			djp->handler(djp, regs);
	}
}

/*
 * Copy post processing instructions
 * Target instructions MUST be relocatable.
 */
static int  arch_prepare_djprobe_instance(struct djprobe_instance *djpi,
						   unsigned long size)
{
	kprobe_opcode_t *stub;
	stub = djpi->stub.insn;

	/* copy arch-dep-instance from template */
	memcpy((void*)stub, (void*)&arch_tmpl_stub_entry, ARCH_STUB_SIZE);

	/* set probe information */
	*((long*)(stub + ARCH_STUB_VAL_IDX)) = (long)djpi;
	/* set probe function */
	*((long*)(stub + ARCH_STUB_CALL_IDX)) = (long)djprobe_callback;

	/* copy instructions into the middle of axporbe instance */
	memcpy((void*)(stub + ARCH_STUB_INST_IDX),
	       (void*)djpi->addr, size);
	djpi->stub.size = size;

	/* set returning jmp instruction at the tail of axporbe instance*/
	__set_jmp_op(stub + ARCH_STUB_INST_IDX + size, 
		     (void*)((long)djpi->addr + size));

	return 0;
}

/*
 * Insert "jmp" instruction into the probing point.
 */
static int arch_install_djprobe_instance(struct djprobe_instance *djpi)
{
	kprobe_opcode_t *stub;
	stub = djpi->stub.insn;
	__set_jmp_op((void*)djpi->addr, (void*)stub);
	return 0;
}
/* Write back original instructions & kprobe */
void  arch_uninstall_djprobe_instance(struct djprobe_instance *djpi)
{
	kprobe_opcode_t *stub;
	stub = &djpi->stub.insn[ARCH_STUB_INST_IDX];
	__recover_ops((void*)djpi->addr, (void*)stub);
}

// common code

/*
 * The djprobe do not refer instances list when probe function called.
 * This list is operated on registering and unregistering djprobe.
 */

#include <linux/spinlock.h>

#ifndef DEFINE_SPINLOCK
#define DEFINE_SPINLOCK(x) spinlock_t x = SPIN_LOCK_UNLOCKED
#endif
static LIST_HEAD(djprobe_list);
static DEFINE_SPINLOCK(djprobe_lock);

static inline struct djprobe_instance * 
	__get_djprobe_instance(void *addr, int size)
{
	struct djprobe_instance *djpi;
	list_for_each_entry(djpi, &djprobe_list, list) {
		if (((long)addr < (long)djpi->addr + DJPI_ARCH_SIZE(djpi))
		    && ((long)djpi->addr < (long)addr + size)) {
			return djpi;
		}
	}
	return NULL;
}

/* Instruction pages for djprobe's stub code */
static inline void __free_djprobe_instance(struct djprobe_instance *djpi)
{
	list_del(&djpi->list);
	kfree(djpi);
}

#ifdef CONFIG_SMP
static DEFINE_SPINLOCK(check_lock);
/* smp code */
static void __work_check_djprobe_instance(struct djprobe_instance *djpi)
{
	unsigned flags;
	spin_lock_irqsave(&check_lock, flags);
	if (DJPI_READY(djpi)) {
		if (DJPI_EMPTY(djpi)) {
			arch_uninstall_djprobe_instance(djpi);
		} else {
			arch_install_djprobe_instance(djpi);
		}
	}
	atomic_inc(&djpi->checked);
	spin_unlock_irqrestore(&check_lock, flags);
}

static void work_check_djprobe_instance(void *data)
{
	struct djprobe_instance *djpi = (struct djprobe_instance *)data;
	if (DJPI_CHECKED(djpi)) 
		BUG();
	if ( __check_stack_contents(djpi->addr) != 0)
		atomic_inc(&djpi->failed);
	__work_check_djprobe_instance(djpi);
	/*wait loop*/
	while (!DJPI_CHECKED(djpi))
		barrier();
	if (!DJPI_FAILED(djpi)) __local_serialize_cpu();
}

static void 
	schedule_check_djprobe_instance(struct djprobe_instance *djpi) 
{
	do {
		barrier();
		atomic_set(&djpi->failed, 0);
		atomic_set(&djpi->checked, 0);
		__work_check_djprobe_instance(djpi);
		smp_call_function(work_check_djprobe_instance, djpi, 0, 1);
	} while (DJPI_FAILED(djpi)); /*failed to check*/
	__local_serialize_cpu();
}
#else

static void 
	schedule_check_djprobe_instance(struct djprobe_instance *djpi) 
{
	unsigned long flags;
	local_irq_save(flags);
	atomic_set(&djpi->failed, 0);
	atomic_set(&djpi->checked, 1);
	if (DJPI_EMPTY(djpi)) {
		arch_uninstall_djprobe_instance(djpi);
	} else {
		arch_install_djprobe_instance(djpi);
	}
	__local_serialize_cpu();
	local_irq_restore(flags);
}
#endif


int register_djprobe(struct djprobe * djp)
{
	struct djprobe_instance *djpi;
	int ret = 0;
	
	if (djp == NULL || djp->addr == NULL || 
	    djp->size > ARCH_STUB_INSN_MAX || 
	    djp->size < ARCH_STUB_INSN_MIN || 
	    djp->inst != NULL)
		return -EINVAL;

	spin_lock(&djprobe_lock);
	INIT_LIST_HEAD(&djp->plist);
	/* check confliction with other djprobes */
	djpi = __get_djprobe_instance(djp->addr, djp->size);
	if (djpi) {
		if (djpi->addr == djp->addr) {
			djp->inst = djpi;
			list_add_rcu(&djp->plist, &djpi->plist);
			goto out;
		} else {
			ret = -EEXIST; /* a djprobe were inserted */
			goto out;
		}
	}
	/* make a new instance */
	djpi = kmalloc(sizeof(struct djprobe_instance),GFP_KERNEL);
	if (djpi == NULL) {
		ret = -ENOMEM; /* memory allocation error */
		goto out;
	}
	memset(djpi, 0, sizeof(struct djprobe_instance)); /* for kprobe */
	/* attach */
	INIT_LIST_HEAD(&djpi->plist);
	djp->inst = djpi;
	list_add_rcu(&djp->plist, &djpi->plist);
	djpi->addr = djp->addr;
	INIT_LIST_HEAD(&djpi->list);
	list_add(&djpi->list, &djprobe_list);

	arch_prepare_djprobe_instance(djpi, djp->size); /*TODO : remove size*/
	schedule_check_djprobe_instance(djpi); /*check and install*/
out:
	spin_unlock(&djprobe_lock);
	return ret;
}

void unregister_djprobe(struct djprobe * djp)
{
	struct djprobe_instance *djpi;
	if (djp == NULL || djp->inst == NULL) 
		return ;

	djpi = djp->inst;
	spin_lock(&djprobe_lock);
	djp->inst = NULL;
	list_del_rcu(&djp->plist);
	if (DJPI_EMPTY(djpi)) {
		schedule_check_djprobe_instance(djpi);
		__free_djprobe_instance(djpi);
	}
	spin_unlock(&djprobe_lock);
}

EXPORT_SYMBOL_NOVERS(register_djprobe);
EXPORT_SYMBOL_NOVERS(unregister_djprobe);

/*
 * Initiation/Termination Functions.
 */ 

static int mod_init_djprobe(void) 
{
	return 0;
}

static void mod_exit_djprobe(void)
{
	return ;
}

module_init(mod_init_djprobe);
module_exit(mod_exit_djprobe);
MODULE_AUTHOR("M.Hiramatsu <hiramatu@sdl.hitachi.co.jp>");
MODULE_LICENSE("GPL");


