/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_ar_parse.c - /proc/modules and /proc/PID/maps parser                  */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2006                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <sys/utsname.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include "bt_ar_parse.h"

static struct pid_range *ranges;

/*----------------------------------------------------------------------------*/
/*  range_to_name support functions                                           */
/*----------------------------------------------------------------------------*/
void dump_r2n(struct r2n_info *r2i)
{
	int i;
	struct range_to_name *p;

	if (!r2i->all_r2n)
		return;
	for (i = 0; i < r2i->r2n_max; i++) {
		p = r2i->all_r2n[i];
		if (p) {
			printf("====== 0x%08lx:0x%08lx\t%s/%s ======\n",
			       p->begin, p->end, p->dirname, p->basename);
			dump_bfd_symbols(&p->bi, p->begin, p->end, p->offset);
			dump_path_tree(p->pt, p->cnt);
		}
	}
}

void free_r2n(struct r2n_info *r2i)
{
	int i;
	struct range_to_name *p;

	if (!r2i->all_r2n)
		return;
	for (i = 0; i < r2i->r2n_max; i++) {
		p = r2i->all_r2n[i];
		if (p) {
			//free(p->func_names);
			free(p);
		}
	}
	free(r2i->all_r2n);
}

struct range_to_name* addr_to_r2n(struct r2n_info *r2i, unsigned long addr)
{
	int i;
	struct range_to_name *p;

	for (i = 0; i < r2i->r2n_max; i++) {
		p = r2i->all_r2n[i];
		if (p && addr >= p->begin && addr <= p->end)
			return p;
	}
	return NULL;
}

static int is_same_name_registered(struct r2n_info *r2i, char *name)
{
	int i, len;
	struct range_to_name *r2n;

	if (!r2i->all_r2n)
		return 0;
	for (i = 0; i < r2i->r2n_max; i++) {
		r2n = r2i->all_r2n[i];
		if (!r2n)
			continue;
		len = strlen(r2n->dirname);
		if (strstr(name, r2n->dirname) == name && name[len] == '/' &&
		    strstr(name, r2n->basename) == name + len + 1) {
			return 1;
		}
	}
	return 0;
}

static int prepare_obj_file(struct range_to_name *r2n, char *kallsyms,
			    int with_pt, int verbose)
{
	char path[MAX_LINE_LEN];

	snprintf(path, MAX_LINE_LEN, "%s/%s", r2n->dirname, r2n->basename);
	/* In case of without path-tree, do not print 'checking ...' message */
	if (verbose)
		printf("checking %s...\n", path);
	if (init_bfd_if(&r2n->bi, strdup(path), kallsyms) < 0)
		return -1;
	if (!r2n->begin && !r2n->end) {
		struct addr_range *range = get_pid_addr_range(ALL_PID);
		bfd_vma min, max;

		get_kernel_min_max_addr(&r2n->bi, &min, &max);
		if (is_addr_range_match(min, max, range)) {
			r2n->begin = min;
			r2n->end = max;
		} else
			return 1;
	} else {
		r2n->offset = get_offset_addr(&r2n->bi, r2n->begin);
		/* for DEBUG
		printf("OFFSET: ");
		printf_bfd_vma(r2n->offset);
		printf("\n");
		*/
	}
	remove_useless_fsyms(&r2n->bi, r2n->begin, r2n->end, r2n->offset);
	if (with_pt)
		return chk_path_tree(&r2n->bi, &r2n->pt, &r2n->cnt,r2n->offset);
	else
		return 0;
}

static int add_range_to_name(struct r2n_info *r2i,
			     unsigned long begin, unsigned long end,
			     char *name, char *kallsyms, int is_module,
			     int with_pt, int verbose)
{
	int rc, i, step = 128, r2n_max;
	struct range_to_name *r2n, **all_r2n;

	all_r2n = r2i->all_r2n;
	r2n_max = r2i->r2n_max;
	if (is_same_name_registered(r2i, name))
		return 0;
	r2n = xcalloc(1, sizeof(struct range_to_name));
	r2n->begin = begin;
	r2n->end = end;
	snprintf(r2n->name, PATH_MAX, "%s", name);
	r2n->basename = basename(r2n->name);
	r2n->dirname = dirname(r2n->name);

	for (i = 0; i < r2n_max && all_r2n[i]; i++);
	if (i >= r2n_max) {
		all_r2n = xrealloc(all_r2n, (r2n_max + step) * sizeof(r2n));
		memset(&all_r2n[i], 0, step * sizeof(r2n));
		i = r2n_max;
		r2n_max += step;
	}
	all_r2n[i] = r2n;
	rc = prepare_obj_file(r2n, kallsyms, with_pt, verbose);
	if (rc < 0)
		//return 0;
		return -1;
	else if (rc > 0) {
		free(r2n);
		all_r2n[i] = NULL;
	}
	return 0;
}

/* return value =  1 : fixed
 *                 0 : out of range
 *                -1 : error
 */
static int fix_from_addr_and_get_info(struct r2n_info *r2i,
				      unsigned long *from, unsigned long to,
				      struct range_to_name **__r2n,
				      struct path **__p, long *__id)
{
	struct range_to_name *r2n = NULL;
	struct path *p = NULL;
	unsigned long tmp;
	long i;

	/* The from address that logged by the Pentium-M's BTS facility isn't
	 * same as Pentium-4 or Xeon's.
	 * Pentium-M logged next instruction's address as from address.
	 * But note that in interrupt branch record, Pentium-M logged current
	 * instruction's address as from address.
	 */
	if (!r2i->from_is_next)
		goto FIXED;
	tmp = *from;
	if (r2i->from_is_next)
		tmp -= 1;
	r2n = addr_to_r2n(r2i, tmp);
	if (!r2n) {
		/* from-address is already correct (interrupt case) ? */
		r2n = addr_to_r2n(r2i, *from);
		if (!r2n)
			return 0;
		goto FIXED;
	}
	i = r2n->cnt;
	p = find_path_from_addr(r2n->pt, &i, tmp);
	if (!p) {
		/* from-address is already correct (interrupt case) ? */
		if (r2n->end == tmp) {
			r2n = addr_to_r2n(r2i, *from);
			if (!r2n)
				return 0;
			i = r2n->cnt;
		}
		p = find_path_from_addr(r2n->pt, &i, *from);
		if (!p) {
			/*
			fprintf(stderr, "path not found(0x%08lx:%s).\n",
				tmp, r2n->basename);
				*/
			return 0;
		}
		goto FIXED;
	}

	/* check from address is next instruction address or not */
	switch (p->type) {
	case BTYPE_BRANCH:
	case BTYPE_JMP:
	case BTYPE_CALL:
		if (p->next == *from &&
		    (p->jmp == UNKNOWN_BADDR || p->jmp == to)) {
			*from = p->base;
			goto FIXED;
		}
		break;
	case BTYPE_RET:
	case BTYPE_IRET:
		if (p->next == *from) {
			*from = p->base;
			goto FIXED;
		}
		break;
	}
	/* Case of interrupt branch record.
	 * There is no need to change the from address.
	 */
	if (*from < r2n->begin || *from > r2n->end) {
		r2n = NULL;
		p = NULL;
	} else if (*from < p->addr || *from >= p->next) {
		if (i < r2n->cnt - 1) {
			p++;
			if (*from < p->addr || *from >= p->next)
				p = NULL;
		} else
			p = NULL;
	}
FIXED:
	if ((__r2n || __p || __id) && !r2n) {
		r2n = addr_to_r2n(r2i, *from);
		if (!r2n)
			return 0;
	}
	if (__r2n)
		*__r2n = r2n;
	if ((__p || __id) && !p) {
		i = r2n->cnt;
		p = find_path_from_addr(r2n->pt, &i, *from);
		if (!p) {
			/*
			fprintf(stderr, "path not found(0x%08lx:%s).\n",
				*from, r2n->basename);
				*/
			return 0;
		}
	}
	if (__p)
		*__p = p;
	if (__id)
		*__id = i;
	return 1;
}

int chk_fix_from_cache(struct r2n_info *r2i,
		       unsigned long *from, unsigned long to,
		       struct range_to_name **__r2n, struct path **__path,
		       long *__id)
{
	int i, rc;
	struct fix_from_cache *p, *p_prev;
	struct range_to_name *r2n;

	r2n = addr_to_r2n(r2i, *from);
	if (!r2n) {
		if (!r2i->from_is_next)
			return 0;
		r2n = addr_to_r2n(r2i, *from - 1);
		if (!r2n)
			return 0;
	}
	for (i = 0, p_prev = NULL, p = r2n->fix_from_cache; p;
	     i++, p_prev = p, p = p->next) {
		if (p->from == *from && p->to == to) {
			if (p_prev) {
				p_prev->next = p->next;
				p->next = r2n->fix_from_cache;
				r2n->fix_from_cache = p;
			}
			/* else case means found element is already cache top.*/
			*from = p->fixed_from;
			if (__r2n)
				*__r2n = p->r2n;
			if (__path)
				*__path = p->path;
			if (__id)
				*__id = p->id;
			//printf("CACHE HIT\n");
			return 1;
		}
	}
	//printf("CACHE MISS\n");
	if (i >= MAX_FIX_FROM_CACHE) {
		/* free last element */
		for (p = r2n->fix_from_cache; p && p->next && p->next->next;
		     p = p->next);
		free(p->next);
		p->next = NULL;
	}
	/* add new element */
	p = xmalloc(sizeof(struct fix_from_cache));
	p->from = *from;
	p->to = to;
	rc = fix_from_addr_and_get_info(r2i, from, to, __r2n, __path, __id);
	if (rc <= 0) {
		free(p);
		return rc;
	}
	p->fixed_from = *from;
	p->r2n = __r2n ? *__r2n : NULL;
	p->path = __path ? *__path : NULL;
	p->id = __id ? *__id : -1;
	p->next = r2n->fix_from_cache;
	r2n->fix_from_cache = p;
	return 1;
}

/*----------------------------------------------------------------------------*/
/*  parse address range support functions                                     */
/*----------------------------------------------------------------------------*/
int alloc_pid_range(pid_t pid)
{
	struct pid_range **p, *p_prev;

	p = &ranges;
	p_prev = *p;
	while (*p) {
		if ((*p)->pid == pid)
			return 0;
		p_prev = *p;
		p = &(*p)->next;
	}
	*p = xmalloc(sizeof(struct pid_range));
	(*p)->pid = pid;
	(*p)->range = NULL;
	(*p)->next = NULL;
	if (p_prev)
		p_prev->next = *p;
	return 0;
}

static struct pid_range* get_pid_range(pid_t pid)
{
	struct pid_range *p;

	p = ranges;
	while (p) {
		if (p->pid == pid)
			return p;
		p = p->next;
	}
	return p;
}

static void free_one_range(struct addr_range **p_ar)
{
	struct addr_range *p_tmp;

	if (!*p_ar)
		return;
	p_tmp = (*p_ar)->next;
	free(*p_ar);
	*p_ar = p_tmp;
}

void free_ranges(void)
{
	struct pid_range *p, *p_next;

	p = ranges;
	while (p) {
		while (p->range)
			free_one_range(&p->range);
		p_next = p->next;
		free(p);
		p = p_next;
	}
	ranges = NULL;
}

char* range2ulongs(char *p, unsigned long *begin, unsigned long *end)
{
	char *p_end;

	*begin = strtoul(p, &p_end, 0);
	if (*p_end != ':') {
		eprintf("begin address invalid(%s)\n", p);
		return NULL;
	}
	*end = strtoul(p_end + 1, &p_end, 0);
	if (*p_end != '\0' && *p_end != '\n' && *p_end != ' ') {
		eprintf("end address invalid(%s)\n", p);
		return NULL;
	}
	if (*begin >= *end) {
		eprintf("begin address greater or equal end address(%s)\n", p);
		return NULL;
	}
	return p_end;
}

static int is_overlap(unsigned long b1, unsigned long e1,
		      unsigned long b2, unsigned long e2, int continue_ok)
{
	unsigned long long len1, len2;

	len1 = e1 - b1;
	len2 = e2 - b2;

	if (continue_ok) {
		len1++;
		len2++;
	}
	if (b2 == b1 ||
	    (b2 > b1 && b2 - b1 <= len1) ||
	    (b2 < b1 && b1 - b2 <= len2))
		return 1;
	return 0;
}

int add_range(pid_t pid, unsigned long begin, unsigned long end)
{
	struct pid_range *p;
	struct addr_range **pp_ar, *p_ar, *p_tmp;

	ddprintf("------ ADD RANGE (0x%08lx, 0x%08lx) ------\n", begin, end);
	if (!(p = get_pid_range(pid))) {
		// error print
		return -1;
	}
	/* detect overlapped range */
	pp_ar = &p->range;
	while ((p_ar = *pp_ar)) {
		ddprintf("CHECK\n");
		/* overlapped ? */
		if (is_overlap(p_ar->begin, p_ar->end, begin, end, 1)) {
			ddprintf("OVERLAP DETECT\n");
			/* expand the range */
			if (begin < p_ar->begin)
				p_ar->begin = begin;
			if (end > p_ar->end) {
				p_ar->end = end;
				/* check upper ranges */
				p_tmp = p_ar->next;
				while (p_tmp) {
					if (p_tmp->begin <= p_ar->end ||
					    (p_tmp->begin > p_ar->end &&
					     p_tmp->begin - 1 == p_ar->end)){
						if (p_tmp->end > p_ar->end)
							p_ar->end = p_tmp->end;
						free_one_range(&p_ar->next);
						p_tmp = p_ar->next;
					} else {
						break;
					}
				}
			}
			return 0;
		}
		if (p_ar->begin > end) {
			ddprintf("BREAK\n");
			break;
		}
		pp_ar = &p_ar->next;
	}
	/* not overlapped */
	p_tmp = xmalloc(sizeof(struct addr_range));
	p_tmp->begin = begin;
	p_tmp->end = end;
	if (p->range) {
		p_tmp->next = *pp_ar;
		*pp_ar = p_tmp;
	} else {
		p_tmp->next = NULL;
		p->range = p_tmp;
	}
	return 0;
}

int del_range(pid_t pid, unsigned long begin, unsigned long end)
{
	struct pid_range *p;
	struct addr_range **pp_ar, *p_ar, *p_tmp;

	ddprintf("------ DEL RANGE (0x%08lx, 0x%08lx) ------\n", begin, end);
	if (!(p = get_pid_range(pid))) {
		// error print
		return -1;
	}
	/* detect overlapped range */
	pp_ar = &p->range;
	while ((p_ar = *pp_ar)) {
		ddprintf("CHECK\n");
		/* overlapped ? */
		if (is_overlap(p_ar->begin, p_ar->end, begin, end, 0)) {
			ddprintf("OVERLAP DETECT\n");
			if (p_ar->begin >= begin && p_ar->end <= end) {
				/* delete range include existance range */
				free_one_range(pp_ar);
			} else if (p_ar->begin < begin && p_ar->end > end){
				/* existence range include delete range */
				p_tmp = xmalloc(sizeof(struct addr_range));
				p_tmp->begin = end + 1;
				p_tmp->end = p_ar->end;
				p_ar->end = begin - 1;
				p_tmp->next = p_ar->next;
				p_ar->next = p_tmp;
				pp_ar = &p_ar->next;
			} else if (p_ar->begin < begin) {
				/* existence range overlapped (upper part) */
				p_ar->end = begin - 1;
				pp_ar = &p_ar->next;
			} else {
				/* existence range overlapped (lower part) */
				p_ar->begin = end + 1;
				pp_ar = &p_ar->next;
			}
		}
		if (p_ar->begin > end) {
			ddprintf("BREAK\n");
			break;
		}
		pp_ar = &p_ar->next;
	}
	return 0;
}

void dump_ranges(void)
{
	struct pid_range *p;
	struct addr_range *p_ar;

	p = ranges;
	while (p) {
		printf("------ pid: %d ------\n", p->pid);
		p_ar = p->range;
		while (p_ar) {
			printf("0x%08lx:0x%08lx\n", p_ar->begin, p_ar->end);
			p_ar = p_ar->next;
		}
		p = p->next;
	}
}

struct pid_range *get_all_ranges(void)
{
	return ranges;
}

struct addr_range *get_pid_addr_range(pid_t pid)
{
	struct pid_range *p;

	p = ranges;
	while (p) {
		if (p->pid == pid)
			return p->range;
		p = p->next;
	}
	return NULL;
}

/* for speed up, this routine should be convert to the macro... */
int is_addr_range_match(unsigned long from, unsigned long to,
			struct addr_range *r)
{
	while (r) {
		if ((from >= r->begin && from <= r->end) ||
		    (to >= r->begin && to <= r->end))
			return 1;
		r = r->next;
	}
	return 0;
}

static int find_under_path(char *dirname, int max_len, const char *name,
			   int down)
{
	DIR *dir;
	struct dirent *d;
	int dlen, rc;
	struct stat stat;

	//printf("chk dir =>%s< (for >%s<)\n", dirname, name);
	rc = -1;
	if (!(dir = opendir(dirname)))
		return -1;
	while ((d = readdir(dir)) != NULL) {
		if (strcmp(".", d->d_name) == 0 || strcmp("..", d->d_name) == 0)
			continue;
		//printf("chk child =>%s<\n", d->d_name);
		dlen = strlen(dirname);
		if (strcmp(name, d->d_name) == 0) {
			snprintf(dirname + dlen, max_len - dlen,
				 "/%s", d->d_name);
			rc = 0;
			goto EXIT;
		}
		if (!down)
			continue;
		snprintf(dirname + dlen, max_len - dlen, "/%s", d->d_name);
		if (lstat(dirname, &stat) < 0)
			goto EXIT;
		if (S_ISDIR(stat.st_mode)) {
			if (find_under_path(dirname, max_len, name, down) == 0){
				rc = 0;
				goto EXIT;
			}
		}
		dirname[dlen] = '\0';
	}
EXIT:
	closedir(dir);
	return rc;
}

static int get_module_full_path(char *buf, int buf_len, const char *name)
{
	struct utsname utsn;

	if (uname(&utsn) < 0)
		return -1;
	if (buf[0] == '\0')
		snprintf(buf, buf_len, "/lib/modules/%s", utsn.release);
	return find_under_path(buf, buf_len, name, 1);
}

static int __get_vmlinux_full_path(char *buf, int buf_len)
{
	int rc;

	rc = find_under_path(buf, buf_len, "vmlinux", 0);
	if (rc >= 0)
		return rc;
	return find_under_path(buf, buf_len, "vmlinux", 1);
}

static int get_vmlinux_full_path(char *buf, int buf_len)
{
	struct utsname utsn;

	if (buf[0] != '\0') {	/* search start from "buf" path */
		if (__get_vmlinux_full_path(buf, buf_len) >= 0)
			return 0;
	}
	if (uname(&utsn) < 0)
		return -1;
	snprintf(buf, buf_len, "/usr/lib/debug/lib/modules/%s", utsn.release);
	if (__get_vmlinux_full_path(buf, buf_len) >= 0)
		return 0;
	snprintf(buf, buf_len, "/lib/modules/%s/build", utsn.release);
	if (__get_vmlinux_full_path(buf, buf_len) >= 0)
		return 0;
	return -1;
}

//#define HOME_DEBUG
#ifdef HOME_DEBUG
#define DBG_MOD_PATH	"/home/tetsu/siken/to/copied_objcts/modules"
#endif

int parse_modules(struct r2n_info *r2i, char *dir, int with_pt, int verbose)
{
	FILE *fd;
	int rc, num, len;
	char path[128], kallsyms[128];
	char buf[MAX_LINE_LEN + 1], *p, fullname[MAX_LINE_LEN + 1];
	struct addr_range *range;
	unsigned long from, to;

	rc = -1;
	if (!dir)
		goto CHK_KERNEL;
	snprintf(kallsyms, sizeof(kallsyms), "%s/kallsyms", dir);
	snprintf(path, sizeof(path), "%s/modules", dir);
	if ((fd = fopen(path, "r")) == NULL) {
		fprintf(stderr, "can't open %s.(%s)\n", path, strerror(errno));
		goto EXIT;
	}
	buf[MAX_LINE_LEN] = fullname[MAX_LINE_LEN] = '\0';
	range = get_pid_addr_range(ALL_PID);
	while ((p = fgets(buf, MAX_LINE_LEN, fd)) != NULL) {
		len = strlen(p) - 1;
		buf[len] = '\0';
		num = sscanf(buf, "%s %ld %*d %*s %*s %lx",
			     fullname, &to, &from);
		if (num != 3) {
			fprintf(stderr, "modules format error(%d)\n", num);
			goto EXIT;
		}
		to += from;
		if (!is_addr_range_match(from, to, range))
			continue;
		snprintf(buf, MAX_LINE_LEN, "%s.ko", fullname);
#ifdef HOME_DEBUG
		sprintf(fullname, "%s", DBG_MOD_PATH);
#else
		fullname[0] = '\0';
#endif
		if (get_module_full_path(fullname, MAX_LINE_LEN, buf) < 0) {
			if (verbose)
				printf("WARN: %s not found.\n", buf);
			continue;
		}
		if (add_range_to_name(r2i, from, to, fullname, kallsyms, 1,
				      with_pt, verbose) < 0)
			goto EXIT;
	}
CHK_KERNEL:
#ifdef HOME_DEBUG
	sprintf(fullname, "%s", DBG_MOD_PATH);
#else
	fullname[0] = '\0';
#endif
	if (get_vmlinux_full_path(fullname, MAX_LINE_LEN) < 0) {
		if (verbose)
			printf("WARN: vmlinux not found.\n");
	} else {
		if (add_range_to_name(r2i, 0, 0, fullname, NULL, 0,
				      with_pt, verbose) < 0)
			goto EXIT;
	}
	rc = 0;
EXIT:
	if (fd)
		fclose(fd);
	return rc;
}

/* This function returns the number that a function name was found in.
 * In the 'addr' variable, an address of a function found first is set.
 */
int get_symbol_addr(struct r2n_info *r2i,
		    const char *funcsym, unsigned long *addr)
{
	int i, found, tmp;
	struct range_to_name *r2n;
	bfd_vma __addr;

	if (!r2i->all_r2n)
		return 0;
	for (found = 0, i = 0; i < r2i->r2n_max; i++) {
		r2n = r2i->all_r2n[i];
		if (r2n) {
			tmp = get_addr_of_symbol(&r2n->bi, r2n->begin, r2n->end,
						 r2n->offset, funcsym, &__addr);
			if (!found && tmp)
				*addr = __addr;
			found += tmp;
		}
	}
	return found;
}

int get_symbol_addr_all(struct r2n_info *r2i, const char *funcsym,
			unsigned long **__addrs)
{
	int i, j, rc, cnt;
	struct range_to_name *r2n;
	unsigned long *addrs;
	bfd_vma *c_addrs;

	if (!r2i->all_r2n)
		return 0;
	cnt = 0;
	for (i = 0; i < r2i->r2n_max; i++) {
		r2n = r2i->all_r2n[i];
		if (r2n) {
			rc = get_addr_of_symbol_all(&r2n->bi, r2n->begin,
						    r2n->end, r2n->offset,
						    funcsym, &c_addrs);
			if (rc < 0)
				return -1;
			if (rc == 0)
				continue;
			if (cnt)
				addrs = xrealloc(addrs, (cnt + rc) *
						 	 sizeof(unsigned long));
			else
				addrs = xmalloc(rc * sizeof(unsigned long));
			for (j = 0; j < rc; j++)
				addrs[cnt + j] = c_addrs[j];
			cnt += rc;
			free(c_addrs);
		}
	}
	if (cnt)
		*__addrs = addrs;
	return cnt;
}

int get_func_info(struct r2n_info *r2i, unsigned long begin,
		  unsigned long *p_end, struct range_to_name **p_r2n)
{
	int i;
	struct range_to_name *r2n;
	bfd_vma end;

	if (!r2i->all_r2n)
		return 0;
	for (i = 0; i < r2i->r2n_max; i++) {
		r2n = r2i->all_r2n[i];
		if (r2n) {
			end = r2n->end;
			if (get_end_of_func(&r2n->bi, begin, &end)) {
				*p_end = end;
				*p_r2n = r2n;
				return 1;
			}
		}
	}
	return 0;
}

int is_exists_maps(char *dir, char *fname)
{
	char path[128];
	struct stat st;

	snprintf(path, sizeof(path), "%s/%s.maps", dir, fname);
	return stat(path, &st) == 0;
}

int parse_maps(struct r2n_info *r2i, char *dir, char *fname,
	       int with_pt, int verbose)
{
	FILE *fd;
	int rc, num, len;
	char path[128];
	char mode[6], buf[MAX_LINE_LEN + 1], *p;
	struct addr_range *range;
	unsigned long from, to;
	long size;

	rc = -1;
	snprintf(path, sizeof(path), "%s/%s.maps", dir, fname);
	if ((fd = fopen(path, "r")) == NULL) {
		fprintf(stderr, "can't open %s.(%s)\n", path, strerror(errno));
		goto EXIT;
	}
	buf[MAX_LINE_LEN] = '\0';
	range = get_pid_addr_range(ALL_PID);
	while ((p = fgets(buf, MAX_LINE_LEN, fd)) != NULL) {
		len = strlen(p) - 1;
		buf[len] = '\0';
		num = sscanf(buf, "%lx-%lx %s %*s %*x %*s %ld",
			     &from, &to, mode, &size);
		if (num != 4) {
			fprintf(stderr, "map format error(%d)\n", num);
			goto EXIT;
		}
		if (size == 0 || mode[2] != 'x')
			continue;
		for (; *p != '/' && *p != '\0'; p++);
		if (*p != '/' || buf[len-1] == ')')
			continue;	// e.g. /SYSV00000000 (deleted)
		to -= 1;
		if (!is_addr_range_match(from, to, range))
			continue;
		if (add_range_to_name(r2i, from, to, p, NULL, 0,
				      with_pt, verbose) < 0)
			goto EXIT;
	}
	rc = 0;
EXIT:
	if (fd)
		fclose(fd);
	return rc;
}

int chk_from_is_next(char *dir)
{
	char path[PATH_MAX];
	struct stat st;

	snprintf(path, PATH_MAX, "%s/%s", dir, FROM_IS_NEXT_FNAME);
	return (stat(path, &st) == 0);
}

