/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_hook.c - branch trace module (hook interface)                         */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2007                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

//#include <linux/interrupt.h> // for DEBUG (preemptible)
#include <linux/module.h>
#ifdef USE_SYS_KPROBES
#  include <linux/kprobes.h>
#else
#  include "djprobe.h"
#endif
#include <asm/unistd.h>
#include "ctr.h"
#include "bt.h"	// for DEBUG

extern unsigned long switch_to_addr;
extern int switch_to_size;
extern int chk_sctime;

extern cpumask_t enable;
extern int mode;

extern DEFINE_PER_CPU(struct info_per_cpu, bt_info_per_cpu);
extern unsigned long *pid_tbl;
extern unsigned long *syscall_pid_tbl;
extern unsigned long syscall_exe_tbl[];
extern unsigned long syscall_filter_tbl[];

pid_t bt_collect_pid;

void write_pid_record(pid_t);
void bts_log_write(void);
int is_trace_pid(pid_t, unsigned long*);
void add_pid_tbl(pid_t, unsigned long*);
void remove_pid_tbl(pid_t, unsigned long*);
int is_trace_syscall(long);
void add_syscall_tbl(long, unsigned long*);
void bts_facility_on(void);
void bts_facility_off(void);
void bts_facility_save_and_off(int*);
void bts_facility_restore(int);
void bts_on_and_set_pid_info(int, struct task_struct*);
void bt_start(void);
void bt_stop(void);

void check_and_wait_relfs_write(void);

#define is_kernel_thread(p) ((p)->pid < 2 || !(p)->mm)

/* If there is a time delay for pid_tbl bit setting by 'clone' call,
 * child process's pid_tbl check could be faster than parent process.
 * For this case, we also check the parent pid.
 */
#define is_target_user_process(p, pid_tbl)	\
	(!is_kernel_thread(p) \
	 && ((is_syscall(mode) \
	      && (is_trace_pid((p)->real_parent->pid, pid_tbl) \
		  || is_trace_pid((p)->group_leader->pid, pid_tbl))) \
	     || is_trace_pid((p)->pid, pid_tbl)))

static void context_switch_probe(struct pt_regs *regs)
{
	/* !!! do not printk in this routine. */
	struct task_struct *prev, *next;
	int trace_prev, trace_next;
	int cpu, en, save;
	unsigned long *ptbl;

	cpu = get_cpu();
	if (!(en = cpu_isset(cpu, enable)))
		goto EXIT;
	prev = (struct task_struct*)regs->eax;
	next = (struct task_struct*)regs->edx;
	if (prev->pid == next->pid)
		goto EXIT;

	/*
	serial_prints("cs(%d) %d -> %d\n",
		      smp_processor_id(), prev->pid, next->pid);
		      */
	ptbl = is_syscall(mode) ? syscall_pid_tbl : pid_tbl;
	bts_facility_save_and_off(&save);
	if (is_kern_all_by_hook(mode)) {
		trace_prev = 1;
		trace_next = en;
	} else if (is_upid(mode)) {
		trace_prev = is_target_user_process(prev, ptbl);
		trace_next = en && is_target_user_process(next, ptbl);
	} else {
		trace_prev = is_trace_pid(prev->pid, ptbl);
		trace_next = en && is_trace_pid(next->pid, ptbl);
	}
	bts_facility_restore(save);

	if (trace_prev) {
		bts_facility_off();
		bts_log_write();
	}
	if (trace_next)
		bts_on_and_set_pid_info(cpu, next);
EXIT:
	put_cpu();
}

static void sysenter_probe(const ctr_handler_t h)
{
	int cpu, save;
	long n_syscall;
	struct info_per_cpu *info;
	struct pid_manage *pid_mng;

	cpu = get_cpu();
	if (!cpu_isset(cpu, enable))
		goto EXIT;
	n_syscall = h.regs.orig_eax;
	/*
	serial_prints("ENT:%d:%d:%d\n",
		      smp_processor_id(), current->pid, n_syscall);
		      */
	bts_facility_save_and_off(&save);
	if (!is_target_user_process(current, pid_tbl) ||
	    (is_syscall(mode) && !is_trace_syscall(n_syscall))) {
		bts_facility_restore(save);
		goto EXIT;
	}
	put_cpu();

	check_and_wait_relfs_write();

	/* At this point, there is posibilities of cpu changing by scheduler. */
	cpu = get_cpu();
	if (!cpu_isset(cpu, enable))
		goto EXIT;

	if (is_syscall(mode)) {
		/* check system call number */
		add_syscall_tbl(n_syscall, syscall_exe_tbl);

		add_pid_tbl(current->pid, syscall_pid_tbl);
		if (chk_sctime != CHK_SCTIME_OFF) {
			info = &per_cpu(bt_info_per_cpu, cpu);
			pid_mng = &info->pid_manage;
			pid_mng->n_syscall = n_syscall;
			rdtscll(pid_mng->syscall_start);
		}
		bts_on_and_set_pid_info(cpu, current);
	} else {
		bts_log_write();
	}
EXIT:
	put_cpu();
}

static void sysexit_probe(const ctr_handler_t h)
{
	int cpu, save;
	long n_syscall;
	struct info_per_cpu *info;
	struct pid_manage *pid_mng;
	
	cpu = get_cpu();
	if (!cpu_isset(cpu, enable))
		goto EXIT;
	n_syscall = h.regs.orig_eax;
	/*
	serial_prints("EXT:%d:%d:%d\n",
		      smp_processor_id(), current->pid, n_syscall);
		      */
	bts_facility_save_and_off(&save);
	if (!is_target_user_process(current, pid_tbl)) {
		bts_facility_restore(save);
		goto EXIT;
	}
	if (is_syscall(mode)) {
		if (n_syscall == 120) { /* sys_clone? */
			/*
			serial_prints("ADD:%d->%d\n",
				      current->pid, h.regs.eax);
				      */
			add_pid_tbl(h.regs.eax, pid_tbl);
		}
		if (!is_trace_syscall(n_syscall)) {
			bts_facility_restore(save);
			goto EXIT;
		}
	}
	info = &per_cpu(bt_info_per_cpu, cpu);
	pid_mng = &info->pid_manage;
	if (is_syscall(mode)) {
		if (chk_sctime != CHK_SCTIME_OFF)
			rdtscll(pid_mng->syscall_end);
		remove_pid_tbl(current->pid, syscall_pid_tbl);
		bts_log_write();
	} else
		bts_on_and_set_pid_info(cpu, current);
EXIT:
	put_cpu();
}

#ifdef USE_SYS_KPROBES
static struct kprobe kp;
static int context_switch_kp_wrapper(struct kprobe *kp, struct pt_regs *regs)
{
	context_switch_probe(regs);
	return 0;
}
#else
static struct djprobe djp;
static void context_switch_djp_wrapper(struct djprobe *djp,
				       struct pt_regs *regs)
{
	context_switch_probe(regs);
}
#endif

int reg_probe(void)
{
	int rc;

	if (!switch_to_addr || !switch_to_size)
		return -EINVAL;
#ifdef USE_SYS_KPROBES
	kp.addr = (kprobe_opcode_t*)switch_to_addr;
	kp.pre_handler = context_switch_kp_wrapper;
	if ((rc = register_kprobe(&kp)) < 0)
		return rc;
#else
	djp.addr = (void*)switch_to_addr;
	djp.handler = context_switch_djp_wrapper;
	djp.size = switch_to_size;
	if ((rc = register_djprobe(&djp)) < 0)
		return rc;
#endif
	if (is_enable_by_proc(mode)) {
		if ((rc = register_ctr(sysenter_probe, sysexit_probe)) < 0)
			return rc;
	}
	return 0;
}

void unreg_probe(void)
{
#ifdef USE_SYS_KPROBES
	unregister_kprobe(&kp);
#else
	unregister_djprobe(&djp);
#endif
	if (is_enable_by_proc(mode))
		unregister_ctr();
	return;
}
