/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_relfs.c - branch trace module (relayfs interface)                     */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2007                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#ifdef USE_SYS_RELAYFS
#  ifdef IS_6UP_RELFS
#    include <linux/relay.h>
#    include <linux/debugfs.h>
#  else
#    include <linux/relayfs_fs.h>
#  endif
#else
#  include "relayfs_fs.h"
#endif
#include "bt.h"

static char *relfs_dirname = "btrax";
static char *relfs_basename = "cpu";
static struct dentry *dir = NULL;

struct rchan *bt_channel = NULL;
size_t bt_dropped;

static DECLARE_WAIT_QUEUE_HEAD(relfs_write_wait);
static int is_cleanup;

/* produced/consumed control files */
extern struct proc_dir_entry *proc_btrax;
static struct proc_dir_entry *proc_subbuf_sz;
static struct proc_dir_entry *proc_n_subbufs;
static struct proc_dir_entry *proc_subbuf_threshold;
static struct proc_dir_entry *proc_dropped;

extern DEFINE_PER_CPU(struct info_per_cpu, bt_info_per_cpu);

extern size_t subbuf_size;
extern size_t subbuf_num;
extern size_t subbuf_sleep_threshold;
extern int mode;

#ifdef DEBUG
unsigned long wcnt[NR_CPUS];
static void relfs_relay_write(struct rchan *channel, void *p, size_t size)
{
	int cpu = smp_processor_id();
	struct bt_record rec = {0,0,BT_FLAG_DEBUG};
	size_t i;

	for (i = 0; i < size / sizeof(rec); i++) {
		rec.from = wcnt[cpu]++;
		relay_write(channel, &rec, sizeof(rec));
	}
}
#else
static void relfs_relay_write(struct rchan *channel, void *p, size_t size)
{
	struct bt_record *p_rec, *p_max;

	p_rec = (struct bt_record*)p;
	p_max = (struct bt_record*)((char*)p + size);
	for (; p_rec < p_max; p_rec++)
		relay_write(channel, p_rec, sizeof(struct bt_record));
}
#endif

void relfs_flush(void)
{
	serial_prints("relay_flush called\n");
	relay_flush(bt_channel);
}

void write_bt_records(void *p, size_t size)
{
	relfs_relay_write(bt_channel, p, size);
}

void write_pid_record(pid_t pid, char *comm)
{
	struct pid_record rec;
	struct comm_record comm_rec;
	struct timeval tv;

	do_gettimeofday(&tv);
	rec.pid = pid;
	rec.tv_sec = tv.tv_sec;
	rec.tv_usec = tv.tv_usec | BT_FLAG_PID;
	relfs_relay_write(bt_channel, &rec, sizeof(rec));

	memcpy(comm_rec.comm, comm, BT_COMM_LEN);
	comm_rec.flags = BT_FLAG_COMM;
	relfs_relay_write(bt_channel, &comm_rec, sizeof(comm_rec));
}

void write_warn_record(unsigned long left)
{
	struct warn_record rec;

	rec.left = left;
	rec.reserve = 0;
	rec.flags = BT_FLAG_WARN;

	relfs_relay_write(bt_channel, &rec, sizeof(rec));
}

void write_tmr_record(unsigned long long timestamp, long n_syscall)
{
	struct tmr_record rec;

	rec.timestamp = timestamp;
	rec.flags =
		n_syscall >= 0 ? BT_FLAG_T_START | n_syscall : BT_FLAG_T_STOP;
	relfs_relay_write(bt_channel, &rec, sizeof(rec));
}

size_t get_ready_subbuf_num(struct rchan_buf *buf)
{
#if RELAYFS_CHANNEL_VERSION >= 5	// btrax relayfs
	return buf->subbufs_produced - buf->subbufs_consumed;
#else
	return atomic_read(&buf->subbufs_produced)
		- atomic_read(&buf->subbufs_consumed);
#endif
}

static int is_relfs_writeable(void)
{
	unsigned long flags;
	struct rchan_buf *buf;
	size_t ready;
	int rc, cpu;

	if (is_cleanup)
		return 1;
	local_irq_save(flags);
	cpu = smp_processor_id();
	buf = bt_channel->buf[cpu];
	ready = get_ready_subbuf_num(buf);
	rc = (ready < subbuf_sleep_threshold);
	/* for DEBUG*/
	if (!rc)
		serial_prints("(%d)unwritable (p:%d,c:%d)\n", cpu,
			      buf->subbufs_produced, buf->subbufs_consumed);
	/**/
	local_irq_restore(flags);
	return rc;
}

void check_and_wait_relfs_write(void)
{
#if 0
	int sleep = 0;
	int len = sizeof(current->comm);
	char buf[len + 1];

	if (!is_relfs_writeable()) {
		sleep = 1;
		buf[len] = '\0';
		memcpy(buf, current->comm, len);
		serial_prints("Zzz (%s)\n", buf);
	}
#endif
	wait_event_interruptible(relfs_write_wait, is_relfs_writeable());
#if 0
	if (sleep)
		serial_prints("!!! (%s)\n", buf);
#endif
}

#if RELAYFS_CHANNEL_VERSION < 5
static inline int relay_buf_full(struct rchan_buf *buf)
{
	int produced, consumed;

	/* Check buffer full even though mode is overwrite */
	/*
	if (buf->chan->overwrite)
		return 0;
		*/
	produced = atomic_read(&buf->subbufs_produced);
	consumed = atomic_read(&buf->subbufs_consumed);

	return (produced - consumed > buf->chan->n_subbufs - 1) ? 1 : 0;
}

static int subbuf_start_handler(struct rchan_buf *buf, void *subbuf,
				unsigned prev_subbuf_idx, void *prev_subbuf)
{
	unsigned padding = buf->padding[prev_subbuf_idx];

	/* for DEBUG
	serial_prints("CB:subbuf_start %d, %p, %d\n",
		      prev_subbuf_idx, prev_subbuf, padding);
		      */
	/* When do-not-overwrite mode, log data is never dropped. */
	if (!is_kern_all_by_hook(mode) &&
	    !is_start(mode) && relay_buf_full(buf))
		bt_dropped++;
	if (prev_subbuf)
		*((unsigned*)prev_subbuf) = padding;
	return sizeof(padding);	/* reserve space for padding */
}

static void buf_full_handler(struct rchan_buf *buf, unsigned subbuf_idx,
			     void *subbuf)
{
	unsigned padding = buf->padding[subbuf_idx];

	*((unsigned*)subbuf) = padding;
}

static int get_cpu_from_rchan_buf(struct rchan_buf *rbuf)
{
	struct rchan *chan = rbuf->chan;
	int i;

	for (i = 0; i < NR_CPUS; i++)
		if (chan->buf[i] == rbuf)
			return i;
	return -1;
}

#else
static int subbuf_start_handler(struct rchan_buf *buf, void *subbuf,
				void *prev_subbuf, unsigned int prev_padding)
{
	/* for DEBUG
	serial_prints("subbuf_start(cpu:%d, full:%d, p:%d)\n",
		      smp_processor_id(), relay_buf_full(buf), prev_padding);
		      */
	if (prev_subbuf)
		*((unsigned*)prev_subbuf) = prev_padding;
	subbuf_start_reserve(buf, sizeof(prev_padding));

	if (relay_buf_full(buf)) {
		if (is_start(mode))
			return 0;
		else if (!is_kern_all_by_hook(mode))
			/* When STOP or FR mode, log data is never dropped. */
			bt_dropped++;
	}
	return 1;
}
#define get_cpu_from_rchan_buf(rbuf)	(rbuf)->cpu
#if RELAYFS_CHANNEL_VERSION >= 6
static struct dentry *create_buf_file_handler(const char *filename,
					      struct dentry *parent, int mode,
					      struct rchan_buf *buf,
					      int *is_global)
{
	return debugfs_create_file(filename, mode, parent, buf,
				   &relay_file_operations);
}

static int remove_buf_file_handler(struct dentry *dentry)
{
	debugfs_remove(dentry);
	return 0;
}
#endif
#endif

static struct rchan_callbacks relayfs_callbacks =
{
	.subbuf_start = subbuf_start_handler,
#if RELAYFS_CHANNEL_VERSION < 5
	.buf_full = buf_full_handler,
#elif RELAYFS_CHANNEL_VERSION >= 6
	.create_buf_file = create_buf_file_handler,
	.remove_buf_file = remove_buf_file_handler,
#endif
};

static int subbuf_sz_read(char *buffer, char **start, off_t off,
			  int count, int *eof, void *data)
{
	return sprintf(buffer, "%d\n", subbuf_size);
}

static int n_subbufs_read(char *buffer, char **start, off_t off,
			  int count, int *eof, void *data)
{
	return sprintf(buffer, "%d\n", subbuf_num);
}

static int subbuf_sleep_threshold_read(char *buffer, char **start, off_t off,
				       int count, int *eof, void *data)
{
	return sprintf(buffer, "%d\n", subbuf_sleep_threshold);
}

static int dropped_read(char *buffer, char **start, off_t off,
			int count, int *eof, void *data)
{
	return sprintf(buffer, "%d\n", bt_dropped);
}

static int produced_read(char *buffer, char **start, off_t off,
			 int count, int *eof, void *data)
{
	struct rchan_buf *rbuf = (struct rchan_buf*)data;
#if RELAYFS_CHANNEL_VERSION < 5
	int produced = atomic_read(&rbuf->subbufs_produced);
#else
	int produced = rbuf->subbufs_produced;
#endif
	int len = sizeof(produced);

	memcpy(buffer, &produced, len);
	return len;
}

static int consumed_read(char *buffer, char **start, off_t off,
			 int count, int *eof, void *data)
{
	struct rchan_buf *rbuf = (struct rchan_buf*)data;
#if RELAYFS_CHANNEL_VERSION < 5
	int consumed = atomic_read(&rbuf->subbufs_consumed);
#else
	int consumed = rbuf->subbufs_consumed;
#endif
	int len = sizeof(consumed);

	memcpy(buffer, &consumed, len);
	return len;
}

static int consumed_write(struct file *file, const char *user_buf,
			  unsigned long count, void *data)
{
	struct rchan_buf *rbuf = (struct rchan_buf*)data;
	size_t consumed;

	if (copy_from_user(&consumed, user_buf, sizeof(consumed))) {
		printk("%s: copy_from_user failed.\n", MOD_NAME);
		return -EFAULT;
	}
		
	relay_subbufs_consumed(rbuf->chan, get_cpu_from_rchan_buf(rbuf),
			       consumed);

	if (is_relfs_writeable())
		wake_up_interruptible(&relfs_write_wait);

	return count;
}

int relfs_init(void)
{
	int i;
	struct info_per_cpu *info;

#if RELAYFS_CHANNEL_VERSION >= 6
	dir = debugfs_create_dir(relfs_dirname, NULL);
#else
	dir = relayfs_create_dir(relfs_dirname, NULL);
#endif
	if (dir == NULL) {
		printk("%s: cannot create relayfs directory\n", MOD_NAME);
		return -ENOMEM;
	}
#if RELAYFS_CHANNEL_VERSION < 5
	bt_channel = relay_open(relfs_basename, dir, subbuf_size, subbuf_num,
				!is_start(mode), &relayfs_callbacks);
#else
	bt_channel = relay_open(relfs_basename, dir, subbuf_size, subbuf_num,
				&relayfs_callbacks);
#endif
	if (bt_channel == NULL) {
		printk("%s: cannot open relayfs channel\n", MOD_NAME);
		return -ENOMEM;
	}
	proc_subbuf_sz = create_proc_entry("subbuf_size", 0400, proc_btrax);
	if (!proc_subbuf_sz)
		return -ENOMEM;
	proc_subbuf_sz->read_proc = subbuf_sz_read;
	proc_n_subbufs = create_proc_entry("n_subbufs", 0400, proc_btrax);
	if (!proc_n_subbufs)
		return -ENOMEM;
	proc_n_subbufs->read_proc = n_subbufs_read;
	proc_subbuf_threshold = create_proc_entry("subbuf_threshold", 0400,
						  proc_btrax);
	if (!proc_subbuf_threshold)
		return -ENOMEM;
	proc_subbuf_threshold->read_proc = subbuf_sleep_threshold_read;
	proc_dropped = create_proc_entry("dropped", 0400, proc_btrax);
	if (!proc_dropped)
		return -ENOMEM;
	proc_dropped->read_proc = dropped_read;

	for_each_online_cpu(i) {
		info = &per_cpu(bt_info_per_cpu, i);
		info->p_produced = create_proc_entry("produced", 0400,
						     info->p_cpuN);
		if (!info->p_produced)
			return -ENOMEM;
		info->p_produced->data = bt_channel->buf[i];
		info->p_produced->read_proc = produced_read;

		info->p_consumed = create_proc_entry("consumed", 0600,
						     info->p_cpuN);
		if (!info->p_consumed)
			return -ENOMEM;
		info->p_consumed->data = bt_channel->buf[i];
		info->p_consumed->read_proc = consumed_read;
		info->p_consumed->write_proc = consumed_write;
	}
	return 0;
}

static void kill_procs_using_relfs(void)
{
	struct dentry *child;
	int max = 16;
	pid_t pid[max], *p, *p_max;

	if (!dir)
		return;
	do {
		p = pid;
		p_max = p + max;
		spin_lock(&dcache_lock);
		list_for_each_entry(child, &dir->d_subdirs, bt_d_child) {
			chk_procs_using_relfs(child, &p, p_max);
			if (p >= p_max)
				break;
		}
		spin_unlock(&dcache_lock);
		p_max = p;
		for (p = pid; p < p_max; p++) {
			//serial_prints("  kill(%d)\n", *p);
			kill_proc(*p, SIGKILL, 1);
		}
		cpu_relax();
	} while (p != pid);
}

void relfs_cleanup(void)
{
	int i;
	struct info_per_cpu *info;

	is_cleanup = 1;
	wake_up_interruptible_all(&relfs_write_wait);
	do {
		yield();
	} while (waitqueue_active(&relfs_write_wait));

	if (proc_subbuf_sz)
		remove_proc_entry(proc_subbuf_sz->name, proc_btrax);
	if (proc_n_subbufs)
		remove_proc_entry(proc_n_subbufs->name, proc_btrax);
	if (proc_subbuf_threshold)
		remove_proc_entry(proc_subbuf_threshold->name, proc_btrax);
	if (proc_dropped)
		remove_proc_entry(proc_dropped->name, proc_btrax);

	kill_procs_using_relfs();

	for_each_online_cpu(i) {
		info = &per_cpu(bt_info_per_cpu, i);
		if (info->p_produced)
			remove_proc_entry(info->p_produced->name, info->p_cpuN);
		if (info->p_consumed)
			remove_proc_entry(info->p_consumed->name, info->p_cpuN);
	}
	if (bt_channel)
		relay_close(bt_channel);
	if (dir)
#if RELAYFS_CHANNEL_VERSION >= 6
		debugfs_remove(dir);
#else
		relayfs_remove_dir(dir);
#endif
}
