/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_coverage.c - coverage information display program                     */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2007                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <getopt.h>
#include "bt_ar_parse.h"
#include "bt_utils.h"
#include "bt_hconv.h"

#define BT_COVERAGE_VER	BT_VER
#define	COPYRIGHT	"Copyright (c) Hitachi, Ltd. 2005-2007"

int output_summary = 0;
int verbose = 0;
char *outdir;
char *srcdir;
char *includes;
char *excludes;
node *include_funcs;	/* avl tree of unsigned long address */
node *exclude_funcs;
node *include_fcs;	/* avl tree of func_chk */
struct r2n_info r2n_info = { NULL, 128, 0 };
#define HTML_OUTPUT	(outdir != NULL)
#define LIMIT_BY_FUNCS	(include_funcs != NULL)

#define INC_CNT(p_cnt)						\
	do {							\
		if (*(p_cnt) != 0x7fffffff) *(p_cnt) += 1;	\
	} while(0)

#define ADD_CNT(p_cnt, cnt)					\
	do {							\
		*(p_cnt) += (cnt);				\
		if (*(p_cnt) < (cnt))				\
			*(p_cnt) = 0x7fffffff;			\
	} while(0)

int inc_tracking_cnt_with_addr(struct path *p, unsigned long addr)
{
	struct unknown *uk;

	for (uk = (struct unknown *)p->jmp_cnt; uk; uk = uk->next) {
		if (uk->addr == addr) {
			INC_CNT(&uk->cnt);
			return 0;
		}
	}
	uk = xcalloc(1, sizeof(struct unknown));
	uk->addr = addr;
	uk->cnt = 1;
	uk->next = (struct unknown*)p->jmp_cnt;
	p->jmp_cnt = (long)uk;
	return 0;
}

struct proc_each_rec_data {
	struct addr_range *range;
	unsigned long last;
};

int proc_each_record(struct bt_record *__rec, off_t i_rec, void *tmp)
{
	struct bt_record rec = *__rec;
	struct proc_each_rec_data *data = (struct proc_each_rec_data*)tmp;
	long i, idx_last, idx_from, chk_fallthrough;
	struct range_to_name *r2n, *r2n_to;
	struct path *p, *p_last, *p_from, *p_to = NULL;
	int rc;

	if (is_warn_record(&rec)) {
		struct warn_record *warn = (struct warn_record*)&rec;
		printf("WARN: bts left only: %ld\n", warn->left);
		goto EXIT;
	}
	if (is_pid_record(&rec) || is_comm_record(&rec))
		goto EXIT;
	rc = chk_fix_from_cache(&r2n_info, &rec.from, rec.to,
				&r2n, &p_from, &idx_from);
	if (rc <= 0)
		goto EXIT;
	/*
	if (!is_addr_range_match(rec.from, rec.to, data->range))
		goto EXIT;
		*/

	idx_last = r2n->cnt;
	p_last = find_path_from_addr(r2n->pt, &idx_last, data->last);

	/*
	printf("DBG:");
	printf_bfd_vma(data->last);
	printf("->");
	printf_bfd_vma(rec.from);
	printf("->");
	printf_bfd_vma(rec.to);
	printf(" ");
	printf_path(p_from);
	if (p_to)
		printf_path(p_to);
		*/
	if (p_last) {
		/* check fallthrough */
		chk_fallthrough = 1;
		for (i = idx_last; i < idx_from; i++) {
			p = r2n->pt[i];
			/* The address becomes an execution address when
			 * interrupt enters the branch ahead when returning
			 * from interrupt.
			 * Therefore, after it returns from the int instruction,
			 * it becomes fallthrough check from passing the int
			 * instruction.
			 */
			if (i == idx_last && p->type == BTYPE_INT)
				continue;
			if (p->type != BTYPE_BRANCH && p->type != BTYPE_OTHER) {
				chk_fallthrough = 0;
				/* The error occurs because the address between
				 * from turning off the bts facility to turning
				 * on seems to drop off.
				 * When the error occurs in bt_mod.ko, the
				 * fallthrough check is not done, and the
				 * following processing is done.
				 */
				if (strcmp(r2n->basename, MOD_NAME ".ko") != 0){
					printf("WARN: detect lack of log");
					printf(" (0x%08lx) -> 0x%08lx -> 0x%08lx\n",
					       data->last, rec.from, rec.to);
					break;
					/*
					fprintf(stdout, "!!! NG path tree\n");
					dump_r2n();
					return -1;
					*/
				}
			}
		}
		if (chk_fallthrough)
			for (i = idx_last; i < idx_from; i++) {
				p = r2n->pt[i];
				if (i != idx_last)
					INC_CNT(&p->cnt);
				if (p->type == BTYPE_BRANCH)
					INC_CNT(&p->next_cnt);
			}
	}
	if (p_last != p_from)
		INC_CNT(&p_from->cnt);

	/* check branch */
	if (p_from->base == rec.from) {
		if (p_from->jmp == UNKNOWN_BADDR) {
			if (inc_tracking_cnt_with_addr(p_from, rec.to) < 0) {
				printf("!!! Error: (0x%08lx) -> 0x%08lx -> 0x%08lx\n",
				       data->last, rec.from, rec.to);
				return -1;
			}
		} else if (p_from->jmp == rec.to)
			INC_CNT(&p_from->jmp_cnt);
	}
	/* check to */
	r2n_to = addr_to_r2n(&r2n_info, rec.to);
	if (r2n_to) {
		i = r2n_to->cnt;
		p_to = find_path_from_addr(r2n_to->pt, &i, rec.to);
		if (p_to)
			INC_CNT(&p_to->cnt);
	}
EXIT:
	data->last = rec.to;
	return 0;
}

void printf_funcname(struct bfd_if *bi, bfd_vma addr)
{
	int rc, lno;
	size_t offset;
	const char *src_name, *func_name;

	rc = addr_to_func_name_and_offset(bi, addr, &func_name, &offset);
	if (rc >= 0) {
		printf_func_name(func_name, offset);
		rc = get_source_info(bi, addr, &src_name, &func_name, &lno);
		if (rc >= 0 && src_name && lno)
			printf(":%s,%d", src_name, lno);
	} else
		printf_bfd_vma(addr);
}

void printf_srcname_and_lno(struct bfd_if *bi, bfd_vma addr)
{
	int rc, lno;
	const char *src_name, *func_name;

	rc = get_source_info(bi, addr, &src_name, &func_name, &lno);
	if (rc >= 0 && src_name && lno)
		printf("%s,%d", src_name, lno);
	else
		printf_bfd_vma(addr);
		//printf_funcname(bi, addr);
}

/*-----------------------------------------------------------------------------
 *  filter by function name support
 *-----------------------------------------------------------------------------
 */
static int f_cmp_faddr(void *target, void *each)
{
	unsigned long a1 = (unsigned long)target;
	unsigned long a2 = (unsigned long)each;

	if (a1 < a2)
		return -1;
	if (a1 > a2)
		return 1;
	return 0;
}

int create_filter_funcs(char *funcs, node **list)
{
	char *p;
	unsigned long *addrs;
	int i, n;

	p = strtok(strdup(funcs), ",");
	while (p) {
		n = get_symbol_addr_all(&r2n_info, p, &addrs);
		if (n < 0)
			return -1;
		for (i = 0; i < n; i++) {
			if (!search_tree((void*)addrs[i], *list, f_cmp_faddr))
				*list = insert_tree((void*)addrs[i], *list,
						    f_cmp_faddr);
		}
		if (n)
			free(addrs);
		p = strtok(NULL, ",");
	}
	if (!*list) {
		printf("WARN: include functions were not found.\n");
		return -1;
	}
	return 0;
}

int is_include_faddr(unsigned long addr)
{
	return (search_tree((void*)addr, include_funcs, f_cmp_faddr)
		&& !search_tree((void*)addr, exclude_funcs, f_cmp_faddr));
}

int is_exclude_faddr(unsigned long addr)
{
	return (search_tree((void*)addr, exclude_funcs, f_cmp_faddr) != NULL);
}

/*-----------------------------------------------------------------------------
 *  display function coverage
 *-----------------------------------------------------------------------------
 */
static int f_cmp_addr2fc(void *__target, void *__dt)
{
	unsigned long addr = (unsigned long)__target;
	struct func_chk *fc = (struct func_chk*)__dt;

	if (addr < fc->addr)
		return -1;
	if (addr > fc->addr)
		return 1;
	return 0;
}

static int f_cmp_fc(void *__target, void *__dt)
{
	struct func_chk *t_fc = (struct func_chk*)__target;
	struct func_chk *fc = (struct func_chk*)__dt;

	if (t_fc->addr < fc->addr)
		return -1;
	if (t_fc->addr > fc->addr)
		return 1;
	return 0;
}

/* Meaning of return value is -1:error, 0:already checked, 1: checked. */
int __check_func(node **fc_list, unsigned long addr, unsigned long cnt)
{
	struct func_chk *fc;

	fc = search_tree((void*)addr, *fc_list, f_cmp_addr2fc);
	if (fc) {
		ADD_CNT(&fc->cnt, cnt);
		return 0;
	}
	fc = xmalloc(sizeof(struct func_chk));
	fc->addr = addr;
	fc->end = 0;
	fc->cnt = cnt;
	SET_FC_VALID(fc);
	fc->childs = NULL;
	fc->excluded_tree_cnt = 0;

	*fc_list = insert_tree(fc, *fc_list, f_cmp_fc);
	if (!(*fc_list))
		return -1;
	return 1;
}

/* Meaning of return value is -1:error, 0:normal-end. */
int __check_func_by_jaddr(node **fc_list, unsigned long addr,
			     unsigned long cnt)
{
	struct func_chk *fc;

	if (!cnt)
		return 0;
	fc = search_tree((void*)addr, *fc_list, f_cmp_addr2fc);
	if (fc)
		ADD_CNT(&fc->cnt, cnt);
	return 0;
}

int check_func(node **fc_list, struct path *p)
{
	struct unknown *uk;
	int (*func)(node**, unsigned long, unsigned long);

	func = LIMIT_BY_FUNCS ? __check_func_by_jaddr : __check_func;
	if (p->jmp == UNKNOWN_BADDR) {
		for (uk = (struct unknown*)p->jmp_cnt; uk; uk = uk->next) {
			if (func(fc_list, uk->addr, uk->cnt) < 0)
				return -1;
		}
	} else if (func(fc_list, p->jmp, p->jmp_cnt) < 0)
		return -1;
	return 0;
}

struct get_valid_funcs_pack {
	node		*valid_funcs;
	func_compare	f_cmp;
};

static void f_free_unnecessary_fc(void *__dt)
{
	struct func_chk *fc = (struct func_chk*)__dt;

	if (!IS_CHECKED_FC(fc))
		free(fc);
}

static int f_slim_down_flist(void *__dt, void *user_data)
{
	struct func_chk *fc = (struct func_chk*)__dt;
	struct get_valid_funcs_pack *pack =
		(struct get_valid_funcs_pack*)user_data;

	if (!IS_CHECKED_FC(fc))
		return 0;
	pack->valid_funcs = insert_tree(fc, pack->valid_funcs, pack->f_cmp);
	return 0;
}

static void slim_down_flist(node **fc_list, func_compare f_compare)
{
	struct get_valid_funcs_pack pack;

	pack.valid_funcs = NULL;
	pack.f_cmp = f_compare;
	for_each_node(*fc_list, f_slim_down_flist, &pack);
	free_tree(*fc_list, f_free_unnecessary_fc);
	*fc_list = pack.valid_funcs;
}

static int f_cmp_fc_excluded_tcnt(void *__target, void *__dt)
{
	struct func_chk *t_fc = (struct func_chk*)__target;
	struct func_chk *fc = (struct func_chk*)__dt;

	if (t_fc->excluded_tree_cnt > fc->excluded_tree_cnt)
		return -1;
	if (t_fc->excluded_tree_cnt < fc->excluded_tree_cnt)
		return 1;
	if (t_fc->addr < fc->addr)
		return -1;
	if (t_fc->addr > fc->addr)
		return 1;
	return 0;
}

static inline void get_excluded_tcnt_func_list(node **fc_list)
{
	slim_down_flist(fc_list, f_cmp_fc_excluded_tcnt);
}

int mark_valid_path(struct range_to_name *r2n, struct func_chk *fc,
		    unsigned long end, node *fc_list);

int __mark_valid_path(struct range_to_name *r2n, struct func_chk *fc,
		      unsigned long faddr, node *fc_list, int from_uk)
{
	struct range_to_name *c_r2n;
	unsigned long c_end;
	struct func_chk *c_fc;

	if (!is_exclude_faddr(faddr) &&
	    get_func_info(&r2n_info, faddr, &c_end, &c_r2n)) {
		c_fc = search_tree((void*)faddr, fc_list, f_cmp_addr2fc);
		if (c_fc) {
			if (from_uk) {
				if (IS_CHECKED_FC(c_fc))
					return 0;
				if (c_fc->cnt)
					SET_FC_UT(c_fc);
				return 0;
			}
			fc->childs = insert_tree(c_fc, fc->childs, f_cmp_fc);
			if (!(fc->childs))
				return -1;
			if (IS_CHECKED_FC(c_fc))
				return 0;
			SET_FC_VALID(c_fc);
			if (mark_valid_path(c_r2n, c_fc, c_end, fc_list) < 0)
				return -1;
		}
	}
	return 0;
}

int mark_valid_path(struct range_to_name *r2n, struct func_chk *fc,
		    unsigned long end, node *fc_list)
{
	struct path *p;
	long i, top;
	struct unknown *uk;

	top = r2n->cnt;
	p = find_path_from_addr(r2n->pt, &top, fc->addr);
	if (!p) {
		fprintf(stderr,
			"function is not in path-tree.(%s:0x%08lx:0x%08lx)\n",
			r2n->basename, fc->addr, end);
		return -1;
	}
	fc->i = top;
	fc->end = end;
	for (i = top; i < r2n->cnt; i++) {
		p = r2n->pt[i];
		if (p->next > end)
			break;
		p->type &= ~IS_INVALID_PATH;
		switch (p->type) {
		case BTYPE_CALL:
		case BTYPE_JMP:
		case BTYPE_BRANCH:
			/* Function calls by the indirect addressing are not
			 * included in the function call tree.
			 */
			if (p->jmp == UNKNOWN_BADDR) {
				for (uk = (struct unknown*)p->jmp_cnt; uk;
				     uk = uk->next) {
					if (__mark_valid_path(r2n, fc, uk->addr,
							      fc_list, 1) < 0)
						return -1;
				}
			} else {
				if (__mark_valid_path(r2n, fc, p->jmp, fc_list,
						      0) < 0)
					return -1;
			}
			break;
		}
	}
	return 0;
}

struct f_chk_func_data {
	struct range_to_name *r2n;
	node **fc_list;
};

static int f_check_func(bfd_vma addr, const char *name, void *data)
{
	struct f_chk_func_data *dt = (struct f_chk_func_data*)data;

	if (__check_func(dt->fc_list, addr, 0) >= 0)
		return 0;
	return -1;
}

int chk_all_func_syms(node **fc_list, struct range_to_name *r2n)
{
	struct f_chk_func_data dt;
	int rc;

	dt.r2n = r2n;
	dt.fc_list = fc_list;
	rc = for_each_fsymbols(&r2n->bi, r2n->offset, f_check_func, &dt);
	if (rc < 0)
		return rc;
	return 0;
}

static void f_fc_free(void *__dt)
{
	struct func_chk *fc = (struct func_chk*)__dt;

	free(fc);
}

static int f_mark_include_funcs(void *__dt, void *user_data)
{
	unsigned long addr = (unsigned long)__dt;
	node *fc_list = (node*)user_data;
	struct range_to_name *r2n;
	unsigned long end;
	struct func_chk *fc;
	
	if (is_exclude_faddr(addr) ||
	    !get_func_info(&r2n_info, addr, &end, &r2n))
		return 0;
	fc = search_tree((void*)addr, fc_list, f_cmp_addr2fc);
	if (!fc)
		return 0;
	include_fcs = insert_tree(fc, include_fcs, f_cmp_fc);
	if (IS_CHECKED_FC(fc))
		return 0;
	SET_FC_VALID(fc);
	if (mark_valid_path(r2n, fc, end, fc_list) < 0)
		return -1;
	return 0;
}

static int f_all_fc_mark_invalid(void *__dt, void *user_data)
{
	struct func_chk *cc = (struct func_chk*)__dt;

	SET_FC_NOT_CHECKED(cc);
	return 0;
}

int chk_func_coverage(node **fc_list)
{
	unsigned long i, j;
	struct range_to_name *r2n;
	struct path *p;
	int type;

	*fc_list = NULL;
	for (i = 0; i < r2n_info.r2n_max; i++) {
		r2n = r2n_info.all_r2n[i];
		if (r2n) {
			if (chk_all_func_syms(fc_list, r2n) < 0)
				goto ERR_EXIT;
		}
	}
	for (i = 0; i < r2n_info.r2n_max; i++) {
		r2n = r2n_info.all_r2n[i];
		if (r2n) {
			for (j = 0; j < r2n->cnt; j++) {
				p = r2n->pt[j];
				type = p->type & ~IS_INVALID_PATH;
				if (type == BTYPE_CALL) {
					if (check_func(fc_list, p) < 0)
						goto ERR_EXIT;
				} else if (type == BTYPE_JMP ||
					   type == BTYPE_BRANCH) {
					if (__check_func_by_jaddr(fc_list,
								  p->jmp,
								  p->jmp_cnt)<0)
						goto ERR_EXIT;
				}
			}
		}
	}
	if (LIMIT_BY_FUNCS) {
		/* When limiting the coverage output by function-list, mark all
		 * paths as invalid. Then, mark the valid function paths as
		 * valid.
		 */
		for_each_node(*fc_list, f_all_fc_mark_invalid, NULL);
		for (i = 0; i < r2n_info.r2n_max; i++) {
			r2n = r2n_info.all_r2n[i];
			if (r2n) {
				for (j = 0; j < r2n->cnt; j++) {
					p = r2n->pt[j];
					p->type |= IS_INVALID_PATH;
				}
			}
		}
		for_each_node(include_funcs, f_mark_include_funcs, *fc_list);
		slim_down_flist(fc_list, f_cmp_fc);
	}
	return 0;

ERR_EXIT:
	free_tree(*fc_list, f_fc_free);
	return -1;
}

struct call_cnt_pack {
	unsigned long		called;
	unsigned long		all;
	struct range_to_name	*r2n;
};

static int f_cnt_each_fc(void *__dt, void *user_data)
{
	struct call_cnt_pack *pack = (struct call_cnt_pack*)user_data;
	struct range_to_name *r2n = pack->r2n;
	struct func_chk *cc = (struct func_chk*)__dt;

	if (cc->addr >= r2n->begin && cc->addr <= r2n->end) {
		if (IS_VALID_FC(cc)) {
			pack->all++;
			if (cc->cnt)
				pack->called++;
		}
	} else if (cc->addr > r2n->end)
		return 1;	/* break */
	return 0;
}

struct dump_each_fc_data {
	struct range_to_name	*r2n;
	struct html_out_data	*html_out_dt;
};

#define __print_one_func_cov(r2n, fc) \
do { \
	printf_funcname(&(r2n)->bi, (fc)->addr - (r2n)->offset); \
	if ((fc)->excluded_tree_cnt > 0) { \
		if ((fc)->cnt) \
			printf("\t(%ld, F:%ld)\n", (fc)->cnt, \
			       (fc)->excluded_tree_cnt); \
		else \
			printf("\t(0, F:%ld)\n", (fc)->excluded_tree_cnt); \
	} else { \
		if ((fc)->cnt) \
			printf("\t(%ld)\n", (fc)->cnt); \
		else \
			printf("\n"); \
	} \
} while(0)

static int f_dump_each_fc(void *__dt, void *user_data)
{
	struct dump_each_fc_data *fc_dt = (struct dump_each_fc_data*)user_data;
	struct func_chk *cc = (struct func_chk*)__dt;
	struct range_to_name *r2n = fc_dt->r2n;

	if (cc->addr >= r2n->begin && cc->addr <= r2n->end) {
		if (!IS_VALID_FC(cc))
			return 0;
		if (HTML_OUTPUT)
			out_func_html_each(fc_dt->html_out_dt, r2n, cc);
		else {
			printf(cc->cnt ? "(OK) " : "(NT) ");
			__print_one_func_cov(r2n, cc);
		}
	} else if (cc->addr > r2n->end)
		return 1;	/* break */
	return 0;
}

struct excluded_cnt_pack {
	unsigned long		cnt;
	node			*checked_fc;
	node			*fc_list;
};

static int f_cnt_excluded_fc(void *__dt, void *user_data)
{
	struct func_chk *fc = (struct func_chk*)__dt;
	struct excluded_cnt_pack *pack = (struct excluded_cnt_pack*)user_data;

	if (search_tree(fc, pack->checked_fc, f_cmp_fc))
		return 0;
	pack->cnt++;
	pack->checked_fc = insert_tree(fc, pack->checked_fc, f_cmp_fc);
	return for_each_node(fc->childs, f_cnt_excluded_fc, pack);
}

static void get_excluded_tree_cnt(node *fc_list, struct func_chk *fc,
				  unsigned long all_cnt)
{
	struct excluded_cnt_pack pack;

	if (fc->excluded_tree_cnt > 0)
		return;
	pack.cnt = 0;
	pack.checked_fc = insert_tree(fc, NULL, f_cmp_fc);
	pack.fc_list = fc_list;
	for_each_node(include_fcs, f_cnt_excluded_fc, &pack);
	free_tree(pack.checked_fc, NULL);
	fc->excluded_tree_cnt = all_cnt - pack.cnt;
	return;
}

struct print_ftree_data {
	node			*fc_list;
	node			**printed;
	int			nest;
	struct html_out_data	*html_out_dt;
	unsigned long		tree_cnt;
};

static void __f_print_func_tree(struct func_chk *fc,
				struct print_ftree_data *p_ftree_dt,
				int type, int has_child)
{
	int i, nest = p_ftree_dt->nest;
	struct html_out_data *dt = p_ftree_dt->html_out_dt;
	node *fc_list = p_ftree_dt->fc_list;
	struct range_to_name *r2n;

	r2n = addr_to_r2n(&r2n_info, fc->addr);
	if (!r2n)
		return;
	get_excluded_tree_cnt(fc_list, fc, p_ftree_dt->tree_cnt);
	if (HTML_OUTPUT)
		out_func_tree_html_each_enter(dt, r2n, fc, nest, type,
					      has_child);
	else {
		switch (type) {
		case BCOV_TYPE_OK:
			printf("(OK) ");
			break;
		case BCOV_TYPE_NT:
			printf("(NT) ");
			break;
		default:
			printf("(--) ");
			break;
		}
		for (i = 0; i < nest; i++)
			printf("+-");
		__print_one_func_cov(r2n, fc);
	}
}

static int f_print_invalid_func(void *__dt, void *user_data)
{
	struct print_ftree_data *dt = (struct print_ftree_data*)user_data;
	struct func_chk *fc = (struct func_chk*)__dt;
	struct range_to_name *r2n;

	r2n = addr_to_r2n(&r2n_info, fc->addr);
	if (!r2n)
		return 0;
	if (IS_UT_FC(fc)) {
		if (HTML_OUTPUT)
			out_func_tree_html_each_invalid(dt->html_out_dt, r2n,
							fc);
		else {
			printf("(UT) ");
			__print_one_func_cov(r2n, fc);
		}
	}
	return 0;
}

static int f_print_func_tree(void *__dt, void *user_data)
{
	struct print_ftree_data dt = *(struct print_ftree_data*)user_data;
	struct func_chk *fc = (struct func_chk*)__dt;
	int printed, type, has_child;

	printed = search_tree((void*)fc->addr, *dt.printed, f_cmp_addr2fc)
									!= NULL;
	if (printed) {
		if (dt.nest) {
			has_child = 0;
			__f_print_func_tree(fc, &dt, BCOV_TYPE_HT, has_child);
			goto EXIT;
		}
		return 0;
	}
	*dt.printed = insert_tree(fc, *dt.printed, f_cmp_fc);
	if (!(*dt.printed))
		return -1;
	type = fc->cnt ? BCOV_TYPE_OK : BCOV_TYPE_NT;
	has_child = fc->childs != NULL;
	__f_print_func_tree(fc, &dt, type, has_child);
	dt.nest++;
	if (for_each_node(fc->childs, f_print_func_tree, &dt) < 0)
		return -1;
	dt.nest--;
EXIT:
	if (HTML_OUTPUT)
		out_func_tree_html_each_exit(dt.html_out_dt, dt.nest,has_child);
	return 0;
}

void print_func_tree(node *fc_list, struct html_out_data *html_out_dt)
{
	int i;
	struct range_to_name *r2n;
	struct call_cnt_pack pack = { 0, 0, NULL };
	node *printed = NULL;
	struct print_ftree_data dt;
	char *s_inc, *s_exc;

	s_inc = includes ? includes : "(--)";
	s_exc = excludes ? excludes : "(--)";
	if (HTML_OUTPUT)
		out_func_tree_html_start(html_out_dt, s_inc, s_exc);
	for (i = 0; i < r2n_info.r2n_max; i++) {
		r2n = r2n_info.all_r2n[i];
		if (r2n) {
			pack.r2n = r2n;
			for_each_node(fc_list, f_cnt_each_fc, &pack);
		}
	}
	dt.fc_list = fc_list;
	dt.printed = &printed;
	dt.nest = 0;
	dt.html_out_dt = html_out_dt;
	dt.tree_cnt = pack.all;
	if (!HTML_OUTPUT) {
		printf("====== includes: %s ======\n", s_inc);
		printf("====== excludes: %s ======\n", s_exc);
		printf("====== function tree (%ld/%ld=%.2f%%) ======\n",
		       pack.called, pack.all,
		       get_percent(pack.called, pack.all));
	}
	for_each_node(include_fcs, f_print_func_tree, &dt);
	for_each_node(fc_list, f_print_invalid_func, &dt);
	if (HTML_OUTPUT)
		out_func_tree_html_end(html_out_dt);
}

/*-----------------------------------------------------------------------------
 *  display branch coverage
 *-----------------------------------------------------------------------------
 */
static long get_unknown_bcnt(struct path *p)
{
	long i = 0;
	struct unknown *uk;

	for (uk = (struct unknown*)p->jmp_cnt; uk; uk = uk->next)
		i++;
	return i;
}

void __dump_one_branch_coverage(struct range_to_name *r2n,
				struct html_out_data *dt,
				int bcov_type, unsigned long base,
				unsigned long branch, long b_cnt,
				unsigned long fall, long f_cnt)
{
	if (HTML_OUTPUT)
		out_branch_html_each(dt, r2n, base, branch, b_cnt, fall, f_cnt);
	else {
		switch (bcov_type) {
		case BCOV_TYPE_OK:	printf("(OK) "); break;
		case BCOV_TYPE_HT:	printf("(HT) "); break;
		case BCOV_TYPE_NT:	printf("(NT) "); break;
		case BCOV_TYPE_UT:	printf("(UT) "); break;
		case BCOV_TYPE_UN:	printf("(UN) "); break;
		}
		printf_srcname_and_lno(&r2n->bi, base);
		printf(" [%ld/", b_cnt);
		if (fall == UNKNOWN_BADDR)
			printf("x] ");
		else
			printf("%ld] ", f_cnt);
		if (branch == UNKNOWN_BADDR)
			printf("----------:");
		else {
			printf_srcname_and_lno(&r2n->bi, branch);
			printf(":");
		}
		if (fall == UNKNOWN_BADDR)
			printf("xxxxxxxxxx\n");
		else {
			printf_srcname_and_lno(&r2n->bi, fall);
			printf("\n");
		}
	}
}

inline void do_one_branch_coverage(struct range_to_name *r2n,
				   struct html_out_data *dt, struct path *p,
				   func_do_one_bcov func)
{
	struct unknown *uk;
	int bcov_type;
	unsigned long base, branch, fall;
	long b_cnt, f_cnt;

	base = p->base - r2n->offset;
	if (IS_SWITCH_JMP(p) ||		/* switch case */
	    p->jmp == UNKNOWN_BADDR) {	/* indirect addressing branch */
		fall = UNKNOWN_BADDR;
		f_cnt = 0;
		if (get_unknown_bcnt(p)) {
			for (uk = (struct unknown*)p->jmp_cnt; uk;
			     uk = uk->next) {
				bcov_type = BCOV_TYPE_UT;
				branch = uk->addr - r2n->offset;
				b_cnt = uk->cnt;
				func(r2n, dt, bcov_type, base, branch, b_cnt,
				     fall, f_cnt);
			}
		} else {
			bcov_type = BCOV_TYPE_UN;
			branch = UNKNOWN_BADDR;
			b_cnt = 0;
			func(r2n, dt, bcov_type, base, branch, b_cnt,
			     fall, f_cnt);
		}
	} else {
		if (p->jmp_cnt != 0 && p->next_cnt != 0) {
			bcov_type = BCOV_TYPE_OK;
		} else if (p->jmp_cnt == 0 && p->next_cnt == 0) {
			bcov_type = BCOV_TYPE_NT;
		} else {
			bcov_type = BCOV_TYPE_HT;
		}
		branch = p->jmp - r2n->offset;
		b_cnt = p->jmp_cnt;
		fall = p->next - r2n->offset;
		f_cnt = p->next_cnt;
		func(r2n, dt, bcov_type, base, branch, b_cnt, fall, f_cnt);
	}
}

void dump_one_branch_coverage(struct range_to_name *r2n,
			      struct html_out_data *dt, struct path *p)
{
	do_one_branch_coverage(r2n, dt, p, __dump_one_branch_coverage);
}

struct branch_cnts {
	unsigned long ok;
	unsigned long nt;
	unsigned long ht;
	unsigned long uk_through;
	unsigned long uk_not_through;
};

void dump_branch_coverage(struct range_to_name *r2n, struct html_out_data *dt)
{
	long i, uk_bcnt;
	struct branch_cnts bcs = {0,0,0,0,0};
	struct path *p;
	unsigned long n_all_branch, n_all_uk;

	for (i = 0; i < r2n->cnt; i++) {
		p = r2n->pt[i];
		if ((p->type & IS_INVALID_PATH) ||
		    (p->type != BTYPE_BRANCH && !IS_SWITCH_JMP(p)))
			continue;
		if (p->jmp == UNKNOWN_BADDR) {
			uk_bcnt = get_unknown_bcnt(p);
			if (!uk_bcnt)
				bcs.uk_not_through++;
			else
				bcs.uk_through++;
		} else if (p->jmp_cnt != 0 && p->next_cnt != 0)
			bcs.ok++;
		else if (p->jmp_cnt == 0 && p->next_cnt == 0)
			bcs.nt++;
		else
			bcs.ht++;
	}
	n_all_branch = (bcs.ok + bcs.ht + bcs.nt) * 2;
	n_all_uk = bcs.uk_through + bcs.uk_not_through;
	if (HTML_OUTPUT)
		out_summary_html_branch(dt, bcs.ok, bcs.uk_through, bcs.ht,
					bcs.nt, n_all_branch, n_all_uk);
	else {
		printf("------ branch coverage (OK:%ld,HT:%ld,NT:%ld/%ld=%.2f%%",
		       bcs.ok, bcs.ht, bcs.nt, n_all_branch,
		       get_percent(bcs.ok * 2 + bcs.ht, n_all_branch));
		printf(" UK:%ld/%ld=%.2f%%) ------\n",
		       bcs.uk_through, n_all_uk,
		       get_percent(bcs.uk_through, n_all_uk));
	}
	if (output_summary)
		return;
	if (HTML_OUTPUT)
		out_branch_html_start(dt);
	for (i = 0; i < r2n->cnt; i++) {
		p = r2n->pt[i];
		if ((p->type & IS_INVALID_PATH) ||
		    (p->type != BTYPE_BRANCH && !IS_SWITCH_JMP(p)))
			continue;
		dump_one_branch_coverage(r2n, dt, p);
	}
	if (HTML_OUTPUT)
		out_branch_html_end(dt);
}

struct state_cnts {
	unsigned long ok;
	unsigned long nt;
};

/*-----------------------------------------------------------------------------
 *  display state coverage
 *-----------------------------------------------------------------------------
 */
void dump_state_coverage(struct range_to_name *r2n, struct html_out_data *dt)
{
	long i;
	struct state_cnts scs = {0,0};
	struct path *p;
	int is_exec;

	for (i = 0; i < r2n->cnt; i++) {
		p = r2n->pt[i];
		if ((p->type & IS_INVALID_PATH))
			continue;
		if (p->cnt > 0)
			scs.ok += 1;
		else
			scs.nt += 1;
	}
	if (HTML_OUTPUT)
		out_summary_html_state(dt, scs.ok, scs.ok + scs.nt);
	else
		printf("------ state coverage (%ld/%ld=%.2f%%) ------\n",
		       scs.ok, scs.ok + scs.nt,
		       get_percent(scs.ok, scs.ok + scs.nt));
	if (output_summary)
		return;
	if (HTML_OUTPUT)
		out_state_html_start(dt);
	for (i = 0; i < r2n->cnt; i++) {
		p = r2n->pt[i];
		if ((p->type & IS_INVALID_PATH))
			continue;
		is_exec = p->cnt > 0;
		if (HTML_OUTPUT)
			out_state_html_each(dt, r2n, is_exec, p->addr);
		else {
			if (is_exec)
				printf("(OK) ");
			else
				printf("(NT) ");
			printf_srcname_and_lno(&r2n->bi, p->addr);
			printf("\n");
		}
	}
	if (HTML_OUTPUT)
		out_state_html_end(dt);
}

/*-----------------------------------------------------------------------------
 *  display coverage
 *-----------------------------------------------------------------------------
 */
static int f_chk_ifuncs_are_executed(void *__dt, void *user_data)
{
	unsigned long addr = (unsigned long)__dt;
	struct range_to_name *r2n;
	unsigned long end;
	struct func_chk *fc;

	if (is_exclude_faddr(addr) ||
	    !get_func_info(&r2n_info, addr, &end, &r2n))
		return 0;
	fc = search_tree((void*)addr, include_fcs, f_cmp_addr2fc);
	if (!fc || !fc->cnt) {
		printf("WARN: ");
		printf_funcname(&r2n->bi, addr - r2n->offset);
		printf(" was not executed.\n");
	}
	return 0;
}

int dump_coverage(void)
{
	int i;
	struct range_to_name *r2n;
	node *fc_list;
	struct call_cnt_pack pack = { 0, 0, NULL };
	struct html_out_data dt;
	struct dump_each_fc_data u_dt;

	if (!r2n_info.all_r2n)
		return 0;

	if (HTML_OUTPUT) {
		memset(&dt, 0, sizeof(struct html_out_data));
		dt.srcdir = srcdir;
		dt.outdir = outdir;
		dt.limit_by_funcs = LIMIT_BY_FUNCS;

		if (init_html_output(&dt) < 0)
			return -1;
	}
	if (chk_func_coverage(&fc_list) < 0)
		return -1;

	u_dt.html_out_dt = &dt;
	if (LIMIT_BY_FUNCS) {
		/* Check if the inculde functions are executed. */
		for_each_node(include_funcs, f_chk_ifuncs_are_executed, NULL);
		if (!output_summary)
			print_func_tree(fc_list, &dt);
	}
	for (i = 0; i < r2n_info.r2n_max; i++) {
		r2n = r2n_info.all_r2n[i];
		if (!r2n)
			continue;
		pack.called = 0;
		pack.all = 0;
		pack.r2n = r2n;
		for_each_node(fc_list, f_cnt_each_fc, &pack);
		if (!pack.all)
			continue;
		dt.name = r2n->basename;
		if (HTML_OUTPUT) {
			out_summary_html_name(&dt);
			out_summary_html_func(&dt, pack.called, pack.all);
			out_func_html_start(&dt);
		} else {
			printf("====== %s coverage ======\n", r2n->basename);
			printf("------ function coverage (%ld/%ld=%.2f%%) ------\n",
			       pack.called, pack.all,
			       get_percent(pack.called, pack.all));
		}
		if (!output_summary) {
			u_dt.r2n = r2n;
			if (LIMIT_BY_FUNCS)
				get_excluded_tcnt_func_list(&fc_list);
			for_each_node(fc_list, f_dump_each_fc, &u_dt);
		}
		if (HTML_OUTPUT)
			out_func_html_end(&dt);
		dump_branch_coverage(r2n, &dt);
		dump_state_coverage(r2n, &dt);
	}
	if (HTML_OUTPUT) {
		if (exit_html_output(&dt, LIMIT_BY_FUNCS) < 0)
			return -1;
	}
	return 0;
}

int proc_logfile(char *logfile, int id, char *uname_r)
{
	int fd, rc = -1, mode;
	off_t size;
	char *dir, *fname;
	struct proc_each_rec_data data;

	if ((fd = u_open(logfile, &size)) < 0)
		goto EXIT;
	mode = output_summary || HTML_OUTPUT ? PM_SILENT : PM_VERBOSE;
	if (!id) {
		char *tmp = strdup(logfile);

		fname = basename(tmp);
		dir = dirname(tmp);
		if (is_exists_maps(dir, fname)) {
			if (parse_maps(&r2n_info, dir, fname,
				       PM_WITH_PT, mode) < 0)
				goto EXIT;
		}
		if (parse_modules(&r2n_info, dir, uname_r, PM_WITH_PT, mode) <0)
			goto EXIT;
 		r2n_info.from_is_next = chk_from_is_next(dir);
 
		if (mode == PM_VERBOSE)
			printf("start\n");
	}
	data.range = get_pid_addr_range(ALL_PID);
	data.last = UNKNOWN_BADDR;
	if (for_each_bt_record(fd, size, proc_each_record, &data) < 0)
		goto EXIT;
	rc = 0;
EXIT:
	u_close(fd);
	return rc;
}

void err_exit(void)
{
	exit(1);
}

void usage(void)
{
	fprintf(stderr, "bt_coverage %s\n", BT_COVERAGE_VER);
	fprintf(stderr, "    %s\n\n", COPYRIGHT);
	fprintf(stderr, "bt_coverage [-s] [-a top:end] [-d top:end] [--usr|--ker|--all] -f logfile[,...]\n");
	fprintf(stderr, "            [-I func[,...]] [-E func[,...]] [[-S src_dir] -o html_out_dir]\n");
	fprintf(stderr, "  -s: output coverage summary\n");
	fprintf(stderr, "  -a: add address range\n");
	fprintf(stderr, "  -d: delete address range\n");
	fprintf(stderr, "  --usr: alias of '-a 0:0xbfffffff'\n");
	fprintf(stderr, "  --ker: alias of '-a 0xc0000000:0xffffffff'\n");
	fprintf(stderr, "  --all: alias of '-a 0:0xffffffff'\n");
	fprintf(stderr, "  -f: logfile(s)\n");
	fprintf(stderr, "  -I: include function name(s)\n");
	fprintf(stderr, "  -E: exclude function name(s)\n");
	fprintf(stderr, "      This option excludes only included function by -I option.\n");
	fprintf(stderr, "  -S: source directory\n");
	fprintf(stderr, "  -o: html output directory\n");
	fprintf(stderr, "      -o and -S options cannot be used with -s option.\n");
}

int main(int argc, char *argv[])
{
	int i, opt_index, n_logfs = 0;
	char c, *logfs_top, *prev, *cur;
	char *uname_r = NULL;
	unsigned long begin, end;
	struct option long_options[] = {
		{"usr", no_argument, NULL, 0},
		{"ker", no_argument, NULL, 0},
		{"all", no_argument, NULL, 0},
	};

	if (alloc_pid_range(ALL_PID) < 0)
		err_exit();
	while ((c = getopt_long(argc, argv, "sa:d:f:I:E:S:o:u:v", long_options,
				&opt_index)) != -1) {
		switch (c) {
		case 0:
			switch (opt_index) {
			case 0:
				add_range(ALL_PID, 0, 0xbfffffff);
				break;
			case 1:
				add_range(ALL_PID, 0xc0000000, 0xffffffff);
				break;
			case 2:
				add_range(ALL_PID, 0, 0xffffffff);
				break;
			}
			break;
		case 's':
			output_summary = 1;
			break;
		case 'a':
		case 'd':
			if (!range2ulongs(optarg, &begin, &end))
				err_exit();
			if (c == 'a')
				add_range(ALL_PID, begin, end);
			else
				del_range(ALL_PID, begin, end);
			break;
		case 'f':
			logfs_top = optarg;
			prev = NULL;
			cur = strtok(logfs_top, ",");
			while (cur) {
				n_logfs++;
				if ((!prev && cur != logfs_top) ||
				    (prev && cur != prev + strlen(prev) + 1)) {
					usage();
					err_exit();
				}
				prev = cur;
				cur = strtok(NULL, ",");
			}
			break;
		case 'I':
			includes = optarg;
			break;
		case 'E':
			excludes = optarg;
			break;
		case 'S':
		case 'o':
			if (c == 'S')
				srcdir = optarg;
			else
				outdir = optarg;
			break;
		case 'u'://////// TEMPORARY CHANGE
			uname_r = optarg;
			break;
		case 'v':
			verbose = 1;
			break;
		default:
			usage();
			err_exit();
		}
	}
	if (optind < argc || !n_logfs || (output_summary && outdir) ||
	    (!includes && excludes)) {
		usage();
		err_exit();
	}
	if (outdir) {
		if (dir_chk_and_create(outdir, 1) < 0)
			err_exit();
	}
	r2n_info.all_r2n = xcalloc(r2n_info.r2n_max,
				   sizeof(struct range_to_name*));
	for (cur = logfs_top, i = 0; i < n_logfs; i++) {
		if (proc_logfile(cur, i, uname_r) < 0)
			err_exit();
		cur += strlen(cur) + 1;
	}
	if (includes) {
		if (create_filter_funcs(includes, &include_funcs) < 0)
			err_exit();
	}
	if (excludes) {
		if (create_filter_funcs(excludes, &exclude_funcs) < 0)
			err_exit();
	}
	if (verbose)	/* hidden option for debug */
		dump_r2n(&r2n_info);
	if (dump_coverage() < 0)
		err_exit();
	exit(0);
}
