/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_read.c - relayfs read program                                         */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2006                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/poll.h>
#include <fcntl.h>
#include <dirent.h>
#include <errno.h>
#include <linux/threads.h>
#include <sys/mman.h>
#include <getopt.h>
#include "bt.h"

#define	BT_READ_VER	BT_VER
#define	COPYRIGHT	"Copyright (c) Hitachi, Ltd. 2005-2006"

static int verbose;
static int do_stdio;
static int terminate;

static size_t subbuf_size = 0x400000;
static size_t n_subbufs = 8;

char* chk_directory(char *name, int create)
{
	DIR *d;

	if ((d = opendir(name))) {
		closedir(d);
		return name;
	}
	if (create) {
		if (mkdir(name, S_IRWXU) < 0) {
			fprintf(stderr, "mkdir failed.(%s)\n", strerror(errno));
			return NULL;
		}
		return name;
	} else {
		fprintf(stderr, "%s not exists.\n", name);
		return NULL;
	}
}

int onefile_copy(int in, int out)
{
	char buf[256], *p;
	ssize_t r, w;
	struct pollfd fds;

	/*
	dprintf("read start ...\n");
	*/
	for (;;) {
		fds.fd = in;
		fds.events = POLLIN;
		/* relayfs's poll return success only subbuffer change timing */
		r = poll(&fds, 1, 1);
		if (r < 0)
			continue;
		r = read(in, buf, sizeof(buf));
		if (r == 0) {
			if (do_stdio)
				break;
			continue;
		}
		if (r < 0) {
			if (errno == EINTR)
				continue;
			fprintf(stderr, "read failed.(%s)\n", strerror(errno));
			return -1;
		}
		p = buf;
		while (r) {
			w = write(out, p, r);
			if (w < 0) {
				fprintf(stderr, "write failed.(%s)\n",
					strerror(errno));
				return -1;
			}
			p += w;
			r -= w;
		}
	}
	/*
	dprintf("\tresult: pid=%lld, add=%lld, del=%lld\n",
		cnt_pid, cnt_add, cnt_del);
		*/
	return 0;
}

static int process_subbufs(char *basename, size_t produced, size_t consumed,
			   char *relay_buf, int out)
{
	size_t i, idx, start, end, ready = produced - consumed;
	char *p;
	unsigned padding;
	int len;

	if (ready > n_subbufs) {
		if (!terminate)
			fprintf(stderr, "Buffer full. " \
				"Use more large buffer size on relayfs.\n");
		start = produced % n_subbufs;
		end = start + n_subbufs;
	} else {
		start = consumed % n_subbufs;
		end = start + ready;
	}
	for (i = start; i < end; i++) {
		idx = i % n_subbufs;
		if (verbose)
			printf("(%s)proc #%d buf (p:%d, c:%d)\n",
			       basename + 3, idx, produced, consumed);
		p = relay_buf + idx * subbuf_size;
		padding = *(unsigned*)p;
		p += sizeof(padding);
		len = subbuf_size - sizeof(padding) - padding;
		/* for DEBUG
		printf("\tlen:%d, size:%d, padding:%d\n",
		       len, subbuf_size, padding);
		       */
		if (!len)
			break;
		if (write(out, p, len) < 0) {
			fprintf(stderr, "write failed.(%s)\n", strerror(errno));
			return -1;
		}
	}
	return produced - consumed;
}

int onefile_mmap_copy(int in, int out, char *basename)
{
	char buf[256], *relay_buf = MAP_FAILED;
	ssize_t total_bufsize = subbuf_size * n_subbufs;
	struct pollfd fds;
	int fd_produced = -1, fd_consumed = -1;
	int rc;
	size_t produced = 0, consumed = 0, tmp;

	/*
	dprintf("read start ...\n");
	*/
	sprintf(buf, "/proc/btrax/%s/produced", basename);
	fd_produced = open(buf, O_RDONLY);
	if (fd_produced < 0) {
		fprintf(stderr, "open failed.(%s)\n", strerror(errno));
		rc = -1;
		goto EXIT;
	}
	sprintf(buf, "/proc/btrax/%s/consumed", basename);
	fd_consumed = open(buf, O_RDWR);
	if (fd_consumed < 0) {
		fprintf(stderr, "open failed.(%s)\n", strerror(errno));
		rc = -1;
		goto EXIT;
	}
	relay_buf = mmap(NULL, total_bufsize, PROT_READ,
			 MAP_PRIVATE | MAP_POPULATE, in, 0);
	if (relay_buf == MAP_FAILED) {
		fprintf(stderr, "mmap failed.(%s)\n", strerror(errno));
		rc = -1;
		goto EXIT;
	}

	for (;;) {
		fds.fd = in;
		fds.events = POLLIN;
		/* relayfs's poll return success only subbuffer change timing */
		//if (poll(&fds, 1, 1000) < 0) {
		if (poll(&fds, 1, -1) < 0) {
			//printf("peE\n");
			if (errno != EINTR) {
				fprintf(stderr, "poll failed.(%s)\n",
					strerror(errno));
				rc = -1;
				//printf("EXIT1\n"); fflush(stdout);
				goto EXIT;
			}
			continue;
		}
		//printf("pe\n");
		lseek(fd_produced, 0, SEEK_SET);
		if (read(fd_produced, &produced, sizeof(produced)) < 0) {
			fprintf(stderr, "Couldn't read from produced file\n");
			rc = -1;
			//printf("EXIT2\n"); fflush(stdout);
			goto EXIT;
		}
		//printf(":%d:%d\n", produced, consumed);
		tmp = process_subbufs(basename,
				      produced, consumed, relay_buf, out);
		if (tmp < 0) {
			rc = -1;
			//printf("EXIT3\n"); fflush(stdout);
			goto EXIT;
		}
		consumed += tmp;
		if (write(fd_consumed, &tmp, sizeof(tmp)) < 0) {
			fprintf(stderr, "Couldn't write to consumed file(%s)\n",
				strerror(errno));
			rc = -1;
			//printf("EXIT4\n"); fflush(stdout);
			goto EXIT;
		}
		if (terminate)
			break;
	}
	rc = 0;
EXIT:
	if (fd_produced >= 0)
		close(fd_produced);
	if (fd_consumed >= 0)
		close(fd_consumed);
	if (relay_buf != MAP_FAILED)
		munmap(relay_buf, total_bufsize);
	/*
	dprintf("\tresult: pid=%lld, add=%lld, del=%lld\n",
		cnt_pid, cnt_add, cnt_del);
		*/
	return rc;
}

struct in_out_pair {
	pid_t	pid;
	int	in;
	int	out;
	char	basename[NAME_MAX+1];
};

static struct in_out_pair pair_tbl[NR_CPUS];

int do_read(char *in_dir, char *out_dir)
{
	DIR *dir;
	struct dirent* d;
	int max = 128;
	char path[max];
	struct stat st;
	struct in_out_pair *p, *p_max;
	int rc;

	if (do_stdio)
		return onefile_copy(0, 1);

	/* first, check in/out files and get file descriptors */
	if ((dir = opendir(in_dir)) == NULL)
		return -1;
	p = p_max = pair_tbl;
	rc = 0;
	while ((d = readdir(dir)) != NULL) {
		if (strcmp(".", d->d_name) == 0 || strcmp("..", d->d_name) == 0)
			continue;
		snprintf(path, max, "%s/%s", in_dir, d->d_name);
		if (lstat(path, &st) < 0) {
			fprintf(stderr, "%s lstat failed.(%s)\n", path,
				strerror(errno));
			rc = -1;
			break;
		}
		if (S_ISDIR(st.st_mode))
			continue;

		/* process cpuX file */
		strcpy(p->basename, d->d_name);
		if ((p->in = open(path, O_RDONLY | O_NONBLOCK)) < 0) {
			fprintf(stderr, "%s open failed.(%s)\n", path,
				strerror(errno));
			rc = -1;
			break;
		}
		snprintf(path, max, "%s/%s", out_dir, d->d_name);
		if ((p->out = open(path, O_CREAT|O_EXCL|O_RDWR,
				   S_IRUSR|S_IWUSR)) < 0) {
			fprintf(stderr, "%s open failed.(%s)\n", path,
				strerror(errno));
			rc = -1;
			break;
		}
		p++;
	}
	closedir(dir);
	if (rc < 0)
		return rc;

	/* fork each read processes */
	p_max = p;
	for (p = pair_tbl; p < p_max; p++) {
		switch ((p->pid = fork())) {
		case -1:
			fprintf(stderr, "fork failed.(%s)\n", strerror(errno));
			close(p->in);
			close(p->out);
			break;
		case 0:
			onefile_mmap_copy(p->in, p->out, p->basename);
			close(p->in);
			close(p->out);
			_exit(0);
			break;
		default:
			break;
		}
	}
	for (p = pair_tbl; p < p_max; p++) {
		if (p->pid > 0)
			waitpid(p->pid, NULL, 0);
	}
	return 0;
}

void err_exit(void)
{
	exit(1);
}

void usage(void)
{
	fprintf(stderr, "bt_read %s\n", BT_READ_VER);
	fprintf(stderr, "    %s\n\n", COPYRIGHT);
	fprintf(stderr, "bt_read [-T] -s subbuf_size -n n_subbufs -i dir -o dir\n");
	fprintf(stderr, "  -T: terminate after current produced buffer read\n");
	fprintf(stderr, "  -s: relayfs sub-buffer size\n");
	fprintf(stderr, "  -n: relayfs sub-buffer number\n");
	fprintf(stderr, "  -i: input directory\n");
	fprintf(stderr, "  -o: output directory\n");
}

int main(int argc, char* argv[])
{
	int do_not_read = 0;
	char c, *in_dir = NULL, *out_dir = NULL;

	while ((c = getopt(argc, argv, "Ts:n:i:o:vDS")) != -1) {
		switch (c) {
		case 'T':
			terminate = 1;
			break;
		case 's':
			subbuf_size = atoi(optarg);
			break;
		case 'n':
			n_subbufs = atoi(optarg);
			break;
		case 'i':
			if ((in_dir = chk_directory(optarg, 0)) == NULL)
				err_exit();
			break;
		case 'o':
			if ((out_dir = chk_directory(optarg, 0)) == NULL)
				err_exit();
			break;
		case 'v':
			verbose = 1;
			break;
		case 'D':
			do_not_read = 1;
			break;
		case 'S':
			do_stdio = 1;
			break;
		default:
			usage();
			err_exit();
		}
	}
	if (optind < argc ||
	    (!do_stdio && (in_dir == NULL || out_dir == NULL))) {
		usage();
		err_exit();
	}
	if (!do_not_read)
		do_read(in_dir, out_dir);
	exit(0);
}
