/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt.h - branch tracer for linux common header                             */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2008                         */
/*             Authors: Yumiko Sugita (yumiko.sugita.yf@hitachi.com),        */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#ifndef __BT_H__
#define __BT_H__

#ifdef __KERNEL__
#  include <linux/version.h>
#  include <linux/file.h>
#  include <linux/proc_fs.h>
#  if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,19)
#    include <linux/irq.h>
#  endif
#else
#  include <sys/types.h>
#  ifdef HAVE_CONFIG_H
#    include "config.h"
#  endif
#endif

#define MOD_NAME		"bt_mod"

#define BT_FLAG_T_START		(1 << 26)
#define BT_FLAG_T_STOP		(1 << 27)
#define BT_FLAG_WARN		(1 << 28)
#define BT_FLAG_DEBUG		(1 << 29)
#define BT_FLAG_COMM		(1 << 30)
#define BT_FLAG_PID		(1 << 31)
#define	BT_FLAG_PID_LL		((unsigned long long)BT_FLAG_PID << 32)
#define	is_bt_record(r)		\
	(!((r)->flags & (BT_FLAG_WARN|BT_FLAG_DEBUG|BT_FLAG_PID)))
#define	is_pid_record(r)	((r)->flags & BT_FLAG_PID)
#define	is_comm_record(r)	((r)->flags & BT_FLAG_COMM)
#define	is_warn_record(r)	((r)->flags & BT_FLAG_WARN)

#define BT_COMM_LEN		8

struct bt_record {
	unsigned long	from;
	unsigned long	to;
	unsigned long	flags;
} __attribute__((packed));

struct pid_record {
	unsigned long		pid;
	unsigned long long	clocks;
} __attribute__((packed));

struct comm_record {
	char		comm[BT_COMM_LEN];
	unsigned long	flags;
} __attribute__((packed));

struct warn_record {
	unsigned long	left;
	unsigned long	reserve;
	unsigned long	flags;
} __attribute__((packed));

struct tmr_record {
	unsigned long long	timestamp;
	unsigned long		flags;
} __attribute__((packed));

struct ds_manage {
	unsigned long	bts_base;
	unsigned long	bts_index;
	unsigned long	bts_max;
	unsigned long	bts_threshold;
	unsigned long	pebs_base;
	unsigned long	pebs_index;
	unsigned long	pebs_max;
	unsigned long	pebs_threshold;
	unsigned long	pebs_cnt_reset;
	char		reserved[24];
};

struct pid_manage {
	pid_t			pid;
	char			comm[BT_COMM_LEN];
	int			is_wrote;

	/* rdtsc value for checking the syscall execution time */
	long			n_syscall;
	unsigned long long	syscall_start;
	unsigned long long	syscall_end;
};

struct info_per_cpu {
	struct ds_manage	*ds_manage;
	struct pid_manage	pid_manage;
	struct proc_dir_entry	*p_cpuN;
	struct proc_dir_entry	*p_on_off_cnt;
	struct proc_dir_entry	*p_produced;
	struct proc_dir_entry	*p_consumed;
	unsigned long		on_off_cnt;
};

#ifdef __KERNEL__
#include <asm/types.h>

/* MSRs */
#ifndef MSR_IA32_DS_AREA
#  define MSR_IA32_DS_AREA	0x600
#endif
#define MSR_DEBUGCTL		0x1d9
struct debugctl_bits {
	int	lbr;
	int	tr;
	int	bts;
	int	btint;
};
#define MSR_DEBUGCTL_P4_BITS	{ 1<<0, 1<<2, 1<<3, 1<<4 }
#define MSR_DEBUGCTL_PM_BITS	{ 1<<0, 1<<6, 1<<7, 1<<8 }

#define	BTS_BUF_MIN_SIZE	(sizeof(struct bt_record) * 64 * 1024)
#define	MIN_INT_MARGIN_RECS	(1 * 1024)
#define	DEFAULT_INT_MARGIN_RECS	(8 * 1024)

/* chk_syscall_time parameter value */
#define CHK_SCTIME_OFF			0
#define CHK_SCTIME_ON			1
#define CHK_SCTIME_ON_WITHOUT_TRACE	2

void bt_enable_per_cpu(void *data);
void bt_disable_per_cpu(void *data);
void bt_enable(void);
void bt_disable(void);

#define is_both(mode)		((mode) == 0)
#define is_start(mode)		((mode) == 1)
#define is_stop(mode)		((mode) == 2)
#define is_fr(mode)  		((mode) == 3)
#define is_syscall_pid(mode) 	((mode) == 4)
#define is_upid(mode)  		((mode) == 5)

#define is_user(mode) 		((mode) == 5)
#define is_kern(mode)		((mode) < 5)

#define is_kern_pid_by_hook(mode)	(is_both(mode) || is_start(mode))
#define is_kern_all_by_hook(mode)	(is_stop(mode) || is_fr(mode))
#define is_syscall(mode)	is_syscall_pid(mode)
#define is_enable_by_proc(mode)	(is_syscall(mode) || is_upid(mode))

static inline void task_list_lock(void)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,18)
	rcu_read_lock();
#else
	read_lock_irq(&tasklist_lock);
#endif
}

static inline void task_list_unlock(void)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,18)
	rcu_read_unlock();
#else
	read_unlock_irq(&tasklist_lock);
#endif
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,18)
typedef struct irq_chip bt_int_t;
#define	bt_handler chip
#else
typedef struct hw_interrupt_type bt_int_t;
#define	bt_handler handler
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,18)
#define	bt_d_child d_u.d_child
#else
#define	bt_d_child d_child
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,19)
typedef unsigned long (*kallsyms_lookup_name_t)(const char *name);
typedef void (*set_irq_chip_and_handler_t)(unsigned int irq,
					   struct irq_chip *chip,
					   irq_flow_handler_t handle);
typedef void (*handle_irq_t)(unsigned int irq, struct irq_desc *desc);
#endif

static inline void chk_procs_using_relfs(struct dentry *d,
					 pid_t **pp_pid, pid_t *p_max)
{
	struct task_struct *p;
	struct files_struct *files;
	struct file *file = NULL;
	unsigned int i;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,14)
	struct fdtable *fdt;
#endif

	task_list_lock();
	for_each_process(p) {
		files = p->files;
		if (!files)
			continue;
		spin_lock(&files->file_lock);
		//serial_prints("chk-pid(%d)\n", p->pid);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,14)
		fdt = files_fdtable(files);
		for (i = 0; i < fdt->max_fds; i++) {
			file = fdt->fd[i];
#else
		for (i = 0; i < files->max_fds; i++) {
			file = files->fd[i];
#endif
			if (file && file->f_dentry == d) {
				*(*pp_pid)++ = p->pid;
				if (*pp_pid >= p_max) {
					spin_unlock(&files->file_lock);
					goto EXIT;
				}
				break;
			}
		}
		spin_unlock(&files->file_lock);
	} while (p != &init_task);
EXIT:
	task_list_unlock();
}

/* for serial console prints */
int serial_init(int);
asmlinkage int serial_prints(const char *, ...);

#else

//#define MAX_LINE_LEN	256
#define MAX_LINE_LEN	4096

#endif /* __KERNEL__ */

#endif /*__BT_H__*/
