/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_irq.c - branch trace module (interrupt handler)                       */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2008                         */
/*             Authors: Yumiko Sugita (yumiko.sugita.yf@hitachi.com),        */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <linux/interrupt.h>
#include <linux/irq.h>
#include <asm/apic.h>
#include "bt.h"

extern struct debugctl_bits ctl_bits;
extern DEFINE_PER_CPU(struct info_per_cpu, bt_info_per_cpu);
extern unsigned long irq_desc_p;
extern unsigned long no_irq_type_p;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,19)
extern set_irq_chip_and_handler_t set_irq_chip_and_handler_f;
extern handle_irq_t handle_edge_irq_f;
#endif

static char *irq_name = "bts";
static int pmi_irq = -1;

void bts_log_write(void);

static void do_nothing(unsigned int irq)
{
}

static unsigned int startup_pmi(unsigned int irq)
{
	return 0;
}

static void set_affinity_pmi(unsigned int irq, cpumask_t dest) {
}

static void irq_mask_clear(void) {
	unsigned long dt;

	dt = apic_read(APIC_LVTPC);
	dt &= ~APIC_LVT_MASKED;
	apic_write(APIC_LVTPC, dt);
}

static void ack(unsigned int irq)
{
	int cpu;
	unsigned int low, high;
	irq_desc_t *desc;
	struct info_per_cpu *info;

	desc = (irq_desc_t*)irq_desc_p;
	spin_unlock(&desc[pmi_irq].lock);

	/* for DEBUG
	if (!irqs_disabled())
		serial_prints("IF is 1 on BTS ISR entry\n");
		*/
	ack_APIC_irq();

	cpu = smp_processor_id();

	info = &per_cpu(bt_info_per_cpu, cpu);
	if (!info->ds_manage) {
		spin_lock(&desc[pmi_irq].lock);
		return;
	}
	rdmsr(MSR_DEBUGCTL, low, high);
	wrmsr(MSR_DEBUGCTL, low & ~ctl_bits.tr, high);

	/* for DEBUG
	{
		irq_desc_t *desc = (irq_desc_t*)irq_desc_p;
		serial_prints("ack(cpu:%d) (pend:%d)\n", smp_processor_id(),
			      desc[pmi_irq].status & IRQ_PENDING);
	}
	*/
	bts_log_write();
	irq_mask_clear();
	spin_lock(&desc[pmi_irq].lock);
	wrmsr(MSR_DEBUGCTL, low, high);
}

static void end(unsigned int irq)
{
	/* for DEBUG
	irq_desc_t *desc = (irq_desc_t*)irq_desc_p;
	serial_prints("end(cpu:%d) (pend:%d)\n", smp_processor_id(),
		      desc[pmi_irq].status & IRQ_PENDING);
		      */
}

static struct hw_interrupt_type	bt_int_type = {
	.typename = "BTRAX",
	.startup = startup_pmi,
	.shutdown = do_nothing,
	.enable = do_nothing,
	.disable = do_nothing,
	.ack = ack,
	.end = end,
	.set_affinity = set_affinity_pmi
};

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,19)
irqreturn_t bt_interrupt(int irq, void *dev_id)
#else
irqreturn_t bt_interrupt(int irq, void *dev_id, struct pt_regs *regs)
#endif
{
	return IRQ_HANDLED;
}

/* Here are the macros for the old kernel. */
#ifndef IRQF_DISABLED
#  define IRQF_DISABLED		SA_INTERRUPT
#endif
#ifndef IRQF_SHARED
#  define IRQF_SHARED		SA_SHIRQ
#endif
static int init_isr_per_system(void)
{
	int irq;
	int rc;
	irq_desc_t *desc = (irq_desc_t*)irq_desc_p;
	bt_int_t *type = (bt_int_t*)no_irq_type_p;

	for (irq = 95; irq > 15; irq--) {
		if (desc[irq].bt_handler == type)
			break;
	}
	if (irq == 15)
		return -EBUSY;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,19)
	set_irq_chip_and_handler_f(irq, &bt_int_type, handle_edge_irq_f);
#else
	desc[irq].bt_handler = &bt_int_type;
#endif
	rc = request_irq(irq, bt_interrupt, IRQF_DISABLED | IRQF_SHARED,
			 irq_name, irq_name);
	if (rc) {
		printk("%s: cannot setup irq %d for BTS\n", MOD_NAME, irq);
		desc[irq].bt_handler = type;
		return rc;
	}
	pmi_irq = irq;

	return 0;
}

static void init_isr_per_cpu(void* data)
{
	int vector;
	unsigned long dt, current_vector;
	int cpu;
	cpumask_t *failed_map = data;

	cpu = smp_processor_id();
	vector = pmi_irq + FIRST_EXTERNAL_VECTOR;

#if 1
	dt = apic_read(APIC_LVTPC);
	if (dt & APIC_DM_SMI) {
		printk("%s: (%d)BTS facility handled by SMI\n", MOD_NAME, cpu);
		return;
	}
#endif
	current_vector = dt & APIC_VECTOR_MASK;
	/* check whether a vector already exists, temporarily masked? */	
	if (current_vector != 0 && current_vector != APIC_VECTOR_MASK) {
		printk("%s: (%d)BTS LVT vector (0x%016lx) already installed\n",
		       MOD_NAME, cpu, (dt & APIC_VECTOR_MASK));
		return;
	}
	apic_write(APIC_LVTPC, vector | APIC_DM_FIXED | APIC_LVT_MASKED);

	dt = apic_read(APIC_LVTPC);
	apic_write_around(APIC_LVTPC, dt & ~APIC_LVT_MASKED);
	serial_prints("CPU%d: BTS overflow interrupt enabled\n", cpu);
	serial_prints("\t(0x%016lx)\n", apic_read(APIC_LVTPC));
	cpu_clear(cpu, *failed_map);
}

static void cleanup_isr_per_system(void)
{
	irq_desc_t *desc = (irq_desc_t*)irq_desc_p;
	bt_int_t *type = (bt_int_t*)no_irq_type_p;

	free_irq(pmi_irq, irq_name);
	desc[pmi_irq].bt_handler = type;

	return;
}

static void cleanup_isr_per_cpu(void *data)
{
	unsigned long dt;
	int cpu;

	cpu = smp_processor_id();
	dt = apic_read(APIC_LVTPC);
	if ((dt & APIC_VECTOR_MASK) != pmi_irq + FIRST_EXTERNAL_VECTOR)
		return;

	apic_write(APIC_LVTPC, APIC_LVT_MASKED | APIC_VECTOR_MASK);
	serial_prints("CPU:%d LVTPC cleanup(0x%016lx) exit\n", cpu,
		      apic_read(APIC_LVTPC));

	return;
}

int setup_isr(void)
{
	int rc;
	cpumask_t failed_map = cpu_online_map;

	if ((rc = init_isr_per_system()) < 0)
		return rc;
	on_each_cpu(init_isr_per_cpu, &failed_map, 1, 1);
	if (!cpus_empty(failed_map))
		return -EBUSY;
	return 0;
}

void cleanup_isr(void)
{
	if (pmi_irq < 0)
		return;
	cleanup_isr_per_system();
	on_each_cpu(cleanup_isr_per_cpu, NULL, 1, 1);
	pmi_irq = -1;
}
