/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_relfs.c - branch trace module (proc file system interface)            */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2008                         */
/*             Authors: Yumiko Sugita (yumiko.sugita.yf@hitachi.com),        */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <asm/apic.h>
#include <asm/msr.h>
#include <asm/processor.h>
#include <asm/uaccess.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/smp.h>
#include <linux/vmalloc.h>
#include <linux/mm.h>
#include <asm/unistd.h>
#include "bt.h"

struct proc_dir_entry *proc_btrax;
static struct proc_dir_entry *proc_enable;
static struct proc_dir_entry *proc_pid;
static struct proc_dir_entry *proc_filter_syscall;
static struct proc_dir_entry *proc_exec_syscall;
static struct proc_dir_entry *proc_btsbuf_sz;
static struct proc_dir_entry *proc_bts_mrecs;
static struct proc_dir_entry *proc_irq_addrs;
extern DEFINE_PER_CPU(struct info_per_cpu, bt_info_per_cpu);
extern cpumask_t enable;
extern unsigned long idt_table_p;
extern unsigned long *pid_tbl;
extern unsigned long *syscall_pid_tbl;
extern unsigned long syscall_exe_tbl[];
extern unsigned long syscall_filter_tbl[];
extern int pid_max;
extern int mode;
extern size_t btsbuf_size;
extern size_t bts_int_margin_recs;

#define BT_COLLECT_NAME	"bt_collect_log"
extern pid_t bt_collect_pid;

void bt_enable(void);
void bt_disable(void);
void add_pid_tbl(pid_t, unsigned long*);
void remove_pid_tbl(pid_t, unsigned long*);
void add_syscall_tbl(int, unsigned long*);

static int proc_irq_addrs_read(char *buf, char **start, off_t off,
			       int count, int *eof, void *data)
{
	int i, len, max = 10;
	off_t loff;
	char lbuf[max + 1], *p;
	struct desc_struct *idt_table = (struct desc_struct*)idt_table_p;

	loff = 0;
	*start = p = buf;
	
	for (i = 0; i < 256 + 1; i++) {
		if (i == 256) {
			unsigned long low, high;

			rdmsr(MSR_IA32_SYSENTER_EIP, low, high);
			len = snprintf(lbuf, max, "%08lx\n", low);
		} else
			len = snprintf(lbuf, max, "%08lx\n",
				       (idt_table[i].a & 0x0000ffff) |
				       (idt_table[i].b & 0xffff0000));
		if (len < 0 || len >= max)
			return -EFAULT;
		if (loff >= off && loff + len <= off + count) {
			memcpy(p, lbuf, len);
			p += len;
		} else if (loff + len > off + count)
			break;
		loff += len;
	}
	return p - buf;
}

static int proc_read_bitmap(char *buf, char **start, off_t off,
			    int count, int *eof, void *data,
			    unsigned long *tbl, int n_max)
{
	int len, max = 10;
	int n, n_max_in_tbl = 0;
	off_t loff;
	char *fmt, lbuf[max + 1], *p;

	loff = 0;
	*start = p = buf;
	
	/* first, check n_max_in_tbl */
	n_max_in_tbl = 0;
	for (n = -1;;) {
		if (n < 0)
			n = find_first_bit(tbl, n_max);
		else
			n = find_next_bit(tbl, n_max, n + 1);
		if (n >= n_max)
			break;
		n_max_in_tbl = n;
	}

	for (n = -1;;) {
		if (n < 0)
			n = find_first_bit(tbl, n_max);
		else
			n = find_next_bit(tbl, n_max, n + 1);
		if (n >= n_max)
			break;
		fmt = (n == n_max_in_tbl) ? "%d\n" : "%d,";
		len = snprintf(lbuf, max, fmt, n);
		if (len < 0 || len >= max)
			return -EFAULT;
		if (loff >= off && loff + len <= off + count) {
			memcpy(p, lbuf, len);
			p += len;
		} else if (loff + len > off + count)
			break;
		loff += len;
	}
	return p - buf;
}

typedef void (*func_write_bitmap)(int val, unsigned long *tbl);

static int __proc_write_bitmap(char *buf, unsigned long len,
			       unsigned long *consumed, unsigned long *tbl,
			       func_write_bitmap f)
{
	int val;
	char *p, *p_max, *p_end;

	p_max = buf + len;
	for (p = buf; p < p_max; p = p_end + 1) {
		val = simple_strtol(p, &p_end, 0);
		switch (*p_end) {
		case ',':
			f(val, tbl);
			break;
		case '\n':
			if (p_max != p_end + 1)
				return -EINVAL;
			f(val, tbl);
			*consumed = p_max - buf;
			return 1;
		case '\0':
			if (p_max != p_end)
				return -EINVAL;
			*consumed = p - buf;
			return 0;
			break;
		}
	}
	*consumed = p - buf;
	return 0;
}

#define WRITE_BUF_SIZE	64

/* The data length ('count' variable) might become shorter than the string
 * length of one element. At this time, it is necessary to process the string
 * because it terminates abnormally when returns 0.
 * Therefore, it is necessary to preserve it in the static variable buf.
 */
static int in_bitmap_write;
static int proc_write_bitmap(struct file *file, const char *user_buf,
			     unsigned long count, void *data,
			     unsigned long *tbl, int n_max,
			     func_write_bitmap f)
{
	unsigned long max = WRITE_BUF_SIZE, len, consumed;
	static unsigned long left;
	static char buf[WRITE_BUF_SIZE + 1];
	char *p, *p_max;
	int rc;

	if (!in_bitmap_write) {
		bitmap_zero(tbl, n_max);
		left = 0;
	}
	in_bitmap_write = 1;

	for (p = (char*)user_buf, p_max = (char*)user_buf + count;
	     p < p_max; p += len) {
		len = max - left;
		if (len > p_max - p)
			len = p_max - p;
		if (copy_from_user(buf + left, p, len))
			return -EFAULT;
		buf[left + len] = '\0';
		rc = __proc_write_bitmap(buf, left + len, &consumed, tbl, f);
		if (rc < 0)
			return rc;
		//printk("BUF:%s:%ld:%d:%ld\n", buf, left + len, rc, consumed);
		if (rc)	/* finished ? */
			in_bitmap_write = 0;
		if (consumed) {
			len = consumed - left;
			left = 0;
		} else {
			left = len;
		}
	}
	return p - user_buf;
}

static int proc_pid_read(char *buf, char **start, off_t off,
			 int count, int *eof, void *data)
{
	return proc_read_bitmap(buf, start, off, count, eof, data,
				pid_tbl, pid_max);
}

static int proc_pid_write(struct file *file, const char *user_buf,
			  unsigned long count, void *data)
{
	if (!is_enable_by_proc(mode) || !cpus_empty(enable))
		return -EFAULT;
	return proc_write_bitmap(file, user_buf, count, data,
				 pid_tbl, pid_max, add_pid_tbl);
}

static int proc_exec_syscall_read(char *buf, char **start, off_t off,
				  int count, int *eof, void *data)
{
	return proc_read_bitmap(buf, start, off, count, eof, data,
				syscall_exe_tbl, NR_syscalls);
}

static int proc_filter_syscall_read(char *buf, char **start, off_t off,
				    int count, int *eof, void *data)
{
	return proc_read_bitmap(buf, start, off, count, eof, data,
				syscall_filter_tbl, NR_syscalls);
}

static int proc_filter_syscall_write(struct file *file, const char *user_buf,
				     unsigned long count, void *data)
{
	if (!is_syscall(mode) || !cpus_empty(enable))
		return -EFAULT;
	return proc_write_bitmap(file, user_buf, count, data,
				 syscall_filter_tbl, NR_syscalls,
				 add_syscall_tbl);
}

#if 0
static void otameshi_print(void)
{
	struct task_struct *t;
	struct vm_area_struct *v;

	serial_prints("------ otameshi print all maps (from) ------\n");
	task_list_lock();
	t = &init_task;
	for (t = next_task(t); t != &init_task; t = next_task(t)) {
		if (!t->mm)
			continue;
		serial_prints("(%d)\n", t->pid);
		for (v = t->mm->mmap; v; v = v->vm_next) {
			if (v->vm_flags & VM_EXEC)
				serial_prints("%08lx-%08lx\n",
					      v->vm_start, v->vm_end);
		}
	}
	task_list_unlock();
	serial_prints("------ otameshi print all maps (to) ------\n");
}
#endif

static int proc_enable_read(char *buf, char **start, off_t off,
			    int count, int *eof, void *data)
{
	int n;

	n = snprintf(buf, count, "%d\n", !cpus_empty(enable));

	return n;
}

static int proc_enable_write(struct file *file, const char *user_buf,
			     unsigned long count, void *data)
{
	char *p_end;
	long val;
	char buf[16];
	unsigned long size;

	size = (sizeof(buf) < count) ? sizeof(buf) : count;

	if (copy_from_user(buf, user_buf, size))
		return -EFAULT;

	buf[size-1] = '\0';
	val = simple_strtol(buf, &p_end, 0);
	if (*p_end != '\0')
		return -EINVAL;

	//serial_prints("BTRAX:enable write(%ld)\n", val);
	if (val) {
		//otameshi_print();
		bt_enable();
	} else
		bt_disable();
	return count;
}

static int proc_btsbuf_sz_read(char *buffer, char **start, off_t off,
			       int count, int *eof, void *data)
{
	return sprintf(buffer, "%d\n", btsbuf_size);
}

static int proc_bts_mrecs_read(char *buffer, char **start, off_t off,
			       int count, int *eof, void *data)
{
	return sprintf(buffer, "%d\n", bts_int_margin_recs);
}

static int proc_on_off_cnt_read(char *buf, char **start, off_t off,
				int count, int *eof, void *data)
{
	int n, cpu;
	struct info_per_cpu *info;

	cpu = (int)data;
	info = &per_cpu(bt_info_per_cpu, cpu);
	n = snprintf(buf, count, "%ld\n", info->on_off_cnt);

	return n;
}

int proc_init(void)
{
	int i, bytes;
	char tmp[3 + 4 + 1]; /* cpuXXXX */
	struct info_per_cpu *info;

	proc_btrax = proc_mkdir("btrax", &proc_root);
	if (!proc_btrax)
		return -ENOMEM;

	bytes = BITS_TO_LONGS(pid_max) * (sizeof(long) / sizeof(char));
	pid_tbl = vmalloc(bytes);
	if (!pid_tbl)
		return -ENOMEM;
	bitmap_zero(pid_tbl, pid_max);
	syscall_pid_tbl = vmalloc(bytes);
	if (!syscall_pid_tbl)
		return -ENOMEM;
	bitmap_zero(syscall_pid_tbl, pid_max);

	proc_pid = create_proc_entry("pid", 0600, proc_btrax);
	if (!proc_pid)
		return -ENOMEM;
	proc_pid->read_proc = proc_pid_read;
	proc_pid->write_proc = proc_pid_write;

	bitmap_fill(syscall_filter_tbl, NR_syscalls);
	proc_filter_syscall = create_proc_entry("filter_syscall", 0400,
						proc_btrax);
	if (!proc_filter_syscall)
		return -ENOMEM;
	proc_filter_syscall->read_proc = proc_filter_syscall_read;
	proc_filter_syscall->write_proc = proc_filter_syscall_write;

	proc_exec_syscall = create_proc_entry("exec_syscall", 0400, proc_btrax);
	if (!proc_exec_syscall)
		return -ENOMEM;
	proc_exec_syscall->read_proc = proc_exec_syscall_read;

	proc_enable = create_proc_entry("enable", 0600, proc_btrax);
	if (!proc_enable)
		return -ENOMEM;
	proc_enable->read_proc = proc_enable_read;
	proc_enable->write_proc = proc_enable_write;

	proc_btsbuf_sz = create_proc_entry("btsbuf_size", 0400, proc_btrax);
	if (!proc_btsbuf_sz)
		return -ENOMEM;
	proc_btsbuf_sz->read_proc = proc_btsbuf_sz_read;

	proc_bts_mrecs = create_proc_entry("bts_mrecs", 0400, proc_btrax);
	if (!proc_bts_mrecs)
		return -ENOMEM;
	proc_bts_mrecs->read_proc = proc_bts_mrecs_read;

	proc_irq_addrs = create_proc_entry("irq_addrs", 0400, proc_btrax);
	if (!proc_irq_addrs)
		return -ENOMEM;
	proc_irq_addrs->read_proc = proc_irq_addrs_read;

	for_each_online_cpu(i) {
		info = &per_cpu(bt_info_per_cpu, i);
		snprintf(tmp, sizeof(tmp), "cpu%d", i);
		info->p_cpuN = proc_mkdir(tmp, proc_btrax);
		if (!info->p_cpuN)
			return -ENOMEM;
		info->p_on_off_cnt = create_proc_entry("on_off_cnt", 0400,
						       info->p_cpuN);
		if (!info->p_on_off_cnt)
			return -ENOMEM;
		info->p_on_off_cnt->data = (void*)i;
		info->p_on_off_cnt->read_proc = proc_on_off_cnt_read;
	}
	return 0;
}

void proc_cleanup(void)
{
	int i;
	struct info_per_cpu *info;

	for_each_online_cpu(i) {
		info = &per_cpu(bt_info_per_cpu, i);
		if (info->p_on_off_cnt)
			remove_proc_entry(info->p_on_off_cnt->name,
					  info->p_cpuN);
		if (info->p_cpuN)
			remove_proc_entry(info->p_cpuN->name, proc_btrax);
	}
	if (proc_enable)
		remove_proc_entry(proc_enable->name, proc_btrax);
	if (pid_tbl)
		vfree(pid_tbl);
	if (proc_pid)
		remove_proc_entry(proc_pid->name, proc_btrax);
	if (proc_filter_syscall)
		remove_proc_entry(proc_filter_syscall->name, proc_btrax);
	if (proc_exec_syscall)
		remove_proc_entry(proc_exec_syscall->name, proc_btrax);
	if (proc_btsbuf_sz)
		remove_proc_entry(proc_btsbuf_sz->name, proc_btrax);
	if (proc_bts_mrecs)
		remove_proc_entry(proc_bts_mrecs->name, proc_btrax);
	if (proc_irq_addrs)
		remove_proc_entry(proc_irq_addrs->name, proc_btrax);
	if (proc_btrax)
		remove_proc_entry(proc_btrax->name, &proc_root);
}

