/*
 * trace_probe.c -- djprobe simple kernel tracer
 * Copyright (c) 2005 Hitachi,Ltd.,
 * Created by Masami Hiramatsu<hiramatu@sdl.hitachi.co.jp>
 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */
#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include "djprobe.h"
#include <linux/seq_file.h>
#include <linux/vmalloc.h>
#include <linux/slab.h>
#include <linux/smp.h>
#include <asm/uaccess.h>
#include <linux/proc_fs.h>

struct trace_log {
	int cpu;
	void *addr;
	unsigned long long tsc;
	pid_t pid;
};

#define BUFFER_SIZE 1024
static long bsize=BUFFER_SIZE;
module_param(bsize, long, 0444);

struct trace_buffer {
	struct trace_log *buf;
	int index;
};

static struct trace_buffer percpu_buffer[NR_CPUS];

/* iterator */
static void *log_seq_start(struct seq_file *m, loff_t *pos)
{
	long n = (long)*pos;
	int ncpu = num_online_cpus();
	int cpu, i;

	if (bsize == 0 || n >= bsize*ncpu) return NULL;
	i = n/bsize;
	for_each_online_cpu(cpu) {
		if (0 == i--) break;
	}
	return &percpu_buffer[cpu].buf[n%bsize];
}

static void *log_seq_next(struct seq_file *m, void *v, loff_t *pos)
{
	long n;
	int cpu, i;
	int ncpu = num_online_cpus();

	(*pos)++;
	n = (long)*pos;

	if (n >= bsize*ncpu) return NULL;
	i = n/bsize;
	for_each_online_cpu(cpu) {
		if (0 == i--) break;
	}
	return &percpu_buffer[cpu].buf[n%bsize];
}

static void log_seq_stop(struct seq_file *m, void *v)
{
	//
}

#define LOG_ENTRY_SIZE (3+1+5+1+10+1+20+1)
static int log_seq_show(struct seq_file *m, void *v)
{
	struct trace_log *entry = v;
	
	if (entry == NULL) return 0;
	if ((void*)entry <= (void*)0xc0000000) {
		printk("error! %p\n", entry);
		return -1;
	}
	seq_printf(m, "% 3d,% 5d,0x%p,% 20llu\n",
		   entry->cpu, entry->pid, entry->addr, entry->tsc);
	return 0;
}

static struct seq_operations log_seq_op = {
	.start  = log_seq_start,
	.next   = log_seq_next,
	.stop   = log_seq_stop,
	.show   = log_seq_show
};

static int log_open(struct inode *inode, struct file *file)
{
	return seq_open(file, &log_seq_op);
}

static struct file_operations proc_log_operations = {
	.owner          = THIS_MODULE,
	.open           = log_open,
	.read           = seq_read,
	.llseek         = seq_lseek,
	.release        = seq_release,
};
static struct proc_dir_entry *log_proc_ent = NULL;

static inline void log_entry(void *addr, pid_t pid, unsigned long long tsc)
{
	struct trace_log *entry;
	int idx;
	idx = percpu_buffer[smp_processor_id()].index++;
	idx %= bsize;
	get_cpu();
	entry = &percpu_buffer[smp_processor_id()].buf[idx];
	entry->cpu = smp_processor_id();
	entry->addr = addr;
	entry->tsc = tsc;
	entry->pid = pid;
	put_cpu();
}

static void cleanup_buffers(void)
{
	int cpu;
	for_each_online_cpu(cpu) {
		if (percpu_buffer[cpu].buf)
			vfree(percpu_buffer[cpu].buf);
	}
}

static int init_buffers(void)
{
	int cpu;
	for_each_online_cpu(cpu) {
		percpu_buffer[cpu].buf = 
			vmalloc(sizeof(struct trace_log)*bsize);
		if (percpu_buffer[cpu].buf == NULL) goto failed;
		percpu_buffer[cpu].index = 0;
		percpu_buffer[cpu].buf[bsize-1].cpu = 255;
	}
	return 0;
       failed:
	cleanup_buffers();
	return -ENOMEM;
}

static void trace_probe(struct djprobe *djp, struct pt_regs *regs)
{
	unsigned long long tsc;
	rdtscll(tsc);
	log_entry(djp->inst->kp.addr, current->pid, tsc);
	return ;
}

struct trace_probe {
	struct djprobe djp;
	void *addr;
	int size;
	struct list_head list;
};

static LIST_HEAD(probe_list);
static int nr_probes = 0;
#ifndef DEFINE_SPINLOCK
#define DEFINE_SPINLOCK(x) spinlock_t x = SPIN_LOCK_UNLOCKED
#endif
static DEFINE_SPINLOCK(probe_lock);
#define PROC_PROBE_SIZE 12
static struct proc_dir_entry *probe_proc_ent = NULL;

static int add_probe(void *addr, int size)
{
	int ret;
	struct trace_probe *probe;
	if (addr < (void *)0xc0000000 || size < 5 || size > 16+5-1) {
		printk("simple tracer: probe param error: %p, %d\n",
		       addr, size);
		return -EINVAL;
	}
	spin_lock(&probe_lock);
	probe = kcalloc(1, sizeof(struct trace_probe), GFP_KERNEL);
	if (probe == NULL) {
		ret = -ENOMEM;
		goto out;
	}
	probe->djp.handler = trace_probe;
	probe->djp.addr = addr;
	probe->djp.size = size;
	ret = register_djprobe(&(probe->djp));
//	ret = register_djprobe(&(probe->djp), addr, size);
	if (ret != 0) {
		printk("simple tracer: probe install error: %d\n", ret);
		kfree(probe);
		goto out;
	}
	probe->addr = addr;
	probe->size = size;
	INIT_LIST_HEAD(&probe->list);
	list_add_tail(&probe->list, &probe_list);
	nr_probes++;
	probe_proc_ent->size = (nr_probes*PROC_PROBE_SIZE);

	printk("simple tracer: probe installed to %p, size %d\n",
	       (void*)addr, size);
	
       out:
	spin_unlock(&probe_lock);
	return ret;
}

static inline void __del_probe(struct trace_probe *probe) 
{
	unregister_djprobe(&probe->djp);
	printk("simple tracer: probe uninstalled from %p\n", probe->addr);
	nr_probes--;
	list_del(&probe->list);
	kfree(probe);
}

static int del_probe(void *addr)
{
	struct trace_probe *probe;
	spin_lock(&probe_lock);
	list_for_each_entry(probe, &probe_list, list){
		if (probe->addr == addr) {
			__del_probe(probe);
			spin_unlock(&probe_lock);
			return 0;
		}
	}
	spin_unlock(&probe_lock);
	printk("simple tracer: probe delete error: Not found %p\n", addr);
	return -EINVAL;
}

static void cleanup_probes(void)
{
	struct trace_probe *probe;
	spin_lock(&probe_lock);
	while(!list_empty(&probe_list)){
		probe = list_entry(probe_list.next, struct trace_probe, list);
		__del_probe(probe);
	}
	spin_unlock(&probe_lock);
	flush_scheduled_work();
}

static DECLARE_MUTEX(proc_mutex);

/* iterator */
static void *probe_seq_start(struct seq_file *m, loff_t *pos)
{
	struct trace_probe *probe;
	loff_t n = *pos;

	if (down_interruptible(&proc_mutex) != 0) 
		return NULL;

	if (nr_probes == 0 || n >= nr_probes) return NULL;
	list_for_each_entry(probe, &probe_list, list){
		if (0 == n--) break;
	}
	return probe;
}

static void *probe_seq_next(struct seq_file *m, void *v, loff_t *pos)
{
	struct trace_probe *probe = v;
	
	(*pos)++;
	if (probe->list.next == &probe_list) {
		probe = NULL;
	} else {
		probe = list_entry(probe->list.next, struct trace_probe, list);
	}
	return probe;
}

static void probe_seq_stop(struct seq_file *m, void *v)
{
	up(&proc_mutex);
}

static int probe_seq_show(struct seq_file *m, void *v)
{
	struct trace_probe *probe = v;
	
	if (probe == NULL) return 0;
	seq_printf(m, "0x%p %-2d\n", probe->addr, probe->size);
	return 0;
}

static struct seq_operations probe_seq_op = {
	.start  = probe_seq_start,
	.next   = probe_seq_next,
	.stop   = probe_seq_stop,
	.show   = probe_seq_show
};

static int probe_open(struct inode *inode, struct file *file)
{
	return seq_open(file, &probe_seq_op);
}

#define MAX_PROBE_WRITE 20

ssize_t probe_write(struct file *file, const char __user *buffer,
		    size_t count, loff_t *ppos)
{
	char kbuf[MAX_PROBE_WRITE+1], *tmp, *tmp2;
	unsigned long addr;
	int size, ret;

	ret = down_interruptible(&proc_mutex);
	if (ret) return -EINTR;

	if (count > MAX_PROBE_WRITE) {
		ret = -EINVAL;
		goto out;
	}
	if (copy_from_user(&kbuf, buffer, count)) {
		ret = -EFAULT;
		goto out;
	}
	kbuf[MAX_PROBE_WRITE] = '\0';

	ret = -EINVAL;
	tmp = strchr(kbuf, ' ');
	if (!tmp) {
		goto out;
	}
	*tmp = '\0';
	tmp++;
	
	addr = simple_strtoul(tmp, &tmp2, 0);
	if (tmp2 == tmp) 
		goto out;
	
	if (strcmp(kbuf, "add") == 0) {
		if (*tmp2 != ' ') goto out;
		tmp2++;
		size = (int)simple_strtol(tmp2, &tmp, 0);
		if (tmp2 == tmp) goto out;
		ret = add_probe((void *)addr, size);
	} else if (strcmp(kbuf, "del") == 0) {
		ret = del_probe((void *)addr);
	}
	if (ret >= 0)
		ret = count;
       out:
	up(&proc_mutex);
	return ret;
}

static struct file_operations proc_probe_operations = {
	.owner          = THIS_MODULE,
	.open           = probe_open,
	.read           = seq_read,
	.llseek         = seq_lseek,
	.release        = seq_release,
	.write		= probe_write,
};

static int install_probe(void) 
{
	int ret;
	ret = init_buffers();
	if (ret < 0) return ret;

	probe_proc_ent = create_proc_entry("traceprobes", S_IRUSR | S_IWUSR,
					   &proc_root);
	if (!probe_proc_ent) {
		cleanup_buffers();
		return -ENOMEM;
	}
	probe_proc_ent->proc_fops = &proc_probe_operations;
	probe_proc_ent->size = 0;

	log_proc_ent = create_proc_entry("tracelogs", S_IRUSR, &proc_root);
	if (!log_proc_ent) {
		remove_proc_entry("traceprobes", &proc_root);
		cleanup_buffers();
		return -ENOMEM;
	}
	log_proc_ent->proc_fops = &proc_log_operations;
	log_proc_ent->size = bsize*LOG_ENTRY_SIZE*num_online_cpus();
	printk("simple tracer: initialized bsize=%ld\n", bsize);
	return 0;
}

static void uninstall_probe(void)
{
	remove_proc_entry("traceprobes", &proc_root);
	remove_proc_entry("tracelogs", &proc_root);
	cleanup_probes();
	cleanup_buffers();
	printk("simple tracer: removed\n");
}

module_init(install_probe);
module_exit(uninstall_probe);
MODULE_AUTHOR("M.Hiramatsu <hiramatu@sdl.hitachi.co.jp>");
MODULE_LICENSE("GPL");

