/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_split.c - log split by each pid program                               */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2008                         */
/*             Authors: Yumiko Sugita (yumiko.sugita.yf@hitachi.com),        */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <sys/mman.h>
#include <getopt.h>
#include "bt_utils.h"
#include "avltree.h"

#define BT_SPLIT_VER	VERSION
#define	COPYRIGHT	"Copyright (c) Hitachi, Ltd. 2005-" RELEASE_YEAR

#define MODE_SPLIT_ONLY			0
#define MODE_SEARCH_ONLY		1
#define MODE_SEARCH_AND_SPLIT		2
#define MODE_PRINT_PID			3
#define MODE_PRINT_PID_SYSCALL_TIME	4

int do_chk_pid_pos(int mode, char *cpu_files[], unsigned long addr)
{
	if (mode == MODE_SEARCH_ONLY || mode == MODE_SEARCH_AND_SPLIT) {
		printf("sec          usec   rec-index    cpu pid    address\n");
		printf("------------ ------ ------------ --- ------ ----------\n");
	}
	return chk_pid_pos(cpu_files,
			   !(mode == MODE_SPLIT_ONLY || mode == MODE_PRINT_PID),
			   addr);
}

static int cp_records(void *__dt, void *user_data)
{
	struct __pid_info *__p = (struct __pid_info*)__dt;
	struct log_file_info *log_info = get_log_file_info(__p);
	int to_fd = (int)user_data;
	off_t all_len, len, max = sizeof(struct bt_record) * 1024;
	ssize_t rc;
	char buf[sizeof(struct bt_record) * 1024];

	if (u_lseek(log_info->fd, __p->i_rec * sizeof(struct bt_record),
		    SEEK_SET) < 0)
		return -1;
	all_len = __p->n_rec * sizeof(struct bt_record);
	while (all_len) {
		len = all_len > max ? max : all_len;
		all_len -= len;
		rc = u_read(log_info->fd, buf, len);
		if (rc < 0)
			return -1;
		rc = u_write(to_fd, buf, len);
		if (rc < 0)
			return -1;
	}
	return 0;
}

int __create_pid_files(struct pid_log_info *p, void *data)
{
	int fd;
	char *dir_path = (char*)data;
	int path_max = 128;
	char path[path_max];

	snprintf(path, path_max, "%s/%d", dir_path, p->pid);
	if ((fd = open(path, O_CREAT|O_EXCL|O_RDWR, S_IRUSR|S_IWUSR))
	    < 0) {
		fprintf(stderr, "can't create %d.(%s)\n", p->pid,
			strerror(errno));
		return -1;
	}
	if (for_each_node(p->info, cp_records, (void*)fd) < 0) {
		close(fd);
		return -1;
	}
	close(fd);
	return 0;
}

int create_pid_files(char *dir_path)
{
	return for_each_pid_log_info(__create_pid_files, dir_path);
}

struct syscall_time {
	long 			n_syscall; /* -1 means wait BT_FLAG_T_START */
	unsigned long long	timestamp;
};

#define NR_syscalls	285
struct syscall_time_tbl {
	long			n;
	unsigned long long	total_time;
	unsigned long long	min_time;
	unsigned long long	max_time;
} syscall_time_tbl[NR_syscalls];

static int print_syscall_time(void *__dt, void *user_data)
{
	struct __pid_info *__p = (struct __pid_info*)__dt;
	struct log_file_info *log_info = get_log_file_info(__p);
	struct syscall_time *st = (struct syscall_time*)user_data;
	off_t all_len, i, len, max = sizeof(struct bt_record) * 1024;
	ssize_t rc;
	char buf[sizeof(struct bt_record) * 1024];
	struct tmr_record *rec;
	struct syscall_time_tbl *tbl;
	unsigned long long syscall_t;

	if (u_lseek(log_info->fd, __p->i_rec * sizeof(struct bt_record),
		    SEEK_SET) < 0)
		return -1;
	all_len = __p->n_rec * sizeof(struct bt_record);
	while (all_len) {
		len = all_len > max ? max : all_len;
		all_len -= len;
		rc = u_read(log_info->fd, buf, len);
		if (rc < 0)
			return -1;
		rec = (struct tmr_record*)buf;
		for (i = 0; i < len / sizeof(struct bt_record); i++) {
			if (rec->flags & BT_FLAG_T_START) {
				if (st->n_syscall >= 0) {
					fprintf(stderr,
						"There is no system-call end timestamp (%ld:%20llu -> %ld:%20llu)\n",
						st->n_syscall, st->timestamp,
						rec->flags & ~BT_FLAG_T_START,
						rec->timestamp);
					//return -1;
				}
				st->n_syscall = rec->flags & ~BT_FLAG_T_START;
				st->timestamp = rec->timestamp;
			} else if (rec->flags & BT_FLAG_T_STOP) {
				/* trace start when during system-call */
				if (st->n_syscall < 0)
					continue;
				syscall_t = rec->timestamp - st->timestamp;
				printf("  %3ld %20llu\n",
				       st->n_syscall, syscall_t);
				tbl = &syscall_time_tbl[st->n_syscall];
				tbl->n++;
				tbl->total_time += syscall_t;
				if (tbl->total_time < syscall_t)
					fprintf(stderr,
						"total time too short\n");
				if (syscall_t < tbl->min_time)
					tbl->min_time = syscall_t;
				if (syscall_t > tbl->max_time)
					tbl->max_time = syscall_t;
				st->n_syscall = -1;
			}
			rec++;
		}
	}
	return 0;
}

int __print_pid_syscall_time(struct pid_log_info *p, void *data)
{
	struct syscall_time st;

	printf("====== System call time of %d(%s) ======\n",
	       p->pid, p->comm);
	printf("  n   cnt\n");
	printf("  --- --------------------\n");
	st.n_syscall = -1;
	if (for_each_node(p->info, print_syscall_time, &st) < 0)
		return -1;
	return 0;
}

int print_pid_syscall_time(void)
{
	int rc;
	size_t i;
	struct syscall_time_tbl *tbl;
	long double avrg;

	for (i = 0; i < NR_syscalls; i++) {
		tbl = &syscall_time_tbl[i];
		tbl->min_time = 0xffffffffffffffffULL;
	}
	rc = for_each_pid_log_info(__print_pid_syscall_time, NULL);
	if (rc < 0)
		return -1;
	for (i = 0; i < NR_syscalls; i++) {
		tbl = &syscall_time_tbl[i];
		if (!tbl->n) {
			tbl->min_time = 0;
			avrg = 0.0;
		} else
			avrg = (long double)tbl->total_time / tbl->n;
		printf("%3d,%lu,%llu,%llu,%.2Lf,%llu\n",
		       i, tbl->n, tbl->min_time, tbl->max_time, avrg,
		       tbl->total_time);
	}
	return 0;
}

char *cpu_files[MAX_CPU_FILES + 1];

static int __get_cpu_files(int cpu, char *cpu_path, size_t size, void *dt)
{
	int index = *(int*)dt;

	cpu_files[index] = strdup(cpu_path);
	*(int*)dt = index + 1;
	return 0;
}

char** get_cpu_files(char *dir)
{
	int rc, index = 0;

	rc = proc_cpu_files(dir, __get_cpu_files, &index);
	if (rc < 0)
		return NULL;
	return cpu_files;
}

void err_exit(void)
{
	finalize_log_info();
	exit(1);
}

void usage(void)
{
	fprintf(stderr, "bt_split %s\n", BT_SPLIT_VER);
	fprintf(stderr, "    %s\n\n", COPYRIGHT);
#if 0
	fprintf(stderr, "bt_split [-p] [-S addr] [-s addr] -d dir\n");
	fprintf(stderr, "  -p: print pid list\n");
	fprintf(stderr, "  -S: search address, and do not split logfile\n");
	fprintf(stderr, "  -s: search address\n");
#else
	fprintf(stderr, "bt_split -d dir\n");
#endif
	fprintf(stderr, "  -d: log directory\n");
}

int main(int argc, char *argv[])
{
	int verbose = 0, mode = MODE_SPLIT_ONLY;
	unsigned long addr = 0;
	char c, *dir = NULL, *p_end;
	char **cpu_files;

	while ((c = getopt(argc, argv, "pPs:S:d:v")) != -1) {
		switch (c) {
		case 'p':
		case 'P':
			if (mode != MODE_SPLIT_ONLY) {
				usage();
				err_exit();
			}
			mode = c == 'p' ?
				MODE_PRINT_PID : MODE_PRINT_PID_SYSCALL_TIME;
			break;
		case 's':
		case 'S':
			if (mode != MODE_SPLIT_ONLY) {
				usage();
				err_exit();
			}
			mode = c == 'S' ?
				MODE_SEARCH_ONLY : MODE_SEARCH_AND_SPLIT;
			addr = strtoul(optarg, &p_end, 0);
			if (*p_end != '\0') {
				fprintf(stderr, "invalid address(%s).\n",
					optarg);
				err_exit();
			}
			break;
		case 'd':
			dir = optarg;
			break;
		case 'v':
			verbose = 1;
			break;
		default:
			usage();
			err_exit();
		}
	}
	if (optind < argc || !dir) {
		usage();
		err_exit();
	}
	cpu_files = get_cpu_files(dir);
	if (initialize_log_info(cpu_files) < 0)
		err_exit();
	if (do_chk_pid_pos(mode, cpu_files, addr) < 0)
		err_exit();
	if (mode == MODE_PRINT_PID) {
		print_pid_info();
		goto EXIT;
	} else if (mode == MODE_PRINT_PID_SYSCALL_TIME) {
		if (print_pid_syscall_time() < 0)
			err_exit();
		goto EXIT;
	}
	if (verbose)
		dump_pid_pos();
	if (mode != MODE_SEARCH_ONLY) {
		if (create_pid_files(dir) < 0)
			err_exit();
	}
EXIT:
	finalize_log_info();

	exit(0);
}
