/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_ar_parse.c - /proc/modules and /proc/PID/maps parser                  */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2008                         */
/*             Authors: Yumiko Sugita (yumiko.sugita.yf@hitachi.com),        */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <sys/utsname.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include "bt_ar_parse.h"

static struct pid_range *ranges;

/*----------------------------------------------------------------------------*/
/*  range_to_name support functions                                           */
/*----------------------------------------------------------------------------*/
void dump_r2n(struct r2n_info *r2i)
{
	int i;
	struct range_to_name *p;

	if (!r2i->all_r2n)
		return;
	for (i = 0; i < r2i->r2n_max; i++) {
		p = r2i->all_r2n[i];
		if (p) {
			printf("====== 0x%08lx:0x%08lx\t%s/%s ======\n",
			       p->begin, p->end, p->dirname, p->basename);
			dump_bfd_symbols(&p->bi, p->begin, p->end, p->offset);
			dump_path_tree(p->pt, p->cnt);
		}
	}
}

static void free_path_tree(struct range_to_name *r2n)
{
	long i;

	if (!r2n || !r2n->pt)
		return;
	for (i = 0; i < r2n->cnt; i++)
		free_path(r2n->pt[i]);
	free(r2n->pt);
	r2n->pt = NULL;
	r2n->cnt = 0;
}

static void free_fix_from_cache(struct range_to_name *r2n)
{
	struct fix_from_cache *p, *p_next;

	for (p = r2n->fix_from_cache; p; p = p_next) {
		p_next = p->next;
		free(p);
	}
	r2n->fix_from_cache = NULL;
}

void free_r2n(struct r2n_info *r2i)
{
	int i;
	struct range_to_name *p;

	if (!r2i->all_r2n)
		return;
	for (i = 0; i < r2i->r2n_max; i++) {
		p = r2i->all_r2n[i];
		if (p) {
			//free(p->func_names);
			free_path_tree(p);
			free_fix_from_cache(p);
			free(p);
		}
	}
	free(r2i->all_r2n);
}

/* This function is used for merging the kernel coverage. */
void free_r2n_but_kernel(struct r2n_info *r2i)
{
	int i;
	struct range_to_name *p;

	if (!r2i->all_r2n)
		return;
	for (i = 0; i < r2i->r2n_max; i++) {
		p = r2i->all_r2n[i];
		if (p) {
			if (is_kernel(p))
				continue;
			//free(p->func_names);
			free_path_tree(p);
			free_fix_from_cache(p);
			free(p);
			r2i->all_r2n[i] = NULL;
		}
	}
}

struct range_to_name* addr_to_r2n(struct r2n_info *r2i, unsigned long addr)
{
	int i;
	struct range_to_name *p;

	for (i = 0; i < r2i->r2n_max; i++) {
		p = r2i->all_r2n[i];
		if (p && addr >= p->begin && addr <= p->end)
			return p;
	}
	return NULL;
}

static int is_same_name_registered(struct r2n_info *r2i, char *name)
{
	int i, len;
	struct range_to_name *r2n;

	if (!r2i->all_r2n)
		return 0;
	for (i = 0; i < r2i->r2n_max; i++) {
		r2n = r2i->all_r2n[i];
		if (!r2n)
			continue;
		len = strlen(r2n->dirname);
		if (strstr(name, r2n->dirname) == name && name[len] == '/' &&
		    strstr(name, r2n->basename) == name + len + 1) {
			return 1;
		}
	}
	return 0;
}

/* Size of the 'buf' must be PATH_MAX */
static char* get_pt_fpath(char *buf, char *name, char *uname_r, int is_read)
{
	int i;
	struct stat st;
	char *dirs[] = { "/tmp", "/var/tmp" };

	for (i = 0; i < sizeof(dirs) / sizeof(dirs[0]); i++) {
		snprintf(buf, PATH_MAX, "%s/btrax.pt.%s-%s",
			 dirs[i], name, uname_r);
		if (is_read) {
			if (lstat(buf, &st) >= 0 && S_ISREG(st.st_mode))
				return buf;
		} else {
			if (lstat(dirs[i], &st) >= 0 && S_ISDIR(st.st_mode))
				return buf;
		}
	}
	return NULL;
}

#if 1
static int chk_self_and_pt_mtime(char *self, char *pt)
{
	struct stat self_st, pt_st;

	if (lstat(self, &self_st) < 0 || lstat(pt, &pt_st) < 0)
		return -1;
	if (self_st.st_mtime >= pt_st.st_mtime)
		return -1;
	return 0;
}
#endif

#define read_dt(fd, dt) \
	if (read((fd), &(dt), sizeof(dt)) != sizeof(dt)) goto READ_ERR_EXIT

static int read_pt_from_file(struct range_to_name *r2n, char *vmlinux_path,
			     struct stat *vmlinux_st, char *uname_r)
{
	unsigned long i;
	int fd, len;
	struct stat st;
	char ver[16], fpath[PATH_MAX], path[PATH_MAX];
	struct path *pt;

	if (!get_pt_fpath(fpath, r2n->basename, uname_r, 1))
		return -1;
	if ((fd = open(fpath, O_RDONLY)) < 0) {
		fprintf(stderr, "'%s' open failed.(%s)\n",
			fpath, strerror(errno));
		return -1;
	}
#if 1
	/* If this program's modified time is newer than path-tree file,
	 * there is posibility of updating of binutils.
	 */
	if ((len = readlink("/proc/self/exe", path, PATH_MAX)) < 0) {
		fprintf(stderr, "readlink failed.(%s)\n", strerror(errno));
		goto ERR_EXIT;
	}
	path[len] = '\0';
	if (chk_self_and_pt_mtime(path, fpath) < 0)
		return -1;
#endif

	read_dt(fd, len);
	if (len >= 16)
		goto ERR_EXIT;
	if (read(fd, ver, len) != len)
		goto READ_ERR_EXIT;
	ver[len] = '\0';
	if (strcmp(VERSION, ver) != 0)
		goto ERR_EXIT;
	read_dt(fd, len);
	if (read(fd, path, len) != len)
		goto READ_ERR_EXIT;
	path[len] = '\0';
	if (strcmp(vmlinux_path, path) != 0)
		goto ERR_EXIT;
	read_dt(fd, st.st_size);
	if (vmlinux_st->st_size != st.st_size)
		goto ERR_EXIT;
	read_dt(fd, st.st_mtime);
	if (vmlinux_st->st_mtime != st.st_mtime)
		goto ERR_EXIT;
	read_dt(fd, r2n->cnt);
	r2n->pt = xmalloc(sizeof(struct path*) * r2n->cnt);
	for (i = 0; i < r2n->cnt; i++) {
		pt = xmalloc(sizeof(struct path));
		read_dt(fd, *pt);
		r2n->pt[i] = pt;
	}
	close(fd);
	return 0;

READ_ERR_EXIT:
	fprintf(stderr, "read failed.(%s)\n", strerror(errno));
ERR_EXIT:
	close(fd);
	return -1;
}

#define write_dt(fd, dt) \
	if (write((fd), &(dt), sizeof(dt)) < 0) goto ERR_EXIT

static int write_pt_to_file(struct range_to_name *r2n, char *vmlinux_path,
			    struct stat *st, char *uname_r)
{
	char fpath[PATH_MAX];
	unsigned long i;
	int fd, len;

	if (!get_pt_fpath(fpath, r2n->basename, uname_r, 0))
		return -1;
	if ((fd = creat(fpath, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH))
	    < 0) {
		fprintf(stderr, "'%s' open failed.(%s)\n",
			fpath, strerror(errno));
		return -1;
	}
	/* The 'sticky' bit on a temporary directory is typically set.
	 * In this case, the 'creat' function can not set the write permission
	 * for another users. So, we call 'chmod' function.
	 */
	chmod(fpath, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH);

	len = strlen(VERSION);
	write_dt(fd, len);
	if (write(fd, VERSION, len) != len)
		goto ERR_EXIT;
	len = strlen(vmlinux_path);
	write_dt(fd, len);
	if (write(fd, vmlinux_path, len) != len)
		goto ERR_EXIT;
	write_dt(fd, st->st_size);
	write_dt(fd, st->st_mtime);
	write_dt(fd, r2n->cnt);
	for (i = 0; i < r2n->cnt; i++)
		write_dt(fd, *r2n->pt[i]);
	close(fd);
	return 0;
ERR_EXIT:
	fprintf(stderr, "write failed.(%s)\n", strerror(errno));
	ftruncate(fd, 0);
	close(fd);
	return -1;
}

static int get_kernel_offset(struct range_to_name *r2n, char *kallsyms)
{
	char *ksym_name = "_stext";
	unsigned long addr1, addr2;
	FILE *f = NULL;
	char buf[MAX_LINE_LEN + 1], type, sym[MAX_LINE_LEN];
	int found, num;

	if (get_addr_of_symbol(&r2n->bi, 0, ksym_name, &addr1) < 1){
		fprintf(stderr, "Symbol '%s' not found in kernel.\n",
			ksym_name);
		return -1;
	}
	if (kallsyms == NULL) {
		fprintf(stderr, "kallsyms was not logged.\n");
		return -1;
	}
	if ((f = fopen(kallsyms, "r")) == NULL) {
		fprintf(stderr, "fopen \"%s\" failed.(%s)\n",
			kallsyms, strerror(errno));
		return -1;
	}
	buf[MAX_LINE_LEN] = '\0';
	found = 0;
	while (fgets(buf, MAX_LINE_LEN, f)) {
		num = sscanf(buf, "%lx %c %s\n", &addr2, &type, sym);
		if (num == 3 || strcmp(sym, ksym_name) == 0) {
			found = 1;
			break;
		}
	}
	fclose(f);
	if (!found) {
		fprintf(stderr, "Symbol '%s' not found in '%s'.\n",
			ksym_name, kallsyms);
		return -1;
	}
	r2n->offset = addr2 - addr1;
	return 0;
}

static int prepare_obj_file(struct range_to_name *r2n, char *kallsyms,
			    char *uname_r, int with_pt, int verbose)
{
	char path[MAX_LINE_LEN];
	struct stat st;

	snprintf(path, MAX_LINE_LEN, "%s/%s", r2n->dirname, r2n->basename);
	/* In case of without path-tree, do not print 'checking ...' message */
	if (verbose)
		printf("checking %s...\n", path);
	if (stat(path, &st) < 0) {
		fprintf(stderr, "'%s' stat failed.(%s)\n",
			path, strerror(errno));
		return -1;
	}
	if (init_bfd_if(&r2n->bi, strdup(path),
			r2n->is_kernel ? NULL : kallsyms) < 0)
		return -1;
	if (!r2n->begin && !r2n->end) {
		struct addr_range *range = get_pid_addr_range(ALL_PID);
		unsigned long min, max;

#ifdef USE_ELFUTILS
		if (init_dwarf(&r2n->bi, (const char*)path) < 0)
			return -1;
#endif
		if (get_kernel_offset(r2n, kallsyms) < 0)
			return -1;
		get_kernel_min_max_addr(&r2n->bi, &min, &max);
		if (is_addr_range_match(min, max, range)) {
			r2n->begin = min + r2n->offset;
			r2n->end = max + r2n->offset;
		} else
			return 1;
	} else {
		r2n->offset = get_offset_addr(&r2n->bi, r2n->begin);
		/* for DEBUG
		printf("OFFSET: 0x%08lx\n", r2n->offset);
		*/
	}
	remove_useless_fsyms(&r2n->bi, r2n->begin, r2n->end, r2n->offset);
	if (with_pt) {
		if (is_kernel(r2n)) {
			if (read_pt_from_file(r2n, path, &st, uname_r) >= 0)
				return 0;
			r2n->skip_ud2_srcinfo =
				chk_kernel_has_ud2_src_info(&r2n->bi);
			/*
			printf("Kernel has src-info with ud2=>%d\n",
			       r2n->skip_ud2_srcinfo);
			       */
		}
		if (chk_path_tree(&r2n->bi, &r2n->pt, &r2n->cnt, r2n->offset,
				  r2n->skip_ud2_srcinfo) < 0)
			return -1;
		if (is_kernel(r2n))
			return write_pt_to_file(r2n, path, &st, uname_r);
	}
	return 0;
}

int add_range_to_name(struct r2n_info *r2i,
		      unsigned long begin, unsigned long end, char *name,
		      char *kallsyms, int is_module, int with_pt, int verbose)
{
	int rc, i, step = 8;
	struct range_to_name *r2n;

	if (is_same_name_registered(r2i, name))
		return 0;
	r2n = xcalloc(1, sizeof(struct range_to_name));
	r2n->begin = begin;
	r2n->end = end;
	snprintf(r2n->name, PATH_MAX, "%s", name);
	r2n->basename = basename(r2n->name);
	r2n->dirname = dirname(r2n->name);
	r2n->is_kernel = strcmp(r2n->basename, KERNEL) == 0;

	for (i = 0; i < r2i->r2n_max && r2i->all_r2n[i]; i++);
	if (i >= r2i->r2n_max) {
		r2i->all_r2n = xrealloc(r2i->all_r2n,
					(r2i->r2n_max + step) * sizeof(r2n));
		memset(&r2i->all_r2n[i], 0, step * sizeof(r2n));
		i = r2i->r2n_max;
		r2i->r2n_max += step;
	}
	r2i->all_r2n[i] = r2n;
	rc = prepare_obj_file(r2n, kallsyms, r2i->uname_r, with_pt, verbose);
	if (rc < 0)
		//return 0;
		return -1;
	else if (rc > 0) {
		free(r2n);
		r2i->all_r2n[i] = NULL;
	}
	return 0;
}

/* return value =  1 : fixed
 *                 0 : out of range
 *                -1 : error
 */
static int fix_from_addr_and_get_info(struct r2n_info *r2i,
				      unsigned long *from, unsigned long to,
				      struct range_to_name **__r2n,
				      struct path **__p, long *__id)
{
	struct range_to_name *r2n = NULL;
	struct path *p = NULL;
	unsigned long tmp;
	long i;

	/* The from address that logged by the Pentium-M's BTS facility isn't
	 * same as Pentium-4 or Xeon's.
	 * Pentium-M logged next instruction's address as from address.
	 * But note that in interrupt branch record, Pentium-M logged current
	 * instruction's address as from address.
	 */
	if (!r2i->from_is_next)
		goto FIXED;
	tmp = *from;
	if (r2i->from_is_next)
		tmp -= 1;
	r2n = addr_to_r2n(r2i, tmp);
	if (!r2n) {
		/* from-address is already correct (interrupt case) ? */
		r2n = addr_to_r2n(r2i, *from);
		if (!r2n)
			return 0;
		goto FIXED;
	}
	i = r2n->cnt;
	p = find_path_from_addr(r2n->pt, &i, tmp);
	if (!p) {
		/* from-address is already correct (interrupt case) ? */
		if (r2n->end == tmp) {
			r2n = addr_to_r2n(r2i, *from);
			if (!r2n)
				return 0;
			i = r2n->cnt;
		}
		p = find_path_from_addr(r2n->pt, &i, *from);
		if (!p) {
			/*
			fprintf(stderr, "path not found(0x%08lx:%s).\n",
				tmp, r2n->basename);
				*/
			return 0;
		}
		goto FIXED;
	}

	/* check from address is next instruction address or not */
	switch (p->type) {
	case BTYPE_BRANCH:
	case BTYPE_JMP:
	case BTYPE_CALL:
		if (p->next == *from &&
		    (p->jmp == UNKNOWN_BADDR || p->jmp == to)) {
			*from = p->base;
			goto FIXED;
		}
		break;
	case BTYPE_RET:
	case BTYPE_IRET:
		if (p->next == *from) {
			*from = p->base;
			goto FIXED;
		}
		break;
	}
	/* Case of interrupt branch record.
	 * There is no need to change the from address.
	 */
	if (*from < r2n->begin || *from > r2n->end) {
		r2n = NULL;
		p = NULL;
	} else if (*from < p->addr || *from >= p->next) {
		if (i < r2n->cnt - 1) {
			p++;
			if (*from < p->addr || *from >= p->next)
				p = NULL;
		} else
			p = NULL;
	}
FIXED:
	if ((__r2n || __p || __id) && !r2n) {
		r2n = addr_to_r2n(r2i, *from);
		if (!r2n)
			return 0;
	}
	if (__r2n)
		*__r2n = r2n;
	if ((__p || __id) && !p) {
		i = r2n->cnt;
		p = find_path_from_addr(r2n->pt, &i, *from);
		if (!p) {
			/*
			fprintf(stderr, "path not found(0x%08lx:%s).\n",
				*from, r2n->basename);
				*/
			return 0;
		}
	}
	if (__p)
		*__p = p;
	if (__id)
		*__id = i;
	return 1;
}

int chk_fix_from_cache(struct r2n_info *r2i,
		       unsigned long *from, unsigned long to,
		       struct range_to_name **__r2n, struct path **__path,
		       long *__id)
{
	int i, rc;
	struct fix_from_cache *p, *p_prev;
	struct range_to_name *r2n;

	r2n = addr_to_r2n(r2i, *from);
	if (!r2n) {
		if (!r2i->from_is_next)
			return 0;
		r2n = addr_to_r2n(r2i, *from - 1);
		if (!r2n)
			return 0;
	}
	for (i = 0, p_prev = NULL, p = r2n->fix_from_cache; p;
	     i++, p_prev = p, p = p->next) {
		if (p->from == *from && p->to == to) {
			if (p_prev) {
				p_prev->next = p->next;
				p->next = r2n->fix_from_cache;
				r2n->fix_from_cache = p;
			}
			/* else case means found element is already cache top.*/
			*from = p->fixed_from;
			if (__r2n)
				*__r2n = p->r2n;
			if (__path)
				*__path = p->path;
			if (__id)
				*__id = p->id;
			//printf("CACHE HIT\n");
			return 1;
		}
	}
	//printf("CACHE MISS\n");
	if (i >= MAX_FIX_FROM_CACHE) {
		/* free last element */
		for (p = r2n->fix_from_cache; p && p->next && p->next->next;
		     p = p->next);
		free(p->next);
		p->next = NULL;
	}
	/* add new element */
	p = xmalloc(sizeof(struct fix_from_cache));
	p->from = *from;
	p->to = to;
	rc = fix_from_addr_and_get_info(r2i, from, to, __r2n, __path, __id);
	if (rc <= 0) {
		free(p);
		return rc;
	}
	p->fixed_from = *from;
	p->r2n = __r2n ? *__r2n : NULL;
	p->path = __path ? *__path : NULL;
	p->id = __id ? *__id : -1;
	p->next = r2n->fix_from_cache;
	r2n->fix_from_cache = p;
	return 1;
}

/*----------------------------------------------------------------------------*/
/*  parse address range support functions                                     */
/*----------------------------------------------------------------------------*/
int alloc_pid_range(pid_t pid)
{
	struct pid_range **p, *p_prev;

	p = &ranges;
	p_prev = *p;
	while (*p) {
		if ((*p)->pid == pid)
			return 0;
		p_prev = *p;
		p = &(*p)->next;
	}
	*p = xmalloc(sizeof(struct pid_range));
	(*p)->pid = pid;
	(*p)->range = NULL;
	(*p)->next = NULL;
	if (p_prev)
		p_prev->next = *p;
	return 0;
}

static struct pid_range* get_pid_range(pid_t pid)
{
	struct pid_range *p;

	p = ranges;
	while (p) {
		if (p->pid == pid)
			return p;
		p = p->next;
	}
	return p;
}

static void free_one_range(struct addr_range **p_ar)
{
	struct addr_range *p_tmp;

	if (!*p_ar)
		return;
	p_tmp = (*p_ar)->next;
	free(*p_ar);
	*p_ar = p_tmp;
}

void free_ranges(void)
{
	struct pid_range *p, *p_next;

	p = ranges;
	while (p) {
		while (p->range)
			free_one_range(&p->range);
		p_next = p->next;
		free(p);
		p = p_next;
	}
	ranges = NULL;
}

char* range2ulongs(char *p, unsigned long *begin, unsigned long *end)
{
	char *p_end;

	*begin = strtoul(p, &p_end, 0);
	if (*p_end != ':') {
		eprintf("begin address invalid(%s)\n", p);
		return NULL;
	}
	*end = strtoul(p_end + 1, &p_end, 0);
	if (*p_end != '\0' && *p_end != '\n' && *p_end != ' ') {
		eprintf("end address invalid(%s)\n", p);
		return NULL;
	}
	if (*begin >= *end) {
		eprintf("begin address greater or equal end address(%s)\n", p);
		return NULL;
	}
	return p_end;
}

static int is_overlap(unsigned long b1, unsigned long e1,
		      unsigned long b2, unsigned long e2, int continue_ok)
{
	unsigned long long len1, len2;

	len1 = e1 - b1;
	len2 = e2 - b2;

	if (continue_ok) {
		len1++;
		len2++;
	}
	if (b2 == b1 ||
	    (b2 > b1 && b2 - b1 <= len1) ||
	    (b2 < b1 && b1 - b2 <= len2))
		return 1;
	return 0;
}

int add_range(pid_t pid, unsigned long begin, unsigned long end)
{
	struct pid_range *p;
	struct addr_range **pp_ar, *p_ar, *p_tmp;

	ddprintf("------ ADD RANGE (0x%08lx, 0x%08lx) ------\n", begin, end);
	if (!(p = get_pid_range(pid))) {
		// error print
		return -1;
	}
	/* detect overlapped range */
	pp_ar = &p->range;
	while ((p_ar = *pp_ar)) {
		ddprintf("CHECK\n");
		/* overlapped ? */
		if (is_overlap(p_ar->begin, p_ar->end, begin, end, 1)) {
			ddprintf("OVERLAP DETECT\n");
			/* expand the range */
			if (begin < p_ar->begin)
				p_ar->begin = begin;
			if (end > p_ar->end) {
				p_ar->end = end;
				/* check upper ranges */
				p_tmp = p_ar->next;
				while (p_tmp) {
					if (p_tmp->begin <= p_ar->end ||
					    (p_tmp->begin > p_ar->end &&
					     p_tmp->begin - 1 == p_ar->end)){
						if (p_tmp->end > p_ar->end)
							p_ar->end = p_tmp->end;
						free_one_range(&p_ar->next);
						p_tmp = p_ar->next;
					} else {
						break;
					}
				}
			}
			return 0;
		}
		if (p_ar->begin > end) {
			ddprintf("BREAK\n");
			break;
		}
		pp_ar = &p_ar->next;
	}
	/* not overlapped */
	p_tmp = xmalloc(sizeof(struct addr_range));
	p_tmp->begin = begin;
	p_tmp->end = end;
	if (p->range) {
		p_tmp->next = *pp_ar;
		*pp_ar = p_tmp;
	} else {
		p_tmp->next = NULL;
		p->range = p_tmp;
	}
	return 0;
}

int del_range(pid_t pid, unsigned long begin, unsigned long end)
{
	struct pid_range *p;
	struct addr_range **pp_ar, *p_ar, *p_tmp;

	ddprintf("------ DEL RANGE (0x%08lx, 0x%08lx) ------\n", begin, end);
	if (!(p = get_pid_range(pid))) {
		// error print
		return -1;
	}
	/* detect overlapped range */
	pp_ar = &p->range;
	while ((p_ar = *pp_ar)) {
		ddprintf("CHECK\n");
		/* overlapped ? */
		if (is_overlap(p_ar->begin, p_ar->end, begin, end, 0)) {
			ddprintf("OVERLAP DETECT\n");
			if (p_ar->begin >= begin && p_ar->end <= end) {
				/* delete range include existance range */
				free_one_range(pp_ar);
			} else if (p_ar->begin < begin && p_ar->end > end){
				/* existence range include delete range */
				p_tmp = xmalloc(sizeof(struct addr_range));
				p_tmp->begin = end + 1;
				p_tmp->end = p_ar->end;
				p_ar->end = begin - 1;
				p_tmp->next = p_ar->next;
				p_ar->next = p_tmp;
				pp_ar = &p_ar->next;
			} else if (p_ar->begin < begin) {
				/* existence range overlapped (upper part) */
				p_ar->end = begin - 1;
				pp_ar = &p_ar->next;
			} else {
				/* existence range overlapped (lower part) */
				p_ar->begin = end + 1;
				pp_ar = &p_ar->next;
			}
		}
		if (p_ar->begin > end) {
			ddprintf("BREAK\n");
			break;
		}
		pp_ar = &p_ar->next;
	}
	return 0;
}

void dump_ranges(void)
{
	struct pid_range *p;
	struct addr_range *p_ar;

	p = ranges;
	while (p) {
		printf("------ pid: %d ------\n", p->pid);
		p_ar = p->range;
		while (p_ar) {
			printf("0x%08lx:0x%08lx\n", p_ar->begin, p_ar->end);
			p_ar = p_ar->next;
		}
		p = p->next;
	}
}

struct pid_range *get_all_ranges(void)
{
	return ranges;
}

struct addr_range *get_pid_addr_range(pid_t pid)
{
	struct pid_range *p;

	p = ranges;
	while (p) {
		if (p->pid == pid)
			return p->range;
		p = p->next;
	}
	return NULL;
}

/* for speed up, this routine should be convert to the macro... */
int is_addr_range_match(unsigned long from, unsigned long to,
			struct addr_range *r)
{
	while (r) {
		if ((from >= r->begin && from <= r->end) ||
		    (to >= r->begin && to <= r->end))
			return 1;
		r = r->next;
	}
	return 0;
}

static int find_under_path(char *dirname, int max_len, const char *name,
			   int down)
{
	DIR *dir;
	struct dirent *d;
	int dlen, rc;
	struct stat stat;

	//printf("chk dir =>%s< (for >%s<)\n", dirname, name);
	rc = -1;
	if (!(dir = opendir(dirname)))
		return -1;
	while ((d = readdir(dir)) != NULL) {
		if (strcmp(".", d->d_name) == 0 || strcmp("..", d->d_name) == 0)
			continue;
		//printf("chk child =>%s<\n", d->d_name);
		dlen = strlen(dirname);
		if (strcmp(name, d->d_name) == 0) {
			snprintf(dirname + dlen, max_len - dlen,
				 "/%s", d->d_name);
			rc = 0;
			goto EXIT;
		}
		if (!down)
			continue;
		snprintf(dirname + dlen, max_len - dlen, "/%s", d->d_name);
		if (lstat(dirname, &stat) < 0)
			goto EXIT;
		if (S_ISDIR(stat.st_mode)) {
			if (find_under_path(dirname, max_len, name, down) == 0){
				rc = 0;
				goto EXIT;
			}
		}
		dirname[dlen] = '\0';
	}
EXIT:
	closedir(dir);
	return rc;
}

static char *elf_path_prefix = "";

void set_elf_path_prefix(char *prefix)
{
	elf_path_prefix = strdup(prefix);
}

char* get_elf_path_prefix(void)
{
	return elf_path_prefix;
}

static int get_module_full_path(char *buf, int buf_len, const char *name,
				char *uname_r)
{
	int len;

	len = snprintf(buf, buf_len, "%s/lib/modules/%s",
		       elf_path_prefix, uname_r);
	if (len >= buf_len) {
		fprintf(stderr, "!!! too short buf size(%d)\n", buf_len);
		return -1;
	}
	return find_under_path(buf, buf_len, name, 1);
}

static int __get_vmlinux_full_path(char *buf, int buf_len)
{
	int rc;

	rc = find_under_path(buf, buf_len, KERNEL, 0);
	if (rc >= 0)
		return rc;
	return find_under_path(buf, buf_len, KERNEL, 1);
}

static int get_vmlinux_full_path(char *buf, int buf_len, char *uname_r)
{
	snprintf(buf, buf_len, "%s/usr/lib/debug/lib/modules/%s",
		 elf_path_prefix, uname_r);
	if (__get_vmlinux_full_path(buf, buf_len) >= 0)
		return 0;
	snprintf(buf, buf_len, "%s/lib/modules/%s/build",
		 elf_path_prefix, uname_r);
	if (__get_vmlinux_full_path(buf, buf_len) >= 0)
		return 0;
	return -1;
}

static int change_module_name(char *name)
{
	int changed;
	char *p;

	changed = 0;
	for (p = name; p && *p; p++) {
		switch (*p) {
		case '-':
		case '_':
			changed = 1;
			if (*p == '-')
				*p = '_';
			else
				*p = '-';
		}
	}
	return changed;
}

static unsigned long irq_addrs[256 + 1];
static int irq_addrs_cnt;

static int comp_address(const void *p_a1, const void *p_a2)
{
	unsigned long a1 = *(unsigned long*)p_a1;
	unsigned long a2 = *(unsigned long*)p_a2;

	if (a1 < a2)
		return -1;
	if (a1 > a2)
		return 1;
	return 0;
}

/* return address count */
static int get_uniq_and_sorted_irq_addrs(char *dir,
					 unsigned long addrs[256 + 1])
{
	int i, j, already_checked, rc;
	char buf[PATH_MAX];
	FILE *f;
	unsigned long a;

	if (!dir)
		return 0;
	rc = -1;
	snprintf(buf, PATH_MAX, "%s/irq_addrs", dir);
	if ((f = fopen(buf, "r")) == NULL) {
		fprintf(stderr, "fopen \"%s\" failed.(%s)\n",
			buf, strerror(errno));
		goto EXIT;
	}
	i = 0;
	while (fgets(buf, PATH_MAX, f)) {
		if (sscanf(buf, "%lx\n", &a) != 1) {
			fprintf(stderr, "!!! wrong format irq_addrs file\n");
			goto EXIT;
		}
		already_checked = 0;
		for (j = 0; j < i; j++)
			if (addrs[j] == a) {
				already_checked = 1;
				break;
			}
		if (already_checked)
			continue;
		addrs[i++] = a;
		if (i >= 256) {
			fprintf(stderr, "!!! wrong format irq_addrs file\n");
			goto EXIT;
		}
	}
	rc = i;
	qsort(addrs, i, sizeof(unsigned long), comp_address);
EXIT:
	if (f)
		fclose(f);
	return rc;
}

static inline int is_irq_addrs(unsigned long a, unsigned long addrs[256 + 1],
			       int n)
{
	return bsearch(&a, addrs, n, sizeof(unsigned long), comp_address)
			!= NULL;
}
#define IS_IRQ(a)	is_irq_addrs((a), irq_addrs, irq_addrs_cnt)

int parse_modules(struct r2n_info *r2i, char *dir, int with_pt, int verbose)
{
	FILE *fd = NULL;
	int rc, num, len;
	char path[128], kallsyms[128];
	char buf[MAX_LINE_LEN + 1], *p, fullname[MAX_LINE_LEN + 1];
	struct addr_range *range;
	unsigned long from, to;

	rc = -1;
	if (!dir)
		goto CHK_KERNEL;
	if ((irq_addrs_cnt = get_uniq_and_sorted_irq_addrs(dir, irq_addrs)) < 0)
		goto EXIT;
	snprintf(kallsyms, sizeof(kallsyms), "%s/kallsyms", dir);
	snprintf(path, sizeof(path), "%s/modules", dir);
	if ((fd = fopen(path, "r")) == NULL) {
		fprintf(stderr, "can't open %s.(%s)\n", path, strerror(errno));
		goto EXIT;
	}
	buf[MAX_LINE_LEN] = fullname[MAX_LINE_LEN] = '\0';
	range = get_pid_addr_range(ALL_PID);
	while ((p = fgets(buf, MAX_LINE_LEN, fd)) != NULL) {
		len = strlen(p) - 1;
		buf[len] = '\0';
		num = sscanf(buf, "%s %ld %*d %*s %*s %lx",
			     fullname, &to, &from);
		if (num != 3) {
			fprintf(stderr, "modules format error(%d)\n", num);
			goto EXIT;
		}
		to += from;
		if (!is_addr_range_match(from, to, range))
			continue;
		snprintf(buf, MAX_LINE_LEN, "%s.ko", fullname);
		fullname[0] = '\0';
		if (get_module_full_path(fullname, MAX_LINE_LEN, buf,
					 r2i->uname_r) < 0 &&
		    (!change_module_name(buf) ||
		     get_module_full_path(fullname, MAX_LINE_LEN, buf,
					  r2i->uname_r) < 0)) {
			if (verbose)
				printf("WARN: %s not found.\n", buf);
			continue;
		}
		if (add_range_to_name(r2i, from, to, fullname, kallsyms, 1,
				      with_pt, verbose) < 0)
			goto EXIT;
	}
CHK_KERNEL:
	fullname[0] = '\0';
	if (get_vmlinux_full_path(fullname, MAX_LINE_LEN, r2i->uname_r) < 0) {
		if (verbose)
			printf("WARN: vmlinux not found.\n");
	} else {
		if (add_range_to_name(r2i, 0, 0, fullname, kallsyms, 0, with_pt,
				      verbose) < 0)
			goto EXIT;
	}
	rc = 0;
EXIT:
	if (fd)
		fclose(fd);
	return rc;
}

/* This function returns the number that a function name was found in.
 * In the 'addr' variable, an address of a function found first is set.
 */
int get_symbol_addr(struct r2n_info *r2i,
		    const char *funcsym, unsigned long *addr)
{
	int i, found, tmp;
	struct range_to_name *r2n;
	unsigned long __addr;

	if (!r2i->all_r2n)
		return 0;
	for (found = 0, i = 0; i < r2i->r2n_max; i++) {
		r2n = r2i->all_r2n[i];
		if (r2n) {
			tmp = get_addr_of_symbol(&r2n->bi, r2n->offset, funcsym,
						 &__addr);
			if (!found && tmp)
				*addr = __addr;
			found += tmp;
		}
	}
	return found;
}

int get_symbol_addr_all(struct r2n_info *r2i, const char *funcsym,
			unsigned long **__addrs)
{
	int i, j, rc, cnt;
	struct range_to_name *r2n;
	unsigned long *addrs = NULL;
	unsigned long *c_addrs;

	if (!r2i->all_r2n)
		return 0;
	cnt = 0;
	for (i = 0; i < r2i->r2n_max; i++) {
		r2n = r2i->all_r2n[i];
		if (r2n) {
			rc = get_addr_of_symbol_all(&r2n->bi, r2n->offset,
						    funcsym, &c_addrs);
			if (rc < 0)
				return -1;
			if (rc == 0)
				continue;
			if (cnt)
				addrs = xrealloc(addrs, (cnt + rc) *
						 	 sizeof(unsigned long));
			else
				addrs = xmalloc(rc * sizeof(unsigned long));
			for (j = 0; j < rc; j++)
				addrs[cnt + j] = c_addrs[j];
			cnt += rc;
			free(c_addrs);
		}
	}
	if (cnt)
		*__addrs = addrs;
	return cnt;
}

const char* get_fname(struct range_to_name *r2n, unsigned long addr)
{
	const char *fname;
	size_t offset;

	if (addr_to_func_name_and_offset(&r2n->bi, addr - r2n->offset,
					 &fname, &offset) < 0)
		return NULL;
	return fname;
}

#if 0
static int get_func_begin(struct range_to_name *r2n, unsigned long addr,
			  unsigned long *p_begin)
{
	unsigned long begin;

	if (r2n) {
		if (get_begin_of_func(&r2n->bi, addr - r2n->offset, &begin)) {
			*p_begin = begin + r2n->offset;
			return 1;
		}
	}
	return 0;
}
#endif

static int get_func_end(struct range_to_name *r2n, unsigned long begin,
			unsigned long *p_end)
{
	unsigned long end;

	if (r2n) {
		end = r2n->end - r2n->offset;
		if (get_end_of_func(&r2n->bi, begin - r2n->offset, &end)) {
			*p_end = end + r2n->offset;
			return 1;
		}
	}
	return 0;
}

int get_func_info(struct r2n_info *r2i, unsigned long begin,
		  unsigned long *p_end, struct range_to_name **p_r2n)
{
	int i;
	struct range_to_name *r2n;

	if (!r2i->all_r2n)
		return 0;
	for (i = 0; i < r2i->r2n_max; i++) {
		r2n = r2i->all_r2n[i];
		if (get_func_end(r2n, begin, p_end)) {
			*p_r2n = r2n;
			return 1;
		}
	}
	return 0;
}

int is_exists_maps(char *dir, char *fname)
{
	char path[128];
	struct stat st;

	snprintf(path, sizeof(path), "%s/%s.maps", dir, fname);
	return stat(path, &st) == 0;
}

int parse_maps(struct r2n_info *r2i, char *dir, char *fname,
	       int with_pt, int verbose)
{
	FILE *fd;
	int rc, num, len;
	char path[128];
	char mode[6], buf[MAX_LINE_LEN + 1], *p, fullname[MAX_LINE_LEN + 1];
	struct addr_range *range;
	unsigned long from, to;
	long size;

	rc = -1;
	snprintf(path, sizeof(path), "%s/%s.maps", dir, fname);
	if ((fd = fopen(path, "r")) == NULL) {
		fprintf(stderr, "can't open %s.(%s)\n", path, strerror(errno));
		goto EXIT;
	}
	buf[MAX_LINE_LEN] = fullname[MAX_LINE_LEN] = '\0';
	range = get_pid_addr_range(ALL_PID);
	while ((p = fgets(buf, MAX_LINE_LEN, fd)) != NULL) {
		len = strlen(p) - 1;
		buf[len] = '\0';
		num = sscanf(buf, "%lx-%lx %s %*s %*x %*s %ld",
			     &from, &to, mode, &size);
		if (num != 4) {
			fprintf(stderr, "map format error(%d)\n", num);
			goto EXIT;
		}
		if (size == 0 || mode[2] != 'x')
			continue;
		for (; *p != '/' && *p != '\0'; p++);
		if (*p != '/' || buf[len-1] == ')')
			continue;	// e.g. /SYSV00000000 (deleted)
		to -= 1;
		if (!is_addr_range_match(from, to, range))
			continue;
		snprintf(fullname, MAX_LINE_LEN, "%s%s", elf_path_prefix, p);
		if (add_range_to_name(r2i, from, to, fullname, NULL, 0, with_pt,
				      verbose) < 0)
			goto EXIT;
	}
	rc = 0;
EXIT:
	if (fd)
		fclose(fd);
	return rc;
}

int chk_from_is_next(char *dir)
{
	char path[PATH_MAX];
	struct stat st;

	snprintf(path, PATH_MAX, "%s/%s", dir, FROM_IS_NEXT_FNAME);
	return (stat(path, &st) == 0);
}

int set_uname(struct r2n_info *r2i, char *log_path)
{
	struct utsname utsn;
	int max_uname_len = 80;
	char *dir, *tmp, path[PATH_MAX], uname_str[max_uname_len], *p;
	FILE *f;

	if (r2i->uname_r)
		return 0;
	if (log_path) {
		tmp = strdup(log_path);
		dir = dirname(tmp);
		snprintf(path, PATH_MAX, "%s/uname", dir);
		free(tmp);
		if ((f = fopen(path, "r"))) {
			p = fgets(uname_str, max_uname_len, f);
			fclose(f);
			if (p) {
				p[strlen(p) - 1] = '\0';
				r2i->uname_r = strdup(p);
				return 0;
			}
		}
	}
	if (uname(&utsn) < 0) {
		fprintf(stderr, "uname failed.(%s)\n", strerror(errno));
		return -1;
	}
	r2i->uname_r = strdup(utsn.release);
	return 0;
}

/*----------------------------------------------------------------------------
 *  check enter to and leave from context (function-call and interrupt) 
 *----------------------------------------------------------------------------
 */
#define MAX_NEST	2048

struct ret_chk {
	int			type;
	unsigned long		ret_addr;
};

struct st_ret_chk {
	int		nest;
	struct ret_chk	ret_chk[MAX_NEST];
};

static struct st_ret_chk st_ret_chk;

#ifdef DEBUG
static void dump_ret_chk(struct st_ret_chk *retc, int n)
{
	int i;
	struct ret_chk *p;

	for (i = n; i < retc->nest; i++) {
		p = &retc->ret_chk[i];
		printf("%04d:0x%08lx(%d)\n", n, p->ret_addr, p->type);
	}
}
#endif

static void set_expect_ret_addr(struct st_ret_chk *retc, int type,
				unsigned long ret_addr)
{
	struct ret_chk *p;

	if (retc->nest >= MAX_NEST) {
		printf("WARN: MAX_NEST is too small.\n");
		return;
	}
	p = &retc->ret_chk[retc->nest];
	p->type = type;
	p->ret_addr = ret_addr;
	retc->nest++;
}

static void get_nest_from_ret_addr(struct st_ret_chk *retc, unsigned long addr,
				   int ret_maybe_fixup)
{
	int i;
	struct ret_chk *p;

	for (i = retc->nest - 1; i >= 0; i--) {
		p = &retc->ret_chk[i];
		if (ret_maybe_fixup) {
			if (p->type == BTYPE_INT) {
				retc->nest = i;
				return;
			}
		} else {
			if (p->ret_addr == addr) {
				retc->nest = i;
				return;
			}
		}
	}
	/* If there is trace-off and trace-on, 'ret_chk' data is almost meaning
	 * less because real stack was already changed.
	 * But normally, trace-off and trace-on's both nest are the same.
	 * So, if we couldn't find the return address in 'ret_chk', then
	 * simply minus one from nest value.
	 */
	if (retc->nest)
		retc->nest--;
	return;
}

static int proc_enter(struct st_ret_chk *retc, int type, int nest_chk,
		      struct bt_record *p, unsigned long ret)
{
	switch (type) {
	case BTYPE_CALL:
	case BTYPE_INT:
	case BTYPE_BREAK:
		/*
		printf("ENTER(%d): %d, from:%08lx, to:%08lx, ret:%08lx\n",
		       retc->nest, type, p->from, p->to, ret);
		       */
		set_expect_ret_addr(retc, type, ret);
		return 2;
	}
	return 1;
}

static int proc_leave(struct st_ret_chk *retc, int type, int nest_chk,
		      struct bt_record *p, int ret_maybe_fixup)
{
	switch (type) {
	case BTYPE_CALL:
	case BTYPE_INT:
	case BTYPE_BREAK:
		/*
		printf("LEAVE(%d): %d, from:%08lx, to:%08lx\n",
		       retc->nest, type, p->from, p->to);
		       */
		get_nest_from_ret_addr(retc, p->to, ret_maybe_fixup);
		return 3;
	}
	return 1;
}

void chk_nest_initialize(void)
{
	st_ret_chk.nest = 0;
}

static inline int dest_is_kernel_func(struct range_to_name *r2n,
				      struct bt_record *p)
{
	const char *fname;
	size_t offset;

	return (is_kernel(r2n) &&
		(addr_to_func_name_and_offset(&r2n->bi, p->to - r2n->offset,
					      &fname, &offset) >= 0) &&
		fname && offset == 0);
}

#define CALL(type, nest_chk, p, p_from, rc) \
	do { \
		*(type) = BTYPE_CALL; \
		(rc) = proc_enter(&st_ret_chk, *(type), (nest_chk), (p), \
				  (p_from)->next); \
	} while (0)

#define RET(type, nest_chk, p, rc) \
	do { \
		*(type) = BTYPE_CALL; \
		(rc) = proc_leave(&st_ret_chk, *(type), (nest_chk), (p), 0); \
	} while (0)

#define INT(type, nest_chk, p, rc) \
	do { \
		*(type) = BTYPE_INT; \
		(rc) = proc_enter(&st_ret_chk, *(type), (nest_chk), (p), \
				  (p)->from); \
	} while (0)

#define IRET(type, nest_chk, p, ret_maybe_fixup, rc) \
	do { \
		*(type) = BTYPE_INT; \
		(rc) = proc_leave(&st_ret_chk, *(type), (nest_chk), (p), \
				  (ret_maybe_fixup)); \
	} while (0)

#define JMP(type, nest_chk, p, rc) \
	do { \
		*(type) = BTYPE_JMP; \
		(rc) = proc_enter(&st_ret_chk, *(type), (nest_chk), (p), \
				  UNKNOWN_BADDR); \
	} while (0)

/* return value =  0 : not process ('from' and 'to' are both out of range)
 *                 1 : not process (normal branch or jump etc.)
 *                 2 : enter
 *                 3 : leave
 *                -1 : error
 */
int chk_enter_leave(struct r2n_info *r2i, struct bt_record *p,
		    int nest_chk, int *inst_type, int *type, int *nest,
		    struct range_to_name **p_r2n_from,
		    struct range_to_name **p_r2n_to,
		    struct path **pp_from, long *p_idx_from)
{
	int rc, ret_maybe_fixup = 0;
	long idx_from = -1;
	struct range_to_name *r2n = NULL, *r2n_to = NULL;
	struct path *p_from = NULL;

	*nest = st_ret_chk.nest;
	r2n_to = addr_to_r2n(r2i, p->to);
	rc = chk_fix_from_cache(r2i, &p->from, p->to, &r2n, &p_from, &idx_from);
	if (rc < 0) {
		*inst_type = BTYPE_OTHER;
		goto EXIT;
	}
	/*
	if (!is_addr_range_match(p->from, p->to, data->range))
		return 0;
		*/
	if (rc == 0) {
		/* Branch from unknown-binary */
		if (!r2n_to)
			goto EXIT;
		if (is_kernel(r2n_to)) {
			*inst_type = BTYPE_OTHER;
			if (IS_IRQ(p->to))
				INT(type, nest_chk, p, rc);
			else
				IRET(type, nest_chk, p, ret_maybe_fixup, rc);
		} else {
			/* Branch from unknown-binary to user-land means
			 * iret or ret (from 'vsdo').
			 * We assume it iret because we can't identify these.
			 */
			*inst_type = BTYPE_OTHER;
			IRET(type, nest_chk, p, ret_maybe_fixup, rc);
		}
		goto EXIT;
	}
	*inst_type = p_from->type;
	if (p_from->base != p->from)
		*inst_type = BTYPE_OTHER;

	if (!r2n_to) {
		/* Branch to unknown-binary */
		if (is_kernel(r2n))
			IRET(type, nest_chk, p, ret_maybe_fixup, rc);
		else {
			if (IS_IRQ(p->to))
				INT(type, nest_chk, p, rc);
			else
				CALL(type, nest_chk, p, p_from, rc);
		}
		goto EXIT;
	}

	switch (*inst_type) {
	case BTYPE_BRANCH:
	case BTYPE_JMP:
	case BTYPE_CALL:
		if ((p_from->jmp != UNKNOWN_BADDR && p_from->jmp == p->to) ||
		    (p_from->jmp == UNKNOWN_BADDR && !IS_IRQ(p->to))) {
			switch (*inst_type) {
			case BTYPE_JMP:
				if (dest_is_kernel_func(r2n_to, p))
					CALL(type, nest_chk, p, p_from, rc);
				else
					JMP(type, nest_chk, p, rc);
				break;
			case BTYPE_CALL:
				CALL(type, nest_chk, p, p_from, rc);
				break;
			case BTYPE_BRANCH:
				rc = 1;		/* do nothing */
				break;
			}
		} else 
			INT(type, nest_chk, p, rc);
		goto EXIT;
	case BTYPE_RET:
		if (IS_IRQ(p->to)) {
			INT(type, nest_chk, p, rc);
		} else if (!(is_kernel(r2n) || is_kernel(r2n_to)) &&
			   p_from->next - p->from > 1) {
			/* 'ret n' from user-land to user-land means jump to
			 * the plt library function.
			 */
			JMP(type, nest_chk, p, rc);
		} else
			RET(type, nest_chk, p, rc);
		goto EXIT;
	case BTYPE_INT:
	case BTYPE_BREAK:
	case BTYPE_OTHER:
		INT(type, nest_chk, p, rc);
		goto EXIT;
	case BTYPE_IRET:
		/* If the 'fixup_exception' function executed, then iret address
		 * was changed.
		 * So, in this case, we will not check the return address.
		 */
		ret_maybe_fixup = r2n == r2n_to; /*iret from kernel to kernel?*/
		IRET(type, nest_chk, p, ret_maybe_fixup, rc);
		goto EXIT;
	}
	rc = 0;
EXIT:
	if (rc == 3) /* leave? */
		*nest = st_ret_chk.nest;
	if (p_r2n_from)
		*p_r2n_from = r2n;
	if (p_r2n_to)
		*p_r2n_to = r2n_to;
	if (pp_from)
		*pp_from = p_from;
	if (p_idx_from)
		*p_idx_from = idx_from;
	return rc;
}

