/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_utils.c - utilities                                                   */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2008                         */
/*             Authors: Yumiko Sugita (yumiko.sugita.yf@hitachi.com),        */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <sys/utsname.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <libiberty.h>
#include "bt_utils.h"

/*----------------------------------------------------------------------------*/
/*  utility functions                                                         */
/*----------------------------------------------------------------------------*/
int u_open(const char *path, off_t *size)
{
	int fd, rc;
	struct stat st;

	if ((fd = open(path, O_RDONLY)) < 0) {
		fprintf(stderr, "can't open %s.(%s)\n", path, strerror(errno));
		return fd;
	}
	if ((rc = lstat(path, &st)) < 0) {
		fprintf(stderr, "%s lstat failed.(%s)\n", path,strerror(errno));
		return rc;
	}
	*size = st.st_size;
	return fd;
}

int u_close(int fd)
{
	int rc;

	if (fd < 0)
		return 0;
	if ((rc = close(fd)) < 0)
		fprintf(stderr, "close failed.(%s)\n", strerror(errno));
	return rc;
}

off_t u_lseek(int fd, off_t offset, int whence)
{
	off_t rc;

	if ((rc = lseek(fd, offset, whence)) < 0)
		fprintf(stderr, "lseek failed.(%s)\n", strerror(errno));
	return rc;
}

ssize_t u_read(int fd, void *buf, size_t count)
{
	ssize_t rc = 0;
	size_t tmp;

	for (tmp = count; tmp; tmp -= rc) {
		rc = read(fd, buf, tmp);
		if (rc < 0) {
			if (errno == EINTR)
				continue;
			fprintf(stderr, "read failed.(%s)\n", strerror(errno));
			return rc;
		}
	}
	return count;
}

ssize_t u_write(int fd, const void *buf, size_t count)
{
	ssize_t rc = 0;
	size_t tmp;

	for (tmp = count; tmp; tmp -= rc) {
		rc = write(fd, buf, tmp);
		if (rc < 0) {
			if (errno == EINTR)
				continue;
			fprintf(stderr, "write failed.(%s)\n", strerror(errno));
			return rc;
		}
	}
	return count;
}

static off_t rec_align_size;

static int gcd(int a, int b)
{
	if (a == 0 || b == 0)
		return 0;
	while (a != b) {
		if (a > b)
			a = a - b;
		else
			b = b - a;
	}
	return a;
}

static int lcm(int a, int b)
{
	if (a == 0 || b == 0)
		return 0;
	return (a / gcd(a, b)) * b;	// lcm = a * b / gcd(a, b)
}

static off_t get_rec_align_offset(off_t offset, int get_next_top)
{
	if (!rec_align_size)
		rec_align_size = lcm(getpagesize(), sizeof(struct bt_record));

	offset = offset / rec_align_size * rec_align_size;
	if (get_next_top)
		offset += rec_align_size;
	return offset;
}

#ifdef MMAP_ALIGN_TEST
struct align_test_data {
	off_t	offset;
	off_t	expect1;
	off_t	expect2;
};

static int __test_get_rec_align_offset(struct align_test_data* data)
{
	off_t r;

	r = get_rec_align_offset(data->offset, 0);
	//printf("T:%lld,%d=>%lld\n", data->offset, 0, r);
	if (r != data->expect1)
		return -1;
	r = get_rec_align_offset(data->offset, 1);
	//printf("T:%lld,%d=>%lld\n", data->offset, 1, r);
	if (r != data->expect2)
		return -1;
	return 0;
}

static int test_get_rec_align_offset()
{
	int i;
	struct align_test_data data[] = {
		{0,	0,	12288},
		{1,	0,	12288},
		{12287,	0,	12288},
		{12288,	12288,	24576},
		{12289,	12288,	24576},
	};

	for (i = 0; i < ARRAY_SIZE(data); i++) {
		if (__test_get_rec_align_offset(&data[i]) < 0)
			return -1;
	}
	return 0;
}
#endif

static off_t find_mmappable_size(int fd, off_t size)
{
	size_t size2;
	void *p;

#ifdef MMAP_ALIGN_TEST
	printf("TEST_RESULT:%d\n", test_get_rec_align_offset());
#endif
	for (;;) {
		size2 = size;
#ifdef MMAP_ALIGN_TEST
		if (size == size2 && size <= 12288)
#else
		if (size == size2)
#endif
			break;
		size = get_rec_align_offset((size / 2) - 1, 1);
	}
	for (;;) {
		p = mmap(NULL, size, PROT_READ, MAP_PRIVATE, fd, 0);
		if (p != MAP_FAILED) {
			munmap(p, size);
			break;
		}
		size = get_rec_align_offset((size / 2) - 1, 1);
	}
	return size;
}

/* For the mmap call, offset value is necessary to be page align.
 * Prefix 'poff_' of the local variables mean page-aligned value.
 */
int for_each_block_record(int fd, off_t i_rec_from, off_t i_rec_to,
			  t_func_each_bt_record f, void *dt)
{
	int rc, rec_size = sizeof(struct bt_record);
	off_t poff_from, poff_to, mappable_size, from_diff, size;
	off_t map_size, offset, i_rec;
	char *p_map, *p, *p_max;

	if (i_rec_from == i_rec_to)
		return 0;
	/* First, we find the mmap-able size */
	poff_from = get_rec_align_offset(i_rec_from * rec_size, 0);
	poff_to = get_rec_align_offset(i_rec_to * rec_size, 1);
	mappable_size = find_mmappable_size(fd, poff_to - poff_from);

	from_diff = (i_rec_from * rec_size) - poff_from;
	size = (i_rec_to - i_rec_from) * rec_size;

	/* for DEBUG
	printf("DBG:(i_from=%lld i_to=%lld)\n"
	       "    (poff_from=%lld poff_to=%lld mappable_size=%lld)\n"
	       "    (from_diff=%lld rsize=%lld)\n",
	       i_rec_from, i_rec_to,
	       poff_from, poff_to, mappable_size, from_diff, size);
	       */

	rc = 0;
	i_rec = i_rec_from;
	for (offset = poff_from; size; offset += map_size) {
		if (from_diff + size > mappable_size)
			map_size = mappable_size;
		else
			map_size = from_diff + size;
		size -= map_size - from_diff;
		p_map = mmap(NULL, map_size, PROT_READ, MAP_PRIVATE, fd,offset);
		/* for DEBUG
		printf("MMAP:from_diff=%lld map_size=%lld offset=%lld\n",
		       from_diff, map_size, offset);
		*/
		if (p_map == MAP_FAILED) {
			fprintf(stderr, "mmap failed.(%s)\n", strerror(errno));
			return -1;
		}
		p = p_map;
		p_max = p + map_size;
		if (from_diff) {
			p += from_diff;
			from_diff = 0;
		}
		for (; p < p_max; p += rec_size, i_rec++) {
			rc = f((struct bt_record*)p, i_rec, dt);
			if (rc) {
				munmap(p_map, map_size);
				return rc;
			}
		}
		munmap(p_map, map_size);
	}
	return rc;
}

int for_each_bt_record(int fd, off_t size, t_func_each_bt_record f, void *dt)
{
	return for_each_block_record(fd, 0, size / sizeof(struct bt_record),
				     f, dt);
}

int proc_cpu_files(char *dir_path, proc_cpu_files_t handler, void *dt)
{
	DIR *dir;
	struct dirent* d;
	char path[PATH_MAX];
	struct stat st;
	int rc, cpu;

	if ((dir = opendir(dir_path)) == NULL) {
		fprintf(stderr, "can't open %s\n", dir_path);
		return -1;
	}
	rc = 0;
	while ((d = readdir(dir)) != NULL) {
		if (strcmp(".", d->d_name) == 0 || strcmp("..", d->d_name) == 0)
			continue;
		snprintf(path, PATH_MAX, "%s/%s", dir_path, d->d_name);
		if (lstat(path, &st) < 0) {
			fprintf(stderr, "%s lstat failed.(%s)\n", path,
				strerror(errno));
			rc = -1;
			break;
		}
		if (S_ISDIR(st.st_mode) ||
		    sscanf(d->d_name, "cpu%d", &cpu) != 1)
			continue;
		rc = handler(cpu, path, st.st_size, dt);
		if (rc < 0)
			break;
	}
	closedir(dir);
	return rc;
}

static int __count_cpu_files(int cpu, char *cpu_path, size_t size, void *dt)
{
	*(int*)dt += 1;
	return 0;
}

int count_cpu_files(char *dir_path)
{
	int count = 0;

	if (proc_cpu_files(dir_path, __count_cpu_files, &count) < 0)
		return -1;
	return count;
}

/*----------------------------------------------------------------------------
 *  check the log file's split point
 *----------------------------------------------------------------------------
 */
static int nr_cpu_files;
static struct log_file_info *log_file_info;

static struct pid_log_info **pid_log_info;
static size_t pid_max = 1024;

static int __dump_pid_pos(void *__dt, void *user_data)
{
	struct __pid_info *__p = (struct __pid_info*)__dt;

	printf("   %s, clocks:%lld (%lld:%lld)\n",
	       __p->finfo->fpath, __p->clocks, __p->i_rec, __p->n_rec);
	return 0;
}

void dump_pid_pos(void)
{
	size_t i;
	struct pid_log_info *p;

	if (!pid_log_info)
		return;
	for (i = 0; i < pid_max; i++) {
		p = pid_log_info[i];
		if (!p)
			break;
		printf("pid:%d (%s) ------>\n", p->pid, p->comm);
		for_each_node(p->info, __dump_pid_pos, NULL);
	}
}

void print_pid_info(void)
{
	size_t i;
	struct pid_log_info *p;

	if (!pid_log_info)
		return;
	for (i = 0; i < pid_max; i++) {
		p = pid_log_info[i];
		if (!p)
			break;
		printf("%d %s\n", p->pid, p->comm);
	}
}

static void __free_pid_info(void *__dt)
{
	struct __pid_info *__p = (struct __pid_info*)__dt;

	free(__p);
}

static void free_pid_info(void)
{
	size_t i;
	struct pid_log_info *p;

	if (!pid_log_info)
		return;
	for (i = 0; i < pid_max; i++) {
		p = pid_log_info[i];
		if (!p)
			break;
		free_tree(p->info, __free_pid_info);
		free(p);
		pid_log_info[i] = NULL;
	}
	free(pid_log_info);
	pid_log_info = NULL;
}

static void cpu_file_close(void)
{
	int i;
	struct log_file_info *log_info;

	for (i = 0; i < nr_cpu_files; i++) {
		log_info = &log_file_info[i];
		if (log_info->fpath) {
			free(log_info->fpath);
			log_info->fpath = NULL;
		}
		if (log_info->fd) {
			close(log_info->fd);
			log_info->fd = 0;
		}
	}
	free(log_file_info);
	log_file_info = NULL;
}

int initialize_log_info(char *files[])
{
	if (!files)
		return -1;
	pid_log_info = xcalloc(pid_max, sizeof(struct pid_log_info*));

	for (nr_cpu_files = 0; files[nr_cpu_files]; nr_cpu_files++);
	if (nr_cpu_files == 0)
		return -1;
	log_file_info = xcalloc(nr_cpu_files, sizeof(struct log_file_info));
	return 0;
}

void finalize_log_info(void)
{
	free_pid_info();
	cpu_file_close();
}

static struct pid_log_info* find_pid_info_index(pid_t pid)
{
	size_t i, step = 1024;
	struct pid_log_info *p;

	for (i = 0; i < pid_max; i++) {
		p = pid_log_info[i];
		if (!p)
			goto ALLOC;
		if (p->pid == pid)
			return p;
	}
	/* enlarge pid_log_info area */
	pid_log_info = xrealloc(pid_log_info,
				(pid_max + step) *sizeof(struct pid_log_info*));
	for (; i < pid_max + step; i++)
		pid_log_info[i] = NULL;
	i = pid_max;
	pid_max += step;
ALLOC:
	p = xmalloc(sizeof(struct pid_log_info));
	p->pid = pid;
	p->comm[0] = '\0';
	p->info = NULL;
	pid_log_info[i] = p;
	return p;
}

static inline int compare_timestamp(unsigned long long *t1,
				    unsigned long long *t2)
{
	if (*t1 < *t2)
		return 1;
	else if (*t1 > *t2)
		return -1;
	return 0;
}

static int f_compare_timestamp(void *target, void *each)
{
	unsigned long long *t1 = &((struct __pid_info*)target)->clocks;
	unsigned long long *t2 = &((struct __pid_info*)each)->clocks;

	return compare_timestamp(t2, t1);
}

static struct __pid_info* add_cpu_pid_info(struct log_file_info *finfo,
					   struct pid_record *rec, off_t i_rec)
{
	struct pid_log_info *p;
	struct __pid_info *__p;

	/* get pid-info */
	p = find_pid_info_index(rec->pid);
	if (!p)
		return NULL;

	/* setup pid-sub-info for add*/
	__p = xmalloc(sizeof(struct __pid_info));
	__p->finfo = finfo;
	__p->clocks = rec->clocks & ~BT_FLAG_PID_LL;
	__p->i_rec = i_rec;
	__p->n_rec = 0;

	/* add pid-sub-info */
	p->info = insert_tree(__p, p->info, f_compare_timestamp);
	if (!p->info)
		return NULL;
	return __p;
}

static int set_pid_comm(struct pid_record *pid_rec, char *comm)
{
	struct pid_log_info *p;
	unsigned long long clocks;
	int rc;

	p = find_pid_info_index(pid_rec->pid);
	if (!p)
		return -1;
	clocks = pid_rec->clocks & ~BT_FLAG_PID_LL;
	rc = compare_timestamp(&p->comm_clocks, &clocks);
	if (rc > 0) {
		p->comm_clocks = clocks;
		p->comm[BT_COMM_LEN] = '\0';
		memcpy(p->comm, comm, BT_COMM_LEN);
	}
	return 0;
}

/* data for chk_pid_pos_per_cpu */
struct chk_pid_args {
	int is_search;
	unsigned long search_addr;
	struct log_file_info *finfo;
	struct pid_record last_rec;
	struct __pid_info *last_info;
};

static int chk_pid_pos_per_cpu(struct bt_record *p, off_t i_rec, void *dt)
{
	struct chk_pid_args *args = (struct chk_pid_args*)dt;
	struct warn_record *warn;
	struct __pid_info *__p;

	if (is_pid_record(p)) {
		__p = args->last_info;
		if (__p)
			__p->n_rec = i_rec - __p->i_rec;
		args->last_rec = *(struct pid_record*)p;
		args->last_info = add_cpu_pid_info(args->finfo,
						   (struct pid_record*)p,
						   i_rec);
		if (!args->last_info)
			return -1;
		return 0;
	}
	if (is_comm_record(p) && args->last_rec.clocks) {
		if (set_pid_comm(&args->last_rec,
				 ((struct comm_record*)p)->comm) < 0)
			return - 1;
		return 0;
	}
	if (is_warn_record(p)) {
		warn = (struct warn_record*)p;
		printf("WARN(%s, rec:%lld): bts left only: %ld\n",
		       args->finfo->fpath, i_rec, warn->left);
		return 0;
	}
	if (!args->is_search)
		return 0;
	/* Don't check p->from cause Pentium-M's p->from is not correct. */
	if (!args->last_rec.clocks || p->to != args->search_addr)
		return 0;
	printf("%20lld %12lld %s %6ld 0x%08lx\n",
	       args->last_rec.clocks & ~BT_FLAG_PID_LL, i_rec,
	       args->finfo->fpath, args->last_rec.pid, args->search_addr);
	return 0;
}

int chk_pid_pos(char *files[], int is_search, unsigned long search_addr)
{
	char *f;
	int i, rc;
	struct stat st;
	struct chk_pid_args args;
	struct log_file_info *log_info;
	struct __pid_info *__p;

	if (!files)
		return -1;
	args.is_search = is_search;
	args.search_addr = search_addr;

	for (i = 0; (f = files[i]); i++) {
		if (lstat(f, &st) < 0) {
			fprintf(stderr, "%s lstat failed.(%s)\n", f,
				strerror(errno));
			return -1;
		}
		log_info = &log_file_info[i];
		log_info->fpath = strdup(f);
		if ((log_info->fd = open(f, O_RDONLY)) < 0) {
			fprintf(stderr, "%s open failed.(%s)\n", f,
				strerror(errno));
			return -1;
		}
		if (!st.st_size) {
			log_info->size = 0;
			continue;
		}
		log_info->size = st.st_size;
		args.finfo = log_info;
		memset(&args.last_rec, 0, sizeof(struct pid_record));
		args.last_info = NULL;
		rc = for_each_bt_record(log_info->fd, log_info->size,
					chk_pid_pos_per_cpu, &args);
		if (rc < 0)
			return -1;
		__p = args.last_info;
		if (__p)
			__p->n_rec = st.st_size / sizeof(struct bt_record)
								- __p->i_rec;
	}
	return 0;
}

int for_each_pid_log_info(proc_pid_log_info_t func, void *data)
{
	int rc;
	size_t i;
	struct pid_log_info *p;

	for (i = 0; i < pid_max; i++) {
		p = pid_log_info[i];
		if (!p)
			break;
		rc = func(p, data);
		if (rc < 0)
			return rc;
	}
	return 0;
}

inline struct log_file_info *get_log_file_info(struct __pid_info *p)
{
	if (!p)
		return NULL;
	return p->finfo;
}

